#:::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
# This file is part of G-language Genome Analysis Environment package
#
#     Copyright (C) 2001-2016 Keio University
#:::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
# 
# G-language GAE is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2 of the License, or (at your option) any later version.
# 
# G-language GAE is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
# See the GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public
# License along with G-language GAE -- see the file COPYING.
# If not, write to the Free Software Foundation, Inc.,
# 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
# 
#END_HEADER
#
# written by Kazuharu Arakawa <gaou@sfc.keio.ac.jp> at
# G-language Project, Institute for Advanced Biosciences, Keio University.
#

package G::Shell::Help;

use strict;
use base qw(Exporter);
use autouse 'String::Approx'=>qw(amatch);

use Carp;
use File::Find;
use G::DB::SDB;
use G::Messenger;

our @EXPORT = qw(
		 help
		 buildHelpDB
		 );

#:::::::::::::::::::::::::::::::::
#       Perldoc
#:::::::::::::::::::::::::::::::::


=head1 NAME

  G::Shell::Help - G-language Shell helper module for command references

=head1 DESCRIPTION

  This class is a part of G-language Genome Analysis Environment, 
  providing functions for "help" features of the G-language Shell.

=cut

#::::::::::::::::::::::::::::::
#          Variables
#::::::::::::::::::::::::::::::

my %pdoc;
my %gclass;

#::::::::::::::::::::::::::::::
#    Let the code begin...
#::::::::::::::::::::::::::::::


sub buildHelpDB{
    require G::IO;

    my $gv  = $G::IO::VERSION;

    if(sdb_exists("pdoc-g$gv") && sdb_exists("gclass-$gv")){
	%pdoc   = %{sdb_load("pdoc-g$gv")};
	%gclass = %{sdb_load("gclass-$gv")};

	return;
    }

    my $gPath    = qx/perldoc -l G::IO/;
    my $rcmdPath = qx/perldoc -l Rcmd::Clustering/;
    
    $gPath    =~ s/\/IO.pm//;
    $rcmdPath =~ s/\/Clustering.pm//;
    
    chomp($gPath);
    chomp($rcmdPath);

    find({"wanted"=>\&process_g,  'follow'=>1}, $gPath . '.pm');
    find({"wanted"=>\&process_g,  'follow'=>1}, $gPath);
    find({"wanted"=>\&process_g,  'follow'=>1}, $rcmdPath . '.pm');
    find({"wanted"=>\&process_g,  'follow'=>1}, $rcmdPath);

    sdb_save(\%pdoc,   "pdoc-g$gv");
    sdb_save(\%gclass, "gclass-$gv");
}


=head2 help

  Name: help   -   G-language Shell help command

  Description:
    G-language Shell help command. This command seaches through
    G-language GAE manual to show the corresponding entries for a given keyword.

    You can also search through the documentation with options (See below).

  Usage:
    help <options> <keyword>

    For <keyword>, you can specify G-language GAE function name
        eg. help gcskew

  Options:
   -s         search through documentation with keyword

  Author: 
    Kazuharu Arakawa (gaou@sfc.keio.ac.jp)

  History:
   20120829-01 removed -g,  -w, and -bp option (search BioMOBY and BioPerl document)
   20080227-01 added -w option
   20080214-01 replaced print with msg_send, and also returns the string
   20070531-01 initial posting

=cut

sub help{
    buildHelpDB() unless(scalar %pdoc);

    my $key = shift || 'help';

    if ($key =~ /^\-(\S+)/){
	my $option = $1;
	
	$key = join(' ', @_);
	my $lckey = lc($key);
	my $ans = "     Keyword \"$key\" was not found in the documentations.\n";

	my @tmp;

	foreach my $keyword (sort keys %pdoc){
	    if(lc($pdoc{$keyword}) =~ /$lckey/){
		push(@tmp, $keyword);
	    }elsif(length($lckey) < 1){
		push(@tmp, $keyword);
	    }
	    
	    if(scalar(@tmp)){
		$ans = sprintf("     Found keyword \"%s\" in the following manual pages \(%s hits\).\n\n", $key, scalar(@tmp));
		
		foreach my $entry (@tmp){
		    my $tmp = $pdoc{$entry};
		    
		    my $abstract;
		    if($tmp =~ /Name:\s*(.*)(   -   .*?)\_\_\%\%\%\%\%\_\_/){
			$abstract = $2;
		    }
		    
		    if($entry){
			$ans .= sprintf("       %-32s%s\n",$entry, $abstract);
		    }
		}
	    }
	}    
        msg_send($ans);

	return $ans;
    }else{

        my $ans  = $pdoc{$key};
	my $ans2 = $pdoc{'$gb->' . $key . '()'};

        if(length $ans){
            $ans =~ s/\_\_\%\%\%\%\%\_\_/\n/g;

            unless($ans =~ /Name: (.*)   -   (.*)\n/){
                $ans = "\n  Name: $key\n". $ans;
            }
	}elsif(length $ans2){
	    $ans = $ans2;
            $ans =~ s/\_\_\%\%\%\%\%\_\_/\n/g;

            unless($ans =~ /Name: (.*)   -   (.*)\n/){
                $ans = "\n  Name: $key\n". $ans;
            }
        }else{
            $ans = "     Keyword: \"$key\" not found in manual.\n\n";
            my $lckey = lc($key);
            my @tmp;;
            
            foreach my $keyword (sort keys %pdoc){
                if(lc($keyword) =~ /$lckey/){
                    push(@tmp, $keyword);
		    if(lc($keyword) eq $lckey){
			$ans = $pdoc{$keyword};
			$ans =~ s/\_\_\%\%\%\%\%\_\_/\n/g;
			unless($ans =~ /Name: (.*)   -   (.*)\n/){
			    $ans = "\n  Name: $key\n". $ans;
			}
			msg_send($ans);
			return $ans;
		    }
                }
            }
            
            unless(scalar(@tmp)){
                foreach my $keyword (sort keys %pdoc){
                    if(amatch($lckey, ['i 1'], lc($keyword))){
                        push(@tmp, $keyword);
                    }
                }
            }

            if(scalar(@tmp)){
                $ans .= "     Did you mean:\n";
                
                foreach my $entry (@tmp){
                    $ans .= "              " . $entry . "\n";
                }
            }

            $ans .= "\n\n      use \"help -s\" if you want to search, or \"help help\" if desperate.\n";
	}

	msg_send($ans);
	return $ans;
    }
}

sub process_g{
    next unless(/\.pm$/);

    my $class = 'G';

    if ($File::Find::name =~ /\/((?:G|SubOpt|Rcmd).*)\.pm$/){
	$class = $1;
	$class =~ s/\//::/g;
    }else{
	next;
    }

    open(FILE, $File::Find::name) || die($File::Find::name, " ", $!);
    while(<FILE>){
	chomp;
	if(/^\s*\=head2\s+(.*)\s*$/){
	    my $subname = $1;
	    $gclass{$subname} = $class;

	    while(<FILE>){
		chomp;
		last if (/\=cut/);
		$pdoc{$subname} .= $_ . '__%%%%%__';
	    }
	}
    }
    close(FILE);
}


sub process_bp{
#this process is deprecated.
    my $class = 'Bio';

    if ($File::Find::name =~ /\/(Bio.*)\.pm$/){
	$class = $1;
	$class =~ s/\//::/g;
    }else{
	next;
    }

    open(FILE, $File::Find::name) || die($!);
    while(<FILE>){
	chomp;
	if(/^\s*\=head2\s+(\S*)\s*/){
	    my $subname = $class . '::' . $1;
	    next if ($subname =~ /Mailing Lists/ || $subname =~ /Reporting Bugs/);

	    while(<FILE>){
		chomp;
		last if (/\=cut/);
		$pdoc{$subname} .= $_ . '__%%%%%__';
	    }
	}
    }
    close(FILE);
}

1;
