#include "stdafx.h"
#include "KstString.h"

#include <tchar.h>

#include "KstVec3.h"
#include "KstMat4.h"

#include "KstFontMap.h"
#include "KstDefaultFont.h"

#include "CharCodeConvert.h"


namespace gk
{


static const int CHAR_PADDING = 2;

static const int CHAR_HALF_SIZE = 16 + CHAR_PADDING;
static const int CHAR_SIZE      = 32 + CHAR_PADDING;



KstString::KstString(void)
{
	m_Width = 0.0f;
	m_Height = 0.0f;
}

KstString::KstString(const std::string& str)
{
	CreateStroke( str , *this );
}

KstString::KstString(const char* str)
{
	CreateStroke( str , *this );
}

KstString::KstString(const std::string& str, const KstFontMap& font)
{
	CreateStroke( str , font , *this );
}

KstString::KstString(const char* str, const KstFontMap& font)
{
	CreateStroke( str , font , *this );
}



void KstString::Clear(void)
{
	m_Str.clear();
	m_Width = 0.0f;
	m_Height = 0.0f;
}


void KstString::AddStroke(KstStrokeChar& c)
{
	return m_Str.push_back(c);
}


size_t KstString::GetStrokeCount(void) const
{
	return m_Str.size();
}

KstStrokeChar& KstString::GetStroke(size_t i)
{
	return m_Str.at(i);
}

const KstStrokeChar& KstString::GetStroke(size_t i) const
{
	return m_Str.at(i);
}


float KstString::GetWidth(void) const
{
	return m_Width;
}

float KstString::GetHeight(void) const
{
	return m_Height;
}


void KstString::SetStr(const std::string& s)
{
	CreateStroke(s, *this);
}

void KstString::SetStr(const char* s)
{
	CreateStroke(s, *this);
}

void KstString::SetStr(const std::string& s, const KstFontMap& font)
{
	CreateStroke(s, font, *this);
}

//! ̃Xg[N𐶐
void KstString::SetStr(const char* s, const KstFontMap& font)
{
	CreateStroke(s, font, *this);
}


void KstString::CreateStroke(const std::string& s, KstString& stroke)
{
	const KstFontMap& font = KstDefaultFont::GetFont();
	return CreateStroke( s , font , stroke );
}

void KstString::CreateStroke(const char* s, KstString& stroke)
{
	const KstFontMap& font = KstDefaultFont::GetFont();
	return CreateStroke( s , font , stroke );
}

void KstString::CreateStroke(const std::string& s, const KstFontMap& font, KstString& stroke)
{
	return CreateStroke( s.c_str() , font , stroke );
}

void KstString::CreateStroke(const char* s, const KstFontMap& font, KstString& stroke)
{
	stroke.Clear();

	float LastLineLen = 0.0f;

	size_t StrSize = strlen(s);
	for( size_t i = 0 ; i < StrSize ;  )
	{
		// multibyte char
		if( _ismbblead( s[i] ) )
		{
			std::string js = SJIStoJISc( s[i] , s[i+1] );

			unsigned int c0 = (unsigned int)(unsigned char)js[0];
			unsigned int c1 = (unsigned int)(unsigned char)js[1];

			unsigned int code = c0 * 0x100 + c1;

			KstStrokeChar c;
			c.Create(code, true, font);

			stroke.AddStroke(c);

			if( stroke.GetStrokeCount() == 1 )
				stroke.m_Height = (float)CHAR_SIZE;

			LastLineLen += CHAR_SIZE;
			stroke.m_Width = (std::max)( stroke.m_Width , LastLineLen );

			i += 2;
			continue;
		}

		// check newline
		if( s[i] == '\r' || s[i] == '\n' )
		{
			KstStrokeChar c;
			c.CreateNewline();
			stroke.AddStroke(c);

			if( stroke.GetStrokeCount() == 1 )
				stroke.m_Height = (float)CHAR_SIZE;

			LastLineLen = 0.0f;
			stroke.m_Height += CHAR_SIZE;

			if( (i+1) < StrSize )
			{
				if( s[i+1] == '\r' || s[i+1] == '\n' )
					i += 1;
			}

			i += 1;
			continue;
		}

		// check tab( create as two spaces )
		if( s[i] == '\t' )
		{
			unsigned int code = (unsigned int)(unsigned char)' ';

			KstStrokeChar c;
			c.Create(code, false, font);

			stroke.AddStroke(c);

			if( stroke.GetStrokeCount() == 1 )
				stroke.m_Height += CHAR_SIZE;

			stroke.AddStroke(c);

			LastLineLen += CHAR_HALF_SIZE * 2;
			stroke.m_Width = (std::max)( stroke.m_Width , LastLineLen );

			i += 1;
			continue;
		}

		// singlebyte char
		{
			unsigned int code = (unsigned int)(unsigned char)s[i];

			KstStrokeChar c;
			c.Create(code, false, font);

			stroke.AddStroke(c);

			if( stroke.GetStrokeCount() == 1 )
				stroke.m_Height += CHAR_SIZE;

			LastLineLen += CHAR_HALF_SIZE;
			stroke.m_Width = (std::max)( stroke.m_Width , LastLineLen );

			i += 1;
			continue;
		}
	}
}



void KstString::Draw(void) const
{
	Draw( Vec3() , Vec3() , 1.0f );
}

void KstString::Draw(const Vec3& position) const
{
	Draw( position , Vec3() , 1.0f );
}

void KstString::Draw(const Vec3& position, const Vec3& euler_rot) const
{
	Draw( position , euler_rot , 1.0f );
}

void KstString::Draw(const Vec3& position, const Vec3& euler_rot, float scale) const
{
	glPushMatrix();

	SetTransform(position, euler_rot, scale);

	float y_offset = GetHeight() - CHAR_SIZE;
	glTranslatef( (GLfloat)CHAR_PADDING , y_offset , 0.0f );

	DrawStringMain();

	glPopMatrix();
}

void KstString::Draw(const Mat4& transform) const
{
	glPushMatrix();

	SetTransform(transform);

	float y_offset = GetHeight() - CHAR_SIZE;
	glTranslatef( (GLfloat)CHAR_PADDING , y_offset , 0.0f );

	DrawStringMain();

	glPopMatrix();
}

void KstString::DrawStringMain(void) const
{
	glPushMatrix();

	for( size_t i = 0 ; i < m_Str.size() ; ++i )
	{
		const KstStrokeChar& c = m_Str[i];

		if( c.IsNewline() )
		{
			glPopMatrix();
			glTranslated(0, -(double)CHAR_SIZE, 0);
			glPushMatrix();
		}
		else if( c.IsSinglebyteChar() )
		{
			DrawChar( c );

			glTranslated((double)CHAR_HALF_SIZE, 0, 0);
		}
		else if( c.IsMultibyteChar() )
		{
			DrawChar( c );

			glTranslated((double)CHAR_SIZE, 0, 0);
		}
	}

	glPopMatrix();
}


void KstString::DrawBound(void) const
{
	DrawBound( Vec3() , Vec3() , 1.0f );
}

void KstString::DrawBound(const Vec3& position) const
{
	DrawBound( position , Vec3() , 1.0f );
}

void KstString::DrawBound(const Vec3& position, const Vec3& euler_rot) const
{
	DrawBound( position , euler_rot , 1.0f );
}

void KstString::DrawBound(const Vec3& position, const Vec3& euler_rot, float scale) const
{
	glPushMatrix();

	SetTransform(position, euler_rot, scale);

	DrawBoundMain();

	glPopMatrix();
}

void KstString::DrawBound(const Mat4& transform) const
{
	glPushMatrix();

	SetTransform(transform);

	DrawBoundMain();

	glPopMatrix();
}

void KstString::DrawBoundMain(void) const
{
	float w = GetWidth();
	float h = GetHeight();

	glBegin(GL_LINE_LOOP);
	glVertex3f( 0.0f , 0.0f , 0.0f );
	glVertex3f( w    , 0.0f , 0.0f );
	glVertex3f( w    , h    , 0.0f );
	glVertex3f( 0.0f , h    , 0.0f );
	glEnd();
}


// XP[ -> ] -> ړ̏ɕϊ
void KstString::SetTransform(const Vec3& position, const Vec3& euler_rot, float scale) const
{
	glTranslatef( position.x , position.y , position.z );

	glRotatef( euler_rot.z , 0.0f , 0.0f , 1.0f );
	glRotatef( euler_rot.y , 0.0f , 1.0f , 0.0f );
	glRotatef( euler_rot.x , 1.0f , 0.0f , 0.0f );

	glScalef( scale , scale , scale );
}

void KstString::SetTransform(const Mat4& transform) const
{
	glMultMatrixf(transform.ary());
}


void KstString::DrawChar(const KstStrokeChar& stroke) const
{
	glBegin(GL_LINES);
	for( size_t j = 0 ; j < stroke.GetNumSegments() ; ++j )
	{
		const Point2i& p0 = stroke.GetSegment(j).p0;
		const Point2i& p1 = stroke.GetSegment(j).p1;
		glVertex3i( p0.x , p0.y , 0 );
		glVertex3i( p1.x , p1.y , 0 );
	}
	glEnd();
}


}
