#pragma once

#include <math.h>
#include <assert.h>


namespace gk
{


class Vec3
{
public:
	float x , y , z;

public:
	Vec3(void)                         : x(0.0f), y(0.0f), z(0.0f) {}
	Vec3(float _x, float _y, float _z) : x(_x), y(_y), z(_z) {}
	Vec3(const Vec3& _v)               : x(_v.x), y(_v.y), z(_v.z) {}
	Vec3(const float* _v)              : x(_v[0]), y(_v[1]), z(_v[2]) {}

	float*       get_ary(void)       { return &x; }
	const float* get_ary(void) const { return &x; }

	float&       operator[]( size_t idx )       { return get_ary()[ idx ]; }
	const float& operator[]( size_t idx ) const { return get_ary()[ idx ]; }

	float&       at( size_t idx )       { assert(idx < 3); return get_ary()[ idx ]; }
	const float& at( size_t idx ) const { assert(idx < 3); return get_ary()[ idx ]; }

	void set( const Vec3& src );
	void set( const float* _v );
	void set( const float& _x, const float& _y, const float& _z );
	Vec3& operator=( const Vec3& v );

	bool operator==( const Vec3& v ) const;
	bool operator!=( const Vec3& v ) const;

	Vec3& operator+=( const Vec3& _v );
	Vec3& operator-=( const Vec3& _v );
	Vec3& operator*=( const float& val );
	Vec3& operator/=( const float& val );

	Vec3 operator-(void) const;

	void normalize(void);
	Vec3 get_normalize(void) const;

	float length(void) const;
	float square_length(void) const;

	void rotate_x(const float& angle);
	void rotate_y(const float& angle);
	void rotate_z(const float& angle);
	void rotate(const float& angle, const Vec3& axis);

	Vec3 get_rotate_x( const float& angle ) const;
	Vec3 get_rotate_y( const float& angle ) const;
	Vec3 get_rotate_z( const float& angle ) const;
	Vec3 get_rotate( const float& angle, const Vec3& axis ) const;

	friend float dot(const Vec3& _a, const Vec3& _b);
	friend Vec3 operator+(const Vec3& _a, const Vec3& _b);
	friend Vec3 operator-(const Vec3& _a, const Vec3& _b);
	friend Vec3 operator*(const Vec3& _v, const float& _t);
	friend Vec3 operator*(const float& _t, const Vec3& _v);
	friend Vec3 operator/(const Vec3& _v, const float& _t);
};



inline float dot(const Vec3& _a, const Vec3& _b)
{
	return _a.x * _b.x + _a.y * _b.y + _a.z * _b.z ;
}

inline Vec3 cross(const Vec3& _a, const Vec3& _b)
{
	Vec3 ret;
	ret.x = _a.y * _b.z - _a.z * _b.y ;
	ret.y = _a.z * _b.x - _a.x * _b.z ;
	ret.z = _a.x * _b.y - _a.y * _b.x ;
	return ret;
}

inline float scholar_triple(const Vec3& _a, const Vec3& _b, const Vec3& _c)
{
	return dot( cross( _a , _b ) , _c );
}

inline void Vec3::set( const Vec3& src )
{
	set( src.x , src.y , src.z );
}
inline void Vec3::set( const float* _v )
{
	set( _v[0] , _v[1] , _v[2] );
}
inline void Vec3::set( const float& _x, const float& _y, const float& _z )
{
	x = _x ;  y = _y ;  z = _z ;
}
inline Vec3& Vec3::operator=( const Vec3& v )
{
	set( v );
	return (*this);
}

inline bool Vec3::operator==( const Vec3& v ) const
{
	return ( x == v.x && y == v.y && z == v.z );
}
inline bool Vec3::operator!=( const Vec3& v ) const
{
	return !( (*this) == v );
}

inline Vec3& Vec3::operator+=(const Vec3& _v)
{
	x += _v.x ;  y += _v.y ;  z += _v.z ;
	return *this;
}

inline Vec3& Vec3::operator-=(const Vec3& _v)
{
	x -= _v.x ;  y -= _v.y ;  z -= _v.z ;
	return *this;
}

inline Vec3& Vec3::operator*=(const float& val)
{
	x *= val ;  y *= val ;  z *= val ;
	return *this;
}

inline Vec3& Vec3::operator/=(const float& val)
{
	x /= val ;  y /= val ;  z /= val ;
	return *this;
}

inline Vec3 Vec3::operator-(void) const
{
	return Vec3( -x , -y , -z );
}

inline void Vec3::normalize(void)
{
	float l = length();
	if( l <= 0.0f )
		return;
	(*this) /= l;
}

inline Vec3 Vec3::get_normalize(void) const
{
	Vec3 tmp = *this;
	tmp.normalize();
	return tmp;
}

inline float Vec3::length(void) const
{
	return sqrt( square_length() );
}

inline float Vec3::square_length(void) const
{
	return x * x + y * y + z * z ;
}


inline void Vec3::rotate_x(const float& angle)
{
	float c = cos(angle)  ,  s = sin(angle);
	float ty = y * c - z * s ;
	float tz = y * s + z * c ;
	y = ty ;   z = tz ;
}

inline void Vec3::rotate_y(const float& angle)
{
	float c = cos(angle)  ,  s = sin(angle);
	float tz = z * c - x * s ;
	float tx = z * s + x * c ;
	z = tz ;   x = tx ;
}

inline void Vec3::rotate_z(const float& angle)
{
	float c = cos(angle)  ,  s = sin(angle);
	float tx = x * c - y * s ;
	float ty = x * s + y * c ;
	x = tx ;   y = ty ;
}

inline void Vec3::rotate(const float& angle, const Vec3& axis)
{
	float c = cos(angle);  float s = sin(angle);
	float mc = float(1) - c ;

	const float& ax = axis.x ;  const float& ay = axis.y ;  const float& az = axis.z ;
	float tx = x ;  float ty = y ;  float tz = z ; 

	x = ( ax * ax * mc + c      ) * tx + ( ax * ay * mc - az * s ) * ty + ( ax * az * mc + ay * s ) * tz ;
	y = ( ay * ax * mc + az * s ) * tx + ( ay * ay * mc + c      ) * ty + ( ay * az * mc - ax * s ) * tz ;
	z = ( az * ax * mc - ay * s ) * tx + ( az * ay * mc + ax * s ) * ty + ( az * az * mc + c      ) * tz ;
}

inline Vec3 Vec3::get_rotate_x( const float& angle ) const
{
	Vec3 v = *this;
	v.rotate_x( angle );
	return v;
}

inline Vec3 Vec3::get_rotate_y( const float& angle ) const
{
	Vec3 v = *this;
	v.rotate_y( angle );
	return v;
}

inline Vec3 Vec3::get_rotate_z( const float& angle ) const
{
	Vec3 v = *this;
	v.rotate_z( angle );
	return v;
}

inline Vec3 Vec3::get_rotate( const float& angle, const Vec3& axis ) const
{
	Vec3 v = *this;
	v.rotate( angle , axis );
	return v;
}

inline Vec3 operator+(const Vec3& _a, const Vec3& _b)
{
	Vec3 v = _a ;  v += _b ;  return v ;
}

inline Vec3 operator-(const Vec3& _a, const Vec3& _b)
{
	Vec3 v = _a ;  v -= _b ;  return v ;
}

inline Vec3 operator*(const Vec3& _v, const float& _t)
{
	Vec3 v = _v ;  v *= _t ;  return v ;
}

inline Vec3 operator*(const float& _t, const Vec3& _v)
{
	Vec3 v = _v ;  v *= _t ;  return v ;
}

inline Vec3 operator/(const Vec3& _v, const float& _t)
{
	Vec3 v = _v ;  v /= _t ;  return v ;
}


}
