/**********************************************************************
 
	Copyright (C) 2005- Hirohisa MORI <joshua@nichibun.ac.jp>
 
	This program is free software; you can redistribute it 
	and/or modify it under the terms of the GLOBALBASE 
	Library General Public License (G-LGPL) as published by 

	http://www.globalbase.org/
 
	This program is distributed in the hope that it will be 
	useful, but WITHOUT ANY WARRANTY; without even the 
	implied warranty of MERCHANTABILITY or FITNESS FOR A 
	PARTICULAR PURPOSE.

**********************************************************************/

#include	"machine/include.h"
#include	"memory_debug.h"
#include	"pri_level.h"
#include	"lock_level.h"
#include	"utils.h"
#include	"task.h"
#include	"matrix.h"
#include	"xl.h"
#include	"xlerror.h"

MATRIX_SEXP	ms_root;
MATRIX_NODE *	nd_root;
int		nd_cnt;
int		total_ncache_size;
MATRIX *	matrises;
int mn_length;
SEM matrix_lock,matrix_gc_lock;
SYS_QUEUE matrix_que;
XLISP_ENV * matrix_xl_env;

MATRIX *_open_matrix();
void _close_matrix(MATRIX * m);

MATRIX_NODE * _get_matrix_node(int * errp,MATRIX * m,INTEGER64 * dim_code,int);

void _insert_matrix_sexp(MATRIX_SEXP * d);
int _set_matrix_channel_info(MATRIX * m,int ch,MATRIX_CHANNEL_INFO * inf);
int _set_matrix_param(MATRIX * m,MATRIX_PARAM * p);
int _set_matrix_dim_divide(MATRIX * m,int dim,int dim_divide);
int _set_matrix_block_size(MATRIX * m,int dim,int block_size);
int _set_matrix_pixel_size(MATRIX * m,int dim,int pixel_size);
int _set_matrix_cal(MATRIX * m,int id,XL_SEXP * cal);

void _insert_node_ring(MATRIX_NODE * n);
void _delete_node_ring(MATRIX_NODE * n);
void _get_matrix_access(MATRIX_NODE * n);
int  _cmp_dim_code(MATRIX * m,INTEGER64 * dim_code1,INTEGER64 * dim_code2);
INTEGER64 * _copy_dim_code(MATRIX * m,INTEGER64 * dim_code);
MATRIX_NODE * _get_matrix_tree(int * errp,MATRIX * m,INTEGER64 * dim_code,int access);
MATRIX_NODE * _get_matrix_tree_1(int * errp,MATRIX * m,INTEGER64 * dim_code,int access);
MATRIX_NODE * _get_matrix_tree_2(int * errp,MATRIX * m,INTEGER64 * dim_code,int access);
void matrix_exec_cal(MATRIX_TOKEN * t);

void _lock_node(MATRIX_NODE *,MATRIX_TOKEN * );
void _unlock_node(MATRIX_NODE *,MATRIX_TOKEN *);
void _insert_node_wait_token(MATRIX_NODE * ,MATRIX_TOKEN *);
XL_SEXP *  _get_channel_data(MATRIX_NODE * n,int ch,XL_SEXP * s);
MATRIX * _search_matrix(L_CHAR * neturl,L_CHAR * filename,L_CHAR * key);
int _set_channel_sexp(MATRIX_NODE * n,int ch,XL_SEXP *s);
void _matrix_node_channel_unlock(MATRIX_NODE * n);
void _matrix_node_channel_lock(MATRIX_NODE * n);
void _insert_matrix_access(MATRIX_TOKEN * t,void (*func)());
void _set_matrix_mode(MATRIX * m,int mode);
void _insert_matrix_open(MATRIX * m,void (*func)());
void _matrix_trigger_access(MATRIX_NODE * n,int access);
int _check_dim_code(MATRIX_NODE *,INTEGER64 *);
int _get_dim_code_index(MATRIX_NODE * n,INTEGER64 * dim_code);
void _isnert_dim_code_index(MATRIX_NODE *,INTEGER64 *);
void _set_dirty_file(MATRIX_NODE * n);
void _cal_matrix_total_levels(MATRIX * m);

void matrix_que_task();
int _free_matrix_node(MATRIX_NODE *);
void _delete_matrix_sexp(MATRIX_SEXP*);
void _gc_matrix();


void
init_matrix()
{
	ms_root.prev = ms_root.next = &ms_root;
	nd_root = d_alloc(sizeof(MATRIX_NODE_HEADER));
	nd_root->h.next = nd_root->h.prev = nd_root;
	total_ncache_size = MIN_NCACHE_SIZE;
	mn_length = 0;
	matrix_lock = new_lock(LL_MATRIX);
	matrix_gc_lock = new_lock(LL_MATRIX_GC);

	memset(&matrix_que,0,sizeof(SYS_QUEUE));
	matrix_que.flags = QF_PRI_FIFO;
	matrix_que.gc_func = 0;
	matrix_que.gc_get = 0;
	matrix_que.key_func = matrix_que_task;
	matrix_que.pri = PRI_FETCH;
	setup_queue(&matrix_que);

	matrix_xl_env = new_env(gblisp_top_env0);

	init_mxCH(matrix_xl_env);
	init_mxSet(matrix_xl_env);
	init_mxTrigger(matrix_xl_env);
}




void
matrix_que_task()
{
XL_INTERPRETER * xli;
L_CHAR* key;
MATRIX_TOKEN * t;

	key = touch_qkey(&matrix_que);
	if ( key == 0 )
		return;

	xli = new_xl_interpreter();
	xli->a_type = XLA_SELF;
	setup_i(xli);

	for ( ; ; ) {
		gc_push(0,0,"dm_loading_task");

		t = delete_queue(&matrix_que,sq_key_cond,key,0);
		if ( t == 0 ) {
			gc_pop(0,0);
			break;
		}
		
		(*t->func)(t);

		gc_pop(0,0);
	}

	close_self_interpreter();
	release_qkey(&matrix_que,key);
}


void
_insert_node_ring(MATRIX_NODE * n)
{
MATRIX_NODE * n1, * n2;
	n->h.prev = nd_root;
	n->h.next = nd_root->h.next;
	n->h.prev->h.next = n;
	n->h.next->h.prev = n;
	nd_cnt ++;
	n1 = nd_root->h.prev;
	for ( ; nd_cnt > total_ncache_size && n1 != nd_root ; ) {
		n2 = n->h.prev;
		if ( n1 != n )
			_free_matrix_node(n1);
		n1 = n2;
	}
}

void
_delete_node_ring(MATRIX_NODE * n)
{
	n->h.prev->h.next = n->h.next;
	n->h.next->h.prev = n->h.prev;
	nd_cnt --;
}

void
_insert_matrix_sexp(MATRIX_SEXP * d)
{
	lock_task(matrix_gc_lock);
	d->prev = &ms_root;
	d->next = ms_root.next;
	d->prev->next = d;
	d->next->prev = d;
	unlock_task(matrix_gc_lock,"insert_matrix_sexp");
}

void
_delete_matrix_sexp(MATRIX_SEXP * d)
{
	d->prev->next = d->next;
	d->next->prev = d->prev;
}



void
_gc_matrix()
{
MATRIX_SEXP * d, * dn;
	for ( d = ms_root.next ; d != &ms_root ; ) {
		dn = d->next;
		if ( d->del == 0 )
			gc_gb_sexp(d->data);
		else {
			_delete_matrix_sexp(d);
			d_f_ree(d);
		}
		d = dn;
	}
}

void
_matrix_node_channel_lock(MATRIX_NODE * n)
{
	for ( ; n->ch_lock ; ) {
		sleep_task((int)n,matrix_lock);
		lock_task(matrix_lock);
	}
	n->ch_lock = 1;
}

void
matrix_node_channel_lock(MATRIX_NODE * n)
{
	lock_task(matrix_lock);
	_matrix_node_channel_lock(n);
	unlock_task(matrix_lock,"matrix_node_channel_lock");
}



void
_matrix_node_channel_unlock(MATRIX_NODE * n)
{
	n->ch_lock = 0;
	wakeup_task((int)n);
	
}

void
matrix_node_channel_unlock(MATRIX_NODE * n)
{
	lock_task(matrix_lock);
	_matrix_node_channel_unlock(n);
	unlock_task(matrix_lock,"matrix_node_channel_lock");
}



void
_insert_matrix_open(MATRIX * m,void (*func)())
{
MATRIX_TOKEN * t;
	t = d_alloc(sizeof(*t));
	memset(t,0,sizeof(*t));
	t->h.key = nl_copy_str(std_cm,"_insert_matrix_open");
	t->h.pri = 1000;
	t->func = func;
	t->wait_matrix = m;
	insert_queue(&matrix_que,t,0);
}

MATRIX *
_open_matrix(L_CHAR * neturl,L_CHAR * filename,L_CHAR * key,void (*open_method)())
{
MATRIX * ret;
	ret = d_alloc(sizeof(*ret));
	memset(ret,0,sizeof(*ret));
	ret->neturl = ll_copy_str(neturl);
	ret->filename = ll_copy_str(filename);
	ret->key = ll_copy_str(key);
	ret->p.open_method = open_method;
	ret->next = matrises;
	matrises = ret;

	if ( open_method )
		_insert_matrix_open(ret,open_method);
	return ret;
}

MATRIX *
open_matrix(L_CHAR * neturl,L_CHAR * filename,L_CHAR * key,void (*open_method)(MATRIX_TOKEN*))
{
MATRIX * ret;
	lock_task(matrix_lock);
	ret = _open_matrix(neturl,filename,key,open_method);
	unlock_task(matrix_lock,"new_matrix");
	return ret;
}


void
_close_matrix(MATRIX * m)
{
int i;
int f;
MATRIX_SEXP * d;
MATRIX_NODE * n;
MATRIX ** mp;
	if ( m->mode == MM_CLOSE )
		return;
	m->mode = MM_CLOSE;
	for ( mp = &matrises ; *mp && *mp != m ; mp = &(*mp)->next );
	if ( *mp == 0 )
		er_panic("_close_matrix");
	*mp = m->next;
retry:
	if ( m->wait_token ) {
		sleep_task((int)m,matrix_lock);
		lock_task(matrix_lock);
		goto retry;
	}
	f = 0;
	for ( i = 0 ; i < NCACHE_SIZE ; ) {
		if ( m->node_hash[i] == 0 ) {
			i ++; 
			continue;
		}
		f = 1;
		n = m->node_hash[i];
		if ( _free_matrix_node(m->node_hash[i]) < 0 ) {
			sleep_task((int)n,matrix_lock);
			lock_task(matrix_lock);
			goto retry;
		}
	}
	if ( m->p.close_file )
		(*m->p.close_file)(m);
	if ( m->p.close_net )
		(*m->p.close_net)(m);

	if ( m->channel_info )
		d_f_ree(m->channel_info);
	if ( m->dim_divide )
		d_f_ree(m->dim_divide);
	if ( m->block_size )
		d_f_ree(m->block_size);
	if ( m->pixel_size )
		d_f_ree(m->pixel_size);
	for ( i = 0 ; i < MI_MAX ; i ++ ) {
		d = m->cal[i];
		d->del = 1;
	}
	if ( m->neturl )
		d_f_ree(m->neturl);
	if ( m->filename )
		d_f_ree(m->filename);
	if ( m->key )
		d_f_ree(m->key);
}


MATRIX *
_search_matrix(L_CHAR * neturl,L_CHAR * filename,L_CHAR * key)
{
MATRIX * ret;
	for ( ret = matrises ; ret ; ret = ret->next ) {
		if ( neturl ) {
			if ( ret->neturl == 0 )
				continue;
			if ( l_strcmp(neturl,ret->neturl) )
				continue;
		}
		if ( filename ) {
			if ( ret->filename == 0 )
				continue;
			if ( l_strcmp(filename,ret->filename) )
				continue;
		}
		if ( key ) {
			if ( ret->key == 0 )
				continue;
			if ( l_strcmp(key,ret->key) )
				continue;
		}
		return ret;
	}
	return 0;
}

MATRIX *
search_matrix(int * errp,L_CHAR * neturl,L_CHAR * filename,L_CHAR * key,MATRIX_TOKEN * t)
{
MATRIX * ret;
	lock_task(matrix_lock);
	ret = _search_matrix(neturl,filename,key);
	if ( ret ) {
		if ( ret->mode == MM_CREATE && t ) {
			t->wait_matrix = ret;
			t->next = ret->wait_token;
			ret->wait_token = t;
			if ( errp )
				*errp = ME_PROC_NODE;
		}
		else if ( errp )
			*errp = 0;
	}
	else if ( errp )
		*errp = ME_NO_NODE;
	unlock_task(matrix_lock,"search_matrix");
	return ret;
}

int
_set_matrix_channel_info(MATRIX * m,int ch,MATRIX_CHANNEL_INFO * inf)
{
	if ( m->p.channel_nos <= ch || ch < 0 ) 
		return ME_INDEX_OVER;
	m->channel_info[ch] = *inf;
	return 0;
}

void
_cal_matrix_total_levels(MATRIX * m)
{
int pix;
int i;
int lev;
int max;
	m->total_levels = 0;
	if ( m->pixel_size == 0 )
		return;
	if ( m->p.dim == 0 )
		return;
	if ( m->dim_divide )
		return;
	max = 0;
	for ( i = 0 ; i < m->p.dim ; i ++ ) {
		if ( m->pixel_size[i] == 0 )
			return;
		if ( m->dim_divide[i] == 0 )
			return;
		pix = m->pixel_size[i];
		lev = 0;
		for ( ; pix <= 1 ; ) {
			lev ++;
			if ( pix & ((1<<m->dim_divide[i])-1) )
				pix = (pix >> m->dim_divide[i]) + 1;
			else	pix = pix >> m->dim_divide[i];
		}
		lev ++;
		if ( max < lev )
			max = lev;
	}
	m->total_levels = max;
}

int
_set_matrix_param(MATRIX * m,MATRIX_PARAM * p)
{
int i;
	if ( m->p.channel_nos && p->channel_nos )
		return ME_CHANNEL_NOS;
	if ( m->p.dim && p->dim )
		return ME_DIM;
	for ( i = 0 ; i < MI_MAX ; i ++ ) {
		if ( m->p.pri_area[i] && p->pri_area[i] )
			return ME_PRI_AREA;
	}
	if ( m->p.open_method && p->open_method )
		return ME_OPEN_METHOD;
	if ( m->p.read_file && p->read_file )
		return ME_FETCH_METHOD;
	if ( m->p.write_file && p->write_file )
		return ME_FETCH_METHOD;
	if ( m->p.read_net && p->read_net )
		return ME_FETCH_METHOD;
	if ( m->p.close_file && p->close_file )
		return ME_FETCH_METHOD;
	if ( m->p.close_net && p->close_net )
		return ME_FETCH_METHOD;
	if ( p->channel_nos ) {
		m->p.channel_nos = p->channel_nos;
		m->channel_info = d_alloc(sizeof(MATRIX_CHANNEL_INFO)*p->channel_nos);
		memset(m->channel_info,0,sizeof(MATRIX_CHANNEL_INFO)*p->channel_nos);
	}
	if ( p->dim ) {
		m->p.dim = p->dim;
		m->dim_divide = d_alloc(p->dim);
		memset(m->dim_divide,0,p->dim);
		m->block_size = d_alloc(p->dim);
		memset(m->block_size,0,p->dim);
		m->pixel_size = d_alloc(p->dim*sizeof(int));
		memset(m->pixel_size,0,p->dim);
	}
	for ( i = 0 ; i < MI_MAX ;i ++ ) {
		if ( p->pri_area[i] )
			m->p.pri_area[i] = p->pri_area[i];
	}
	if ( p->open_method )
		m->p.open_method = p->open_method;
	if ( p->read_file )
		m->p.read_file = p->read_file;
	if ( p->write_file )
		m->p.write_file = p->write_file;
	if ( p->read_net )
		m->p.read_net = p->read_net;
	if ( p->close_file )
		m->p.close_file = p->close_file;
	if ( p->close_net )
		m->p.close_net = p->close_net;
	return 0;
}

int
_set_matrix_dim_divide(MATRIX * m,int dim,int dim_divide)
{
	if ( m->p.dim == 0 )
		return ME_DIM;
	if ( dim < 0 || dim >= m->p.dim )
		return ME_INDEX_OVER;
	m->dim_divide[dim] = dim_divide;
	_cal_matrix_total_levels(m);
	return 0;
}

int
_set_matrix_block_size(MATRIX * m,int dim,int block_size)
{
	if ( m->p.dim == 0 )
		return ME_DIM;
	if ( dim < 0 || dim >= m->p.dim )
		return ME_INDEX_OVER;
	m->block_size[dim] = block_size;
	return 0;
}

int
_set_matrix_pixel_size(MATRIX * m,int dim,int pixel_size)
{
	if ( m->p.dim == 0 )
		return ME_DIM;
	if ( dim < 0 || dim >= m->p.dim )
		return ME_INDEX_OVER;
	m->pixel_size[dim] = pixel_size;
	_cal_matrix_total_levels(m);
	return 0;
}

int
_set_matrix_cal(MATRIX * m,int id,XL_SEXP * cal)
{
MATRIX_SEXP * d;
	if ( id < 0 || id >= MI_MAX )
		return ME_INDEX_OVER;
	if ( m->cal[id] ) {
		m->cal[id]->data = cal;
	}
	else {
		m->cal[id] = d = d_alloc(sizeof(*d));
		d->data = cal;
		d->del = 0;
		_insert_matrix_sexp(d);
	}
	return 0;
}

void
_set_matrix_mode(MATRIX * m,int mode)
{
MATRIX_TOKEN * t;
	m->mode = mode;
	if ( mode == 0 )
		return;
	for ( ; m->wait_token ; ) {
		t = m->wait_token;
		m->wait_token = t->next;
		t->wait_matrix = 0;
		_insert_matrix_access(t,matrix_exec_cal);
	}
}

void
set_matrix_mode(MATRIX * m,int mode)
{
	lock_task(matrix_lock);
	_set_matrix_mode(m,mode);
	unlock_task(matrix_lock,"set_matrix_mode");
}


int
set_matrix_channel_info(MATRIX * m,int ch,MATRIX_CHANNEL_INFO * inf)
{
int ret;
	lock_task(matrix_lock);
	ret = _set_matrix_channel_info(m,ch,inf);
	unlock_task(matrix_lock,"set_matrix");
	return ret;
}


int 
set_matrix_param(MATRIX * m,MATRIX_PARAM * p)
{
int ret;
	lock_task(matrix_lock);
	ret = _set_matrix_param(m,p);
	unlock_task(matrix_lock,"set_matrix");
	return ret;
}


int 
set_matrix_dim_divide(MATRIX * m,int dim,int dim_divide)
{
int ret;
	lock_task(matrix_lock);
	ret = _set_matrix_dim_divide(m,dim,dim_divide);
	unlock_task(matrix_lock,"set_matrix");
	return ret;
}


int 
set_matrix_block_size(MATRIX * m,int dim,int block_size)
{
int ret;
	lock_task(matrix_lock);
	ret = _set_matrix_block_size(m,dim,block_size);
	unlock_task(matrix_lock,"set_matrix");
	return ret;
}


int 
set_matrix_pixel_size(MATRIX * m,int dim,int pixel_size)
{
int ret;
	lock_task(matrix_lock);
	ret = _set_matrix_pixel_size(m,dim,pixel_size);
	unlock_task(matrix_lock,"set_matrix");
	return ret;
}

int 
set_matrix_cal(MATRIX * m,int id,XL_SEXP * cal)
{
int ret;
	lock_task(matrix_lock);
	ret = _set_matrix_cal(m,id,cal);
	unlock_task(matrix_lock,"set_matrix");
	return ret;
}

int 
_cmp_dim_code(MATRIX * m,INTEGER64 * dim_code1,INTEGER64 * dim_code2)
{
int dim;
int i;
INTEGER64 level_1,level_2;
INTEGER64 mask;
	if ( dim_code1 == 0 && dim_code2 == 0 )
		return 0;
	if ( dim_code1 == 0 && dim_code2 )
		return -1;
	if ( dim_code1 && dim_code2 == 0 )
		return 1;
	level_1 = dim_code1[0];
	level_2 = dim_code2[0];
	dim = m->p.dim;

	if ( level_1 < level_2 )
		return -1;
	if ( level_1 > level_2 )
		return 1;
	for ( i = 1 ; i <= dim ; i ++ ) {
		mask = - (1 << (level_1*m->dim_divide[i] + m->block_size[i]));
		if ( (dim_code1[i] & mask) < (dim_code2[i] & mask)  )
			return -1;
		if ( (dim_code1[i] & mask) > (dim_code2[i] & mask)  )
			return 1;
	}
	return 0;
}

INTEGER64 *
_copy_dim_code(MATRIX * m,INTEGER64 * dim_code)
{
INTEGER64 * ret;
int dim;
	if ( dim_code == 0 )
		return 0;
	dim = m->p.dim;
	ret = d_alloc(sizeof(int)*(dim+1));
	memcpy(ret,dim_code,sizeof(INTEGER64)*(dim+1));
	return ret;
}

INTEGER64 *
copy_dim_code(MATRIX * m,INTEGER64 * dim_code)
{
	return _copy_dim_code(m,dim_code);
}



void
matrix_exec_cal(MATRIX_TOKEN * t)
{
XL_SEXP * ret;
XLISP_ENV * e;
	e = new_env(matrix_xl_env);
	e->e.work = t;
	for( ; get_type(t->cal_target) == XLT_PAIR ; ) {
		gc_push(0,0,"matrix_exec_cal");
		ret = eval(e,car(t->cal_target));
		if ( get_type(ret) != XLT_ERROR ) {
			t->cal_target = cdr(t->cal_target);
			gc_pop(0,0);
			continue;
		}
		switch ( ret->err.code ) {
		case XLE_SYSTEM_EXIT:
			gc_pop(0,0);
			return;
		case XLE_SYSTEM_APPLICATION:
			gc_pop(0,0);
			return;
		}
		log_print_sexp(LOG_ERROR,LOG_LAYER_XL,0,"matrix_exec_cal",ret,0);
		gc_pop(0,0);
		lock_task(matrix_lock);
		t->process_node->status = MS_LOADING_ERR_1;
		goto end;
	}
	t->access_target ++;
	if ( t->access_target < MT_ACCESS_MAX ) {
		if ( t->access[t->access_target] < 0 )
			goto end2;
		t->cal_target = t->process_node->matrix->cal[t->access[t->access_target]];
		_insert_matrix_access(t,matrix_exec_cal);
		return;
	}
	t->process_node->status = MS_OK;
end2:
	lock_task(matrix_lock);
end:
	for ( ; t->locked_node_list ; ) {
		_unlock_node(t->locked_node_list,t);
	}
	wakeup_task((int)t->process_node);
	unlock_task(matrix_lock,"matrix_exec_cal");
}

void
_insert_node_wait_token(MATRIX_NODE * n,MATRIX_TOKEN * t)
{
	if ( t == 0 )
		return;
	if ( t->wait_node )
		er_panic("_insert_node_wait_token");
	t->wait_node = n;
	t->next = n->wait_token;
	n->wait_token = t;
}

void
_lock_node(MATRIX_NODE * n,MATRIX_TOKEN * t)
{
MATRIX_LIST * lst;	n->locked ++;
	if ( t == 0 )
		return;
	lst = d_alloc(sizeof(*lst));
	lst->node = n;
	lst->next = t->locked_node_list;
	t->locked_node_list = lst;
}

void
_unlock_node(MATRIX_NODE * n,MATRIX_TOKEN * t)
{
MATRIX_LIST ** lp,* lst;
	if ( n->locked > 0 )
		n->locked --;
	if ( t ) {
		for ( lp = &t->locked_node_list ; *lp ; lp = &(*lp)->next )
			if ( (*lp)->node == n ) {
				lst = *lp;
				*lp = lst->next;
				d_f_ree(lst);
				break;
			}
	}
	wakeup_task((int)n);
}

void
_insert_matrix_access(MATRIX_TOKEN * t,void (*func)())
{
int pri;
MATRIX * m;
MATRIX_NODE * n;
int b;
	t->h.key = nl_copy_str(std_cm,"_get_matrix_access");
	t->func = func;
	n = t->process_node;
	m = n->matrix;
	b = m->p.pri_area[t->access[t->access_target]];
	switch ( b & 0xffff ) {
	case PR_UP:
		pri = ((b >> 16) & 0xffff) + n->dim_code[0];
		break;
	case PR_DOWN:
		pri = ((b >> 16) & 0xffff) - n->dim_code[0];
			+ m->total_levels * 2;
		break;
	case PR_STRT:
		pri = ((b >> 16) & 0xffff);
		break;
	default:
		er_panic("_get_matrix_access");
	}
	t->h.pri = pri;
	insert_queue(&matrix_que,t,0);
}

void
_get_matrix_access(MATRIX_NODE * n)
{
MATRIX_TOKEN * t;
MATRIX * m;
	t = d_alloc(sizeof(*t));
	memset(t,0,sizeof(*t));
	t->process_node = n;
	m = n->matrix;
	switch ( n->status ) {
	case MS_NOBIND:
		n->status = MS_PROCESS;
		t->access[0] = MI_FETCH_1;
		t->access[1] = MI_FETCH_2;
		t->access[2] = MI_VISU_1;
		t->access[3] = MI_VISU_2;
		t->access[4] = -1;
		t->access_target = 0;
		t->cal_target = m->cal[MI_FETCH_1];
		_insert_matrix_access(t,matrix_exec_cal);
		break;
	case MS_PROCESS:
	case MS_OK:
	case MS_PROCESS_ERR_1:
		break;
	}
}


void
_matrix_trigger_access(MATRIX_NODE * n,int access)
{
MATRIX_TOKEN * t;
MATRIX * m;
int i;
	m = n->matrix;
	if ( n->process_token == 0 ) {
		t = d_alloc(sizeof(*t));
		memset(t,0,sizeof(*t));
		t->process_node = n;
		t->access[0] = -1;
		t->access_target = -1;
		t->cal_target = 0;
	}
	for( i = 0 ; t->access[i] >= 0 ; i ++ )
		if ( t->access[i] == access )
			break;
retry:
	if ( i >= MT_ACCESS_MAX )
		er_panic("_trigger_access");
	if ( t->access[i] < 0 ) {
		if ( i >= MT_ACCESS_MAX-1 )
			er_panic("_trigger_access2");
		t->access[i] = access;
		t->access[i+1] = -1;
		if ( t->access_target < 0 )
			t->access_target = 0;
		if ( t->cal_target == 0 )
			t->cal_target = m->cal[t->access[0]];
	}
	else {
		if ( t->access[i+1] >= 0 ) {
			for ( ; t->access[i] >= 0 ; i ++ );
				goto retry;
		}
		if ( t->access_target == i )
			t->cal_target = m->cal[t->access[i]];
	}

	if ( n->process_token == 0 )
		_insert_matrix_access(t,matrix_exec_cal);
}

void
matrix_trigger_access(MATRIX_NODE * n,int access)
{
	lock_task(matrix_lock);
	_matrix_trigger_access(n,access);
	unlock_task(matrix_lock,"trigger_access");
}



INLINE unsigned int
_get_node_hash(MATRIX * m,INTEGER64 * dim_code)
{
U_INTEGER64 key;
int i;
int dim;
INTEGER64 mask;
INTEGER64 level;
	dim = m->p.dim;
	level = dim_code[0];
	key = dim_code[0];
	for ( i = 1 ; i <= dim ; i ++ ) {
		mask = - (1 << (level*m->dim_divide[i] + m->block_size[i-1]));
		key += (dim_code[i] & mask);
	}
	return key % NHASH_SIZE;
}


int
_free_matrix_node(MATRIX_NODE * n)
{
MATRIX_NODE ** np;
unsigned int key;
MATRIX * m;
int i;
MATRIX_DATA_TYPE * md;
	key = _get_node_hash(n->matrix,n->dim_code);
	m = n->matrix;
	for ( np = m->node_hash[key] ; *np && *np != n ; np = &(*np)->nh_next );
	if ( *np == 0 )
		er_panic("_free_matrix_node");
	_delete_node_ring(n);
	if ( n->locked || n->ch_lock || n->save_lock || n->dirty_file )
		return -1;
	if ( n->wait_token || n->process_token )
		return -2;
	*np = n->nh_next;
	d_f_ree(n->dim_code);
	for ( i = 0 ; i < m->p.channel_nos ; i ++ ) {
		md = m->channel_info[i].data_type;
		(*md->free_data)(md,n->channel[i].data);
	}
	if ( n->nlist_dim_bit_field )
		d_f_ree(n->nlist_dim_bit_field);
	if ( n->nlist_dim_addr )
		d_f_ree(n->nlist_dim_addr);
	d_f_ree(n);
	return 0;
}



MATRIX_NODE *
_get_matrix_node(int * errp,MATRIX * m,INTEGER64 * dim_code,int access)
{
MATRIX_NODE * n;
unsigned int key;
	switch ( m->mode ) {
	case MM_CREATE:
		if ( errp )
			*errp = ME_MATRIX_ERR;
		return 0;
	case MM_STANBY:
		break;
	case MM_ERR:
		if ( errp )
			*errp = ME_MATRIX_ERR;
		return 0;
	case MM_CLOSE:
		if ( errp )
			*errp = ME_MATRIX_ERR;
		return 0;
	default:
		er_panic("_get_matrix_node");
	}
	key = _get_node_hash(m,dim_code);
	for ( n = m->node_hash[key] ; n ; n = n->nh_next )
		if ( _cmp_dim_code(m,n->dim_code,dim_code) == 0 )
			break;
	if ( n ) {
		_delete_node_ring(n);
		_insert_node_ring(n);
		if ( n->status <= MS_PROCESS ) {
			if ( errp )
				*errp = ME_PROC_NODE;
		}
		else {
			if ( errp )
				*errp = 0;
		}
		return n;
	}
	if ( errp )
		*errp = ME_NO_NODE;
	if ( access < 0 )
		return 0;
	n->matrix = m;
	n->dim_code = _copy_dim_code(m,dim_code);
	n->nh_next = m->node_hash[key];
	m->node_hash[key] = n;
	_insert_node_ring(n);
	_get_matrix_access(n);
	return n;
}

MATRIX_NODE *
get_matrix_node(int * errp,MATRIX * m,int * dim_code,int access,MATRIX_TOKEN * t)
{
MATRIX_NODE * n;
int err;
	lock_task(matrix_lock);
	n = _get_matrix_node(&err,m,dim_code,access);
	if ( errp )
		*errp = err;
	if ( err ) {
		if ( err == ME_MATRIX_ERR )
			n = 0;
		else if ( access < 0 )
			n = 0;
		else	_insert_node_wait_token(n,t);
	}
	else	_lock_node(n,t);
	unlock_task(matrix_lock,"get_matrix_node");
	return n;
}

int
_get_dim_code_index(MATRIX_NODE * n,INTEGER64 * dim_code)
{
int ix;
MATRIX * m;
INTEGER64 dd;
int i;
	ix = 0;
	m = n->matrix;
	for ( i = 1 ; i <= m->p.dim ; i ++ ) {
		dd = dim_code[i] >> ((n->dim_code[0]-1)*m->dim_divide[i]
				+ m->block_size[i]);
		dd &= (1<<m->dim_divide[i])-1;
		ix = (ix << m->dim_divide[i]) | dd;
	}
	return ix;
}


int
_check_dim_code(MATRIX_NODE * n,INTEGER64 * dim_code)
{
int pt_st,pt_end;
int mid;
int lev;
int ix;
	if ( n->dim_code[0] == 0 )
		return 0;
	if ( n->nlist_dim_bit_field ) {
		ix = _get_dim_code_index(n,dim_code);
		if ( (n->nlist_dim_bit_field[ix/8]) & (1<<(ix % 8)) )
			return 1;
		return 0;
	}
	else {
		pt_st = 0;
		pt_end = n->nlist_dim_addr_len;
		lev = dim_code[0];
		dim_code[0] = n->dim_code[0]-1;
		for ( ; pt_st + 1 < pt_end ; ) {
			mid = (pt_st + pt_end)/2;
			switch ( _cmp_dim_code(n->matrix,dim_code,
					n->nlist_dim_addr[mid])) {
			case -1:
				pt_end = mid;
				break;
			case 0:
				dim_code[0] = lev;
				return 1;
			case 1:
				pt_st = mid+1;
				break;
			}
		}
		if ( _cmp_dim_code(n->matrix,dim_code,
				n->nlist_dim_addr[pt_st]) == 0 ) {
			dim_code[0] = lev;
			return 1;
		}
		dim_code[0] = lev;
		return 0;
	}
}

void
_isnert_dim_code_index(MATRIX_NODE * n,INTEGER64 * dim_code)
{
int ix;
int ** lst;
int pt_st,pt_end;
int mid;
int lev;
int i;
	if ( _check_dim_code(n,dim_code) )
		return;
	if ( n->nlist_dim_bit_field ) {
		ix = _get_dim_code_index(n,dim_code);
		n->nlist_dim_bit_field[ix/8] |= (1<<(ix % 8));
	}
	lst = n->nlist_dim_addr = d_re_alloc(n->nlist_dim_addr,
			sizeof(int*) * (n->nlist_dim_addr_len + 1));
	pt_st = 0;
	pt_end = n->nlist_dim_addr_len;
	lev = dim_code[0];
	dim_code[0] = n->dim_code[0]-1;
	for ( ; pt_st + 1 < pt_end ; ) {
		mid = (pt_st + pt_end)/2;
		switch ( _cmp_dim_code(n->matrix,dim_code,
				n->nlist_dim_addr[mid])) {
		case -1:
			pt_end = mid;
			break;
		case 0:
			er_panic("_insert_dim_code_index");
		case 1:
			pt_st = mid+1;
			break;
		}
	}
	for ( i = n->nlist_dim_addr_len ; i > pt_st ; i -- )
		n->nlist_dim_addr[i] = n->nlist_dim_addr[i-1];
	n->nlist_dim_addr[pt_st] = _copy_dim_code(n->matrix,dim_code);
	dim_code[0] = lev;
}



MATRIX_NODE *
_get_matrix_tree_1(int * errp,MATRIX * m,INTEGER64 * dim_code,int access)
{
MATRIX_NODE * ret,* _ret;
int err;
	dim_code[0] = m->total_levels-1;
	err = 0;
	ret = _get_matrix_node(&err,m,dim_code,0);
	if ( err )
		goto loading;
	dim_code[0] --;
	for ( ; dim_code[0] >= 0 ; dim_code[0] -- )  {
		_ret = _get_matrix_node(&err,m,dim_code,-1);
		if ( err == 0 ) {
			ret = _ret;
			continue;
		}
		if ( _check_dim_code(ret,dim_code) ) {
			_ret = _get_matrix_node(&err,m,dim_code,0);
			goto loading;
		}
		break;
	}
	if ( ret == 0 && errp )
		*errp = ME_NO_NODE;
	return ret;
loading:
	dim_code[0] --;
	for ( ; dim_code[0] >=  0 ; dim_code[0] -- ) {
		_ret = _get_matrix_node(&err,m,dim_code,0);
	}
	if ( ret == 0 && errp )
		*errp = ME_NO_NODE;
	return ret;
}

MATRIX_NODE *
_get_matrix_tree_2(int * errp,MATRIX * m,INTEGER64 * dim_code,int access)
{
MATRIX_NODE * ret,* _ret;
int err;
	dim_code[0] = m->total_levels-1;
	err = 0;
	for ( ; dim_code[0] >= 0 ; dim_code[0] -- ) {
		_ret = _get_matrix_node(&err,m,dim_code,0);
		if ( err == 0 )
			ret = _ret;
		else {
			_isnert_dim_code_index(ret,dim_code);
		}
	}
	if ( ret == 0 && errp )
		*errp = ME_NO_NODE;
	return ret;
}

MATRIX_NODE *
_get_matrix_tree(int * errp,MATRIX * m,INTEGER64 * dim_code,int access)
{
	if ( access > 0 )
		return _get_matrix_tree_2(errp,m,dim_code,access);
	else	return _get_matrix_tree_1(errp,m,dim_code,access);
}


INTEGER64 *
get_dim_code_from_sexp(XL_SEXP * s)
{
int *ret;
int len;
int i;
XL_SEXP * ss;
	len = list_length(s);
	if ( len < 2 )
		return 0;
	ret = d_alloc(sizeof(int)*len);
	for ( i = 0 ; i < len ; i ++ , s = cdr(s) ) {
		ss = car(s);
		if ( get_type(ss) != XLT_INTEGER )
			ret[i] = 0;
		ret[i] = ss->integer.data;
	}
	return ret;
}

XL_SEXP * get_sexp_from_dim_code(MATRIX * m,INTEGER64* dim_code)
{
XL_SEXP * ret;
int i;
	ret = 0;
	for ( i = 0 ; i <= m->p.dim ; i ++ )
		ret = cons(get_integer(dim_code[i],0),ret);
	return reverse(ret);
}

XL_SEXP * 
matrix_error(char * func,XL_SEXP * s)
{
	return get_error(
		s->h.file,
		s->h.line,
		XLE_SYSTEM_APPLICATION,
		l_string(std_cm,func),
		n_get_string("Application System error"));
}

XL_SEXP * 
_get_channel_data(MATRIX_NODE * n,int ch,XL_SEXP * s)
{
MATRIX * m;
void * d;
MATRIX_DATA_TYPE * md;
MATRIX_SEXP * sx;
XL_SEXP * ret;
	if  ( n->channel == 0 )
		goto no_obj;
	m = n->matrix;
	if ( ch < 0 || ch >= m->p.channel_nos )
		goto no_obj;
	md = m->channel_info[ch].data_type;
	if ( md->type == MDT_SEXP ) {
		_matrix_node_channel_lock(n);
		sx = n->channel[ch].data;
		if ( sx == 0 ) {
			_matrix_node_channel_unlock(n);
			goto no_obj;
		}
		cons(ret = sx->data,0);
		_matrix_node_channel_unlock(n);
		return ret;
	}
	else {
		_matrix_node_channel_lock(n);
		d = n->channel[ch].data;
		if ( d == 0 ) {
			_matrix_node_channel_unlock(n);
			goto no_obj;
		}
		ret = (*md->md2sexp)(md,d);
		_matrix_node_channel_unlock(n);
		return ret;
	}
no_obj:
	if ( s )
		return get_error(
			s->h.file,
			s->h.line,
			XLE_PROTO_INV_OBJECT,
			l_string(std_cm,"mxCH"),
			n_get_string("invalid object in channel of NODE"));
	else	return get_error(
			0,
			0,
			XLE_PROTO_INV_OBJECT,
			l_string(std_cm,"mxCH"),
			n_get_string("invalid object in channel of NODE"));
}


XL_SEXP * 
get_channel_data(MATRIX_NODE * n,int ch,XL_SEXP * s)
{
XL_SEXP * ret;
	lock_task(matrix_lock);
	ret = _get_channel_data(n,ch,s);
	unlock_task(matrix_lock,"get_channel_data");
	return ret;
}

void
_set_dirty_file(MATRIX_NODE * n)
{
	n->dirty_file = 1;
	_matrix_trigger_access(n,MI_SAVE);
}


int
_set_channel_sexp(MATRIX_NODE * n,int ch,XL_SEXP * s)
{
MATRIX * m;
MATRIX_DATA_HEADER * h;
MATRIX_DATA_TYPE * md;
int cf;
	m = n->matrix;
	if ( ch < 0 || ch >= m->p.channel_nos )
		return -1;
	if ( n->channel == 0 ) {
		n->channel = d_alloc(sizeof(MATRIX_CHANNEL)*m->p.channel_nos);
		memset(n->channel,0,sizeof(MATRIX_CHANNEL)*m->p.channel_nos);
	}
	md = m->channel_info[ch].data_type;
	h = (*md->sexp2md)(&cf,md,s);
	if ( h == 0 )
		return -1;
	_matrix_node_channel_lock(n);
	if ( n->channel[ch].data == 0 ) {
		if ( cf )
			n->channel[ch].data = h;
		else	n->channel[ch].data = (md->alloc_copy)(md,h);
		if ( m->channel_info[ch].flags & MF_FILE )
			_set_dirty_file(n);
		_matrix_node_channel_unlock(n);
		return 1;
	}
	switch ( (*md->cmp)(h,n->channel[ch].data) ) {
	case -2:
		if ( cf )
			(*md->free_data)(md,h);
		_matrix_node_channel_unlock(n);
		return -1;
	case -1:
	case 1:
		if ( n->channel[ch].data )
			(*md->free_data)(md,n->channel[ch].data);
		(md->copy)(md,n->channel[ch].data,h);
		if ( cf )
			(*md->free_data)(md,h);
		if ( m->channel_info[ch].flags & MF_FILE )
			n->dirty_file = 1;
		_matrix_node_channel_unlock(n);
		return 1;
	case 0:
		if ( cf )
			(*md->free_data)(md,h);
		_matrix_node_channel_unlock(n);
		return 0;
	default:
		er_panic("_set_channel_sexp");
	}
	_matrix_node_channel_unlock(n);
	return -1;
}


int
set_channel_sexp(MATRIX_NODE * n,int ch,XL_SEXP * s)
{
int ret;
	lock_task(matrix_lock);
	ret = _set_channel_sexp(n,ch,s);
	unlock_task(matrix_lock,"get_channel_data");
	return ret;
}

