/**********************************************************************
 
	Copyright (C) 2008 Hirohisa MORI <joshua@globalbase.org>
 
	This program is free software; you can redistribute it 
	and/or modify it under the terms of the GLOBALBASE 
	Library General Public License (G-LGPL) as published by 

	http://www.globalbase.org/
 
	This program is distributed in the hope that it will be 
	useful, but WITHOUT ANY WARRANTY; without even the 
	implied warranty of MERCHANTABILITY or FITNESS FOR A 
	PARTICULAR PURPOSE.

**********************************************************************/

#include	<stdlib.h>
#include	"memory_debug.h"
#include	"memory_routine.h"
#include	"xlerror.h"
#include	"xl.h"
#include	"utils.h"
#include	"gbview.h"
#include	"win_flame.h"
#include	"lock_level.h"

#include	"mx_format.h"

#include	"pg_1dgraph.h"

#define CENTER_LINE_COLOR		0x3ff
#define GRAPH_LINE_COLOR		(0x3ff<<10)

extern SEM luster_lock;

int
vmt_create_1dgraph(VIEW_METHOD*,MATRIX_NODE*);
void
vmt_draw_1dgraph_int8(VIEW_METHOD*,GBVIEW_FLAME*,RESOURCE*,DRAW_WORK*);
void
vmt_draw_1dgraph_int16(VIEW_METHOD*,GBVIEW_FLAME*,RESOURCE*,DRAW_WORK*);
void
vmt_draw_1dgraph_int32(VIEW_METHOD*,GBVIEW_FLAME*,RESOURCE*,DRAW_WORK*);
void
vmt_draw_1dgraph_uint32(VIEW_METHOD*,GBVIEW_FLAME*,RESOURCE*,DRAW_WORK*);
void
vmt_draw_1dgraph_int64(VIEW_METHOD*,GBVIEW_FLAME*,RESOURCE*,DRAW_WORK*);
void
vmt_draw_1dgraph_uint64(VIEW_METHOD*,GBVIEW_FLAME*,RESOURCE*,DRAW_WORK*);
void
vmt_draw_1dgraph_float(VIEW_METHOD*,GBVIEW_FLAME*,RESOURCE*,DRAW_WORK*);
void
vmt_draw_1dgraph_double(VIEW_METHOD*,GBVIEW_FLAME*,RESOURCE*,DRAW_WORK*);
void flush_area_1dgraph_int(WF_WORK_AREA * a);
int get_interrested_point_1dgraph_int(WF_WORK_AREA * a,I_POINT * ptr,int size);
int set_interrested_point_1dgraph_int(WF_WORK_AREA * a,I_POINT * ptr);
void close_1dgraph_int(WF_WORK_AREA *a);

void flush_area_1dgraph_uint(WF_WORK_AREA * a);
int get_interrested_point_1dgraph_uint(WF_WORK_AREA * a,I_POINT * ptr,int size);
int set_interrested_point_1dgraph_uint(WF_WORK_AREA * a,I_POINT * ptr);
void close_1dgraph_uint(WF_WORK_AREA *a);

void flush_area_1dgraph_double(WF_WORK_AREA * a);
int get_interrested_point_1dgraph_double(WF_WORK_AREA * a,I_POINT * ptr,int size);
int set_interrested_point_1dgraph_double(WF_WORK_AREA * a,I_POINT * ptr);
void close_1dgraph_double(WF_WORK_AREA *a);

static int div_list[] = {-2};


#define VIEW_METHOD_TBL_DEF(str,type,vmt_draw)	\
VIEW_METHOD_TBL str = {			\
	type,				\
	1,div_list,0,			\
	vmt_create_1dgraph,		\
	0,				\
	vmt_draw,			\
	0				\
};

VIEW_METHOD_TBL_DEF(vm_1dgraph_tbl_int8,MDT_VECTOR|MDT_INT8,vmt_draw_1dgraph_int8)
VIEW_METHOD_TBL_DEF(vm_1dgraph_tbl_uint8,MDT_VECTOR|MDT_UINT8,vmt_draw_1dgraph_int8)
VIEW_METHOD_TBL_DEF(vm_1dgraph_tbl_int16,MDT_VECTOR|MDT_INT16,vmt_draw_1dgraph_int16)
VIEW_METHOD_TBL_DEF(vm_1dgraph_tbl_uint16,MDT_VECTOR|MDT_UINT16,vmt_draw_1dgraph_int16)
VIEW_METHOD_TBL_DEF(vm_1dgraph_tbl_int32,MDT_VECTOR|MDT_INT32,vmt_draw_1dgraph_int32)
VIEW_METHOD_TBL_DEF(vm_1dgraph_tbl_uint32,MDT_VECTOR|MDT_UINT32,vmt_draw_1dgraph_uint32)
VIEW_METHOD_TBL_DEF(vm_1dgraph_tbl_int64,MDT_VECTOR|MDT_INT64,vmt_draw_1dgraph_int64)
VIEW_METHOD_TBL_DEF(vm_1dgraph_tbl_uint64,MDT_VECTOR|MDT_UINT64,vmt_draw_1dgraph_uint64)
VIEW_METHOD_TBL_DEF(vm_1dgraph_tbl_float,MDT_VECTOR|MDT_FLOAT,vmt_draw_1dgraph_float)
VIEW_METHOD_TBL_DEF(vm_1dgraph_tbl_double,MDT_VECTOR|MDT_DOUBLE,vmt_draw_1dgraph_double)

WF_WORK_AREA_TBL wa_tbl_1dgraph_int = {
	flush_area_1dgraph_int,
	get_interrested_point_1dgraph_int,
	set_interrested_point_1dgraph_int,
	close_1dgraph_int
};

WF_WORK_AREA_TBL wa_tbl_1dgraph_uint = {
	flush_area_1dgraph_uint,
	get_interrested_point_1dgraph_uint,
	set_interrested_point_1dgraph_uint,
	close_1dgraph_uint
};
WF_WORK_AREA_TBL wa_tbl_1dgraph_double = {
	flush_area_1dgraph_double,
	get_interrested_point_1dgraph_double,
	set_interrested_point_1dgraph_double,
	close_1dgraph_double
};


typedef struct param_1dgraph_header {
	WF_WORK_AREA		a;
	I_POINT			min_point;
	I_POINT			max_point;
	double			pitch;
	unsigned		get_max:1;
} PARAM_1DGRAPH_HEADER;

typedef struct param_1dgraph_int {
	PARAM_1DGRAPH_HEADER	h;
	INTEGER64		min;
	INTEGER64		max;
	struct param_1dgraph_int *	next;
} PARAM_1DGRAPH_INT;

typedef struct param_1dgraph_uint {
	PARAM_1DGRAPH_HEADER	h;
	U_INTEGER64		min;
	U_INTEGER64		max;
	struct param_1dgraph_uint *	next;
} PARAM_1DGRAPH_UINT;

typedef struct param_1dgraph_double {
	PARAM_1DGRAPH_HEADER	h;
	double			min;
	double			max;
	struct param_1dgraph_double *	next;
} PARAM_1DGRAPH_DOUBLE;


typedef struct vm_1dgraph_data {
	int				height;
	int				color;
	int				type;
#define LINE_TYPE_NORMAL		1
#define LINE_TYPE_PADDING		2
} VM_1DGRAPH_DATA;

typedef struct vm_1dgraph {
	VM_WORK_HEADER			wh;
	VM_1DGRAPH_DATA *		dt;
	unsigned			initialize:1;
} VM_1DGRAPH;

XL_SEXP *
xl_pg_1dgraph_line(XLISP_ENV * env,XL_SEXP * s,
	XLISP_ENV * a,XL_SYM_FIELD * sf);
VM_1DGRAPH_DATA *
vm1dg_initialize(RESOURCE * r);


void
init_pg_1dgraph(XLISP_ENV * env)
{
XLISP_ENV * pgv_env;

	pgv_env = new_env(env);
	root_tag(env,l_string(std_cm,"pg-1dgraph"),pgv_env);
	set_env(pgv_env,l_string(std_cm,"line"),
		get_func_prim(xl_pg_1dgraph_line,
			FO_APPLICATIVE,0,1,1));
}




XL_SEXP *
xl_pg_1dgraph_line(XLISP_ENV * env,XL_SEXP * s,XLISP_ENV * a,XL_SYM_FIELD * sf)
{
RESOURCE *r;
XL_SEXP * ret;
VM_1DGRAPH_DATA * dt;
L_CHAR * type;
L_CHAR * color;
L_CHAR * height;
	r = get_resource_ptr(&ret,env,s->h.file,s->h.line);
	if ( r == 0 )
		return ret;
	dt = vm1dg_initialize(r);

	type = get_sf_attribute(sf,l_string(std_cm,"type"));
	if ( type == 0 )
		dt->type = LINE_TYPE_NORMAL;
	else if ( l_strcmp(type,l_string(std_cm,"normal")) == 0 )
		dt->type = LINE_TYPE_NORMAL;
	else if ( l_strcmp(type,l_string(std_cm,"padding")) == 0 )
		dt->type = LINE_TYPE_PADDING;
	else	goto inv_param;

	color = get_sf_attribute(sf,l_string(std_cm,"color"));
	if ( color == 0 )
		dt->color = 0;
	else {
		sscanf(n_string(std_cm,color),"%x",&dt->color);
	}
	height = get_sf_attribute(sf,l_string(std_cm,"height"));
	if ( height == 0 )
		dt->height = E1D_FRAME_HEIGHT;
	else {
		dt->height = atoi(n_string(std_cm,height));
	}
	
	return 0;
inv_param:
	return get_error(
		s->h.file,
		s->h.line,
		XLE_PROTO_INV_PARAM,
		l_string(std_cm,"pg-1dgraph/line"),
		List(n_get_string("invalid parameter"),
			n_get_string("type(attr)"),
			-1));
}





void flush_area_1dgraph_int(WF_WORK_AREA * a)
{
PARAM_1DGRAPH_INT * target;
	target = (PARAM_1DGRAPH_INT*)a;
	target->h.min_point = target->next->h.min_point;
	target->h.max_point = target->next->h.max_point;
	target->h.pitch = target->next->h.pitch;
	target->h.get_max = target->next->h.get_max;
	target->max = target->next->max;
	target->min = target->next->min;
	target->next->h.min_point.x = target->next->h.min_point.y = 0x80000000;
	target->next->h.max_point.x = target->next->h.max_point.y = 0x80000000;
}

int get_interrested_point_1dgraph_int(WF_WORK_AREA * a,I_POINT * ptr,int size)
{
PARAM_1DGRAPH_INT * target;
	target =  (PARAM_1DGRAPH_INT*)a;
	if ( size < 4 )
		return 4;
	ptr[0] = target->h.min_point;
	ptr[1] = target->h.max_point;
	ptr[2] = target->next->h.min_point;
	ptr[3] = target->next->h.max_point;
	return 4;
}

int set_interrested_point_1dgraph_int(WF_WORK_AREA * a,I_POINT * ptr)
{
PARAM_1DGRAPH_INT * target;
	target =  (PARAM_1DGRAPH_INT*)a;
	target->h.min_point = ptr[0];
	target->h.max_point = ptr[1];
	target->next->h.min_point = ptr[2];
	target->next->h.max_point = ptr[3];
	return 4;
}


void close_1dgraph_int(WF_WORK_AREA *a)
{
PARAM_1DGRAPH_INT * target;
	target =  (PARAM_1DGRAPH_INT*)a;
	d_f_ree(target->next);
	d_f_ree(target);
}


void flush_area_1dgraph_uint(WF_WORK_AREA * a)
{
PARAM_1DGRAPH_UINT * target;
	target =  (PARAM_1DGRAPH_UINT*)a;
	target->h.min_point = target->next->h.min_point;
	target->h.max_point = target->next->h.max_point;
	target->h.pitch = target->next->h.pitch;
	target->h.get_max = target->next->h.get_max;
	target->max = target->next->max;
	target->min = target->next->min;
	target->next->h.min_point.x = target->next->h.min_point.y = 0x80000000;
	target->next->h.max_point.x = target->next->h.max_point.y = 0x80000000;
}

int get_interrested_point_1dgraph_uint(WF_WORK_AREA * a,I_POINT * ptr,int size)
{
PARAM_1DGRAPH_UINT * target;
	target = (PARAM_1DGRAPH_UINT*)a;
	if ( size < 4 )
		return 4;
	ptr[0] = target->h.min_point;
	ptr[1] = target->h.max_point;
	ptr[2] = target->next->h.min_point;
	ptr[3] = target->next->h.max_point;
	return 4;
}

int set_interrested_point_1dgraph_uint(WF_WORK_AREA * a,I_POINT * ptr)
{
PARAM_1DGRAPH_UINT * target;
	target = (PARAM_1DGRAPH_UINT*)a;
	target->h.min_point = ptr[0];
	target->h.max_point = ptr[1];
	target->next->h.min_point = ptr[2];
	target->next->h.max_point = ptr[3];
	return 4;
}


void close_1dgraph_uint(WF_WORK_AREA *a)
{
PARAM_1DGRAPH_UINT * target;
	target =(PARAM_1DGRAPH_UINT*) a;
	d_f_ree(target->next);
	d_f_ree(target);
}



void flush_area_1dgraph_double(WF_WORK_AREA * a)
{
PARAM_1DGRAPH_DOUBLE * target;
	target = (PARAM_1DGRAPH_DOUBLE*)a;
	target->h.min_point = target->next->h.min_point;
	target->h.max_point = target->next->h.max_point;
	target->h.pitch = target->next->h.pitch;
	target->h.get_max = target->next->h.get_max;
	target->max = target->next->max;
	target->min = target->next->min;
	target->next->h.min_point.x = target->next->h.min_point.y = 0x80000000;
	target->next->h.max_point.x = target->next->h.max_point.y = 0x80000000;
}

int get_interrested_point_1dgraph_double(WF_WORK_AREA * a,I_POINT * ptr,int size)
{
PARAM_1DGRAPH_DOUBLE * target;
	target = (PARAM_1DGRAPH_DOUBLE*)a;
	if ( size < 4 )
		return 4;
	ptr[0] = target->h.min_point;
	ptr[1] = target->h.max_point;
	ptr[2] = target->next->h.min_point;
	ptr[3] = target->next->h.max_point;
	return 4;
}

int set_interrested_point_1dgraph_double(WF_WORK_AREA * a,I_POINT * ptr)
{
PARAM_1DGRAPH_DOUBLE * target;
	target = (PARAM_1DGRAPH_DOUBLE*)a;
	target->h.min_point = ptr[0];
	target->h.max_point = ptr[1];
	target->next->h.min_point = ptr[2];
	target->next->h.max_point = ptr[3];
	return 4;
}


void close_1dgraph_double(WF_WORK_AREA *a)
{
PARAM_1DGRAPH_DOUBLE * target;
	target = (PARAM_1DGRAPH_DOUBLE*)a;
	d_f_ree(target->next);
	d_f_ree(target);
}


VM_1DGRAPH_DATA *
vm1dg_initialize(RESOURCE * r)
{
VM_1DGRAPH * ret;
VM_1DGRAPH_DATA * d;
	ret = (VM_1DGRAPH*)get_vm_work_header(r,&vm_1dgraph_tbl_int8,sizeof(*ret));
	if ( ret->initialize )
		return ret->dt;
	d = d_alloc(sizeof(*d));
	memset(d,0,sizeof(*d));
	ret->dt = d;
	ret->initialize = 1;
	ret = (VM_1DGRAPH*)get_vm_work_header(r,&vm_1dgraph_tbl_int16,sizeof(*ret));
	ret->dt = d;
	ret->initialize = 1;
	ret = (VM_1DGRAPH*)get_vm_work_header(r,&vm_1dgraph_tbl_int32,sizeof(*ret));
	ret->dt = d;
	ret->initialize = 1;
	ret = (VM_1DGRAPH*)get_vm_work_header(r,&vm_1dgraph_tbl_int64,sizeof(*ret));
	ret->dt = d;
	ret->initialize = 1;
	ret = (VM_1DGRAPH*)get_vm_work_header(r,&vm_1dgraph_tbl_uint16,sizeof(*ret));
	ret->dt = d;
	ret->initialize = 1;
	ret = (VM_1DGRAPH*)get_vm_work_header(r,&vm_1dgraph_tbl_uint32,sizeof(*ret));
	ret->dt = d;
	ret->initialize = 1;
	ret = (VM_1DGRAPH*)get_vm_work_header(r,&vm_1dgraph_tbl_uint64,sizeof(*ret));
	ret->dt = d;
	ret->initialize = 1;
	ret = (VM_1DGRAPH*)get_vm_work_header(r,&vm_1dgraph_tbl_float,sizeof(*ret));
	ret->dt = d;
	ret->initialize = 1;
	ret = (VM_1DGRAPH*)get_vm_work_header(r,&vm_1dgraph_tbl_double,sizeof(*ret));
	ret->dt = d;
	ret->initialize = 1;
	return d;
}


VM_1DGRAPH_DATA * 
_vm1dg_initialize(RESOURCE * r)
{
VM_1DGRAPH * ret;
VM_1DGRAPH_DATA * d;
	ret = (VM_1DGRAPH*)_get_vm_work_header(r,&vm_1dgraph_tbl_int8,sizeof(*ret));
	if ( ret->initialize )
		return ret->dt;
	d = d_alloc(sizeof(*d));
	memset(d,0,sizeof(*d));
	ret->dt = d;
	ret->initialize = 1;
	ret = (VM_1DGRAPH*)_get_vm_work_header(r,&vm_1dgraph_tbl_int16,sizeof(*ret));
	ret->dt = d;
	ret->initialize = 1;
	ret = (VM_1DGRAPH*)_get_vm_work_header(r,&vm_1dgraph_tbl_int32,sizeof(*ret));
	ret->dt = d;
	ret->initialize = 1;
	ret = (VM_1DGRAPH*)_get_vm_work_header(r,&vm_1dgraph_tbl_int64,sizeof(*ret));
	ret->dt = d;
	ret->initialize = 1;
	ret = (VM_1DGRAPH*)_get_vm_work_header(r,&vm_1dgraph_tbl_uint16,sizeof(*ret));
	ret->dt = d;
	ret->initialize = 1;
	ret = (VM_1DGRAPH*)_get_vm_work_header(r,&vm_1dgraph_tbl_uint32,sizeof(*ret));
	ret->dt = d;
	ret->initialize = 1;
	ret = (VM_1DGRAPH*)_get_vm_work_header(r,&vm_1dgraph_tbl_uint64,sizeof(*ret));
	ret->dt = d;
	ret->initialize = 1;
	ret = (VM_1DGRAPH*)_get_vm_work_header(r,&vm_1dgraph_tbl_float,sizeof(*ret));
	ret->dt = d;
	ret->initialize = 1;
	ret = (VM_1DGRAPH*)_get_vm_work_header(r,&vm_1dgraph_tbl_double,sizeof(*ret));
	ret->dt = d;
	ret->initialize = 1;
	return d;
}

int
vmt_create_1dgraph(VIEW_METHOD* vm,MATRIX_NODE* n)
{
VM_1DGRAPH * v1dgt;
VM_1DGRAPH_DATA * dt;
	v1dgt = (VM_1DGRAPH*)_get_vm_work_header(vm->r,vm->tbl,sizeof(*v1dgt));
	vm->work = v1dgt;
	if ( v1dgt->initialize == 0 ) {
		dt = _vm1dg_initialize(vm->r);
		dt->type = LINE_TYPE_NORMAL;
		dt->color = 0;
		dt->height = E1D_FRAME_HEIGHT;
	}
	return 0;
}

void
calc_set_y_pitch_int(PARAM_1DGRAPH_INT * pg,int height)
{
	if ( pg->min >= 0 || (pg->min < 0 && pg->max >= 0 && pg->max > -pg->min) ) {
		if ( pg->max == 0 )
			pg->h.pitch = 1;
		else	pg->h.pitch = (height-3)/pg->max;
		pg->h.get_max = 1;
	}
	else {
		if ( pg->min == 0 )
			pg->h.pitch = 1;
		else	pg->h.pitch = (height-3)/(-pg->min);
		pg->h.get_max = 0;
	}
}

void
set_y_data_int(DRAW_WORK * dw,VIEW_METHOD * vm,WF_PARAM_1D * p1d,PARAM_1DGRAPH_INT * pg,int x,INTEGER64 d)
{
int w;
int y;
int bottom_y,top_y;
int center,height;
unsigned long cc;
int pos;
INTEGER64 pos_l;
VM_1DGRAPH * v1dgt;
VM_1DGRAPH_DATA * vd;
	v1dgt = vm->work;
	vd = v1dgt->dt;
	w = dw->rectondisp->br.x - dw->rectondisp->tl.x;
	top_y = dw->rectondisp->tl.y;
	bottom_y = dw->rectondisp->br.y;
	center = (p1d->y_bottom + p1d->y_top)/2;
	height = (p1d->y_bottom - p1d->y_top)/2;
	pos_l = d*pg->h.pitch;
	if ( pg->next->h.max_point.x == 0x80000000 || pg->next->max < d ) {
		pg->next->max = d;
		pg->next->h.max_point.x = x;
		pg->next->h.max_point.y = 0;
	}
	if ( pg->next->h.min_point.x == 0x80000000 || pg->next->min > d ) {
		pg->next->h.min_point.x = x;
		pg->next->h.min_point.y = 0;
		pg->next->min = d;
	}
	calc_set_y_pitch_int(pg->next,height);
	if ( height <= pos_l ) {
		pos = top_y-1;
		dw->flags |= WFF_FLUSH;
	}
	else if ( pos_l <= -height ) {
		pos = bottom_y+1;
		dw->flags |= WFF_FLUSH;
	}
	else	pos = center - pos_l;
	if ( (dw->flags & WFF_FLUSH) == 0 && (
		(pg->h.get_max && (
			pg->h.max_point.x == 0x80000000 ||
			pg->h.max_point.x < -p1d->width/3 ||
			pg->h.max_point.x > 4*p1d->width/3 )) ||
		(pg->h.get_max == 0 && (
			pg->h.min_point.x == 0x80000000 ||
			pg->h.min_point.x < -p1d->width/3 ||
			pg->h.min_point.x > 4*p1d->width/3 ))) ) {
	
		dw->flags |= WFF_FLUSH;
		pg->next->h.pitch *= 2;
	}
	for ( y = top_y ; y < bottom_y ; y++ ) {
		if ( y < p1d->y_top )
			continue;
		if ( y > p1d->y_bottom )
			break;
		
		switch ( vd->type ) {
		case LINE_TYPE_NORMAL:
			if ( y == pos )
				cc =  COL(((vd->color>>24)&0xff)<<(COL_BIT-8),((vd->color>>16)&0xff)<<(COL_BIT-8),((vd->color>>8)&0xff)<<(COL_BIT-8));
			else if ( center == y )
				cc = CENTER_LINE_COLOR;
			else	cc = C_TRANSPARENT;
			break;
		case LINE_TYPE_PADDING:
			if ( pos < center ) {
				if ( y > center )
					cc = C_TRANSPARENT;
				else if ( y == center )
					cc = CENTER_LINE_COLOR;
				else if ( y >= pos )
					cc = COL(((vd->color>>24)&0xff)<<(COL_BIT-8),((vd->color>>16)&0xff)<<(COL_BIT-8),((vd->color>>8)&0xff)<<(COL_BIT-8));
				else	cc = C_TRANSPARENT;
			}
			else if ( pos > center ) {
				if ( y < center )
					cc = C_TRANSPARENT;
				else if ( y == center )
					cc = CENTER_LINE_COLOR;
				else if ( y <= pos )
					cc = COL(((vd->color>>24)&0xff)<<(COL_BIT-8),((vd->color>>16)&0xff)<<(COL_BIT-8),((vd->color>>8)&0xff)<<(COL_BIT-8));
				else	cc = C_TRANSPARENT;
			}
			else {
				if ( y == center )
					cc = COL(((vd->color>>24)&0xff)<<(COL_BIT-8),((vd->color>>16)&0xff)<<(COL_BIT-8),((vd->color>>8)&0xff)<<(COL_BIT-8));
				else	cc = C_TRANSPARENT;
			}
			break;
		default:
			er_panic("not support");
		}
		dw->pixels[x + w*(y-top_y)] = cc;
	}
}

/*****/


void
calc_set_y_pitch_uint(PARAM_1DGRAPH_UINT * pg,int height)
{
	if( pg->max == 0 )
		pg->h.pitch = 1;
	else	pg->h.pitch = (height-3)/pg->max;
	pg->h.get_max = 1;
}

void
set_y_data_uint(DRAW_WORK * dw,VIEW_METHOD * vm,WF_PARAM_1D * p1d,PARAM_1DGRAPH_UINT * pg,int x,U_INTEGER64 d)
{
int w;
int y;
int bottom_y,top_y;
int center,height;
unsigned long cc;
int pos;
U_INTEGER64 pos_l;
VM_1DGRAPH * v1dgt;
VM_1DGRAPH_DATA * vd;
	v1dgt = vm->work;
	vd = v1dgt->dt;
	w = dw->rectondisp->br.x - dw->rectondisp->tl.x;
	top_y = dw->rectondisp->tl.y;
	bottom_y = dw->rectondisp->br.y;
	center = (p1d->y_bottom + p1d->y_top)/2;
	height = (p1d->y_bottom - p1d->y_top)/2;
	pos_l = d*pg->h.pitch;
	if ( pg->next->h.max_point.x == 0x80000000 || pg->next->max < d ) {
		pg->next->max = d;
		pg->next->h.max_point.x = x;
		pg->next->h.max_point.y = 0;
	}
	if ( pg->next->h.min_point.x == 0x80000000 || pg->next->min > d ) {
		pg->next->h.min_point.x = x;
		pg->next->h.min_point.y = 0;
		pg->next->min = d;
	}
	calc_set_y_pitch_uint(pg->next,height);
	if ( height <= pos_l ) {
		pos = top_y-1;
		dw->flags |= WFF_FLUSH;
	}
	else if ( pos_l <= -height ) {
		pos = bottom_y+1;
		dw->flags |= WFF_FLUSH;
	}
	else	pos = center - pos_l;
	if ( (dw->flags & WFF_FLUSH) == 0 && (
		(pg->h.get_max && (
			pg->h.max_point.x == 0x80000000 ||
			pg->h.max_point.x < -p1d->width/3 ||
			pg->h.max_point.x > 4*p1d->width/3 )) ||
		(pg->h.get_max == 0 && (
			pg->h.min_point.x == 0x80000000 ||
			pg->h.min_point.x < -p1d->width/3 ||
			pg->h.min_point.x > 4*p1d->width/3 ))) ) {
	
		dw->flags |= WFF_FLUSH;
		pg->next->h.pitch *= 2;
	}
	for ( y = top_y ; y < bottom_y ; y++ ) {
		if ( y < p1d->y_top )
			continue;
		if ( y > p1d->y_bottom )
			break;
		
		switch ( vd->type ) {
		case LINE_TYPE_NORMAL:
			if ( y == pos )
				cc =  COL(((vd->color>>24)&0xff)<<(COL_BIT-8),((vd->color>>16)&0xff)<<(COL_BIT-8),((vd->color>>8)&0xff)<<(COL_BIT-8));
			else if ( center == y )
				cc = CENTER_LINE_COLOR;
			else	cc = C_TRANSPARENT;
			break;
		case LINE_TYPE_PADDING:
			if ( pos < center ) {
				if ( y > center )
					cc = C_TRANSPARENT;
				else if ( y == center )
					cc = CENTER_LINE_COLOR;
				else if ( y >= pos )
					cc = COL(((vd->color>>24)&0xff)<<(COL_BIT-8),((vd->color>>16)&0xff)<<(COL_BIT-8),((vd->color>>8)&0xff)<<(COL_BIT-8));
				else	cc = C_TRANSPARENT;
			}
			else if ( pos > center ) {
				if ( y < center )
					cc = C_TRANSPARENT;
				else if ( y == center )
					cc = CENTER_LINE_COLOR;
				else if ( y <= pos )
					cc = COL(((vd->color>>24)&0xff)<<(COL_BIT-8),((vd->color>>16)&0xff)<<(COL_BIT-8),((vd->color>>8)&0xff)<<(COL_BIT-8));
				else	cc = C_TRANSPARENT;
			}
			else {
				if ( y == center )
					cc = COL(((vd->color>>24)&0xff)<<(COL_BIT-8),((vd->color>>16)&0xff)<<(COL_BIT-8),((vd->color>>8)&0xff)<<(COL_BIT-8));
				else	cc = C_TRANSPARENT;
			}
			break;
		default:
			er_panic("not support");
		}
		dw->pixels[x + w*(y-top_y)] = cc;
	}
}



void
calc_set_y_pitch_double(PARAM_1DGRAPH_DOUBLE * pg,int height)
{
	if ( pg->min >= 0 || (pg->min < 0 && pg->max >= 0 && pg->max > -pg->min) ) {
		if ( pg->max == 0 )
			pg->h.pitch = 1;
		else	pg->h.pitch = (height-3)/pg->max;
		pg->h.get_max = 1;
	}
	else {
		if ( pg->min == 0 )
			pg->h.pitch = 1;
		else	pg->h.pitch = (height-3)/(-pg->min);
		pg->h.get_max = 0;
	}
}

void
set_y_data_double(DRAW_WORK * dw,VIEW_METHOD * vm,WF_PARAM_1D * p1d,PARAM_1DGRAPH_DOUBLE * pg,int x,double d)
{
int w;
int y;
int bottom_y,top_y;
int center,height;
unsigned long cc;
int pos;
double pos_l;
VM_1DGRAPH * v1dgt;
VM_1DGRAPH_DATA * vd;
	v1dgt = vm->work;
	vd = v1dgt->dt;
	w = dw->rectondisp->br.x - dw->rectondisp->tl.x;
	top_y = dw->rectondisp->tl.y;
	bottom_y = dw->rectondisp->br.y;
	center = (p1d->y_bottom + p1d->y_top)/2;
	height = (p1d->y_bottom - p1d->y_top)/2;
	pos_l = d*pg->h.pitch;
	if ( pg->next->h.max_point.x == 0x80000000 || pg->next->max < d ) {
		pg->next->max = d;
		pg->next->h.max_point.x = x;
		pg->next->h.max_point.y = 0;
	}
	if ( pg->next->h.min_point.x == 0x80000000 || pg->next->min > d ) {
		pg->next->h.min_point.x = x;
		pg->next->h.min_point.y = 0;
		pg->next->min = d;
	}
	calc_set_y_pitch_double(pg->next,height);
/*
ss_printf("d = %f\n",d);
ss_printf("d = %f %f\n",pg->next->max,pg->next->min);
*/
	if ( height <= pos_l ) {
		pos = top_y-1;
		dw->flags |= WFF_FLUSH;
	}
	else if ( pos_l <= -height ) {
		pos = bottom_y+1;
		dw->flags |= WFF_FLUSH;
	}
	else	pos = center-rint(pos_l);
//ss_printf("F %x %x %i %i %i\n",dw->flags,WFF_FLUSH,pg->h.get_max,pg->h.max_point.x,pg->h.min_point.x);
	if ( (dw->flags & WFF_FLUSH) == 0 && (
		(pg->h.get_max && (
			pg->h.max_point.x == 0x80000000 ||
			pg->h.max_point.x < -p1d->width/3 ||
			pg->h.max_point.x > 4*p1d->width/3 )) ||
		(pg->h.get_max == 0 && (
			pg->h.min_point.x == 0x80000000 ||
			pg->h.min_point.x < -p1d->width/3 ||
			pg->h.min_point.x > 4*p1d->width/3 ))) ) {

ss_printf("FLUSH\n");	
		dw->flags |= WFF_FLUSH;
		pg->next->h.pitch *= 2;
	}
	for ( y = top_y ; y < bottom_y ; y++ ) {
		if ( y < p1d->y_top )
			continue;
		if ( y > p1d->y_bottom )
			break;
		
		switch ( vd->type ) {
		case LINE_TYPE_NORMAL:
			if ( y == pos )
				cc =  COL(((vd->color>>24)&0xff)<<(COL_BIT-8),((vd->color>>16)&0xff)<<(COL_BIT-8),((vd->color>>8)&0xff)<<(COL_BIT-8));
			else if ( center == y )
				cc = CENTER_LINE_COLOR;
			else	cc = C_TRANSPARENT;
			break;
		case LINE_TYPE_PADDING:
			if ( pos < center ) {
				if ( y > center )
					cc = C_TRANSPARENT;
				else if ( y == center )
					cc = CENTER_LINE_COLOR;
				else if ( y >= pos )
					cc = COL(((vd->color>>24)&0xff)<<(COL_BIT-8),((vd->color>>16)&0xff)<<(COL_BIT-8),((vd->color>>8)&0xff)<<(COL_BIT-8));
				else	cc = C_TRANSPARENT;
			}
			else if ( pos > center ) {
				if ( y < center )
					cc = C_TRANSPARENT;
				else if ( y == center )
					cc = CENTER_LINE_COLOR;
				else if ( y <= pos )
					cc = COL(((vd->color>>24)&0xff)<<(COL_BIT-8),((vd->color>>16)&0xff)<<(COL_BIT-8),((vd->color>>8)&0xff)<<(COL_BIT-8));
				else	cc = C_TRANSPARENT;
			}
			else {
				if ( y == center )
					cc = COL(((vd->color>>24)&0xff)<<(COL_BIT-8),((vd->color>>16)&0xff)<<(COL_BIT-8),((vd->color>>8)&0xff)<<(COL_BIT-8));
				else	cc = C_TRANSPARENT;
			}
			break;
		default:
			er_panic("not support");
		}
		dw->pixels[x + w*(y-top_y)] = cc;
	}
}

/*****/



void
init_1dgraph_header(PARAM_1DGRAPH_HEADER * h)
{
	h->min_point.x = h->max_point.x = 0x80000000;
	h->min_point.y = h->max_point.y = 0x80000000;
	h->get_max = 0;
	h->pitch = 1;
}

PARAM_1DGRAPH_INT *
new_work_area_1d_int(WF_PARAM_1D * p1d,RESOURCE * r)
{
PARAM_1DGRAPH_INT * ret;
	ret = (PARAM_1DGRAPH_INT*)wf_checkout_work_area(p1d->aset,r);
	if ( ret )
		return ret;
	ret = (PARAM_1DGRAPH_INT*)wf_new_work_area(p1d->aset,&wa_tbl_1dgraph_int,r,sizeof(*ret));
	init_1dgraph_header(&ret->h);
	ret->min = 0;
	ret->max = 0;
	ret->next = d_alloc(sizeof(*ret));
	init_1dgraph_header(&ret->next->h);
	ret->next->min = ret->next->max = 0;
	return ret;
}

PARAM_1DGRAPH_UINT *
new_work_area_1d_uint(WF_PARAM_1D * p1d,RESOURCE * r)
{
PARAM_1DGRAPH_UINT * ret;
	ret = (PARAM_1DGRAPH_UINT*)wf_checkout_work_area(p1d->aset,r);
	if ( ret )
		return ret;
	ret = (PARAM_1DGRAPH_UINT*)wf_new_work_area(p1d->aset,&wa_tbl_1dgraph_uint,r,sizeof(*ret));
	init_1dgraph_header(&ret->h);
	ret->min = 0;
	ret->max = 0;
	ret->next = d_alloc(sizeof(*ret));
	init_1dgraph_header(&ret->next->h);
	ret->next->min = ret->next->max = 0;
	return ret;
}


PARAM_1DGRAPH_DOUBLE *
new_work_area_1d_double(WF_PARAM_1D * p1d,RESOURCE * r)
{
PARAM_1DGRAPH_DOUBLE * ret;
	ret = (PARAM_1DGRAPH_DOUBLE*)wf_checkout_work_area(p1d->aset,r);
	if ( ret )
		return ret;
	ret = (PARAM_1DGRAPH_DOUBLE*)wf_new_work_area(p1d->aset,&wa_tbl_1dgraph_double,r,sizeof(*ret));
	init_1dgraph_header(&ret->h);
	ret->min = 0;
	ret->max = 0;
	ret->next = d_alloc(sizeof(*ret));
	init_1dgraph_header(&ret->next->h);
	ret->next->min = ret->next->max = 0;
	return ret;
}


/*************/


typedef struct mtx_cache_int8 {
	int		tl_x;
	int		br_x;
	int		level;
	int		rect_logsize;
	MATRIX_DH_SET	ds;
	MATRIX_NODE *	n;
	char *		ds_ptr;
	unsigned	leaf;
} MTX_CACHE_INT8;


INTEGER64
get_mtx_1dgraph_int8(VIEW_METHOD * vm,MTX_CACHE_INT8 * cache,RESOURCE * r,INTEGER64 lev,INTEGER64 x,INTEGER64 * lev_p)
{
int err;
void * d;
MATRIX * m;
MATRIX_NODE * nn;
INTEGER64 dim_code[2];
int xx;
INTEGER64 mx;
int clr;
	m = r->h.mtx;
	clr = 0;
	if ( cache->n ) {
		if ( cache->level == lev &&
			cache->tl_x <= x &&
			cache->br_x > x )
			goto end;
		clr = 1;
		matrix_node_channel_unlock(cache->n,0);
		unlock_node(cache->n,0);
	}
	nn = cache->n;
	
	dim_code[0] = lev;
	dim_code[1] = x;
//ss_printf("get_mtx_data-1\n");

	if ( r->h.mtx->p.flags & MPF_NODE_DIPENDENT ) {
		d = get_matrix_node_channel
				(&err,&cache->n,r->h.mtx,
				&dim_code[0],vm->view_channel[0],
				GN_TREE,GN_NODE_CREATE,0,r->h.loop_no);
	}
	else {
		d = get_matrix_node_channel
				(&err,&cache->n,r->h.mtx,
				&dim_code[0],vm->view_channel[0],
				GN_TREE,GN_STEP_NODE_CREATE,0,r->h.loop_no);
	}
	if ( d == 0 ) {
		if ( nn && clr == 0 ) {
			matrix_node_channel_unlock(nn,0);
			unlock_node(nn,0);
		}
		memset(cache,0,sizeof(*cache));
		*lev_p = m->total_levels;
		return 0;
	}
	get_matrix_dh_set(&cache->ds,d);
	dim_code[0] = cache->n->dim_code[0];
	dim_code[1] = cache->n->dim_code[1];
	cache->tl_x = dim_code[1] & (mx=(-(1<<(dim_code[0]*m->dim_divide[0] + m->block_size[0]))));
	cache->br_x = cache->tl_x + (1<<(dim_code[0]*m->dim_divide[0] + m->block_size[0]));
	if ( cache->br_x > m->pixel_size[0] )
		cache->br_x = m->pixel_size[0];
	cache->level = dim_code[0];
	cache->ds_ptr = cache->ds.offset;
	cache->rect_logsize = m->block_size[0];
	*lev_p = dim_code[0];

end:
	xx = (x - cache->tl_x)>>(cache->level*m->dim_divide[0]);
	return cache->ds_ptr[xx];
}


void
vmt_draw_1dgraph_int8(
	VIEW_METHOD * vm,
	GBVIEW_FLAME * gf,
	RESOURCE* r,
	DRAW_WORK * dw)
{
int i;
GB_POINT p;
REAL1 res;
INTEGER64 xx;
REAL1 dpm1,dpm2;
INTEGER64 level;
REAL1 d;
INTEGER64 lev;
INTEGER64 lev_cnt;

MTX_CACHE_INT8 cache;

INTEGER64 cc;
INTEGER64 tim,tim2;

MATRIX * mtx;
REAL1 divide;
INTEGER64 int_divide;

WF_PARAM_1D * p1d;
PARAM_1DGRAPH_INT * pg;

	if ( r->pr64.exit_lock )
		return;
	if ( get_mov_flag(gf,dw->wfid) != 0 )
		return;
	if ( r->h.mtx == 0 ) {
		gv_new_luster_option(r,0);
		if ( r->h.mtx == 0 )
			return;
	}
	mtx = r->h.mtx;
	if ( mtx->flags & MXF_ERR_DETECTED ) {
		r->h.mtx = 0;
		close_matrix(mtx);
		gv_new_luster_option(r,0);
		if ( r->h.mtx == 0 )
			return;
		mtx = r->h.mtx;
	}
	
	p1d = dw->work;
	pg = new_work_area_1d_int(p1d,r);

	divide = 1;
	for ( i = 0 ; i < mtx->dim_divide[0] ; i ++ )
		divide *= 2;
	int_divide = mtx->dim_divide[0];

	dw->limit_reso = r->pr64.dpm;
	dw->limit_ptr = dw->pt_list[dw->size/2];
	r->h.loop_no ++;
	dw->flags |= WFF_LUSTER; 
	dpm1 = r->pr64.dpm;
	dpm2 = dpm1;
	level = 0;

	res = -1;
	for ( i = 0 ; i < dw->size ; i ++ ) {
		if ( res < dw->pt_reso[i] )
			res = dw->pt_reso[i];
	}

/*
	if ( res > dpm1*LIMIT_RESOLUTION_RATE )
		return;
*/

	if ( dpm2 < res || res < dpm2/divide ) {
		if ( res > dpm1 ) {
			dpm2 = dpm1;
		level = 0;
		}
		else {
			level = 0;
			d = -1;
			for ( dpm2 = dpm1 ;
				res <= dpm2 &&
				level < r->pr64.max_level;
				d = dpm2,
				dpm2 /= divide,
				level ++ );
			dpm2 = d;
			level --;
		}
	}

	lock_task(luster_lock);
	lev_cnt = 0;
	memset(&cache,0,sizeof(cache));
	lev = level;
	tim = get_xltime();
	for ( i = 0 ; i < dw->size ; i ++ ) {
		if ( (i % 1000) == 0 && wf_active_check_2(gf,dw->wfid) < 0 )
{ss_printf("FREE=RETURN-PIXEL\n");
			break;
}
		if ( r->pr64.exit_lock )
			break;
		tim2 = get_xltime();
		if ( tim != tim2 ) {
			if ( get_mov_flag(gf,dw->wfid) != 0 )
				break;
			tim = tim2;
		}
		p = dw->pt_list[i];
		if ( dw->pt_reso[i] < 0 )
			continue;
		xx = rint(p.x*r->pr64.dpm);
		if ( xx < 0 )
			continue;
		if ( xx >= mtx->pixel_size[0] )
			continue;
		if ( lev_cnt && cache.ds_ptr ) {
		int lx;
			if (		cache.tl_x <= xx && xx < cache.br_x ) {
				if ( cache.level == lev ) {

					lx = (xx>>(lev*int_divide))&((((INTEGER64)1)<<mtx->block_size[0])-1);
					if ( lx >= cache.ds.ix[0] ) {
						cc = 0;
					}
					else cc = cache.ds_ptr[lx];
				}
				else if ( cache.leaf ) {
					lx = (xx>>(cache.level*int_divide))&((((INTEGER64)1)<<mtx->block_size[0])-1);
					if ( lx >= cache.ds.ix[0] ) {
						cc = 0;
					}
					else cc = cache.ds_ptr[lx];
				}
				else	goto non_cache;
			}
			else goto non_cache;
			lev_cnt ++;
			if ( lev_cnt >= 100 )
				lev_cnt = 0;
		}
		else {
		non_cache:
			cc = get_mtx_1dgraph_int8(vm,&cache,r,level,xx,&lev);
			if ( level != lev && cache.leaf == 0 )
				dw->data_request ++;
			lev_cnt ++;
		}
		set_y_data_int(dw,vm,p1d,pg,i,cc);
		dw->flags |= WFF_DRAW;
	}

	if ( cache.n ) {
		matrix_node_channel_unlock(cache.n,0);
		unlock_node(cache.n,0);
	}
	
	unlock_task(luster_lock,"draw_pixel");

	wf_checkin_work_area(&pg->h.a);
}




typedef struct mtx_cache_int16 {
	int		tl_x;
	int		br_x;
	int		level;
	int		rect_logsize;
	MATRIX_DH_SET	ds;
	MATRIX_NODE *	n;
	short *		ds_ptr;
	unsigned	leaf;
} MTX_CACHE_INT16;


INTEGER64
get_mtx_1dgraph_int16(VIEW_METHOD * vm,MTX_CACHE_INT16 * cache,RESOURCE * r,INTEGER64 lev,INTEGER64 x,INTEGER64 * lev_p)
{
int err;
void * d;
MATRIX * m;
MATRIX_NODE * nn;
INTEGER64 dim_code[2];
int xx;
INTEGER64 mx;
int clr;
	m = r->h.mtx;
	clr = 0;
	if ( cache->n ) {
		if ( cache->level == lev &&
			cache->tl_x <= x &&
			cache->br_x > x )
			goto end;
		clr = 1;
		matrix_node_channel_unlock(cache->n,0);
		unlock_node(cache->n,0);
	}
	nn = cache->n;
	
	dim_code[0] = lev;
	dim_code[1] = x;
//ss_printf("get_mtx_data-1\n");

	if ( r->h.mtx->p.flags & MPF_NODE_DIPENDENT ) {
		d = get_matrix_node_channel
				(&err,&cache->n,r->h.mtx,
				&dim_code[0],vm->view_channel[0],
				GN_TREE,GN_NODE_CREATE,0,r->h.loop_no);
	}
	else {
		d = get_matrix_node_channel
				(&err,&cache->n,r->h.mtx,
				&dim_code[0],vm->view_channel[0],
				GN_TREE,GN_STEP_NODE_CREATE,0,r->h.loop_no);
	}
	if ( d == 0 ) {
		if ( nn && clr == 0 ) {
			matrix_node_channel_unlock(nn,0);
			unlock_node(nn,0);
		}
		memset(cache,0,sizeof(*cache));
		*lev_p = m->total_levels;
		return 0;
	}
	get_matrix_dh_set(&cache->ds,d);
	dim_code[0] = cache->n->dim_code[0];
	dim_code[1] = cache->n->dim_code[1];
	cache->tl_x = dim_code[1] & (mx=(-(1<<(dim_code[0]*m->dim_divide[0] + m->block_size[0]))));
	cache->br_x = cache->tl_x + (1<<(dim_code[0]*m->dim_divide[0] + m->block_size[0]));
	if ( cache->br_x > m->pixel_size[0] )
		cache->br_x = m->pixel_size[0];
	cache->level = dim_code[0];
	cache->ds_ptr = cache->ds.offset;
	cache->rect_logsize = m->block_size[0];
	*lev_p = dim_code[0];

end:
	xx = (x - cache->tl_x)>>(cache->level*m->dim_divide[0]);
	return cache->ds_ptr[xx];
}


void
vmt_draw_1dgraph_int16(
	VIEW_METHOD * vm,
	GBVIEW_FLAME * gf,
	RESOURCE* r,
	DRAW_WORK * dw)
{
int i;
GB_POINT p;
REAL1 res;
INTEGER64 xx;
REAL1 dpm1,dpm2;
INTEGER64 level;
REAL1 d;
INTEGER64 lev;
INTEGER64 lev_cnt;

MTX_CACHE_INT16 cache;

INTEGER64 cc;
INTEGER64 tim,tim2;

MATRIX * mtx;
REAL1 divide;
INTEGER64 int_divide;

WF_PARAM_1D * p1d;
PARAM_1DGRAPH_INT * pg;

	if ( r->pr64.exit_lock )
		return;
	if ( get_mov_flag(gf,dw->wfid) != 0 )
		return;
	if ( r->h.mtx == 0 ) {
		gv_new_luster_option(r,0);
		if ( r->h.mtx == 0 )
			return;
	}
	mtx = r->h.mtx;
	if ( mtx->flags & MXF_ERR_DETECTED ) {
		r->h.mtx = 0;
		close_matrix(mtx);
		gv_new_luster_option(r,0);
		if ( r->h.mtx == 0 )
			return;
		mtx = r->h.mtx;
	}
	
	p1d = dw->work;
	pg = new_work_area_1d_int(p1d,r);

	divide = 1;
	for ( i = 0 ; i < mtx->dim_divide[0] ; i ++ )
		divide *= 2;
	int_divide = mtx->dim_divide[0];

	dw->limit_reso = r->pr64.dpm;
	dw->limit_ptr = dw->pt_list[dw->size/2];
	r->h.loop_no ++;
	dw->flags |= WFF_LUSTER; 
	dpm1 = r->pr64.dpm;
	dpm2 = dpm1;
	level = 0;

	res = -1;
	for ( i = 0 ; i < dw->size ; i ++ ) {
		if ( res < dw->pt_reso[i] )
			res = dw->pt_reso[i];
	}

/*
	if ( res > dpm1*LIMIT_RESOLUTION_RATE )
		return;
*/

	if ( dpm2 < res || res < dpm2/divide ) {
		if ( res > dpm1 ) {
			dpm2 = dpm1;
		level = 0;
		}
		else {
			level = 0;
			d = -1;
			for ( dpm2 = dpm1 ;
				res <= dpm2 &&
				level < r->pr64.max_level;
				d = dpm2,
				dpm2 /= divide,
				level ++ );
			dpm2 = d;
			level --;
		}
	}

	lock_task(luster_lock);
	lev_cnt = 0;
	memset(&cache,0,sizeof(cache));
	lev = level;
	tim = get_xltime();
	for ( i = 0 ; i < dw->size ; i ++ ) {
		if ( (i % 1000) == 0 && wf_active_check_2(gf,dw->wfid) < 0 )
{ss_printf("FREE=RETURN-PIXEL\n");
			break;
}
		if ( r->pr64.exit_lock )
			break;
		tim2 = get_xltime();
		if ( tim != tim2 ) {
			if ( get_mov_flag(gf,dw->wfid) != 0 )
				break;
			tim = tim2;
		}
		p = dw->pt_list[i];
		if ( dw->pt_reso[i] < 0 )
			continue;
		xx = rint(p.x*r->pr64.dpm);
		if ( xx < 0 )
			continue;
		if ( xx >= mtx->pixel_size[0] )
			continue;
		if ( lev_cnt && cache.ds_ptr ) {
		int lx;
			if (		cache.tl_x <= xx && xx < cache.br_x ) {
				if ( cache.level == lev ) {

					lx = (xx>>(lev*int_divide))&((((INTEGER64)1)<<mtx->block_size[0])-1);
					if ( lx >= cache.ds.ix[0] ) {
						cc = 0;
					}
					else cc = cache.ds_ptr[lx];
				}
				else if ( cache.leaf ) {
					lx = (xx>>(cache.level*int_divide))&((((INTEGER64)1)<<mtx->block_size[0])-1);
					if ( lx >= cache.ds.ix[0] ) {
						cc = 0;
					}
					else cc = cache.ds_ptr[lx];
				}
				else	goto non_cache;
			}
			else goto non_cache;
			lev_cnt ++;
			if ( lev_cnt >= 100 )
				lev_cnt = 0;
		}
		else {
		non_cache:
			cc = get_mtx_1dgraph_int16(vm,&cache,r,level,xx,&lev);
			if ( level != lev && cache.leaf == 0 )
				dw->data_request ++;
			lev_cnt ++;
		}
		set_y_data_int(dw,vm,p1d,pg,i,cc);
		dw->flags |= WFF_DRAW;
	}

	if ( cache.n ) {
		matrix_node_channel_unlock(cache.n,0);
		unlock_node(cache.n,0);
	}
	
	unlock_task(luster_lock,"draw_pixel");

	wf_checkin_work_area(&pg->h.a);
}



typedef struct mtx_cache_int32 {
	int		tl_x;
	int		br_x;
	int		level;
	int		rect_logsize;
	MATRIX_DH_SET	ds;
	MATRIX_NODE *	n;
	int *		ds_ptr;
	unsigned	leaf;
} MTX_CACHE_INT32;


INTEGER64
get_mtx_1dgraph_int32(VIEW_METHOD * vm,MTX_CACHE_INT32 * cache,RESOURCE * r,INTEGER64 lev,INTEGER64 x,INTEGER64 * lev_p)
{
int err;
void * d;
MATRIX * m;
MATRIX_NODE * nn;
INTEGER64 dim_code[2];
int xx;
INTEGER64 mx;
int clr;
	m = r->h.mtx;
	clr = 0;
	if ( cache->n ) {
		if ( cache->level == lev &&
			cache->tl_x <= x &&
			cache->br_x > x )
			goto end;
		clr = 1;
		matrix_node_channel_unlock(cache->n,0);
		unlock_node(cache->n,0);
	}
	nn = cache->n;
	
	dim_code[0] = lev;
	dim_code[1] = x;
//ss_printf("get_mtx_data-1\n");

	if ( r->h.mtx->p.flags & MPF_NODE_DIPENDENT ) {
		d = get_matrix_node_channel
				(&err,&cache->n,r->h.mtx,
				&dim_code[0],vm->view_channel[0],
				GN_TREE,GN_NODE_CREATE,0,r->h.loop_no);
	}
	else {
		d = get_matrix_node_channel
				(&err,&cache->n,r->h.mtx,
				&dim_code[0],vm->view_channel[0],
				GN_TREE,GN_STEP_NODE_CREATE,0,r->h.loop_no);
	}
	if ( d == 0 ) {
		if ( nn && clr == 0 ) {
			matrix_node_channel_unlock(nn,0);
			unlock_node(nn,0);
		}
		memset(cache,0,sizeof(*cache));
		*lev_p = m->total_levels;
		return 0;
	}
	get_matrix_dh_set(&cache->ds,d);
	dim_code[0] = cache->n->dim_code[0];
	dim_code[1] = cache->n->dim_code[1];
	cache->tl_x = dim_code[1] & (mx=(-(1<<(dim_code[0]*m->dim_divide[0] + m->block_size[0]))));
	cache->br_x = cache->tl_x + (1<<(dim_code[0]*m->dim_divide[0] + m->block_size[0]));
	if ( cache->br_x > m->pixel_size[0] )
		cache->br_x = m->pixel_size[0];
	cache->level = dim_code[0];
	cache->ds_ptr = cache->ds.offset;
	cache->rect_logsize = m->block_size[0];
	*lev_p = dim_code[0];

end:
	xx = (x - cache->tl_x)>>(cache->level*m->dim_divide[0]);
	return cache->ds_ptr[xx];
}


void
vmt_draw_1dgraph_int32(
	VIEW_METHOD * vm,
	GBVIEW_FLAME * gf,
	RESOURCE* r,
	DRAW_WORK * dw)
{
int i;
GB_POINT p;
REAL1 res;
INTEGER64 xx;
REAL1 dpm1,dpm2;
INTEGER64 level;
REAL1 d;
INTEGER64 lev;
INTEGER64 lev_cnt;

MTX_CACHE_INT32 cache;

INTEGER64 cc;
INTEGER64 tim,tim2;

MATRIX * mtx;
REAL1 divide;
INTEGER64 int_divide;

WF_PARAM_1D * p1d;
PARAM_1DGRAPH_INT * pg;

	if ( r->pr64.exit_lock )
		return;
	if ( get_mov_flag(gf,dw->wfid) != 0 )
		return;
	if ( r->h.mtx == 0 ) {
		gv_new_luster_option(r,0);
		if ( r->h.mtx == 0 )
			return;
	}
	mtx = r->h.mtx;
	if ( mtx->flags & MXF_ERR_DETECTED ) {
		r->h.mtx = 0;
		close_matrix(mtx);
		gv_new_luster_option(r,0);
		if ( r->h.mtx == 0 )
			return;
		mtx = r->h.mtx;
	}
	
	p1d = dw->work;
	pg = new_work_area_1d_int(p1d,r);

	divide = 1;
	for ( i = 0 ; i < mtx->dim_divide[0] ; i ++ )
		divide *= 2;
	int_divide = mtx->dim_divide[0];

	dw->limit_reso = r->pr64.dpm;
	dw->limit_ptr = dw->pt_list[dw->size/2];
	r->h.loop_no ++;
	dw->flags |= WFF_LUSTER; 
	dpm1 = r->pr64.dpm;
	dpm2 = dpm1;
	level = 0;

	res = -1;
	for ( i = 0 ; i < dw->size ; i ++ ) {
		if ( res < dw->pt_reso[i] )
			res = dw->pt_reso[i];
	}

/*
	if ( res > dpm1*LIMIT_RESOLUTION_RATE )
		return;
*/

	if ( dpm2 < res || res < dpm2/divide ) {
		if ( res > dpm1 ) {
			dpm2 = dpm1;
		level = 0;
		}
		else {
			level = 0;
			d = -1;
			for ( dpm2 = dpm1 ;
				res <= dpm2 &&
				level < r->pr64.max_level;
				d = dpm2,
				dpm2 /= divide,
				level ++ );
			dpm2 = d;
			level --;
		}
	}

	lock_task(luster_lock);
	lev_cnt = 0;
	memset(&cache,0,sizeof(cache));
	lev = level;
	tim = get_xltime();
	for ( i = 0 ; i < dw->size ; i ++ ) {
		if ( (i % 1000) == 0 && wf_active_check_2(gf,dw->wfid) < 0 )
{ss_printf("FREE=RETURN-PIXEL\n");
			break;
}
		if ( r->pr64.exit_lock )
			break;
		tim2 = get_xltime();
		if ( tim != tim2 ) {
			if ( get_mov_flag(gf,dw->wfid) != 0 )
				break;
			tim = tim2;
		}
		p = dw->pt_list[i];
		if ( dw->pt_reso[i] < 0 )
			continue;
		xx = rint(p.x*r->pr64.dpm);
		if ( xx < 0 )
			continue;
		if ( xx >= mtx->pixel_size[0] )
			continue;
		if ( lev_cnt && cache.ds_ptr ) {
		int lx;
			if (		cache.tl_x <= xx && xx < cache.br_x ) {
				if ( cache.level == lev ) {

					lx = (xx>>(lev*int_divide))&((((INTEGER64)1)<<mtx->block_size[0])-1);
					if ( lx >= cache.ds.ix[0] ) {
						cc = 0;
					}
					else cc = cache.ds_ptr[lx];
				}
				else if ( cache.leaf ) {
					lx = (xx>>(cache.level*int_divide))&((((INTEGER64)1)<<mtx->block_size[0])-1);
					if ( lx >= cache.ds.ix[0] ) {
						cc = 0;
					}
					else cc = cache.ds_ptr[lx];
				}
				else	goto non_cache;
			}
			else goto non_cache;
			lev_cnt ++;
			if ( lev_cnt >= 100 )
				lev_cnt = 0;
		}
		else {
		non_cache:
			cc = get_mtx_1dgraph_int32(vm,&cache,r,level,xx,&lev);
			if ( level != lev && cache.leaf == 0 )
				dw->data_request ++;
			lev_cnt ++;
		}
		set_y_data_int(dw,vm,p1d,pg,i,cc);
		dw->flags |= WFF_DRAW;
	}

	if ( cache.n ) {
		matrix_node_channel_unlock(cache.n,0);
		unlock_node(cache.n,0);
	}
	
	unlock_task(luster_lock,"draw_pixel");

	wf_checkin_work_area(&pg->h.a);
}



typedef struct mtx_cache_int64 {
	int		tl_x;
	int		br_x;
	int		level;
	int		rect_logsize;
	MATRIX_DH_SET	ds;
	MATRIX_NODE *	n;
	INTEGER64 *	ds_ptr;
	unsigned	leaf;
} MTX_CACHE_INT64;


INTEGER64
get_mtx_1dgraph_int64(VIEW_METHOD * vm,MTX_CACHE_INT64 * cache,RESOURCE * r,INTEGER64 lev,INTEGER64 x,INTEGER64 * lev_p)
{
int err;
void * d;
MATRIX * m;
MATRIX_NODE * nn;
INTEGER64 dim_code[2];
int xx;
INTEGER64 mx;
int clr;
	m = r->h.mtx;
	clr = 0;
	if ( cache->n ) {
		if ( cache->level == lev &&
			cache->tl_x <= x &&
			cache->br_x > x )
			goto end;
		clr = 1;
		matrix_node_channel_unlock(cache->n,0);
		unlock_node(cache->n,0);
	}
	nn = cache->n;
	
	dim_code[0] = lev;
	dim_code[1] = x;
//ss_printf("get_mtx_data-1\n");

	if ( r->h.mtx->p.flags & MPF_NODE_DIPENDENT ) {
		d = get_matrix_node_channel
				(&err,&cache->n,r->h.mtx,
				&dim_code[0],vm->view_channel[0],
				GN_TREE,GN_NODE_CREATE,0,r->h.loop_no);
	}
	else {
		d = get_matrix_node_channel
				(&err,&cache->n,r->h.mtx,
				&dim_code[0],vm->view_channel[0],
				GN_TREE,GN_STEP_NODE_CREATE,0,r->h.loop_no);
	}
	if ( d == 0 ) {
		if ( nn && clr == 0 ) {
			matrix_node_channel_unlock(nn,0);
			unlock_node(nn,0);
		}
		memset(cache,0,sizeof(*cache));
		*lev_p = m->total_levels;
		return 0;
	}
	get_matrix_dh_set(&cache->ds,d);
	dim_code[0] = cache->n->dim_code[0];
	dim_code[1] = cache->n->dim_code[1];
	cache->tl_x = dim_code[1] & (mx=(-(1<<(dim_code[0]*m->dim_divide[0] + m->block_size[0]))));
	cache->br_x = cache->tl_x + (1<<(dim_code[0]*m->dim_divide[0] + m->block_size[0]));
	if ( cache->br_x > m->pixel_size[0] )
		cache->br_x = m->pixel_size[0];
	cache->level = dim_code[0];
	cache->ds_ptr = cache->ds.offset;
	cache->rect_logsize = m->block_size[0];
	*lev_p = dim_code[0];

end:
	xx = (x - cache->tl_x)>>(cache->level*m->dim_divide[0]);
	return cache->ds_ptr[xx];
}


void
vmt_draw_1dgraph_int64(
	VIEW_METHOD * vm,
	GBVIEW_FLAME * gf,
	RESOURCE* r,
	DRAW_WORK * dw)
{
int i;
GB_POINT p;
REAL1 res;
INTEGER64 xx;
REAL1 dpm1,dpm2;
INTEGER64 level;
REAL1 d;
INTEGER64 lev;
INTEGER64 lev_cnt;

MTX_CACHE_INT64 cache;

INTEGER64 cc;
INTEGER64 tim,tim2;

MATRIX * mtx;
REAL1 divide;
INTEGER64 int_divide;

WF_PARAM_1D * p1d;
PARAM_1DGRAPH_INT * pg;

	if ( r->pr64.exit_lock )
		return;
	if ( get_mov_flag(gf,dw->wfid) != 0 )
		return;
	if ( r->h.mtx == 0 ) {
		gv_new_luster_option(r,0);
		if ( r->h.mtx == 0 )
			return;
	}
	mtx = r->h.mtx;
	if ( mtx->flags & MXF_ERR_DETECTED ) {
		r->h.mtx = 0;
		close_matrix(mtx);
		gv_new_luster_option(r,0);
		if ( r->h.mtx == 0 )
			return;
		mtx = r->h.mtx;
	}
	
	p1d = dw->work;
	pg = new_work_area_1d_int(p1d,r);

	divide = 1;
	for ( i = 0 ; i < mtx->dim_divide[0] ; i ++ )
		divide *= 2;
	int_divide = mtx->dim_divide[0];

	dw->limit_reso = r->pr64.dpm;
	dw->limit_ptr = dw->pt_list[dw->size/2];
	r->h.loop_no ++;
	dw->flags |= WFF_LUSTER; 
	dpm1 = r->pr64.dpm;
	dpm2 = dpm1;
	level = 0;

	res = -1;
	for ( i = 0 ; i < dw->size ; i ++ ) {
		if ( res < dw->pt_reso[i] )
			res = dw->pt_reso[i];
	}

/*
	if ( res > dpm1*LIMIT_RESOLUTION_RATE )
		return;
*/

	if ( dpm2 < res || res < dpm2/divide ) {
		if ( res > dpm1 ) {
			dpm2 = dpm1;
		level = 0;
		}
		else {
			level = 0;
			d = -1;
			for ( dpm2 = dpm1 ;
				res <= dpm2 &&
				level < r->pr64.max_level;
				d = dpm2,
				dpm2 /= divide,
				level ++ );
			dpm2 = d;
			level --;
		}
	}

	lock_task(luster_lock);
	lev_cnt = 0;
	memset(&cache,0,sizeof(cache));
	lev = level;
	tim = get_xltime();
	for ( i = 0 ; i < dw->size ; i ++ ) {
		if ( (i % 1000) == 0 && wf_active_check_2(gf,dw->wfid) < 0 )
{ss_printf("FREE=RETURN-PIXEL\n");
			break;
}
		if ( r->pr64.exit_lock )
			break;
		tim2 = get_xltime();
		if ( tim != tim2 ) {
			if ( get_mov_flag(gf,dw->wfid) != 0 )
				break;
			tim = tim2;
		}
		p = dw->pt_list[i];
		if ( dw->pt_reso[i] < 0 )
			continue;
		xx = rint(p.x*r->pr64.dpm);
		if ( xx < 0 )
			continue;
		if ( xx >= mtx->pixel_size[0] )
			continue;
		if ( lev_cnt && cache.ds_ptr ) {
		int lx;
			if (		cache.tl_x <= xx && xx < cache.br_x ) {
				if ( cache.level == lev ) {

					lx = (xx>>(lev*int_divide))&((((INTEGER64)1)<<mtx->block_size[0])-1);
					if ( lx >= cache.ds.ix[0] ) {
						cc = 0;
					}
					else cc = cache.ds_ptr[lx];
				}
				else if ( cache.leaf ) {
					lx = (xx>>(cache.level*int_divide))&((((INTEGER64)1)<<mtx->block_size[0])-1);
					if ( lx >= cache.ds.ix[0] ) {
						cc = 0;
					}
					else cc = cache.ds_ptr[lx];
				}
				else	goto non_cache;
			}
			else goto non_cache;
			lev_cnt ++;
			if ( lev_cnt >= 100 )
				lev_cnt = 0;
		}
		else {
		non_cache:
			cc = get_mtx_1dgraph_int64(vm,&cache,r,level,xx,&lev);
			if ( level != lev && cache.leaf == 0 )
				dw->data_request ++;
			lev_cnt ++;
		}
		set_y_data_int(dw,vm,p1d,pg,i,cc);
		dw->flags |= WFF_DRAW;
	}

	if ( cache.n ) {
		matrix_node_channel_unlock(cache.n,0);
		unlock_node(cache.n,0);
	}
	
	unlock_task(luster_lock,"draw_pixel");

	wf_checkin_work_area(&pg->h.a);
}





/*************/


typedef struct mtx_cache_uint8 {
	int		tl_x;
	int		br_x;
	int		level;
	int		rect_logsize;
	MATRIX_DH_SET	ds;
	MATRIX_NODE *	n;
	unsigned char *	ds_ptr;
	unsigned	leaf;
} MTX_CACHE_UINT8;


U_INTEGER64
get_mtx_1dgraph_uint8(VIEW_METHOD * vm,MTX_CACHE_UINT8 * cache,RESOURCE * r,INTEGER64 lev,INTEGER64 x,INTEGER64 * lev_p)
{
int err;
void * d;
MATRIX * m;
MATRIX_NODE * nn;
INTEGER64 dim_code[2];
int xx;
INTEGER64 mx;
int clr;
	m = r->h.mtx;
	clr = 0;
	if ( cache->n ) {
		if ( cache->level == lev &&
			cache->tl_x <= x &&
			cache->br_x > x )
			goto end;
		clr = 1;
		matrix_node_channel_unlock(cache->n,0);
		unlock_node(cache->n,0);
	}
	nn = cache->n;
	
	dim_code[0] = lev;
	dim_code[1] = x;
//ss_printf("get_mtx_data-1\n");

	if ( r->h.mtx->p.flags & MPF_NODE_DIPENDENT ) {
		d = get_matrix_node_channel
				(&err,&cache->n,r->h.mtx,
				&dim_code[0],vm->view_channel[0],
				GN_TREE,GN_NODE_CREATE,0,r->h.loop_no);
	}
	else {
		d = get_matrix_node_channel
				(&err,&cache->n,r->h.mtx,
				&dim_code[0],vm->view_channel[0],
				GN_TREE,GN_STEP_NODE_CREATE,0,r->h.loop_no);
	}
	if ( d == 0 ) {
		if ( nn && clr == 0 ) {
			matrix_node_channel_unlock(nn,0);
			unlock_node(nn,0);
		}
		memset(cache,0,sizeof(*cache));
		*lev_p = m->total_levels;
		return 0;
	}
	get_matrix_dh_set(&cache->ds,d);
	dim_code[0] = cache->n->dim_code[0];
	dim_code[1] = cache->n->dim_code[1];
	cache->tl_x = dim_code[1] & (mx=(-(1<<(dim_code[0]*m->dim_divide[0] + m->block_size[0]))));
	cache->br_x = cache->tl_x + (1<<(dim_code[0]*m->dim_divide[0] + m->block_size[0]));
	if ( cache->br_x > m->pixel_size[0] )
		cache->br_x = m->pixel_size[0];
	cache->level = dim_code[0];
	cache->ds_ptr = cache->ds.offset;
	cache->rect_logsize = m->block_size[0];
	*lev_p = dim_code[0];

end:
	xx = (x - cache->tl_x)>>(cache->level*m->dim_divide[0]);
	return cache->ds_ptr[xx];
}


void
vmt_draw_1dgraph_uint8(
	VIEW_METHOD * vm,
	GBVIEW_FLAME * gf,
	RESOURCE* r,
	DRAW_WORK * dw)
{
int i;
GB_POINT p;
REAL1 res;
INTEGER64 xx;
REAL1 dpm1,dpm2;
INTEGER64 level;
REAL1 d;
INTEGER64 lev;
INTEGER64 lev_cnt;

MTX_CACHE_UINT8 cache;

U_INTEGER64 cc;
INTEGER64 tim,tim2;

MATRIX * mtx;
REAL1 divide;
INTEGER64 int_divide;

WF_PARAM_1D * p1d;
PARAM_1DGRAPH_UINT * pg;

	if ( r->pr64.exit_lock )
		return;
	if ( get_mov_flag(gf,dw->wfid) != 0 )
		return;
	if ( r->h.mtx == 0 ) {
		gv_new_luster_option(r,0);
		if ( r->h.mtx == 0 )
			return;
	}
	mtx = r->h.mtx;
	if ( mtx->flags & MXF_ERR_DETECTED ) {
		r->h.mtx = 0;
		close_matrix(mtx);
		gv_new_luster_option(r,0);
		if ( r->h.mtx == 0 )
			return;
		mtx = r->h.mtx;
	}
	
	p1d = dw->work;
	pg = new_work_area_1d_uint(p1d,r);

	divide = 1;
	for ( i = 0 ; i < mtx->dim_divide[0] ; i ++ )
		divide *= 2;
	int_divide = mtx->dim_divide[0];

	dw->limit_reso = r->pr64.dpm;
	dw->limit_ptr = dw->pt_list[dw->size/2];
	r->h.loop_no ++;
	dw->flags |= WFF_LUSTER; 
	dpm1 = r->pr64.dpm;
	dpm2 = dpm1;
	level = 0;

	res = -1;
	for ( i = 0 ; i < dw->size ; i ++ ) {
		if ( res < dw->pt_reso[i] )
			res = dw->pt_reso[i];
	}

/*
	if ( res > dpm1*LIMIT_RESOLUTION_RATE )
		return;
*/

	if ( dpm2 < res || res < dpm2/divide ) {
		if ( res > dpm1 ) {
			dpm2 = dpm1;
		level = 0;
		}
		else {
			level = 0;
			d = -1;
			for ( dpm2 = dpm1 ;
				res <= dpm2 &&
				level < r->pr64.max_level;
				d = dpm2,
				dpm2 /= divide,
				level ++ );
			dpm2 = d;
			level --;
		}
	}

	lock_task(luster_lock);
	lev_cnt = 0;
	memset(&cache,0,sizeof(cache));
	lev = level;
	tim = get_xltime();
	for ( i = 0 ; i < dw->size ; i ++ ) {
		if ( (i % 1000) == 0 && wf_active_check_2(gf,dw->wfid) < 0 )
{ss_printf("FREE=RETURN-PIXEL\n");
			break;
}
		if ( r->pr64.exit_lock )
			break;
		tim2 = get_xltime();
		if ( tim != tim2 ) {
			if ( get_mov_flag(gf,dw->wfid) != 0 )
				break;
			tim = tim2;
		}
		p = dw->pt_list[i];
		if ( dw->pt_reso[i] < 0 )
			continue;
		xx = rint(p.x*r->pr64.dpm);
		if ( xx < 0 )
			continue;
		if ( xx >= mtx->pixel_size[0] )
			continue;
		if ( lev_cnt && cache.ds_ptr ) {
		int lx;
			if (		cache.tl_x <= xx && xx < cache.br_x ) {
				if ( cache.level == lev ) {

					lx = (xx>>(lev*int_divide))&((((INTEGER64)1)<<mtx->block_size[0])-1);
					if ( lx >= cache.ds.ix[0] ) {
						cc = 0;
					}
					else cc = cache.ds_ptr[lx];
				}
				else if ( cache.leaf ) {
					lx = (xx>>(cache.level*int_divide))&((((INTEGER64)1)<<mtx->block_size[0])-1);
					if ( lx >= cache.ds.ix[0] ) {
						cc = 0;
					}
					else cc = cache.ds_ptr[lx];
				}
				else	goto non_cache;
			}
			else goto non_cache;
			lev_cnt ++;
			if ( lev_cnt >= 100 )
				lev_cnt = 0;
		}
		else {
		non_cache:
			cc = get_mtx_1dgraph_uint8(vm,&cache,r,level,xx,&lev);
			if ( level != lev && cache.leaf == 0 )
				dw->data_request ++;
			lev_cnt ++;
		}
		set_y_data_uint(dw,vm,p1d,pg,i,cc);
		dw->flags |= WFF_DRAW;
	}

	if ( cache.n ) {
		matrix_node_channel_unlock(cache.n,0);
		unlock_node(cache.n,0);
	}
	
	unlock_task(luster_lock,"draw_pixel");

	wf_checkin_work_area(&pg->h.a);
}




typedef struct mtx_cache_uint16 {
	int		tl_x;
	int		br_x;
	int		level;
	int		rect_logsize;
	MATRIX_DH_SET	ds;
	MATRIX_NODE *	n;
	unsigned short *ds_ptr;
	unsigned	leaf;
} MTX_CACHE_UINT16;


U_INTEGER64
get_mtx_1dgraph_uint16(VIEW_METHOD * vm,MTX_CACHE_UINT16 * cache,RESOURCE * r,INTEGER64 lev,INTEGER64 x,INTEGER64 * lev_p)
{
int err;
void * d;
MATRIX * m;
MATRIX_NODE * nn;
INTEGER64 dim_code[2];
int xx;
INTEGER64 mx;
int clr;
	m = r->h.mtx;
	clr = 0;
	if ( cache->n ) {
		if ( cache->level == lev &&
			cache->tl_x <= x &&
			cache->br_x > x )
			goto end;
		clr = 1;
		matrix_node_channel_unlock(cache->n,0);
		unlock_node(cache->n,0);
	}
	nn = cache->n;
	
	dim_code[0] = lev;
	dim_code[1] = x;
//ss_printf("get_mtx_data-1\n");

	if ( r->h.mtx->p.flags & MPF_NODE_DIPENDENT ) {
		d = get_matrix_node_channel
				(&err,&cache->n,r->h.mtx,
				&dim_code[0],vm->view_channel[0],
				GN_TREE,GN_NODE_CREATE,0,r->h.loop_no);
	}
	else {
		d = get_matrix_node_channel
				(&err,&cache->n,r->h.mtx,
				&dim_code[0],vm->view_channel[0],
				GN_TREE,GN_STEP_NODE_CREATE,0,r->h.loop_no);
	}
	if ( d == 0 ) {
		if ( nn && clr == 0 ) {
			matrix_node_channel_unlock(nn,0);
			unlock_node(nn,0);
		}
		memset(cache,0,sizeof(*cache));
		*lev_p = m->total_levels;
		return 0;
	}
	get_matrix_dh_set(&cache->ds,d);
	dim_code[0] = cache->n->dim_code[0];
	dim_code[1] = cache->n->dim_code[1];
	cache->tl_x = dim_code[1] & (mx=(-(1<<(dim_code[0]*m->dim_divide[0] + m->block_size[0]))));
	cache->br_x = cache->tl_x + (1<<(dim_code[0]*m->dim_divide[0] + m->block_size[0]));
	if ( cache->br_x > m->pixel_size[0] )
		cache->br_x = m->pixel_size[0];
	cache->level = dim_code[0];
	cache->ds_ptr = cache->ds.offset;
	cache->rect_logsize = m->block_size[0];
	*lev_p = dim_code[0];

end:
	xx = (x - cache->tl_x)>>(cache->level*m->dim_divide[0]);
	return cache->ds_ptr[xx];
}


void
vmt_draw_1dgraph_uint16(
	VIEW_METHOD * vm,
	GBVIEW_FLAME * gf,
	RESOURCE* r,
	DRAW_WORK * dw)
{
int i;
GB_POINT p;
REAL1 res;
INTEGER64 xx;
REAL1 dpm1,dpm2;
INTEGER64 level;
REAL1 d;
INTEGER64 lev;
INTEGER64 lev_cnt;

MTX_CACHE_UINT16 cache;

U_INTEGER64 cc;
INTEGER64 tim,tim2;

MATRIX * mtx;
REAL1 divide;
INTEGER64 int_divide;

WF_PARAM_1D * p1d;
PARAM_1DGRAPH_UINT * pg;

	if ( r->pr64.exit_lock )
		return;
	if ( get_mov_flag(gf,dw->wfid) != 0 )
		return;
	if ( r->h.mtx == 0 ) {
		gv_new_luster_option(r,0);
		if ( r->h.mtx == 0 )
			return;
	}
	mtx = r->h.mtx;
	if ( mtx->flags & MXF_ERR_DETECTED ) {
		r->h.mtx = 0;
		close_matrix(mtx);
		gv_new_luster_option(r,0);
		if ( r->h.mtx == 0 )
			return;
		mtx = r->h.mtx;
	}
	
	p1d = dw->work;
	pg = new_work_area_1d_uint(p1d,r);

	divide = 1;
	for ( i = 0 ; i < mtx->dim_divide[0] ; i ++ )
		divide *= 2;
	int_divide = mtx->dim_divide[0];

	dw->limit_reso = r->pr64.dpm;
	dw->limit_ptr = dw->pt_list[dw->size/2];
	r->h.loop_no ++;
	dw->flags |= WFF_LUSTER; 
	dpm1 = r->pr64.dpm;
	dpm2 = dpm1;
	level = 0;

	res = -1;
	for ( i = 0 ; i < dw->size ; i ++ ) {
		if ( res < dw->pt_reso[i] )
			res = dw->pt_reso[i];
	}

/*
	if ( res > dpm1*LIMIT_RESOLUTION_RATE )
		return;
*/

	if ( dpm2 < res || res < dpm2/divide ) {
		if ( res > dpm1 ) {
			dpm2 = dpm1;
		level = 0;
		}
		else {
			level = 0;
			d = -1;
			for ( dpm2 = dpm1 ;
				res <= dpm2 &&
				level < r->pr64.max_level;
				d = dpm2,
				dpm2 /= divide,
				level ++ );
			dpm2 = d;
			level --;
		}
	}

	lock_task(luster_lock);
	lev_cnt = 0;
	memset(&cache,0,sizeof(cache));
	lev = level;
	tim = get_xltime();
	for ( i = 0 ; i < dw->size ; i ++ ) {
		if ( (i % 1000) == 0 && wf_active_check_2(gf,dw->wfid) < 0 )
{ss_printf("FREE=RETURN-PIXEL\n");
			break;
}
		if ( r->pr64.exit_lock )
			break;
		tim2 = get_xltime();
		if ( tim != tim2 ) {
			if ( get_mov_flag(gf,dw->wfid) != 0 )
				break;
			tim = tim2;
		}
		p = dw->pt_list[i];
		if ( dw->pt_reso[i] < 0 )
			continue;
		xx = rint(p.x*r->pr64.dpm);
		if ( xx < 0 )
			continue;
		if ( xx >= mtx->pixel_size[0] )
			continue;
		if ( lev_cnt && cache.ds_ptr ) {
		int lx;
			if (		cache.tl_x <= xx && xx < cache.br_x ) {
				if ( cache.level == lev ) {

					lx = (xx>>(lev*int_divide))&((((INTEGER64)1)<<mtx->block_size[0])-1);
					if ( lx >= cache.ds.ix[0] ) {
						cc = 0;
					}
					else cc = cache.ds_ptr[lx];
				}
				else if ( cache.leaf ) {
					lx = (xx>>(cache.level*int_divide))&((((INTEGER64)1)<<mtx->block_size[0])-1);
					if ( lx >= cache.ds.ix[0] ) {
						cc = 0;
					}
					else cc = cache.ds_ptr[lx];
				}
				else	goto non_cache;
			}
			else goto non_cache;
			lev_cnt ++;
			if ( lev_cnt >= 100 )
				lev_cnt = 0;
		}
		else {
		non_cache:
			cc = get_mtx_1dgraph_uint16(vm,&cache,r,level,xx,&lev);
			if ( level != lev && cache.leaf == 0 )
				dw->data_request ++;
			lev_cnt ++;
		}
		set_y_data_uint(dw,vm,p1d,pg,i,cc);
		dw->flags |= WFF_DRAW;
	}

	if ( cache.n ) {
		matrix_node_channel_unlock(cache.n,0);
		unlock_node(cache.n,0);
	}
	
	unlock_task(luster_lock,"draw_pixel");

	wf_checkin_work_area(&pg->h.a);
}



typedef struct mtx_cache_uint32 {
	int		tl_x;
	int		br_x;
	int		level;
	int		rect_logsize;
	MATRIX_DH_SET	ds;
	MATRIX_NODE *	n;
	unsigned int *	ds_ptr;
	unsigned	leaf;
} MTX_CACHE_UINT32;


U_INTEGER64
get_mtx_1dgraph_uint32(VIEW_METHOD * vm,MTX_CACHE_UINT32 * cache,RESOURCE * r,INTEGER64 lev,INTEGER64 x,INTEGER64 * lev_p)
{
int err;
void * d;
MATRIX * m;
MATRIX_NODE * nn;
INTEGER64 dim_code[2];
int xx;
INTEGER64 mx;
int clr;
	m = r->h.mtx;
	clr = 0;
	if ( cache->n ) {
		if ( cache->level == lev &&
			cache->tl_x <= x &&
			cache->br_x > x )
			goto end;
		clr = 1;
		matrix_node_channel_unlock(cache->n,0);
		unlock_node(cache->n,0);
	}
	nn = cache->n;
	
	dim_code[0] = lev;
	dim_code[1] = x;
//ss_printf("get_mtx_data-1\n");

	if ( r->h.mtx->p.flags & MPF_NODE_DIPENDENT ) {
		d = get_matrix_node_channel
				(&err,&cache->n,r->h.mtx,
				&dim_code[0],vm->view_channel[0],
				GN_TREE,GN_NODE_CREATE,0,r->h.loop_no);
	}
	else {
		d = get_matrix_node_channel
				(&err,&cache->n,r->h.mtx,
				&dim_code[0],vm->view_channel[0],
				GN_TREE,GN_STEP_NODE_CREATE,0,r->h.loop_no);
	}
	if ( d == 0 ) {
		if ( nn && clr == 0 ) {
			matrix_node_channel_unlock(nn,0);
			unlock_node(nn,0);
		}
		memset(cache,0,sizeof(*cache));
		*lev_p = m->total_levels;
		return 0;
	}
	get_matrix_dh_set(&cache->ds,d);
	dim_code[0] = cache->n->dim_code[0];
	dim_code[1] = cache->n->dim_code[1];
	cache->tl_x = dim_code[1] & (mx=(-(1<<(dim_code[0]*m->dim_divide[0] + m->block_size[0]))));
	cache->br_x = cache->tl_x + (1<<(dim_code[0]*m->dim_divide[0] + m->block_size[0]));
	if ( cache->br_x > m->pixel_size[0] )
		cache->br_x = m->pixel_size[0];
	cache->level = dim_code[0];
	cache->ds_ptr = cache->ds.offset;
	cache->rect_logsize = m->block_size[0];
	*lev_p = dim_code[0];

end:
	xx = (x - cache->tl_x)>>(cache->level*m->dim_divide[0]);
	return cache->ds_ptr[xx];
}


void
vmt_draw_1dgraph_uint32(
	VIEW_METHOD * vm,
	GBVIEW_FLAME * gf,
	RESOURCE* r,
	DRAW_WORK * dw)
{
int i;
GB_POINT p;
REAL1 res;
INTEGER64 xx;
REAL1 dpm1,dpm2;
INTEGER64 level;
REAL1 d;
INTEGER64 lev;
INTEGER64 lev_cnt;

MTX_CACHE_UINT32 cache;

U_INTEGER64 cc;
INTEGER64 tim,tim2;

MATRIX * mtx;
REAL1 divide;
INTEGER64 int_divide;

WF_PARAM_1D * p1d;
PARAM_1DGRAPH_UINT * pg;

	if ( r->pr64.exit_lock )
		return;
	if ( get_mov_flag(gf,dw->wfid) != 0 )
		return;
	if ( r->h.mtx == 0 ) {
		gv_new_luster_option(r,0);
		if ( r->h.mtx == 0 )
			return;
	}
	mtx = r->h.mtx;
	if ( mtx->flags & MXF_ERR_DETECTED ) {
		r->h.mtx = 0;
		close_matrix(mtx);
		gv_new_luster_option(r,0);
		if ( r->h.mtx == 0 )
			return;
		mtx = r->h.mtx;
	}
	
	p1d = dw->work;
	pg = new_work_area_1d_uint(p1d,r);

	divide = 1;
	for ( i = 0 ; i < mtx->dim_divide[0] ; i ++ )
		divide *= 2;
	int_divide = mtx->dim_divide[0];

	dw->limit_reso = r->pr64.dpm;
	dw->limit_ptr = dw->pt_list[dw->size/2];
	r->h.loop_no ++;
	dw->flags |= WFF_LUSTER; 
	dpm1 = r->pr64.dpm;
	dpm2 = dpm1;
	level = 0;

	res = -1;
	for ( i = 0 ; i < dw->size ; i ++ ) {
		if ( res < dw->pt_reso[i] )
			res = dw->pt_reso[i];
	}

/*
	if ( res > dpm1*LIMIT_RESOLUTION_RATE )
		return;
*/

	if ( dpm2 < res || res < dpm2/divide ) {
		if ( res > dpm1 ) {
			dpm2 = dpm1;
		level = 0;
		}
		else {
			level = 0;
			d = -1;
			for ( dpm2 = dpm1 ;
				res <= dpm2 &&
				level < r->pr64.max_level;
				d = dpm2,
				dpm2 /= divide,
				level ++ );
			dpm2 = d;
			level --;
		}
	}

	lock_task(luster_lock);
	lev_cnt = 0;
	memset(&cache,0,sizeof(cache));
	lev = level;
	tim = get_xltime();
	for ( i = 0 ; i < dw->size ; i ++ ) {
		if ( (i % 1000) == 0 && wf_active_check_2(gf,dw->wfid) < 0 )
{ss_printf("FREE=RETURN-PIXEL\n");
			break;
}
		if ( r->pr64.exit_lock )
			break;
		tim2 = get_xltime();
		if ( tim != tim2 ) {
			if ( get_mov_flag(gf,dw->wfid) != 0 )
				break;
			tim = tim2;
		}
		p = dw->pt_list[i];
		if ( dw->pt_reso[i] < 0 )
			continue;
		xx = rint(p.x*r->pr64.dpm);
		if ( xx < 0 )
			continue;
		if ( xx >= mtx->pixel_size[0] )
			continue;
		if ( lev_cnt && cache.ds_ptr ) {
		int lx;
			if (		cache.tl_x <= xx && xx < cache.br_x ) {
				if ( cache.level == lev ) {

					lx = (xx>>(lev*int_divide))&((((INTEGER64)1)<<mtx->block_size[0])-1);
					if ( lx >= cache.ds.ix[0] ) {
						cc = 0;
					}
					else cc = cache.ds_ptr[lx];
				}
				else if ( cache.leaf ) {
					lx = (xx>>(cache.level*int_divide))&((((INTEGER64)1)<<mtx->block_size[0])-1);
					if ( lx >= cache.ds.ix[0] ) {
						cc = 0;
					}
					else cc = cache.ds_ptr[lx];
				}
				else	goto non_cache;
			}
			else goto non_cache;
			lev_cnt ++;
			if ( lev_cnt >= 100 )
				lev_cnt = 0;
		}
		else {
		non_cache:
			cc = get_mtx_1dgraph_uint32(vm,&cache,r,level,xx,&lev);
			if ( level != lev && cache.leaf == 0 )
				dw->data_request ++;
			lev_cnt ++;
		}
		set_y_data_uint(dw,vm,p1d,pg,i,cc);
		dw->flags |= WFF_DRAW;
	}

	if ( cache.n ) {
		matrix_node_channel_unlock(cache.n,0);
		unlock_node(cache.n,0);
	}
	
	unlock_task(luster_lock,"draw_pixel");

	wf_checkin_work_area(&pg->h.a);
}



typedef struct mtx_cache_uint64 {
	int		tl_x;
	int		br_x;
	int		level;
	int		rect_logsize;
	MATRIX_DH_SET	ds;
	MATRIX_NODE *	n;
	U_INTEGER64 *	ds_ptr;
	unsigned	leaf;
} MTX_CACHE_UINT64;


U_INTEGER64
get_mtx_1dgraph_uint64(VIEW_METHOD * vm,MTX_CACHE_UINT64 * cache,RESOURCE * r,INTEGER64 lev,INTEGER64 x,INTEGER64 * lev_p)
{
int err;
void * d;
MATRIX * m;
MATRIX_NODE * nn;
INTEGER64 dim_code[2];
int xx;
INTEGER64 mx;
int clr;
	m = r->h.mtx;
	clr = 0;
	if ( cache->n ) {
		if ( cache->level == lev &&
			cache->tl_x <= x &&
			cache->br_x > x )
			goto end;
		clr = 1;
		matrix_node_channel_unlock(cache->n,0);
		unlock_node(cache->n,0);
	}
	nn = cache->n;
	
	dim_code[0] = lev;
	dim_code[1] = x;
//ss_printf("get_mtx_data-1\n");

	if ( r->h.mtx->p.flags & MPF_NODE_DIPENDENT ) {
		d = get_matrix_node_channel
				(&err,&cache->n,r->h.mtx,
				&dim_code[0],vm->view_channel[0],
				GN_TREE,GN_NODE_CREATE,0,r->h.loop_no);
	}
	else {
		d = get_matrix_node_channel
				(&err,&cache->n,r->h.mtx,
				&dim_code[0],vm->view_channel[0],
				GN_TREE,GN_STEP_NODE_CREATE,0,r->h.loop_no);
	}
	if ( d == 0 ) {
		if ( nn && clr == 0 ) {
			matrix_node_channel_unlock(nn,0);
			unlock_node(nn,0);
		}
		memset(cache,0,sizeof(*cache));
		*lev_p = m->total_levels;
		return 0;
	}
	get_matrix_dh_set(&cache->ds,d);
	dim_code[0] = cache->n->dim_code[0];
	dim_code[1] = cache->n->dim_code[1];
	cache->tl_x = dim_code[1] & (mx=(-(1<<(dim_code[0]*m->dim_divide[0] + m->block_size[0]))));
	cache->br_x = cache->tl_x + (1<<(dim_code[0]*m->dim_divide[0] + m->block_size[0]));
	if ( cache->br_x > m->pixel_size[0] )
		cache->br_x = m->pixel_size[0];
	cache->level = dim_code[0];
	cache->ds_ptr = cache->ds.offset;
	cache->rect_logsize = m->block_size[0];
	*lev_p = dim_code[0];

end:
	xx = (x - cache->tl_x)>>(cache->level*m->dim_divide[0]);
	return cache->ds_ptr[xx];
}


void
vmt_draw_1dgraph_uint64(
	VIEW_METHOD * vm,
	GBVIEW_FLAME * gf,
	RESOURCE* r,
	DRAW_WORK * dw)
{
int i;
GB_POINT p;
REAL1 res;
INTEGER64 xx;
REAL1 dpm1,dpm2;
INTEGER64 level;
REAL1 d;
INTEGER64 lev;
INTEGER64 lev_cnt;

MTX_CACHE_UINT64 cache;

U_INTEGER64 cc;
INTEGER64 tim,tim2;

MATRIX * mtx;
REAL1 divide;
INTEGER64 int_divide;

WF_PARAM_1D * p1d;
PARAM_1DGRAPH_UINT * pg;

	if ( r->pr64.exit_lock )
		return;
	if ( get_mov_flag(gf,dw->wfid) != 0 )
		return;
	if ( r->h.mtx == 0 ) {
		gv_new_luster_option(r,0);
		if ( r->h.mtx == 0 )
			return;
	}
	mtx = r->h.mtx;
	if ( mtx->flags & MXF_ERR_DETECTED ) {
		r->h.mtx = 0;
		close_matrix(mtx);
		gv_new_luster_option(r,0);
		if ( r->h.mtx == 0 )
			return;
		mtx = r->h.mtx;
	}
	
	p1d = dw->work;
	pg = new_work_area_1d_uint(p1d,r);

	divide = 1;
	for ( i = 0 ; i < mtx->dim_divide[0] ; i ++ )
		divide *= 2;
	int_divide = mtx->dim_divide[0];

	dw->limit_reso = r->pr64.dpm;
	dw->limit_ptr = dw->pt_list[dw->size/2];
	r->h.loop_no ++;
	dw->flags |= WFF_LUSTER; 
	dpm1 = r->pr64.dpm;
	dpm2 = dpm1;
	level = 0;

	res = -1;
	for ( i = 0 ; i < dw->size ; i ++ ) {
		if ( res < dw->pt_reso[i] )
			res = dw->pt_reso[i];
	}

/*
	if ( res > dpm1*LIMIT_RESOLUTION_RATE )
		return;
*/

	if ( dpm2 < res || res < dpm2/divide ) {
		if ( res > dpm1 ) {
			dpm2 = dpm1;
		level = 0;
		}
		else {
			level = 0;
			d = -1;
			for ( dpm2 = dpm1 ;
				res <= dpm2 &&
				level < r->pr64.max_level;
				d = dpm2,
				dpm2 /= divide,
				level ++ );
			dpm2 = d;
			level --;
		}
	}

	lock_task(luster_lock);
	lev_cnt = 0;
	memset(&cache,0,sizeof(cache));
	lev = level;
	tim = get_xltime();
	for ( i = 0 ; i < dw->size ; i ++ ) {
		if ( (i % 1000) == 0 && wf_active_check_2(gf,dw->wfid) < 0 )
{ss_printf("FREE=RETURN-PIXEL\n");
			break;
}
		if ( r->pr64.exit_lock )
			break;
		tim2 = get_xltime();
		if ( tim != tim2 ) {
			if ( get_mov_flag(gf,dw->wfid) != 0 )
				break;
			tim = tim2;
		}
		p = dw->pt_list[i];
		if ( dw->pt_reso[i] < 0 )
			continue;
		xx = rint(p.x*r->pr64.dpm);
		if ( xx < 0 )
			continue;
		if ( xx >= mtx->pixel_size[0] )
			continue;
		if ( lev_cnt && cache.ds_ptr ) {
		int lx;
			if (		cache.tl_x <= xx && xx < cache.br_x ) {
				if ( cache.level == lev ) {

					lx = (xx>>(lev*int_divide))&((((INTEGER64)1)<<mtx->block_size[0])-1);
					if ( lx >= cache.ds.ix[0] ) {
						cc = 0;
					}
					else cc = cache.ds_ptr[lx];
				}
				else if ( cache.leaf ) {
					lx = (xx>>(cache.level*int_divide))&((((INTEGER64)1)<<mtx->block_size[0])-1);
					if ( lx >= cache.ds.ix[0] ) {
						cc = 0;
					}
					else cc = cache.ds_ptr[lx];
				}
				else	goto non_cache;
			}
			else goto non_cache;
			lev_cnt ++;
			if ( lev_cnt >= 100 )
				lev_cnt = 0;
		}
		else {
		non_cache:
			cc = get_mtx_1dgraph_uint64(vm,&cache,r,level,xx,&lev);
			if ( level != lev && cache.leaf == 0 )
				dw->data_request ++;
			lev_cnt ++;
		}
		set_y_data_uint(dw,vm,p1d,pg,i,cc);
		dw->flags |= WFF_DRAW;
	}

	if ( cache.n ) {
		matrix_node_channel_unlock(cache.n,0);
		unlock_node(cache.n,0);
	}
	
	unlock_task(luster_lock,"draw_pixel");

	wf_checkin_work_area(&pg->h.a);
}

/**********************/




typedef struct mtx_cache_double {
	int		tl_x;
	int		br_x;
	int		level;
	int		rect_logsize;
	MATRIX_DH_SET	ds;
	MATRIX_NODE *	n;
	double *	ds_ptr;
	unsigned	leaf;
} MTX_CACHE_DOUBLE;


double
get_mtx_1dgraph_double(VIEW_METHOD * vm,MTX_CACHE_DOUBLE * cache,RESOURCE * r,INTEGER64 lev,INTEGER64 x,INTEGER64 * lev_p)
{
int err;
void * d;
MATRIX * m;
MATRIX_NODE * nn;
INTEGER64 dim_code[2];
int xx;
INTEGER64 mx;
int clr;
	m = r->h.mtx;
	clr = 0;
	if ( cache->n ) {
		if ( cache->level == lev &&
			cache->tl_x <= x &&
			cache->br_x > x )
			goto end;
		clr = 1;
		matrix_node_channel_unlock(cache->n,0);
		unlock_node(cache->n,0);
	}
	nn = cache->n;
	
	dim_code[0] = lev;
	dim_code[1] = x;
//ss_printf("get_mtx_data-1\n");

	if ( r->h.mtx->p.flags & MPF_NODE_DIPENDENT ) {
		d = get_matrix_node_channel
				(&err,&cache->n,r->h.mtx,
				&dim_code[0],vm->view_channel[0],
				GN_TREE,GN_NODE_CREATE,0,r->h.loop_no);
	}
	else {
		d = get_matrix_node_channel
				(&err,&cache->n,r->h.mtx,
				&dim_code[0],vm->view_channel[0],
				GN_TREE,GN_STEP_NODE_CREATE,0,r->h.loop_no);
	}
	if ( d == 0 ) {
		if ( nn && clr == 0 ) {
			matrix_node_channel_unlock(nn,0);
			unlock_node(nn,0);
		}
		memset(cache,0,sizeof(*cache));
		*lev_p = m->total_levels;
		return 0;
	}
	get_matrix_dh_set(&cache->ds,d);
	dim_code[0] = cache->n->dim_code[0];
	dim_code[1] = cache->n->dim_code[1];
	cache->tl_x = dim_code[1] & (mx=(-(1<<(dim_code[0]*m->dim_divide[0] + m->block_size[0]))));
	cache->br_x = cache->tl_x + (1<<(dim_code[0]*m->dim_divide[0] + m->block_size[0]));
	if ( cache->br_x > m->pixel_size[0] )
		cache->br_x = m->pixel_size[0];
	cache->level = dim_code[0];
	cache->ds_ptr = cache->ds.offset;
	cache->rect_logsize = m->block_size[0];
	*lev_p = dim_code[0];

end:
	xx = (x - cache->tl_x)>>(cache->level*m->dim_divide[0]);
	return cache->ds_ptr[xx];
}


void
vmt_draw_1dgraph_double(
	VIEW_METHOD * vm,
	GBVIEW_FLAME * gf,
	RESOURCE* r,
	DRAW_WORK * dw)
{
int i;
GB_POINT p;
REAL1 res;
INTEGER64 xx;
REAL1 dpm1,dpm2;
INTEGER64 level;
REAL1 d;
INTEGER64 lev;
INTEGER64 lev_cnt;

MTX_CACHE_DOUBLE cache;

double cc;
INTEGER64 tim,tim2;

MATRIX * mtx;
REAL1 divide;
INTEGER64 int_divide;

WF_PARAM_1D * p1d;
PARAM_1DGRAPH_DOUBLE * pg;

	if ( r->pr64.exit_lock )
		return;
	if ( get_mov_flag(gf,dw->wfid) != 0 )
		return;
	if ( r->h.mtx == 0 ) {
		gv_new_luster_option(r,0);
		if ( r->h.mtx == 0 )
			return;
	}
	mtx = r->h.mtx;
	if ( mtx->flags & MXF_ERR_DETECTED ) {
		r->h.mtx = 0;
		close_matrix(mtx);
		gv_new_luster_option(r,0);
		if ( r->h.mtx == 0 )
			return;
		mtx = r->h.mtx;
	}
	
	p1d = dw->work;
	pg = new_work_area_1d_double(p1d,r);

	divide = 1;
	for ( i = 0 ; i < mtx->dim_divide[0] ; i ++ )
		divide *= 2;
	int_divide = mtx->dim_divide[0];

	dw->limit_reso = r->pr64.dpm;
	dw->limit_ptr = dw->pt_list[dw->size/2];
	r->h.loop_no ++;
	dw->flags |= WFF_LUSTER; 
	dpm1 = r->pr64.dpm;
	dpm2 = dpm1;
	level = 0;

	res = -1;
	for ( i = 0 ; i < dw->size ; i ++ ) {
		if ( res < dw->pt_reso[i] )
			res = dw->pt_reso[i];
	}

/*
	if ( res > dpm1*LIMIT_RESOLUTION_RATE )
		return;
*/

	if ( dpm2 < res || res < dpm2/divide ) {
		if ( res > dpm1 ) {
			dpm2 = dpm1;
		level = 0;
		}
		else {
			level = 0;
			d = -1;
			for ( dpm2 = dpm1 ;
				res <= dpm2 &&
				level < r->pr64.max_level;
				d = dpm2,
				dpm2 /= divide,
				level ++ );
			dpm2 = d;
			level --;
		}
	}

	lock_task(luster_lock);
	lev_cnt = 0;
	memset(&cache,0,sizeof(cache));
	lev = level;
	tim = get_xltime();
	for ( i = 0 ; i < dw->size ; i ++ ) {
		if ( (i % 1000) == 0 && wf_active_check_2(gf,dw->wfid) < 0 )
{ss_printf("FREE=RETURN-PIXEL\n");
			break;
}
		if ( r->pr64.exit_lock )
			break;
		tim2 = get_xltime();
		if ( tim != tim2 ) {
			if ( get_mov_flag(gf,dw->wfid) != 0 )
				break;
			tim = tim2;
		}
		p = dw->pt_list[i];
		if ( dw->pt_reso[i] < 0 )
			continue;
		xx = rint(p.x*r->pr64.dpm);
		if ( xx < 0 )
			continue;
		if ( xx >= mtx->pixel_size[0] )
			continue;
		if ( lev_cnt && cache.ds_ptr ) {
		int lx;
			if (		cache.tl_x <= xx && xx < cache.br_x ) {
				if ( cache.level == lev ) {

					lx = (xx>>(lev*int_divide))&((((INTEGER64)1)<<mtx->block_size[0])-1);
					if ( lx >= cache.ds.ix[0] ) {
						cc = 0;
					}
					else cc = cache.ds_ptr[lx];
				}
				else if ( cache.leaf ) {
					lx = (xx>>(cache.level*int_divide))&((((INTEGER64)1)<<mtx->block_size[0])-1);
					if ( lx >= cache.ds.ix[0] ) {
						cc = 0;
					}
					else cc = cache.ds_ptr[lx];
				}
				else	goto non_cache;
			}
			else goto non_cache;
			lev_cnt ++;
			if ( lev_cnt >= 100 )
				lev_cnt = 0;
		}
		else {
		non_cache:
			cc = get_mtx_1dgraph_double(vm,&cache,r,level,xx,&lev);
			if ( level != lev && cache.leaf == 0 )
				dw->data_request ++;
			lev_cnt ++;
		}
		set_y_data_double(dw,vm,p1d,pg,i,cc);
		dw->flags |= WFF_DRAW;
	}

	if ( cache.n ) {
		matrix_node_channel_unlock(cache.n,0);
		unlock_node(cache.n,0);
	}
	
	unlock_task(luster_lock,"draw_pixel");
/*
ss_printf("F %x %x %i %i %i\n",dw->flags,WFF_FLUSH,pg->h.get_max,pg->h.max_point.x,pg->h.min_point.x);
ss_printf("F %i %i %i\n",pg->next->h.get_max,pg->next->h.max_point.x,pg->next->h.min_point.x);
ss_printf("PITCH = %f %f\n",pg->h.pitch,pg->next->h.pitch);
*/
	wf_checkin_work_area(&pg->h.a);
}




/**********************/




typedef struct mtx_cache_float {
	int		tl_x;
	int		br_x;
	int		level;
	int		rect_logsize;
	MATRIX_DH_SET	ds;
	MATRIX_NODE *	n;
	float *		ds_ptr;
	unsigned	leaf;
} MTX_CACHE_FLOAT;


double
get_mtx_1dgraph_float(VIEW_METHOD * vm,MTX_CACHE_FLOAT * cache,RESOURCE * r,INTEGER64 lev,INTEGER64 x,INTEGER64 * lev_p)
{
int err;
void * d;
MATRIX * m;
MATRIX_NODE * nn;
INTEGER64 dim_code[2];
int xx;
INTEGER64 mx;
int clr;
	m = r->h.mtx;
	clr = 0;
	if ( cache->n ) {
		if ( cache->level == lev &&
			cache->tl_x <= x &&
			cache->br_x > x )
			goto end;
		clr = 1;
		matrix_node_channel_unlock(cache->n,0);
		unlock_node(cache->n,0);
	}
	nn = cache->n;
	
	dim_code[0] = lev;
	dim_code[1] = x;
//ss_printf("get_mtx_data-1\n");

	if ( r->h.mtx->p.flags & MPF_NODE_DIPENDENT ) {
		d = get_matrix_node_channel
				(&err,&cache->n,r->h.mtx,
				&dim_code[0],vm->view_channel[0],
				GN_TREE,GN_NODE_CREATE,0,r->h.loop_no);
	}
	else {
		d = get_matrix_node_channel
				(&err,&cache->n,r->h.mtx,
				&dim_code[0],vm->view_channel[0],
				GN_TREE,GN_STEP_NODE_CREATE,0,r->h.loop_no);
	}
	if ( d == 0 ) {
		if ( nn && clr == 0 ) {
			matrix_node_channel_unlock(nn,0);
			unlock_node(nn,0);
		}
		memset(cache,0,sizeof(*cache));
		*lev_p = m->total_levels;
		return C_TRANSPARENT;
	}
	get_matrix_dh_set(&cache->ds,d);
	dim_code[0] = cache->n->dim_code[0];
	dim_code[1] = cache->n->dim_code[1];
	cache->tl_x = dim_code[1] & (mx=(-(1<<(dim_code[0]*m->dim_divide[0] + m->block_size[0]))));
	cache->br_x = cache->tl_x + (1<<(dim_code[0]*m->dim_divide[0] + m->block_size[0]));
	if ( cache->br_x > m->pixel_size[0] )
		cache->br_x = m->pixel_size[0];
	cache->level = dim_code[0];
	cache->ds_ptr = cache->ds.offset;
	cache->rect_logsize = m->block_size[0];
	*lev_p = dim_code[0];

end:
	xx = (x - cache->tl_x)>>(cache->level*m->dim_divide[0]);
	return cache->ds_ptr[xx];
}


void
vmt_draw_1dgraph_float(
	VIEW_METHOD * vm,
	GBVIEW_FLAME * gf,
	RESOURCE* r,
	DRAW_WORK * dw)
{
int i;
GB_POINT p;
REAL1 res;
INTEGER64 xx;
REAL1 dpm1,dpm2;
INTEGER64 level;
REAL1 d;
INTEGER64 lev;
INTEGER64 lev_cnt;

MTX_CACHE_FLOAT cache;

double cc;
INTEGER64 tim,tim2;

MATRIX * mtx;
REAL1 divide;
INTEGER64 int_divide;

WF_PARAM_1D * p1d;
PARAM_1DGRAPH_DOUBLE * pg;

	if ( r->pr64.exit_lock )
		return;
	if ( get_mov_flag(gf,dw->wfid) != 0 )
		return;
	if ( r->h.mtx == 0 ) {
		gv_new_luster_option(r,0);
		if ( r->h.mtx == 0 )
			return;
	}
	mtx = r->h.mtx;
	if ( mtx->flags & MXF_ERR_DETECTED ) {
		r->h.mtx = 0;
		close_matrix(mtx);
		gv_new_luster_option(r,0);
		if ( r->h.mtx == 0 )
			return;
		mtx = r->h.mtx;
	}
	
	p1d = dw->work;
	pg = new_work_area_1d_double(p1d,r);

	divide = 1;
	for ( i = 0 ; i < mtx->dim_divide[0] ; i ++ )
		divide *= 2;
	int_divide = mtx->dim_divide[0];

	dw->limit_reso = r->pr64.dpm;
	dw->limit_ptr = dw->pt_list[dw->size/2];
	r->h.loop_no ++;
	dw->flags |= WFF_LUSTER; 
	dpm1 = r->pr64.dpm;
	dpm2 = dpm1;
	level = 0;

	res = -1;
	for ( i = 0 ; i < dw->size ; i ++ ) {
		if ( res < dw->pt_reso[i] )
			res = dw->pt_reso[i];
	}

/*
	if ( res > dpm1*LIMIT_RESOLUTION_RATE )
		return;
*/

	if ( dpm2 < res || res < dpm2/divide ) {
		if ( res > dpm1 ) {
			dpm2 = dpm1;
		level = 0;
		}
		else {
			level = 0;
			d = -1;
			for ( dpm2 = dpm1 ;
				res <= dpm2 &&
				level < r->pr64.max_level;
				d = dpm2,
				dpm2 /= divide,
				level ++ );
			dpm2 = d;
			level --;
		}
	}

	lock_task(luster_lock);
	lev_cnt = 0;
	memset(&cache,0,sizeof(cache));
	lev = level;
	tim = get_xltime();
	for ( i = 0 ; i < dw->size ; i ++ ) {
		if ( (i % 1000) == 0 && wf_active_check_2(gf,dw->wfid) < 0 )
{ss_printf("FREE=RETURN-PIXEL\n");
			break;
}
		if ( r->pr64.exit_lock )
			break;
		tim2 = get_xltime();
		if ( tim != tim2 ) {
			if ( get_mov_flag(gf,dw->wfid) != 0 )
				break;
			tim = tim2;
		}
		p = dw->pt_list[i];
		if ( dw->pt_reso[i] < 0 )
			continue;
		xx = rint(p.x*r->pr64.dpm);
		if ( xx < 0 )
			continue;
		if ( xx >= mtx->pixel_size[0] )
			continue;
		if ( lev_cnt && cache.ds_ptr ) {
		int lx;
			if (		cache.tl_x <= xx && xx < cache.br_x ) {
				if ( cache.level == lev ) {

					lx = (xx>>(lev*int_divide))&((((INTEGER64)1)<<mtx->block_size[0])-1);
					if ( lx >= cache.ds.ix[0] ) {
						cc = 0;
					}
					else cc = cache.ds_ptr[lx];
				}
				else if ( cache.leaf ) {
					lx = (xx>>(cache.level*int_divide))&((((INTEGER64)1)<<mtx->block_size[0])-1);
					if ( lx >= cache.ds.ix[0] ) {
						cc = 0;
					}
					else cc = cache.ds_ptr[lx];
				}
				else	goto non_cache;
			}
			else goto non_cache;
			lev_cnt ++;
			if ( lev_cnt >= 100 )
				lev_cnt = 0;
		}
		else {
		non_cache:
			cc = get_mtx_1dgraph_float(vm,&cache,r,level,xx,&lev);
			if ( level != lev && cache.leaf == 0 )
				dw->data_request ++;
			lev_cnt ++;
		}
		set_y_data_double(dw,vm,p1d,pg,i,cc);
		dw->flags |= WFF_DRAW;
	}

	if ( cache.n ) {
		matrix_node_channel_unlock(cache.n,0);
		unlock_node(cache.n,0);
	}
	
	unlock_task(luster_lock,"draw_pixel");


	wf_checkin_work_area(&pg->h.a);
}





