/*
 * Grain Core - A XForms processor for mobile terminals.
 * Copyright (C) 2005-2006 HAW International Inc.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 * Created on 2006/05/19 9:38:14
 * 
 */
package jp.grain.spike.xpath;

import java.util.Stack;
import java.util.Vector;

import jp.grain.spike.Attribute;
import jp.grain.spike.DefaultElement;
import jp.grain.spike.Document;
import jp.grain.spike.Element;
import jp.grain.spike.Node;

public class XPathEvaluator {

    public static final String AXIS_ABSOLUTE = "absolute";
    public static final String AXIS_CHILD = "child";
    public static final String AXIS_ATTRIBUTE = "attribute";
    public static final String AXIS_PARENT = "parent";
    public static final String AXIS_ANCESTOR = "ancestor";
    public static final String AXIS_DESCENDANT = "descendant";
    public static final String AXIS_DESCENDANT_OR_SELF = "descendant-or-self";
    public static final String AXIS_ANCESTOR_OR_SELF = "ancestor-or-self";
    public static final String AXIS_SELF = "self";
    public static final String AXIS_FOLLOWING_SIBLING = "following-sibling";
    public static final String AXIS_PRECEDING_SIBLING = "preceding-sibling";
    
    public static final String OP_PUSH = "push";
    public static final String OP_CLSPUSH = "clspush";
    
    public static final String OP_CTXNLOAD = "ctxnload";
    public static final String OP_CTXPLOAD = "ctxpload";
    
    public static final String OP_STEP = "step";
    public static final String OP_FILTER = "filter";
    public static final String OP_NFILTER = "nfilter";

    public static final String OP_ADD = "add";
    public static final String OP_SUB = "sub";
    public static final String OP_DIV = "div";
    public static final String OP_MUL = "mul";
    public static final String OP_MOD = "mod";
    public static final String OP_NEG = "neg";

    public static final String OP_EQ = "eq";
    public static final String OP_NE = "ne";
    public static final String OP_LT = "lt";
    public static final String OP_LE = "le";
    public static final String OP_GT = "gt";
    public static final String OP_GE = "ge";

    public static final String OP_AND = "and";
    public static final String OP_OR = "or";
    
    public static final String NODETYPE_NODE = "node";
    public static final String NODETYPE_TEXT = "text";
    
    public static XPathEvaluator.Result evaluate(Node context, XPathExpr expr) {
        Frame frame = new Frame(context);
        frame.init(expr);
        exec(frame);
        Result result = new Result(frame._os.peek());
        return result;
    }
    
    private static void exec(Frame f) {
    	try {
        for (f._oc = 0; f._oc < f._operands.length; f._oc++) {
            Object[] op = (Object[])f._operands[f._oc];
            String opname = (String)op[0];
//            System.out.print("*** op = " + opname + " [ ");
//            for (int i = 1; i < op.length; ++i) {
//                if (i == 1) {
//                    System.out.print(number(op[i]));
//                } else {
//                    System.out.print(", " + number(op[i]));
//                }
//            }
//            System.out.println(" ]");
            
            if (OP_PUSH.equals(opname)) {
                f._os.push(op[1]);
                
            } else if (OP_CTXNLOAD.equals(opname)) {
                Node[] param = {f._ctxnode}; 
                f._os.push(param);
            
            } else if (OP_CTXPLOAD.equals(opname)) {
                int index = number(op[1]);
                f.pushInt(f._ctxparam[index]);

            } else if (OP_STEP.equals(opname)) {
                Node[] nodeset = (Node[])f._os.pop();
                System.out.println("STEP: nodeset");
                String axis = (String)f._os.pop();
                System.out.println("STEP: axis");
                f.initNodeset();
                System.out.println("STEP: f.initNodeset()");
                NodeTester nc = new NodeTester(f._nodeset, (String)op[1], (String)op[2]);
                System.out.println("STEP: created NodeTester");
                step(nodeset, axis, nc);
                System.out.println("STEP: step(nodeset, axis, nc)");
                f.pushNodeset();
                System.out.println("STEP: f.pushNodeset()");
                
            } else if (OP_FILTER.equals(opname)) {
                Node[] nodeset = (Node[])f._os.pop();
                Object[] closure = (Object[])f._os.pop();
                f.initNodeset();
                for (int i = 0; i < nodeset.length; ++i) {
                    System.out.println("*** node=" + i);
                    Frame cf = new Frame(nodeset[i], nodeset.length, i + 1);
                    cf.init(closure);
                    exec(cf);
                    Object result = cf._os.peek();
                    System.out.println("*** result=" + result);
                    if (result instanceof Integer) {
                        int index = ((Integer)result).intValue();
                        if (index != i + 1) continue;  
                    } else {
                        if (!bool(result)) continue;
                    }
                    f._nodeset.addElement(nodeset[i]);
                }
                f.pushNodeset();
                
            } else if (OP_NFILTER.equals(opname)) {
                Node[] nodeset = (Node[])f._os.pop();
                int index = number(f._os.pop());
                if (index == 0) index = nodeset.length;
                if (index < 1 || index > nodeset.length) {
                    f._os.push(new Node[0]);
                } else {
                    Node[] filtered = { nodeset[index - 1] };
                    f._os.push(filtered);
                }                
                
            } else if (OP_ADD.equals(opname)) {
                f.binaryNumber();
                f.pushInt(f._number[0] + f._number[1]);
            } else if (OP_SUB.equals(opname)) {
                f.binaryNumber();
                f.pushInt(f._number[0] - f._number[1]);
            } else if (OP_MUL.equals(opname)) {
                f.binaryNumber();
                f.pushInt(f._number[0] * f._number[1]);
            } else if (OP_DIV.equals(opname)) {
                f.binaryNumber();
                f.pushInt(f._number[0] / f._number[1]);
            } else if (OP_MOD.equals(opname)) {
                f.binaryNumber();
                f.pushInt(f._number[0] % f._number[1]);
            } else if (OP_NEG.equals(opname)) {
                f.unaryNumber();
                f.pushInt(-f._number[0]);
            
            } else if (OP_LT.equals(opname)) {
                f.binaryNumber();
                f.pushBoolean(f._number[0] < f._number[1]);
            } else if (OP_LE.equals(opname)) {
                f.binaryNumber();
                f.pushBoolean(f._number[0] <= f._number[1]);
            } else if (OP_GT.equals(opname)) {
                f.binaryNumber();
                f.pushBoolean(f._number[0] > f._number[1]);
            } else if (OP_GE.equals(opname)) {
                f.binaryNumber();
                f.pushBoolean(f._number[0] >= f._number[1]);
            
            } else if (OP_EQ.equals(opname)) {
                f.binaryNumber();
                f.pushBoolean(f._number[0] == f._number[1]);
            } else if (OP_NE.equals(opname)) {
                f.binaryNumber();
                f.pushBoolean(f._number[0] != f._number[1]);

            } else if (OP_AND.equals(opname)) {
                f.binaryBoolean();
                f.pushBoolean(f._bool[0] && f._bool[1]);
            } else if (OP_OR.equals(opname)) {
                f.binaryBoolean();
                f.pushBoolean(f._bool[0] || f._bool[1]);
            }
        }
    	} catch (Exception e) {
    		throw new RuntimeException("operand="+((Object[])f._operands[f._oc])[0]+": "+e.toString());
    	}
    }

    private static Node[] nodeset(Object obj) {
    	if(obj instanceof Node[]) {
    		return (Node[])obj;
    	}
    	
    	DefaultElement element = new DefaultElement("tmp");
    	element.setText(String.valueOf(obj));
    	
    	return new Node[]{element};
    }
    
    private static int number(Object obj) {
        if (obj instanceof Node[]) {
            obj = string(obj);
        }
        if (obj instanceof Integer) {
            return ((Integer)obj).intValue();
        } else if (obj instanceof String) {
            try {
                return Integer.parseInt(((String)obj).trim());
            } catch (NumberFormatException e) {
            }
        } else if (obj instanceof Boolean) {
            return ((Boolean)obj).booleanValue() ? 1 : 0;
        }
        return 0;
    }
    
    private static String string(Object obj) {
        if (obj instanceof Node[]) {
            Node[] nodeset = ((Node[])obj);
            if (nodeset.length == 0) return "";
            return nodeset[0].getSimpleContent();            
        }
        return String.valueOf(obj);
    }
    
    private static boolean bool(Object obj) {
        if (obj instanceof Integer) {
            if (((Integer)obj).intValue() == 0) return false;
        } else if (obj instanceof Node[]) {
            if (((Node[])obj).length == 0) return false;
        } else if (obj instanceof String) {
            if (((String)obj).length() == 0) return false;
        } else if (obj instanceof Boolean) {
            return ((Boolean)obj).booleanValue();
        }
        return true;
    }
    
    private static void step(Node[] nodeset, String axis, NodeTester nc) {
        for (int i = 0; i < nodeset.length; ++i) {
            Node node = nodeset[i];
            if (AXIS_CHILD.equals(axis)) {
                axisDescendant(node, nc, true);
            } else if (AXIS_DESCENDANT.equals(axis)) {
                axisDescendant(node, nc, false);
            } else if (AXIS_DESCENDANT_OR_SELF.equals(axis)) {
                nc.testAndRecord(node);
                axisDescendant(node, nc, false);
            } else if (AXIS_ABSOLUTE.equals(axis)) {
                nc.testAndRecord(node.getDocument());
            } else if (AXIS_ATTRIBUTE.equals(axis)) {
            	Attribute[] attrs = ((Element)node).getAttributes();
            	for(int j=0; j<attrs.length; j++) {
            		nc.testAndRecord(attrs[j]);
            	}
            } else if (AXIS_PARENT.equals(axis)) {
                axisAncestor(node, nc, true);
            } else if (AXIS_ANCESTOR.equals(axis)) {
                axisAncestor(node, nc, false);
            } else if (AXIS_ANCESTOR_OR_SELF.equals(axis)) {
                nc.testAndRecord(node);
                axisAncestor(node, nc, false);
            } else if (AXIS_SELF.equals(axis)) {
                nc.testAndRecord(node);
            } else if (AXIS_FOLLOWING_SIBLING.equals(axis)) {
                axisFollowingSibling(node, nc);
            } else if (AXIS_PRECEDING_SIBLING.equals(axis)) {
                axisPrecedingSibling(node, nc);
            } 
        }        
    }

    private static void axisAncestor(Node node, NodeTester nc, boolean directOnly) {
        if (node instanceof Element) {
            Node parent = ((Element)node).getParent();
            nc.testAndRecord(parent);
            if (!directOnly) axisAncestor(parent, nc, false);
        }
    }
    
    private static void axisDescendant(Node node, NodeTester nc, boolean directOnly) {
        if (node instanceof Element) {
            Element e = (Element)node;
            for (int i = 0; i < e.getChildCount(); ++i) {
                Object child = e.getChild(i);
                nc.testAndRecord(child);
                if (!directOnly && child instanceof Node) {
                    axisDescendant((Node)child, nc, false);
                }
            }
        } else if (node instanceof Document) {
            Element root = ((Document)node).getRootElement();
            nc.testAndRecord(root);
            if (!directOnly) axisDescendant(root, nc, false);
        }
    }

    private static void axisFollowingSibling(Node node, NodeTester nc) {
        Element parent = node.getParentElement();
        boolean following = false;
        for (int i = 0; i < parent.getChildCount(); ++i) {
            Object sibling = parent.getChild(i);
            if (sibling == node) {
                following = true;
            } else {
                if (following) nc.testAndRecord(sibling);
            }
        }
    }
    
    private static void axisPrecedingSibling(Node node, NodeTester nc) {
        Element parent = node.getParentElement();
        boolean preceding = false;
        for (int i = parent.getChildCount() - 1; i >= 0; --i) {
            Object sibling = parent.getChild(i);
            if (sibling == node) {
                preceding = true;
            } else {
                if (preceding) nc.testAndRecord(sibling);
            }
        }
    }
    
    static class Frame {
        
        private Node _ctxnode;
        private int[] _ctxparam = new int[2];
        private Object[] _operands;
        private Stack _os = new Stack();
        private int _oc; // operand counter
        private Vector _nodeset; // nodeset work area
        private int[] _number = new int[2];
        private boolean[] _bool = new boolean[2];

        Frame(Node context) {
            _ctxnode = context;
            _ctxparam[0] = 1;
            _ctxparam[1] = 1;
        }

        Frame(Node context, int contextSize, int contextPos) {
            _ctxnode = context;
            _ctxparam[0] = contextSize;
            _ctxparam[1] = contextPos;
        }        
        
        void init(XPathExpr expr) {
            _operands = expr.getOperands();
        }
        
        void init(Object[] closure) {
            _operands = closure;
        }
        
        void initNodeset() {
            if (_nodeset == null) {
                _nodeset = new Vector();
            }
            _nodeset.removeAllElements();
        }
        
        void binaryNumber() {
            _number[1] = number(_os.pop());
            _number[0] = number(_os.pop());
        }
        
        void unaryNumber() {
            _number[0] = number(_os.pop());
        }

        void binaryBoolean() {
            _bool[1] = bool(_os.pop());
            _bool[0] = bool(_os.pop());
        }        
        
        void pushInt(int number) {
            _os.push(new Integer(number));
        }

        void pushBoolean(boolean b) {
            _os.push(new Boolean(b));
        }
        
        void pushNodeset() {
            Node[] ns = new Node[_nodeset.size()];
            _nodeset.copyInto(ns);
            _os.push(ns);            
        }

    }
    
    static public class Result {
    	
    	private Object result;
    	
    	Result(Object obj) {
    		this.result = obj;
    	}
    	
    	public String string() {
    		return XPathEvaluator.string(result);
    	}
    	
    	public int number() {
    		return XPathEvaluator.number(result);
    	}

    	public boolean bool() {
    		return XPathEvaluator.bool(result);
    	}

    	public Node[] nodeset() {
    		return XPathEvaluator.nodeset(result);
    	}
    }
    
    static class NodeTester {
        
        private Vector _nodes;
        private String _name;
        private String _type;
        
        NodeTester(Vector nodes, String name, String type) {
            _nodes = nodes;
            _name = name;
            _type = type;
        }
        
        void testAndRecord(Object obj) {
            if (match(obj)) _nodes.addElement(obj);
        }
                
        boolean match(Object obj) {
            if (_type != null) {
                if (NODETYPE_NODE.equals(_type)) {
                    return true;
                } else if (NODETYPE_TEXT.equals(_type) && obj instanceof String) {
                    return true;
                }
            } else {
                if (obj instanceof Element) {
                    if (_name == null) return true;
                    if (_name.equals(((Element)obj).getName())) return true;
                } else if (obj instanceof Attribute) {
                    if (_name == null) return true;
                    if (_name.equals(((Attribute)obj)._name)) return true;
                } else if (obj instanceof Document) {
                    if (_name == null) return true;
                }
            }
        	System.out.println("NodeTester[_name="+_name+": _type="+_type+": obj="+obj+"]");
            return false;
        }
        
    }
}
