/*
 * Grain Core - A XForms processor for mobile terminals.
 * Copyright (C) 2005 HAW International Inc.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 * Created on 2005/06/16 14:11:18
 * 
 */
package jp.grain.spike.xpath;

import j2meunit.framework.Test;
import j2meunit.framework.TestCase;
import j2meunit.framework.TestMethod;
import j2meunit.framework.TestSuite;

import java.io.IOException;

import jp.grain.spike.DefaultElement;
import jp.grain.spike.Document;
import jp.grain.spike.Element;
import jp.grain.spike.Node;
//import jp.grain.spike.Text;

/**
 * BinaryXMLParserTest
 * @version $Id: BinaryXMLSerializerTest.java 242 2006-04-09 08:19:37Z go $
 * @author Go Takahashi
 */
public class XPathEvaluatorTest extends TestCase {
	
    private Document doc;
    
    public XPathEvaluatorTest() {
    }

    public XPathEvaluatorTest(String name, TestMethod method) {
        super(name, method);
        
        // <haw type="now" xforms:unit="yen">
        //   <xforms:output ref="test">abc</xforms:output>
        //   <xforms:output ref="x" xmlns=""/>
        //   <xforms:model>
        //     <scf>
        //       <d>test</d>word
        //     </scf>
        //     5
        //   </xforms:model>
        //   1
        // </haw>    
        
        this.doc = new Document();
        Element haw = new DefaultElement("haw", "", "http://haw.co.jp/haw");
        haw.addPrefixMapping("xforms","http://www.w3.org/2002/xforms");
        haw.addPrefixMapping(null,"http://haw.co.jp/haw");
        haw.addAttribute("type", "now");
        haw.addAttribute("unit", "xforms", "http://www.w3.org/2002/xforms", "yen");
        this.doc.setRootElement(haw);

        Element output=new DefaultElement("output", "xforms", "http://www.w3.org/2002/xforms");
        output.addAttribute("ref","test");
        haw.addChild(output);
        
        output.addChildText("abc");
               
        Element output2 = new DefaultElement("output", "xforms", "http://www.w3.org/2002/xforms");
        output2.addAttribute("ref", "x");
        haw.addChild(output2);
            
        Element model = new DefaultElement("model", "xforms", "http://www.w3.org/2002/xforms");
        model.addPrefixMapping(null, "");
        haw.addChild(model);
        haw.addChildText("1");
        
        Element scf = new DefaultElement("scf");
        model.addChild(scf);
        model.addChildText("5");
        
        Element d = new DefaultElement("d");
        scf.addChild(d);
        scf.addChildText("word");
        
        d.addChildText("test");
    }
    
    public void testChild() {

        Element ctxnode = this.doc.getRootElement().getChildElement(2); // xforms:model
        assertEquals("ctxnode name", "model", ctxnode.getName());
        
        // ( scf ) 
        XPathExpr expr = new XPathExpr();
        addOperand(expr, "push", "child") ;
        addOperand(expr, "ctxnload");
        addOperand(expr, "step", "scf", null);
        
        String result = expr.evaluate(ctxnode).string();
        assertEquals("check result", "word", result);
    } 
    
    public void testParent() {

        Element ctxnode = this.doc.getRootElement().getChildElement(2); // xforms:model
        assertEquals("ctxnode name", "model", ctxnode.getName());
        
        // ( parent::*/model ) 
        XPathExpr expr = new XPathExpr();
        addOperand(expr, "push", "child");
        addOperand(expr, "push", "parent");
        addOperand(expr, "ctxnload");
        addOperand(expr, "step", null, null);
        addOperand(expr, "step", "model", null);
        
        String result = expr.evaluate(ctxnode).string();
        assertEquals("check result", "5", result);
    } 
    
    public void testAncestor() {
        Element ctxnode = this.doc.getRootElement().getChildElement(2).getChildElement(0).getChildElement(0); // d
        assertEquals("ctxnode name", "d", ctxnode.getName());
        
        // ( ancestor::haw ) 
        XPathExpr expr = new XPathExpr();
        addOperand(expr, "push", "ancestor");
        addOperand(expr, "ctxnload");
        addOperand(expr, "step", "haw", null);
        
        String result = expr.evaluate(ctxnode).string();
        assertEquals("check result", "1", result);
    }

    public void testDescendant() {
        Element ctxnode = this.doc.getRootElement().getChildElement(2); // xforms:model
        assertEquals("ctxnode name", "model", ctxnode.getName());
        
        // ( descendant::d ) 
        XPathExpr expr = new XPathExpr();
        addOperand(expr, "push", "descendant");
        addOperand(expr, "ctxnload");
        addOperand(expr, "step", "d", null);
        
        String result = expr.evaluate(ctxnode).string();
        assertEquals("check result", "test", result);
    }    
    
    public void testFollowingSibling() {
        Element ctxnode = this.doc.getRootElement().getChildElement(1); // xforms:output
        assertEquals("ctxnode name", "output", ctxnode.getName());
        
        // ( following-sibling::* ) 
        XPathExpr expr = new XPathExpr();
        addOperand(expr, "push", "following-sibling");
        addOperand(expr, "ctxnload");
        addOperand(expr, "step", null, null);
        
        String result = expr.evaluate(ctxnode).string();
        assertEquals("check result", "5", result);
    }   

    public void testPrecedingSibling() {
        Element ctxnode = this.doc.getRootElement().getChildElement(1); // xforms:output
        assertEquals("ctxnode name", "output", ctxnode.getName());
        
        // ( preceding-sibling::* ) 
        XPathExpr expr = new XPathExpr();
        addOperand(expr, "push", "preceding-sibling");
        addOperand(expr, "ctxnload");
        addOperand(expr, "step", null, null);
        
        String result = expr.evaluate(ctxnode).string();
        assertEquals("check result", "abc", result);
    }   
    
    public void testGrandchild() {

        Element ctxnode = this.doc.getRootElement().getChildElement(2); // xforms:model
        assertEquals("ctxnode name", "model", ctxnode.getName());
        
        // ( scf/d )
        XPathExpr expr = new XPathExpr();
        addOperand(expr, "push", "child");
        addOperand(expr, "push", "child");
        addOperand(expr, "ctxnload");
        addOperand(expr, "step", "scf", null);
        addOperand(expr, "step", "d", null);
        
        String result = expr.evaluate(ctxnode).string();
        assertEquals("check result", "test", result);
    } 
    
    public void testAbsolute() {

        Element ctxnode = this.doc.getRootElement().getChildElement(2); // xforms:model
        assertEquals("ctxnode name", "model", ctxnode.getName());
        
        // ( /haw/model/scf/d )
        XPathExpr expr = new XPathExpr();
        addOperand(expr, "push", "child");
        addOperand(expr, "push", "child");
        addOperand(expr, "push", "child");
        addOperand(expr, "push", "child");
        addOperand(expr, "push", "absolute");
        addOperand(expr, "ctxnload");
        addOperand(expr, "step", null, null);        
        addOperand(expr, "step", "haw", null);
        addOperand(expr, "step", "model", null);
        addOperand(expr, "step", "scf", null);
        addOperand(expr, "step", "d", null);
        
        String result = expr.evaluate(ctxnode).string();
        assertEquals("check result", "test", result);
    }
    
    public void testAttribute() {

        Element ctxnode = this.doc.getRootElement().getChildElement(2); // xforms:model
        assertEquals("ctxnode name", "model", ctxnode.getName());
        
        // ( /haw/model/@ref )
        XPathExpr expr = new XPathExpr();
        addOperand(expr, "push", "attribute");
        addOperand(expr, "push", "child");
        addOperand(expr, "push", "child");
        addOperand(expr, "push", "absolute");
        addOperand(expr, "ctxnload");
        addOperand(expr, "step", null, null);        
        addOperand(expr, "step", "haw", null);
        addOperand(expr, "step", "model", null);
        addOperand(expr, "step", "ref", null);
        
        String result = expr.evaluate(ctxnode).string();
        assertEquals("check result", "test", result);
    } 

    public void testCalculate() {

        Element ctxnode = this.doc.getRootElement().getChildElement(2); // xforms:model
        assertEquals("ctxnode name", "model", ctxnode.getName());
        assertEquals("ctxnode simpleContent", "5", ctxnode.getSimpleContent());
        
        // ( 3 + 4 * 5 div 2 mod 4 ) -> -(- (+ 3 (mod (div (* 4 5) 2) 4) 4))
        XPathExpr expr = new XPathExpr();
        addOperand(expr, "push", new Integer(3));
        addOperand(expr, "push", new Integer(4));
        Node[] ns = {ctxnode};
        addOperand(expr, "push", ns);
        addOperand(expr, "mul");
        addOperand(expr, "push", new Integer(2));
        addOperand(expr, "div");
        addOperand(expr, "push", new Integer(4));
        addOperand(expr, "mod");
        addOperand(expr, "add");
        addOperand(expr, "push", "4");
        addOperand(expr, "sub");
        addOperand(expr, "neg");
        
        
        String result = expr.evaluate(ctxnode).string();
        assertEquals("check result", "-1", result);
    } 
        
    public void testCompare() {

        Element ctxnode = this.doc.getRootElement().getChildElement(2); // xforms:model
        assertEquals("ctxnode name", "model", ctxnode.getName());
        
        // ( 5 < 6 and 4 > 3 and 5 = 5 or 3 != 8 and 4 <= 4 and 5 >= 7 ) 
        // (or (and (and (< 5 6) (> 4 3)) (= 5 5)) (and (and (!= 3 8) (<= 4 4)) (>= 5 7)))
        //  (or (and (and     t      t)      t ) (and  (and        t        t)      f   ))
        //  (or (and  t t) (and t f))
        //  (or  t  f)
        //  t
        XPathExpr expr = new XPathExpr();
        addOperand(expr, "push", new Integer(5));
        addOperand(expr, "push", "6");
        addOperand(expr, "lt");
        addOperand(expr, "push", "4");
        addOperand(expr, "push", "3");
        addOperand(expr, "gt");
        addOperand(expr, "and");
        Node[] ns = {ctxnode};
        addOperand(expr, "push", ns);
        addOperand(expr, "push", new Integer(5));
        addOperand(expr, "eq");
        addOperand(expr, "and");

        addOperand(expr, "push", "3");
        addOperand(expr, "push", new Integer(8));
        addOperand(expr, "ne");
        addOperand(expr, "push", new Integer(4));
        addOperand(expr, "push", new Integer(4));
        addOperand(expr, "le");
        addOperand(expr, "and");
        
        addOperand(expr, "push", "5");
        addOperand(expr, "push", new Integer(7));
        addOperand(expr, "ge");        
        addOperand(expr, "and");
        addOperand(expr, "or");        
        
        String result = expr.evaluate(ctxnode).string();
        assertEquals("check result", "true", result);
    } 
    
    public void testNumberedFilter() {

        Element ctxnode = this.doc.getRootElement().getChildElement(2); // xforms:model
        assertEquals("ctxnode name", "model", ctxnode.getName());
        
        // ( ../*[3] ) 
        XPathExpr expr = new XPathExpr();
        addOperand(expr, "push", new Integer(3));        
        addOperand(expr, "push", "child");
        addOperand(expr, "push", "parent");
        addOperand(expr, "ctxnload");
        addOperand(expr, "step", null, null);
        addOperand(expr, "step", null, null);
        addOperand(expr, "nfilter");
        
        String result = expr.evaluate(ctxnode).string();
        assertEquals("check result", "5", result);
    } 
    
    public void testFilter() {
        try {
        Element ctxnode = this.doc.getRootElement().getChildElement(2); // xforms:model
        assertEquals("ctxnode name", "model", ctxnode.getName());
        
        // ( ../*[position() = last() - 1] ) 
        Object[] closure = new Object[5];
        closure[0] = XPathExpr.createOperand("ctxpload", "1");
        closure[1] = XPathExpr.createOperand("ctxpload", "0");
        closure[2] = XPathExpr.createOperand("push", new Integer(1));
        closure[3] = XPathExpr.createOperand("sub");
        closure[4] = XPathExpr.createOperand("eq");
        
        XPathExpr expr = new XPathExpr();
        addOperand(expr, "push", closure);        
        addOperand(expr, "push", "child");
        addOperand(expr, "push", "parent");
        addOperand(expr, "ctxnload");
        addOperand(expr, "step", null, null);
        addOperand(expr, "step", null, null);
        addOperand(expr, "filter");
        
        String result = expr.evaluate(ctxnode).string();
        assertEquals("check result", "", result);
        } catch (Exception e) {
            e.printStackTrace();
        }
    }
    
    private void addOperand(XPathExpr expr, String op) {
        expr.addOperand(XPathExpr.createOperand(op));
    }
    
    private void addOperand(XPathExpr expr, String op, Object param0) {
        expr.addOperand(XPathExpr.createOperand(op, param0));
    }
    
    private void addOperand(XPathExpr expr, String op, Object param0, Object param1) {
        expr.addOperand(XPathExpr.createOperand(op, param0, param1));
    }

    private void addOperand(XPathExpr expr, String op, Object param0, Object param1, Object param2) {
        expr.addOperand(XPathExpr.createOperand(op, param0, param1, param2));
    }

    public Test suite() {
        TestSuite suite = new TestSuite();
        suite.addTest(new XPathEvaluatorTest("testXPath", new TestMethod() {
            public void run(TestCase tc) throws IOException{ ((XPathEvaluatorTest)tc).testChild(); }
        }));
        suite.addTest(new XPathEvaluatorTest("testGrandchild", new TestMethod() {
            public void run(TestCase tc) throws IOException{ ((XPathEvaluatorTest)tc).testGrandchild(); }
        }));
        suite.addTest(new XPathEvaluatorTest("testAbsolute", new TestMethod() {
            public void run(TestCase tc) throws IOException{ ((XPathEvaluatorTest)tc).testAbsolute(); }
        }));
        suite.addTest(new XPathEvaluatorTest("testParent", new TestMethod() {
            public void run(TestCase tc) throws IOException{ ((XPathEvaluatorTest)tc).testParent(); }
        }));
        suite.addTest(new XPathEvaluatorTest("testAncestor", new TestMethod() {
            public void run(TestCase tc) throws IOException{ ((XPathEvaluatorTest)tc).testAncestor(); }
        }));
        suite.addTest(new XPathEvaluatorTest("testFollowingSibling", new TestMethod() {
            public void run(TestCase tc) throws IOException{ ((XPathEvaluatorTest)tc).testFollowingSibling(); }
        }));
        suite.addTest(new XPathEvaluatorTest("testPrecedingSibling", new TestMethod() {
            public void run(TestCase tc) throws IOException{ ((XPathEvaluatorTest)tc).testPrecedingSibling(); }
        }));
        suite.addTest(new XPathEvaluatorTest("testDescendant", new TestMethod() {
            public void run(TestCase tc) throws IOException{ ((XPathEvaluatorTest)tc).testDescendant(); }
        }));
        suite.addTest(new XPathEvaluatorTest("testCalculate", new TestMethod() {
            public void run(TestCase tc) throws IOException{ ((XPathEvaluatorTest)tc).testCalculate(); }
        }));
        suite.addTest(new XPathEvaluatorTest("testCompare", new TestMethod() {
            public void run(TestCase tc) throws IOException{ ((XPathEvaluatorTest)tc).testCompare(); }
        }));
        suite.addTest(new XPathEvaluatorTest("testNumberedFilter", new TestMethod() {
            public void run(TestCase tc) throws IOException{ ((XPathEvaluatorTest)tc).testNumberedFilter(); }
        }));
        suite.addTest(new XPathEvaluatorTest("testFilter", new TestMethod() {
            public void run(TestCase tc) throws IOException{ ((XPathEvaluatorTest)tc).testFilter(); }
        }));
        return suite;
    }
    
}