
#==============================================================================#
# $Id: marker.rb,v 1.2 2002/09/18 15:43:59 yuya Exp $
#==============================================================================#

#====================================================================#

# Graph Module
module Graph

  #==================================================================#

  # Marker Module
  module Marker

    #================================================================#

    # Base Class
    class Base

      # Initialize Method

      def initialize(size, color)
        @size  = size
        @color = color
      end

      # Accessor

      attr_accessor :size, :color

      # Instance Methods

      def draw(image, x, y)
        drawing(image, x, y, @color.allocate(image))
      end

      def drawing(image, x, y, color)
        raise NotImplementedError, "don't use this class"
      end

    end # Base

    #================================================================#

    class FilledBase < Base

      def initialize(size, color, edge = nil)
        super(size, color)
        @edge = edge
      end

      attr_accessor :edge

      def draw(image, x, y)
        if @edge
          drawing(image, x, y, @color.allocate(image), @edge.allocate(image))
        else
          drawing(image, x, y, @color.allocate(image), nil)
        end
      end

      def drawing(image, x, y, color, edge)
        raise NotImpelementedError, "don't use this class"
      end

    end

    #================================================================#

    class Circle < Base

      def drawing(image, x, y, color)
        image.arc(x, y, @size * 2, @size * 2, 0, 360, color)
      end

    end # Circle

    #================================================================#

    class FilledCircle < FilledBase

      def drawing(image, x, y, color, edge)
        image.filledArc(x, y, @size * 2, @size * 2, 0, 360, color, 0)
        image.arc(x, y, @size * 2, @size * 2, 0, 360, edge) if edge
      end

    end # FilledCircle

    #================================================================#

    class Lozenge < Base

      def drawing(image, x, y, color)
        poly = GD::Polygon.new
        poly.addPt(x - @size, y)
        poly.addPt(x, y - @size)
        poly.addPt(x + @size, y)
        poly.addPt(x, y + @size)
        image.polygon(poly, color)
      end

    end # Lozenge

    #================================================================#

    class FilledLozenge < FilledBase

      def drawing(image, x, y, color, edge)
        poly = GD::Polygon.new
        poly.addPt(x - @size, y)
        poly.addPt(x, y - @size)
        poly.addPt(x + @size, y)
        poly.addPt(x, y + @size)
        image.filledPolygon(poly, color)
        image.polygon(poly, edge) if edge
      end

    end # FilledLozenge

    #================================================================#

    class Cross < Base

      def drawing(image, x, y, color)
        image.line(x - @size, y - @size, x + @size, y + @size, color)
        image.line(x + @size, y - @size, x - @size, y + @size, color)
      end

    end # Cross

    #================================================================#

    class Plus < Base

      def drawing(image, x, y, color)
        image.line(x - @size, y, x + @size, y, color)
        image.line(x, y - @size, x, y + @size, color)
      end

    end # Plus

    #================================================================#

    class Square < Base

      def drawing(image, x, y, color)
        image.rectangle(x - @size, y - @size, x + @size, y + @size, color)
      end

    end # Square

    #================================================================#

    class FilledSquare < FilledBase

      def drawing(image, x, y, color, edge)
        image.filledRectangle(x - @size, y - @size, x + @size, y + @size, color)
        image.rectangle(x - @size, y - @size, x + @size, y + @size, edge) if edge
      end

    end # FilledSquare

    #================================================================#

    class Triangle < Base

      def drawing(image, x, y, color)
        poly = GD::Polygon.new
        poly.addPt(x - @size, y + @size)
        poly.addPt(x, y - @size)
        poly.addPt(x + @size, y + @size)
        image.polygon(poly, color)
      end

    end # Triangle

    #================================================================#

    class FilledTriangle < FilledBase

      def drawing(image, x, y, color, edge)
        poly = GD::Polygon.new
        poly.addPt(x - @size, y + @size)
        poly.addPt(x, y - @size)
        poly.addPt(x + @size, y + @size)
        image.filledPolygon(poly, color)
        image.polygon(poly, edge) if edge
      end

    end # FilledTriangle

    #================================================================#

    class InverseTriangle < Base

      def drawing(image, x, y, color)
        poly = GD::Polygon.new
        poly.addPt(x - @size, y - @size)
        poly.addPt(x, y + @size)
        poly.addPt(x + @size, y - @size)
        image.polygon(poly, color)
      end

    end # InverseTriangle

    #================================================================#

    class FilledInverseTriangle < FilledBase

      def drawing(image, x, y, color, edge)
        poly = GD::Polygon.new
        poly.addPt(x - @size, y - @size)
        poly.addPt(x, y + @size)
        poly.addPt(x + @size, y - @size)
        image.filledPolygon(poly, color)
        image.polygon(poly, edge) if edge
      end

    end # FilledInverseTriangle

    #================================================================#

    class Star < Base

      def point(poly, x, y, deg, size)
        xx = x + Math.sin(Math::PI / 180 * (180 - deg)) * size
        yy = y + Math.cos(Math::PI / 180 * (180 - deg)) * size
        poly.addPt(xx, yy)
      end

      def drawing(image, x, y, color)
        full = @size
        half = @size / 2.5

        poly = GD::Polygon.new
        point(poly, x, y,   0, full)
        point(poly, x, y,  36, half)
        point(poly, x, y,  72, full)
        point(poly, x, y, 108, half)
        point(poly, x, y, 144, full)
        point(poly, x, y, 180, half)
        point(poly, x, y, 216, full)
        point(poly, x, y, 252, half)
        point(poly, x, y, 288, full)
        point(poly, x, y, 324, half)
        image.polygon(poly, color)
      end

    end # Star

    #================================================================#

    class FilledStar < FilledBase

      def point(poly, x, y, deg, size)
        xx = x + Math.sin(Math::PI / 180 * (180 - deg)) * size
        yy = y + Math.cos(Math::PI / 180 * (180 - deg)) * size
        poly.addPt(xx, yy)
      end

      def drawing(image, x, y, color, edge)
        full = @size
        half = @size / 2.5

        poly = GD::Polygon.new
        point(poly, x, y,   0, full)
        point(poly, x, y,  36, half)
        point(poly, x, y,  72, full)
        point(poly, x, y, 108, half)
        point(poly, x, y, 144, full)
        point(poly, x, y, 180, half)
        point(poly, x, y, 216, full)
        point(poly, x, y, 252, half)
        point(poly, x, y, 288, full)
        point(poly, x, y, 324, half)
        image.filledPolygon(poly, color)
        image.polygon(poly, edge) if edge
      end

    end # FilledStar

    #================================================================#

    class Pentagon < Base

      def point(poly, x, y, deg)
        xx = x + Math.sin(Math::PI / 180 * (180 - deg)) * @size
        yy = y + Math.cos(Math::PI / 180 * (180 - deg)) * @size
        poly.addPt(xx, yy)
      end

      def drawing(image, x, y, color)
        poly = GD::Polygon.new
        point(poly, x, y,   0)
        point(poly, x, y,  72)
        point(poly, x, y, 144)
        point(poly, x, y, 216)
        point(poly, x, y, 288)
        image.polygon(poly, color)
      end

    end # Pentagon

    #================================================================#

    class FilledPentagon < FilledBase

      def point(poly, x, y, deg)
        xx = x + Math.sin(Math::PI / 180 * (180 - deg)) * @size
        yy = y + Math.cos(Math::PI / 180 * (180 - deg)) * @size
        poly.addPt(xx, yy)
      end

      def drawing(image, x, y, color, edge)
        poly = GD::Polygon.new
        point(poly, x, y,   0)
        point(poly, x, y,  72)
        point(poly, x, y, 144)
        point(poly, x, y, 216)
        point(poly, x, y, 288)
        image.filledPolygon(poly, color)
        image.polygon(poly, edge) if edge
      end

    end # FilledPentagon

    #================================================================#

  end # Marker

end # Graph
