
#==============================================================================#
# gruby/graph/misc/font.rb
# $Id: font.rb,v 1.4 2002/09/19 11:10:02 yuya Exp $
#==============================================================================#

require 'kconv'
require 'uconv'

class GRb

  module Graph

    class Font

      Giant  = 'Giant'
      Large  = 'Large'
      Medium = 'Medium'
      Small  = 'Small'
      Tiny   = 'Tiny'

      def initialize(font)
        @font = font
      end

      def self.gd(name, color = Color.black, vertical = false)
        return self.new(GDFont.new(name, color, vertical))
      end

      def self.gd_giant(color = Color.black, vertical = false)
        return self.new(GDFont.new(Giant, color, vertical))
      end

      def self.gd_large(color = Color.black, vertical = false)
        return self.new(GDFont.new(Large, color, vertical))
      end

      def self.gd_medium(color = Color.black, vertical = false)
        return self.new(GDFont.new(Medium, color, vertical))
      end

      def self.gd_small(color = Color.black, vertical = false)
        return self.new(GDFont.new(Small, color, vertical))
      end

      def self.gd_tiny(color = Color.black, vertical = false)
        return self.new(GDFont.new(Tiny, color, vertical))
      end

      def self.truetype(filepath, size = 12, color = Color.black, angle = 0)
        return self.new(TrueTypeFont.new(filepath, size, color, angle))
      end

      def draw(image, x, y, text)
        @font.draw(image, x, y, text)
      end

      def width(text)
        @font.width(text)
      end

      def height(text)
        @font.height(text)
      end

      #==================================================================#

      class GDFont

        def initialize(name, color, vertical = false)
          @font     = GD::Font.new(name)
          @color    = color
          @vertical = vertical
        end

        def draw(image, x, y, text)
          color = image.set(@color)

          unless @vertical
            image.image.string(@font, x, y, text, color)
          else
            image.image.stringUp(@font, x, y + self.height(text), text, color)
          end
        end

        def width(text)
          unless @vertical
            return @font.width * text.size
          else
            return @font.height
          end
        end

        def height(text)
          unless @vertical
            return @font.height
          else
            return @font.width * text.size
          end
        end

      end

      #==================================================================#

      class TrueTypeFont

        def initialize(filepath, size, color, angle)
          @filepath = filepath
          @size     = size
          @color    = color
          @angle    = angle
        end

        def convert(text)
          sjis = Kconv.tosjis(text)
          utf8 = Uconv.sjistou8(sjis)
          return utf8
        end
        private :convert

        def draw(image, x, y, text)
          color = image.colorAllocate(*@color.to_a)
          box   = self.box(text)
          draw_font(image, color, @filepath, @size, Math::PI / 180 * @angle, x - box.x, y - box.y, text)
        end

        def draw_font(image, color, filepath, size, angle, x, y, text)
          if image.methods.include?('stringFT')
            Graph.debug { "use stringFT" }
            image.stringFT(color, filepath, size, angle, x, y, convert(text))
          elsif image.methods.include?('stringTTF')
            Graph.debug { "use stringTTF" }
            image.stringTTF(color, filepath, size, angle, x, y, convert(text))
          else
            raise "FreeType library isn't linked"
          end
        end

        def box(text)
          error, bounds = GD::Image.stringFT(0, @filepath, @size, @angle * (Math::PI / 180), 0, 0, convert(text))
          x1, y1, x2, y2, x3, y3, x4, y4 = bounds
          xx, yy = [x1, x2, x3, x4], [y1, y2, y3, y4]

          box = Box.new
          box.x  = xx.min
          box.y  = yy.min
          box.dx = xx.max - xx.min
          box.dy = yy.max - yy.min

          return box
        end

        def width(text)
          return box(text).dx
        end

        def height(text)
          return box(text).dy
        end

      end

    end

  end

end # GRb

#==============================================================================#
#==============================================================================#
