#ifndef GAME_SYOKUNIN_COM_GLSL_MATH_GLSL_MATH_H

#define GAME_SYOKUNIN_COM_GLSL_MATH_GLSL_MATH_H

#include <math.h>
#include <boost/assert.hpp>
#include <algorithm>
#include <math.h>
#include <numeric>
#include <functional>

/*
gslib/glsl_math/glsl_math.h

zlib/libpng license
-------------------

Copyright (C) 2004 &o

This software is provided 'as-is', without any express or implied warranty. In n
o event will the authors be held liable for any damages arising from the use of 
this software.

Permission is granted to anyone to use this software for any purpose, including 
commercial applications, and to alter it and redistribute it freely, subject to 
the following restrictions:

The origin of this software must not be misrepresented; you must not claim that 
you wrote the original software. If you use this software in a product, an ackno
wledgment in the product documentation would be appreciated but is not required.

Altered source versions must be plainly marked as such, and must not be misrepre
sented as being the original software.
This notice may not be removed or altered from any source distribution.

project site : https://sourceforge.jp/projects/gslib/
my site : http://www.game-syokunin.com/
--------------------------------------------------------------------------------

@Iɂ́AĽ̂قLȂ̂ŁA茵ɂ͓{Qlɂ
ĂB{́Ahttp://opensource.jp/licenses/zlib-license.html 璸
Ă܂B

zlib/libpngCZX ( { )

Copyright (C) 2004 &o

{\tgEFÁû܂܁vŁAł邩Öقł邩킸A̕ۏ؂
Ȃ񋟂܂B{\tgEFA̎gpɂĐ邢Ȃ鑹QɂĂA
͈؂̐ӔC𕉂Ȃ̂Ƃ܂B ȉ̐ɏ]ApAvP[V
܂߂āA{\tgEFACӂ̖ړIɎgpARɉςčĔЕz邱Ƃ
Ă̐lɋ܂B

{\tgEFȀoɂċU̕\Ă͂Ȃ܂BȂIWĩ\
tgEFA쐬Ǝ咣Ă͂Ȃ܂BȂ{\tgEFA𐻕iŎgp
ꍇAi̕ɎӎĂ΍KłAK{ł͂܂B
\[XύXꍇ́ÂƂ𖾎Ȃ΂Ȃ܂BIWĩ\tgE
FAłƂU̕\Ă͂Ȃ܂B
\[X̔ЕzA̕\폜A\̓eύX肵Ă͂Ȃ܂
B 

project site : https://sourceforge.jp/projects/gslib/
my site : http://www.game-syokunin.com/
*/

namespace gslib {

	///	glsl like math library.
	/**
	 *	based on followings.
	 *	- GLSLangSpec.Full.1.10.59.pdf
	 *	- nv_math ( NVSDK )
	 *	- ATI SDK's Vector.h
	 *	- Game Programming Gems
	 *	- STL containers
	 */
	namespace glsl_math {

		///	base class of vector, matrix, quaternion
		/**
		 *	@pre
		 *	All subclass's front address must be same as float array's front address.
		 *	This class requests that 'this' pointer is float array's front address.
		 */
		template < size_t Size >
		struct tuple {
			typedef size_t				size_type;
			typedef float				value_type;
			typedef value_type*			iterator;
			typedef const value_type*	const_iterator;

			static size_type size() {
				return Size;
			}
			
			iterator begin() {
				return reinterpret_cast< iterator >( this );
			}
			
			iterator end() {
				return begin() + size();
			}
			
			const_iterator begin() const {
				return reinterpret_cast< const_iterator >( this );
			}

			const_iterator end() const {
				return begin() + size();
			}
			
			void swap( tuple& other ) {
				std::swap_ranges( other.begin(), other.end(), begin() );
			}
			
			tuple() {}
			
			tuple( const tuple& other ) {
				std::copy( other.begin(), other.end(), begin() );
			}
			
			tuple& operator = ( const tuple& other ) {
				if ( this != &other ) {
					std::copy( other.begin(), other.end(), begin() );
				}
				return *this;
			}
			
			float& operator [] ( size_type nth ) {
				return begin()[ nth ];
			}

			float operator [] ( size_type nth ) const {
				return begin()[ nth ];
			}

			float& at( size_type nth ) {
				return begin()[ nth ];
			}

			float at( size_type nth ) const {
				return begin()[ nth ];
			}
		};

		///	inner product, dot product
		template < size_t Size >
		float dot( const tuple< Size >& a, const tuple< Size >& b ) {
			return std::inner_product( a.begin(), a.end(), b.begin(), 0.0f );
		}
		
		///	get length
		template < size_t Size >
		float length( const tuple< Size >& a ) {
			return sqrtf( dot( a, a ) );
		}
		
		///	normalize
		/** 
		 *	@pre
		 *	length( a ) must over 0.
		 */
		template < typename GenType >
		GenType normalize( const GenType& a ) {
			GenType	result( a );
			float invLen = 1.0f / length( a );
			return a * invLen;
		}

		///	distance between a, b
		template < size_t Size >
		float distance( const tuple< Size >& a, const tuple< Size >& b ) {
			float result = 0.0f;
			for ( size_t i = 0; i < Size; ++i ) {
				float diff = a[ i ] - b[ i ];
				result += diff * diff;
			}
			return sqrtf( result );
		}
			
		///	this class implements vector, matrix, quaternion.
		/**
		 *	This class provides basic operators.
		 */
		template < typename SubClass, size_t Size >
		struct tuple_mixin : public tuple< Size > {
			SubClass& operator += ( const tuple< Size >& other ) {
				std::transform(
					begin(), end(), other.begin(), begin(), std::plus< value_type >() );
				return *static_cast< SubClass* >( this );
			}
			SubClass operator + ( const tuple< Size >& other ) const {
				SubClass result( *static_cast< const SubClass* >( this ) );
				result += other;
				return result;
			}
			SubClass& operator -= ( const tuple< Size >& other ) {
				std::transform(
					begin(), end(), other.begin(), begin(), std::minus< value_type >() );
				return *static_cast< SubClass* >( this );
			}
			SubClass operator - ( const tuple< Size >& other ) const {
				SubClass result( *static_cast< const SubClass* >( this ) );
				result -= other;
				return result;
			}
			SubClass& operator *= ( value_type v ) {
				std::transform(
					begin(), end(), begin(), std::bind2nd( std::multiplies< value_type >(), v ) );
				return *static_cast< SubClass* >( this );
			}
			SubClass operator * ( value_type v ) const {
				SubClass result( *static_cast< const SubClass* >( this ) );
				result *= v;
				return result;
			}
			friend SubClass operator * ( value_type v, const tuple_mixin& other ) {
				return other * v;
			}
			SubClass& operator /= ( value_type v ) {
				return operator *= ( 1.0f / v );
			}
			SubClass operator / ( value_type v ) const {
				return ( *this ) * ( 1.0f / v );
			}
			SubClass& operator + () const {
				return *static_cast< SubClass* >( this );
			}
			SubClass operator - () const {
				SubClass result;
				std::transform( begin(), end(), result.begin(), std::negate< value_type >() );
				return result;
			}
			bool operator < ( const tuple< Size >& other ) const {
				return std::lexicographical_compare( begin(), end(), other.begin(), other.end() );
			}
			bool operator == ( const tuple< Size >& other ) const {
				return std::equal( begin(), end(), other.begin() );
			}
			bool operator != ( const tuple< Size >& other ) const {
				return !operator == ( other );
			}
		};
		
		///	this class implements vector common member
		template < typename SubClass, size_t Size >
		struct vector_mixin : public tuple_mixin< SubClass, Size > {
			SubClass& operator *= ( value_type v ) {
				return tuple_mixin< SubClass, Size >::operator *= ( v );
			}
			SubClass operator * ( value_type v ) const {
				return tuple_mixin< SubClass, Size >::operator * ( v );
			}
			SubClass& operator /= ( value_type v ) {
				return tuple_mixin< SubClass, Size >::operator /= ( v );
			}
			SubClass operator / ( value_type v ) const {
				return tuple_mixin< SubClass, Size >::operator / ( v );
			}
			
			SubClass& operator *= ( const tuple< Size >& other ) {
				std::transform(
					begin(), end(), other.begin(), begin(), std::multiplies< value_type >() );
				return *static_cast< SubClass* >( this );
			}
			SubClass operator * ( const tuple< Size >& other ) {
				SubClass result( *this );
				result *= other;
				return result;
			}
			SubClass& operator /= ( const tuple< Size >& other ) {
				std::transform(
					begin(), end(), other.begin(), begin(), std::divides< value_type >() );
				return *static_cast< SubClass* >( this );
			}
			SubClass operator / ( const tuple< Size >& other ) {
				SubClass result( *this );
				result /= other;
				return result;
			}
		};


		struct vec2;
		struct vec3;
		struct vec4;

		struct vec2 : public vector_mixin< vec2, 2 > {
			union {
				struct {
					float x, y;
				};
				struct {
					float r, g;
				};
				struct {
					float s, t;
				};
				float array[ 2 ];
			};
			vec2() {}
			vec2( float v ) : x( v ), y( v ) {}
			vec2( float inX, float inY ) : x( inX ), y( inY ) {}
			vec2( const vec3& xy );
		};

		struct vec3 : public vector_mixin< vec3, 3 > {
			union {
				struct {
					float x, y, z;
				};
				struct {
					float r, g, b;
				};
				struct {
					float s, t, p;
				};
				float array[ 3 ];
			};

			vec3() {}
			vec3( float v ) : x( v ), y( v ), z( v ) {}
			vec3( float inX, float inY, float inZ ) : x( inX ), y( inY ), z( inZ ) {}
			vec3( const vec4& xyz );
			/// vec3.x = vec2.x, vec3.y = vec2.y, vec3.z = float
			vec3( const vec2& xy, float inZ ) :
				x( xy.x ), y( xy.y ), z( inZ ) {
			}
			/// vec3.x = float, vec3.y = vec2.x, vec3.z = vec2.y
			vec3( float inX, const vec2& yz ) :
				x( inX ), y( yz.x ), z( yz.y ) {
			}
			
			///	swizzle ( cast )
			vec2& xy() {
				return reinterpret_cast< vec2& >( array[ 0 ] );
			}
			///	swizzle ( cast )
			const vec2& xy() const {
				return reinterpret_cast< const vec2& >( array[ 0 ] );
			}
			///	swizzle ( cast )
			vec2& yz() {
				return reinterpret_cast< vec2& >( array[ 1 ] );
			}
			///	swizzle ( cast )
			const vec2& yz() const {
				return reinterpret_cast< const vec2& >( array[ 1 ] );
			}
		};

		struct vec4 : public vector_mixin< vec4, 4 > {
			union {
				struct {
					float x, y, z, w;
				};
				struct {
					float r, g, b, a;
				};
				struct {
					float s, t, p, q;
				};
				float array[ 4 ];
			};

			vec4() {}
			vec4( float v ) : x( v ), y( v ), z( v ), w( v ) {}
			vec4( float inX, float inY, float inZ, float inW ) : x( inX ), y( inY ), z( inZ ), w( inW ) {}
			vec4( const vec3& xyz, float inW ) :
				x( xyz.x ), y( xyz.y ), z( xyz.z ), w( inW ) {
			}
			vec4( float inX, const vec3& yzw ) :
				x( inX ), y( yzw.x ), z( yzw.y ), w( yzw.z ) {
			}
			vec4( const vec2& xy, const vec2& zw ) :
				x( xy.x ), y( xy.y ), z( zw.x ), w( zw.y ) {
			}

			///	swizzle ( cast )
			vec2& xy() {
				return reinterpret_cast< vec2& >( array[ 0 ] );
			}
			///	swizzle ( cast )
			const vec2& xy() const {
				return reinterpret_cast< const vec2& >( array[ 0 ] );
			}
			///	swizzle ( cast )
			vec2& yz() {
				return reinterpret_cast< vec2& >( array[ 1 ] );
			}
			///	swizzle ( cast )
			const vec2& yz() const {
				return reinterpret_cast< const vec2& >( array[ 1 ] );
			}
			///	swizzle ( cast )
			vec2& zw() {
				return reinterpret_cast< vec2& >( array[ 2 ] );
			}
			///	swizzle ( cast )
			const vec2& zw() const {
				return reinterpret_cast< const vec2& >( array[ 2 ] );
			}
			///	swizzle ( cast )
			vec3& xyz() {
				return reinterpret_cast< vec3& >( array[ 0 ] );
			}
			///	swizzle ( cast )
			const vec3& xyz() const {
				return reinterpret_cast< const vec3& >( array[ 0 ] );
			}
			///	swizzle ( cast )
			vec3& yzw() {
				return reinterpret_cast< vec3& >( array[ 1 ] );
			}
			///	swizzle ( cast )
			const vec3& yzw() const {
				return reinterpret_cast< const vec3& >( array[ 1 ] );
			}
		};

		inline vec2::vec2( const vec3& xy ) : x( xy.x ), y( xy.y ) {}
		inline vec3::vec3( const vec4& xyz ) : x( xyz.x ), y( xyz.y ), z( xyz.z ) {}
		
		///	outer product, cross product
		/**
		 *	same as GLSLangSpec.Full.1.10.59.pdf '8.4 Geometric Functions'.
		 */
		inline vec3 cross( const vec3& x, const vec3& y ) {
			return vec3(
				x[ 1 ] * y[ 2 ] - y[ 1 ] * x[ 2 ],
				x[ 2 ] * y[ 0 ] - y[ 2 ] * x[ 0 ],
				x[ 0 ] * y[ 1 ] - y[ 0 ] * x[ 1 ] );
		}
		
		///	this class implements matrix common member
		template < typename SubClass, typename FriendVector, size_t RowSize, size_t ColumnSize >
		struct matrix_mixin : public tuple_mixin< SubClass, RowSize * ColumnSize > {
			typedef tuple_mixin< SubClass, RowSize * ColumnSize >	TupleMixin;
			SubClass& operator *= ( value_type v ) {
				return TupleMixin::operator *= ( v );
			}
			SubClass operator * ( value_type v ) const {
				return TupleMixin::operator * ( v );
			}
			SubClass& operator /= ( value_type v ) {
				return TupleMixin::operator /= ( v );
			}
			SubClass operator / ( value_type v ) const {
				return TupleMixin::operator / ( v );
			}
			
			///	get nthColumn column vector's reference
			FriendVector& operator [] ( size_type nthColumn ) {
				BOOST_ASSERT( nthColumn < ColumnSize );
				return *( reinterpret_cast< FriendVector* >( begin() ) + nthColumn );
			}
			///	get nthColumn column vector's const reference
			const FriendVector& operator [] ( size_type nthColumn ) const {
				BOOST_ASSERT( nthColumn < ColumnSize );
				return *( reinterpret_cast< const FriendVector* >( begin() ) + nthColumn );
			}
			SubClass operator * ( const SubClass& other ) const {
				#define AT( i, j )	operator [] ( i )[ j ]
				
				SubClass result( 0 );
				for ( size_type i = 0; i < ColumnSize; ++i ) {
					for ( size_type j = 0; j < RowSize; ++j ) {
						for ( size_type k = 0; k < ColumnSize; ++k ) {
							result[ i ][ j ] += AT( j, k ) * other[ k ][ i ];
						}
					}
				}
				
				#undef AT
				
				return result;
			}
			SubClass& operator *= ( const SubClass& other ) {
				*this = ( *this ) * other;
				return *this;
			}
			///	multiply with column vector ( Right Hand System, OpenGL )
			FriendVector operator * ( const FriendVector& v ) {
				FriendVector result( 0 );
				for ( size_type i = 0; i < FriendVector::size(); ++i ) {
					for ( size_type j = 0; j < FriendVector::size(); ++j ) {
						result[ i ] += v[ j ] * operator [] ( j )[ i ];
					}
				}
				return result;
			}
			///	multiply with column vector ( Left Hand Syste, DirectX )
			friend FriendVector operator * ( const FriendVector& v, const SubClass& other ) {
				FriendVector result;
				for ( size_type i = 0; i < FriendVector::size(); ++i ) {
					result[ i ] = dot( v, other[ i ] );
				}
				return result;
			}
			
			static size_type columnSize() {
				return ColumnSize;
			}
			static size_type rowSize() {
				return RowSize;
			}
		};
		
		struct mat2 : public matrix_mixin< mat2, vec2, 2, 2 > {
			value_type array[ 2 * 2 ];
			
			mat2() {}
			mat2( value_type v ) {
				std::fill( begin(), end(), v );
			}
			mat2( const vec2& column0, const vec2& column1 ) {
				operator [] ( 0 ) = column0;
				operator [] ( 1 ) = column1;
			}
			
			///	The floats are assigned to elements in column major order.
			/**
			 *	@code
			 *	//	( 1, 2 )
			 *  //	( 3, 4 )
			 *	mat2( 1, 3,
			 *		  2, 4 );
			 *	@endcode
			 */
			mat2(
				value_type _00, value_type _10,
				value_type _01, value_type _11 ) {
				
				array[ 0 ] = _00;
				array[ 1 ] = _10;
				array[ 2 ] = _01;
				array[ 3 ] = _11;
			}

			static mat2 identity() {
				return mat2( 1, 0, 0, 1 );
			}
		};
		
		struct mat3 : public matrix_mixin< mat3, vec3, 3, 3 > {
			value_type array[ 3 * 3 ];
			
			mat3() {}
			mat3( value_type v ) {
				std::fill( begin(), end(), v );
			}
			mat3( const vec3& column0, const vec3& column1, const vec3& column2 ) {
				operator [] ( 0 ) = column0;
				operator [] ( 1 ) = column1;
				operator [] ( 2 ) = column2;
			}
			
			mat3(
				value_type _00, value_type _10, value_type _20,
				value_type _01, value_type _11, value_type _21,
				value_type _02, value_type _12, value_type _22 ) {
				
				array[ 0 ] = _00;
				array[ 1 ] = _10;
				array[ 2 ] = _20;
				array[ 3 ] = _01;
				array[ 4 ] = _11;
				array[ 5 ] = _21;
				array[ 6 ] = _02;
				array[ 7 ] = _12;
				array[ 8 ] = _22;
			}

			static mat3 identity() {
				return mat3( 1, 0, 0, 0, 1, 0, 0, 0, 1 );
			}
		};

		struct mat4 : public matrix_mixin< mat4, vec4, 4, 4 > {
			value_type array[ 4 * 4 ];
			
			mat4() {}
			mat4( value_type v ) {
				std::fill( begin(), end(), v );
			}
			mat4( const vec4& column0, const vec4& column1, const vec4& column2, const vec4& column3 ) {
				operator [] ( 0 ) = column0;
				operator [] ( 1 ) = column1;
				operator [] ( 2 ) = column2;
				operator [] ( 3 ) = column3;
			}
			
			mat4(
				value_type _00, value_type _10, value_type _20, value_type _30,
				value_type _01, value_type _11, value_type _21, value_type _31,
				value_type _02, value_type _12, value_type _22, value_type _32,
				value_type _03, value_type _13, value_type _23, value_type _33 ) {
				
				array[  0 ] = _00;
				array[  1 ] = _10;
				array[  2 ] = _20;
				array[  3 ] = _30;
				array[  4 ] = _01;
				array[  5 ] = _11;
				array[  6 ] = _21;
				array[  7 ] = _31;
				array[  8 ] = _02;
				array[  9 ] = _12;
				array[ 10 ] = _22;
				array[ 11 ] = _32;
				array[ 12 ] = _03;
				array[ 13 ] = _13;
				array[ 14 ] = _23;
				array[ 15 ] = _33;
			}
			
			static mat4 identity() {
				return mat4( 1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 1 );
			}
		};
		
		float det( const mat2& m ) {
			return m[ 0 ][ 0 ] * m[ 1 ][ 1 ] - m[ 0 ][ 1 ] * m[ 1 ][ 0 ];
		}
		float cofactor( const mat2& m, mat2::size_type col, mat2::size_type row ) {
			return m[ col ? 0 : 1 ][ row ? 0 : 1 ];
		}
		
		template < typename Mat, typename SmallMat >
		struct CofactorCalc {
			static float calc( const Mat& m, typename Mat::size_type col, typename Mat::size_type row ) {
				SmallMat	result;
				float* f = result.array;
				for ( Mat::size_type i = 0; i < Mat::columnSize(); ++i ) {
					if ( i != col ) {
						for ( Mat::size_type j = 0; j < Mat::rowSize(); ++j ) {
							if ( j != row ) {
								*f = m[ i ][ j ];
								++f;
							}
						}
					}
				}
				return det( result );
			}
		};
		template < typename Mat >
		struct DetCalc {
			static float calc( const Mat& m ) {
				float result = 0;
				for ( Mat::size_type i = 0; i < Mat::columnSize(); ++i ) {
					result += ( ( 0 == ( i & 1 ) ) ? 1 : -1 ) * m[ 0 ][ i ] * cofactor( m, 0, i );
				}
				return result;
			}
		};
		template < typename Mat >
		struct InverseCalc {
			static Mat calc( const Mat& m ) {
				Mat result;
				float invDetM = 1.0f / det( m );
				for ( Mat::size_type i = 0; i < Mat::columnSize(); ++i ) {
					for ( Mat::size_type j = 0; j < Mat::rowSize(); ++j ) {
						result[ i ][ j ] = ( ( 0 == ( ( i + j ) & 1 ) ) ? invDetM : -invDetM ) * cofactor( m, j, i );
					}
				}
				return result;
			}
		};
		
		mat2 inverse( const mat2& m ) {
			return InverseCalc< mat2 >::calc( m );
		}
		
		float cofactor( const mat3& m, mat3::size_type col, mat3::size_type row ) {
			return CofactorCalc< mat3, mat2 >::calc( m, col, row );
		}
		float det( const mat3& m ) {
			return DetCalc< mat3 >::calc( m );
		}
		mat3 inverse( const mat3& m ) {
			return InverseCalc< mat3 >::calc( m );
		}
		
		float cofactor( const mat4& m, mat3::size_type col, mat3::size_type row ) {
			return CofactorCalc< mat4, mat3 >::calc( m, col, row );
		}
		float det( const mat4& m ) {
			return DetCalc< mat4 >::calc( m );
		}
		mat4 inverse( const mat4& m ) {
			return InverseCalc< mat4 >::calc( m );
		}
		
		template < typename Mat >
		Mat transpose( const Mat& m ) {
			Mat result;
			for ( size_type i = 0; i < m.colSize(); ++i ) {
				for ( size_type j = 0; j < m.rowSize(); ++j ) {
					result[ j ][ i ] = m[ i ][ j ];
				}
			}
			return result;
		}
		
/*		#undef min
		#undef max
		
		template < typename Tuple >
		Tuple min( const Tuple& a, const Tuple& b ) {
			Tuple result;
			for ( typename Tuple::size_type i = 0; i < Tuple::size(); ++i ) {
				result.at( i ) = std::_cpp_min( a.at( i ), b.at( i ) );
			}
			return result;
		}

		template < typename Tuple >
		Tuple max( const Tuple& a, const Tuple& b ) {
			Tuple result;
			for ( typename Tuple::size_type i = 0; i < Tuple::size(); ++i ) {
				result.at( i ) = std::_cpp_max( a.at( i ), b.at( i ) );
			}
			return result;
		}*/
		
		struct quat : public tuple_mixin< quat, 4 > {
			union {
				struct {
					float x, y, z, w;
				};
				float array[ 4 ];
			};
			quat() {}
			quat( value_type v ) : x( v ), y( v ), z( v ), w( v ) {}
			quat( value_type inX, value_type inY, value_type inZ, value_type inW ) : x( inX ), y( inY ), z( inZ ), w( inW ) {}
			quat( const vec3& axis, value_type theta ) {
				value_type halfCos = cosf( theta * 0.5f );
				value_type halfSin = sinf( theta * 0.5f );
				xyz() = halfCos * axis;
				w = halfSin;
			}

			///	swizzle ( cast to vec4 )
			vec4& xyzw() {
				return *reinterpret_cast< vec4* >( array );
			}

			///	swizzle ( cast to vec4 )
			const vec4& xyzw() const {
				return *reinterpret_cast< const vec4* >( array );
			}
			
			///	swizzle
			vec3& xyz() {
				return *reinterpret_cast< vec3* >( array );
			}

			///	swizzle
			const vec3& xyz() const {
				return *reinterpret_cast< const vec3* >( array );
			}
			
			///	swizzle
			vec2& xy() {
				return *reinterpret_cast< vec2* >( array );
			}

			///	swizzle
			const vec2& xy() const {
				return *reinterpret_cast< const vec2* >( array );
			}

			typedef tuple_mixin< quat, 4 >	TupleMixin;
			quat& operator *= ( value_type v ) {
				return TupleMixin::operator *= ( v );
			}
			quat operator * ( value_type v ) const {
				return TupleMixin::operator * ( v );
			}
			quat& operator /= ( value_type v ) {
				return TupleMixin::operator /= ( v );
			}
			quat operator / ( value_type v ) const {
				return TupleMixin::operator / ( v );
			}
			quat operator * ( const quat& q ) const {
				return quat(
					w * q.x + x * q.w + y * q.z - z * q.y,
					w * q.y + y * q.w + z * q.x - x * q.z,
					w * q.z + z * q.w + x * q.y - y * q.x,
					w * q.w - x * q.x - y * q.y - z * q.z
				);
			}
			quat& operator *= ( const quat& q ) {
				*this = *this * q;
				return *this;
			}
			
			static quat identity() {
				return quat( 0, 0, 0, 1 );
			}
		};
		
		inline quat conj( const quat& q ) {
			return quat(
				-q.x,
				-q.y,
				-q.z,
				q.w );
		}
		
		inline quat inverse( const quat& q ) {
			return conj( q ) / dot( q, q );
		}
		
		///	Converts degrees to radians and returns the result, i.e., result = PI/180*degrees.
		inline float radians( float deg ) {
			const float pi = 3.14159265358979323846264338327950288419716939937510582;
			return deg * ( pi / 180 );
		}
		
		///	Converts radians to degrees and returns the result, i.e., result = 180/PI*radians.
		inline float degrees( float rad ) {
			const float pi = 3.14159265358979323846264338327950288419716939937510582;
			return rad * ( 180 / pi );
		}

		///	linear interpolation
		/**
		 *	@pre
		 *	0 <= blendRate <= 1
		 *
		 *	@return ( 1 - blendRate ) * a + blendRate * b
		 */
		template < typename GenType >
		inline GenType lerp( const GenType& a, const GenType& b, float blendRate ) {
			BOOST_ASSERT( 0 <= blendRate && blendRate <= 1 );
			return ( 1 - blendRate ) * a + blendRate * b;
		}
		
		///	log
		/**
		 *	This function is from Game Programming Gems 2.9.
		 *	This function has not been tested yet.
		 *
		 *	Logarithm of a quaternion, given as:
		 *	log(q) = v*a where q = [cos(a),v*sin(a)]
		 */
		inline quat log( const quat &q ) {
			float a = static_cast<float>(acosf(q.w));
			float sina = static_cast<float>(sinf(a));
			quat ret;
			ret.w = 0;
			if (sina > 0) {
			/*	ret.x = a*q.x/sina;
				ret.y = a*q.y/sina;
				ret.z = a*q.z/sina;*/
				ret.xyz() = q.xyz() * ( a / sina );
			} else {
				ret.x=ret.y=ret.z=0;
			}
			return ret;
		}

		///	exp
		/**
		 *	This function is from Game Programming Gems 2.9.
		 *	This function has not been tested yet.
		 *
		 *	e^quaternion given as:
		 *	exp(v*a) = [cos(a),vsin(a)]
		 */
		inline quat exp( const quat& q ) {
			float a = static_cast< float >( sqrtf( q.x*q.x + q.y*q.y + q.z*q.z ) );
			float sina = static_cast< float >( sinf( a ) );
			float cosa = static_cast< float >( cosf( a ) );
			quat ret;
			ret.w = cosa;
			if ( a > 0 ) {
				ret.xyz() = q.xyz() * ( sina / a );
			/*	ret.x = sina * q.x / a;
				ret.y = sina * q.y / a;
				ret.z = sina * q.z / a;*/
			} else {
				ret.x = ret.y = ret.z = 0;
			}

			return ret;
		}

		///	Spherical linear interpolation between two quaternions
		/**
		 *	This function is from Game Programming Gems 2.9.
		 *	This function has not been tested yet.
		 */
		inline quat slerp( const quat& q1,const quat& q2,float t ) {
			quat q3;
			float d = dot( q1, q2 );

			/*
			dot = cos(theta)
			if (dot < 0), q1 and q2 are more than 90 degrees apart,
			so we can invert one to reduce spinning
			*/
			if ( d < 0 ) {
				d = -d;
				q3 = -q2;
			}
			else
			{
				q3 = q2;
			}

			
			if ( d < 0.95f ) {
				float angle = static_cast<float>(acosf(d));
				float sina,sinat,sinaomt;
				sina = static_cast<float>(sinf(angle));
				sinat = static_cast<float>(sinf(angle*t));
				sinaomt = static_cast<float>(sinf(angle*(1-t)));
				return (q1*sinaomt+q3*sinat)/sina;
			} else {
				//	if the angle is small, use linear interpolation
				return lerp(q1,q3,t);
			}
		}

		///	This version of slerp, used by squad, does not check for theta > 90.
		/**
		 *	This function is from Game Programming Gems 2.9.
		 *	This function has not been tested yet.
		 */
		inline quat slerpNoInvert(const quat &q1,const quat &q2,float t) {
			float d = dot( q1, q2 );

			if (d > -0.95f && d < 0.95f)
			{
				float angle = static_cast<float>(acosf(d));
				float sina,sinat,sinaomt;
				sina = static_cast<float>(sinf(angle));
				sinat = static_cast<float>(sinf(angle*t));
				sinaomt = static_cast<float>(sinf(angle*(1-t)));
				return (q1*sinaomt+q2*sinat)/sina;
			}
			/*
			if the angle is small, use linear interpolation
			*/
			else
			{
				return lerp(q1,q2,t);
			}
		}


		///	Spherical cubic interpolation
		/**
		 *	This function is from Game Programming Gems 2.9.
		 *	This function has not been tested yet.
		 */
		inline quat squad(const quat &q1,const quat &q2,const quat &a,const quat &b,float t) {
			quat c,d;
			c = slerpNoInvert(q1,q2,t);
			d = slerpNoInvert(a,b,t);
			return slerpNoInvert(c,d,2*t*(1-t));
		}


		///	Given 3 quaternions, qn-1,qn and qn+1, calculate a control point to be used in spline interpolation
		/**
		 *	This function is from Game Programming Gems 2.9.
		 *	This function has not been tested yet.
		 */
		inline quat spline(const quat &qnm1,const quat &qn,const quat &qnp1) {
			quat qni = conj( qn );
			return qn * exp((log(qni*qnm1)+log(qni*qnp1))/-4);
		}
	}
}

#endif