/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/* 
 * Copyright (C) 2005 Hiroyuki Ikezoe <poincare@ikezoe.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <gnome.h>
#include <glade/glade.h>
#include <gconf/gconf-client.h>

#include "gsynaptics.h"
#include "gsynaptics-conf.h"

#define WID(s) glade_xml_get_widget (dialog, s)

static GQuark synaptics_quark = 0;

static GConfClient *gconf = NULL;

enum
{
	RESPONSE_APPLY = 1,
	RESPONSE_CLOSE
};

static void
dialog_button_clicked_cb (GtkDialog *dialog, gint response_id) 
{
	if (response_id == GTK_RESPONSE_HELP)
		g_warning("help");
	else
		gtk_main_quit ();
}

static GladeXML *
create_dialog (void)
{
	GladeXML *dialog;

	dialog = glade_xml_new (DATADIR "/gsynaptics.glade", "gsynaptics_dialog", NULL);
	return dialog;
}

static void
cb_touchpad_toggled (GtkToggleButton *button, GladeXML *dialog)
{
	GtkWidget *widget;
	GSynaptics *synaptics;
	gboolean check;

	widget = WID ("touchpad_box");
	check = gtk_toggle_button_get_active (button);
	gtk_widget_set_sensitive (widget, check);

	synaptics = g_object_get_qdata (G_OBJECT(dialog), synaptics_quark);

	if (check)
	{
		g_synaptics_set_enabled (synaptics, 0);
	}
	else
	{
		g_synaptics_set_enabled (synaptics, 1);
	}
	gconf_client_set_bool (gconf, OFF_KEY, !check, NULL);
}

static void
cb_vertical_scroll_toggled (GtkToggleButton *button, GladeXML *dialog)
{
	GtkWidget *widget;
	GSynaptics *synaptics;
	gboolean check;
	gint value;

	widget = WID ("vertical_scroll_box");
	check = gtk_toggle_button_get_active (button);
	gtk_widget_set_sensitive (widget, check);

	widget = WID ("vertical_scroll_scale");
	value = (gint) gtk_range_get_value (GTK_RANGE(widget));
	
	synaptics = g_object_get_qdata (G_OBJECT(dialog), synaptics_quark);

	if (!check)
		value = 0;

	g_synaptics_set_vertical_scroll_delta (synaptics, value * -1);
	gconf_client_set_int (gconf, VERTSCROLLDELTA_KEY, value * -1, NULL);
}

static void
cb_horizontal_scroll_toggled (GtkToggleButton *button, GladeXML *dialog)
{
	GtkWidget *widget;
	GSynaptics *synaptics;
	gboolean check;
	gint value;

	widget = WID ("horizontal_scroll_box");
	check = gtk_toggle_button_get_active (button);
	gtk_widget_set_sensitive (widget, check);
	
	widget = WID ("horizontal_scroll_scale");
	value = (gint) gtk_range_get_value (GTK_RANGE(widget));
	
	synaptics = g_object_get_qdata (G_OBJECT(dialog), synaptics_quark);

	if (!check)
		value = 0;

	g_synaptics_set_horizontal_scroll_delta (synaptics, value * -1);
	gconf_client_set_int (gconf, HORIZSCROLLDELTA_KEY, value * -1, NULL);
}

static void
cb_circular_scroll_toggled (GtkToggleButton *button, GladeXML *dialog)
{
	GtkWidget *widget;
	GSynaptics *synaptics;
	gboolean check;
	gint value;

	widget = WID ("circular_scroll_box");
	check = gtk_toggle_button_get_active (button);
	gtk_widget_set_sensitive (widget, check);
	
	widget = WID ("circular_scroll_scale");
	value = (gint) gtk_range_get_value (GTK_RANGE(widget));
	
	synaptics = g_object_get_qdata (G_OBJECT(dialog), synaptics_quark);

	if (!check)
		value = 0;

	g_synaptics_set_circular_scroll_delta (synaptics, value * -1);
	g_synaptics_set_circular_scroll_enabled (synaptics, check);
	gconf_client_set_int (gconf, CIRCSCROLLDELTA_KEY, value * -1, NULL);
	gconf_client_set_bool (gconf, CIRCULARSCROLLING_KEY, check, NULL);
}

static void
cb_tapping_toggled (GtkToggleButton *button, GladeXML *dialog)
{
	GtkWidget *widget;
	GSynaptics *synaptics;
	gboolean check;
	gint time;

	widget = WID ("tapping_box");
	check = gtk_toggle_button_get_active (button);
	gtk_widget_set_sensitive (widget, check);
	
	widget = WID ("tapping_time_scale");
	time = (gint) gtk_range_get_value (GTK_RANGE(widget));
	
	synaptics = g_object_get_qdata (G_OBJECT(dialog), synaptics_quark);

	if (!check)
		time = 0;

	g_synaptics_set_tap_time (synaptics, time);
	gconf_client_set_int (gconf, MAXTAPTIME_KEY, time, NULL);
}

static void
cb_fast_taps_toggled (GtkToggleButton *button, GladeXML *dialog)
{
	GSynaptics *synaptics;
	gboolean value;

	value = gtk_toggle_button_get_active (button);
	
	synaptics = g_object_get_qdata (G_OBJECT(dialog), synaptics_quark);

	g_synaptics_set_fast_taps (synaptics, value);
	gconf_client_set_bool (gconf, FASTTAPS_KEY, value, NULL);
}

static gboolean
cb_vertical_scroll_change (GtkRange *range, GtkScrollType scroll,
			   gdouble value, GladeXML *dialog)
{
	GSynaptics *synaptics;
	
	synaptics = g_object_get_qdata (G_OBJECT(dialog), synaptics_quark);

	g_synaptics_set_vertical_scroll_delta (synaptics, (gint)value * -1);
	gconf_client_set_int (gconf, VERTSCROLLDELTA_KEY, (gint)value * -1, NULL);

	return FALSE;
}

static gboolean
cb_horizontal_scroll_change (GtkRange *range, GtkScrollType scroll,
			     gdouble value, GladeXML *dialog)
{
	GSynaptics *synaptics;
	
	synaptics = g_object_get_qdata (G_OBJECT(dialog), synaptics_quark);

	g_synaptics_set_horizontal_scroll_delta (synaptics, (gint)value * -1);
	gconf_client_set_int (gconf, HORIZSCROLLDELTA_KEY, (gint)value * -1, NULL);

	return FALSE;
}

static gboolean
cb_circular_scroll_change (GtkRange *range, GtkScrollType scroll,
			   gdouble value, GladeXML *dialog)
{
	GSynaptics *synaptics;
	
	synaptics = g_object_get_qdata (G_OBJECT(dialog), synaptics_quark);

	g_synaptics_set_circular_scroll_delta (synaptics, (gint)value * -1);
	gconf_client_set_int (gconf, CIRCSCROLLDELTA_KEY, (gint)value * -1, NULL);

	return FALSE;
}

static gboolean
cb_sensitivity_change (GtkRange *range, GtkScrollType scroll,
		       gdouble value, GladeXML *dialog)
{
	GSynaptics *synaptics;
	
	synaptics = g_object_get_qdata (G_OBJECT(dialog), synaptics_quark);

	g_synaptics_set_sensitivity (synaptics, (gint)value);
	gconf_client_set_int (gconf, SENSITIVITY_KEY, (gint)value, NULL);

	return FALSE;
}


static gboolean
cb_tap_time_change (GtkRange *range, GtkScrollType scroll,
		    gdouble value, GladeXML *dialog)
{
	GSynaptics *synaptics;
	
	synaptics = g_object_get_qdata (G_OBJECT(dialog), synaptics_quark);

	g_synaptics_set_tap_time (synaptics, (gint)value);
	gconf_client_set_int (gconf, MAXTAPTIME_KEY, (gint)value, NULL);

	return FALSE;
}

static void
set_circular_scroll_trigger_state (GladeXML *dialog)
{
	GSynaptics *synaptics;
	GtkToggleButton *top, *bottom, *left, *right;
	GtkToggleButton *top_left, *top_right, *bottom_left, *bottom_right;
	gboolean enable = FALSE;
	ScrollTrigger trigger;

	synaptics = g_object_get_qdata (G_OBJECT(dialog), synaptics_quark);

	top          = GTK_TOGGLE_BUTTON (WID ("trigger_top_edge_toggle"));
	bottom       = GTK_TOGGLE_BUTTON (WID ("trigger_bottom_edge_toggle"));
	left         = GTK_TOGGLE_BUTTON (WID ("trigger_left_edge_toggle"));
	right        = GTK_TOGGLE_BUTTON (WID ("trigger_right_edge_toggle"));
	top_left     = GTK_TOGGLE_BUTTON (WID ("trigger_top_left_corner_toggle"));
	top_right    = GTK_TOGGLE_BUTTON (WID ("trigger_top_right_corner_toggle"));
	bottom_left  = GTK_TOGGLE_BUTTON (WID ("trigger_bottom_left_corner_toggle"));
	bottom_right = GTK_TOGGLE_BUTTON (WID ("trigger_bottom_right_corner_toggle"));

	trigger = g_synaptics_circular_scroll_trigger (synaptics);

	if (trigger == AllCorners)
		enable = TRUE;
	gtk_toggle_button_set_active (top,          enable);
	gtk_toggle_button_set_active (bottom,       enable);
	gtk_toggle_button_set_active (left,         enable);
	gtk_toggle_button_set_active (right,        enable);
	gtk_toggle_button_set_active (top_left,     enable);
	gtk_toggle_button_set_active (top_right,    enable);
	gtk_toggle_button_set_active (bottom_left,  enable);
	gtk_toggle_button_set_active (bottom_right, enable);

	switch (trigger) {
	case TopEdge:
		gtk_toggle_button_set_active (top, TRUE);
		break;
	case BottomEdge:
		gtk_toggle_button_set_active (bottom, TRUE);
		break;
	case LeftEdge:
		gtk_toggle_button_set_active (left, TRUE);
		break;
	case RightEdge:
		gtk_toggle_button_set_active (right, TRUE);
		break;
	case TopLeftCorner:
		gtk_toggle_button_set_active (top_left, TRUE);
		break;
	case TopRightCorner:
		gtk_toggle_button_set_active (top_right, TRUE);
		break;
	case BottomLeftCorner:
		gtk_toggle_button_set_active (bottom_left, TRUE);
		break;
	case BottomRightCorner:
		gtk_toggle_button_set_active (bottom_right, TRUE);
		break;
	case AllCorners:
	case NoTrigger:
	default:
		break;
	}
}

static gboolean
set_circular_scroll_trigger (GladeXML *dialog, ScrollTrigger trigger)
{
	GSynaptics *synaptics;
	synaptics = g_object_get_qdata (G_OBJECT(dialog), synaptics_quark);
	g_synaptics_set_circular_scroll_trigger (synaptics, trigger);
	gconf_client_set_int (gconf, CIRCSCROLLTRIGGER_KEY, (gint)trigger, NULL);
	set_circular_scroll_trigger_state (dialog);
	return TRUE;
}

static gboolean
cb_trigger_top_pressed (GtkWidget *widget, GdkEventKey *event, GladeXML *dialog)
{
	return set_circular_scroll_trigger (dialog, TopEdge);
}

static gboolean
cb_trigger_bottom_pressed (GtkWidget *widget, GdkEventKey *event, GladeXML *dialog)
{
	return set_circular_scroll_trigger (dialog, BottomEdge);
}

static gboolean
cb_trigger_left_pressed (GtkWidget *widget, GdkEventKey *event, GladeXML *dialog)
{
	return set_circular_scroll_trigger (dialog, LeftEdge);
}

static gboolean
cb_trigger_right_pressed (GtkWidget *widget, GdkEventKey *event, GladeXML *dialog)
{
	return set_circular_scroll_trigger (dialog, RightEdge);
}

static gboolean
cb_trigger_top_left_pressed (GtkWidget *widget, GdkEventKey *event, GladeXML *dialog)
{
	return set_circular_scroll_trigger (dialog, TopLeftCorner);
}

static gboolean
cb_trigger_top_right_pressed (GtkWidget *widget, GdkEventKey *event, GladeXML *dialog)
{
	return set_circular_scroll_trigger (dialog, TopRightCorner);
}

static gboolean
cb_trigger_bottom_left_pressed (GtkWidget *widget, GdkEventKey *event, GladeXML *dialog)
{
	return set_circular_scroll_trigger (dialog, BottomLeftCorner);
}

static gboolean
cb_trigger_bottom_right_pressed (GtkWidget *widget, GdkEventKey *event, GladeXML *dialog)
{
	return set_circular_scroll_trigger (dialog, BottomRightCorner);
}

static void
cb_trigger_all_clicked (GtkButton *button, GladeXML *dialog)
{
	GSynaptics *synaptics;

	synaptics = g_object_get_qdata (G_OBJECT(dialog), synaptics_quark);
	if (g_synaptics_circular_scroll_trigger (synaptics) != AllCorners)
		g_synaptics_set_circular_scroll_trigger (synaptics, AllCorners);
	else
		g_synaptics_set_circular_scroll_trigger (synaptics, NoTrigger);
	set_circular_scroll_trigger_state (dialog);
}

static void
cb_continue_scrolling_toggled (GtkToggleButton *button, GladeXML *dialog)
{
	GSynaptics *synaptics;
	gboolean value;
	
	value = gtk_toggle_button_get_active (button);

	synaptics = g_object_get_qdata (G_OBJECT(dialog), synaptics_quark);

	g_synaptics_set_coasting_enabled (synaptics, value);
	gconf_client_set_bool (gconf, COASTENABLE_KEY, value, NULL);
}

static void
cb_cursor_movement_toggled (GtkToggleButton *button, GladeXML *dialog)
{
	GSynaptics *synaptics;
	gboolean value;
	
	value = gtk_toggle_button_get_active (button);

	synaptics = g_object_get_qdata (G_OBJECT(dialog), synaptics_quark);

	g_synaptics_set_edge_motion_enabled (synaptics, value);
	gconf_client_set_bool (gconf, EDGEMOTIONUSEALWAYS_KEY, value, NULL);
}

static void
setup_dialog (GladeXML *dialog)
{
	GdkPixbuf *icon_pixbuf = NULL;
	GtkWidget *widget, *button;
	GtkWidget *check, *scale;
	GSynaptics *synaptics;
	
	gchar *path;
	gboolean value;
	gint num;
	
	synaptics = g_object_get_qdata (G_OBJECT(dialog), synaptics_quark);
	
	/* first pain */
	check = WID ("touchpad_check");
	g_signal_connect (G_OBJECT (check), "toggled",
			  G_CALLBACK (cb_touchpad_toggled), dialog);
	value = g_synaptics_is_enabled (synaptics);
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (check),
	    			      value);

	scale = WID ("sensitivity_scale");
	num = gconf_client_get_int (gconf, SENSITIVITY_KEY, NULL);
	if (num == 0)
		num = g_synaptics_sensitivity (synaptics);
	gtk_range_set_value (GTK_RANGE (scale), num);
	g_signal_connect (G_OBJECT (scale), "change-value",
			  G_CALLBACK (cb_sensitivity_change), dialog);

	/* second pain (tapping) */
	check = WID ("faster_tapping_check");
	g_signal_connect (G_OBJECT (check), "toggled",
			  G_CALLBACK (cb_fast_taps_toggled), dialog);
	value = g_synaptics_are_fast_taps_enabled (synaptics);
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (check),
	    			      value);

	scale = WID ("tapping_time_scale");
	num = g_synaptics_tap_time (synaptics);
	gtk_range_set_value (GTK_RANGE (scale), num);
	g_signal_connect (G_OBJECT (scale), "change-value",
			  G_CALLBACK (cb_tap_time_change), dialog);

	check = WID ("tapping_check");
	g_signal_connect (G_OBJECT (check), "toggled",
			  G_CALLBACK (cb_tapping_toggled), dialog);
	value = g_synaptics_is_tapping_enabled (synaptics);
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (check),
	    			      value);

	/* third pain (scroll) */
	scale = WID ("vertical_scroll_scale");
	num = g_synaptics_vertical_scroll_delta (synaptics);
	gtk_range_set_value (GTK_RANGE (scale), num * -1);
	g_signal_connect (G_OBJECT (scale), "change-value",
			  G_CALLBACK (cb_vertical_scroll_change), dialog);

	check = WID ("vertical_scroll_check");
	value = g_synaptics_is_vertical_scroll_enabled (synaptics);
	g_signal_connect (G_OBJECT (check), "toggled",
			  G_CALLBACK (cb_vertical_scroll_toggled), dialog);
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (check),
	    			      value);

	scale = WID ("horizontal_scroll_scale");
	num = g_synaptics_horizontal_scroll_delta (synaptics);
	gtk_range_set_value (GTK_RANGE (scale), num * -1);
	g_signal_connect (G_OBJECT (scale), "change-value",
			  G_CALLBACK (cb_horizontal_scroll_change), dialog);

	check = WID ("horizontal_scroll_check");
	value = g_synaptics_is_horizontal_scroll_enabled (synaptics);
	g_signal_connect (G_OBJECT (check), "toggled",
			  G_CALLBACK (cb_horizontal_scroll_toggled), dialog);
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (check),
	    			      value);

	scale = WID ("circular_scroll_scale");
	num = g_synaptics_circular_scroll_delta (synaptics);
	gtk_range_set_value (GTK_RANGE (scale), num * -1);
	g_signal_connect (G_OBJECT (scale), "change-value",
			  G_CALLBACK (cb_circular_scroll_change), dialog);

	check = WID ("circular_scroll_check");
	value = g_synaptics_is_circular_scroll_enabled (synaptics);
	g_signal_connect (G_OBJECT (check), "toggled",
			  G_CALLBACK (cb_circular_scroll_toggled), dialog);
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (check),
	    			      value);

	button = WID ("trigger_top_edge_toggle");
	g_signal_connect (G_OBJECT (button), "button_press_event",
			  G_CALLBACK (cb_trigger_top_pressed),dialog);
	button = WID ("trigger_bottom_edge_toggle");
	g_signal_connect (G_OBJECT (button), "button_press_event",
			  G_CALLBACK (cb_trigger_bottom_pressed),dialog);
	button = WID ("trigger_left_edge_toggle");
	g_signal_connect (G_OBJECT (button), "button_press_event",
			  G_CALLBACK (cb_trigger_left_pressed),dialog);
	button = WID ("trigger_right_edge_toggle");
	g_signal_connect (G_OBJECT (button), "button_press_event",
			  G_CALLBACK (cb_trigger_right_pressed),dialog);
	button = WID ("trigger_top_left_corner_toggle");
	g_signal_connect (G_OBJECT (button), "button_press_event",
			  G_CALLBACK (cb_trigger_top_left_pressed),dialog);
	button = WID ("trigger_top_right_corner_toggle");
	g_signal_connect (G_OBJECT (button), "button_press_event",
			  G_CALLBACK (cb_trigger_top_right_pressed),dialog);
	button = WID ("trigger_bottom_left_corner_toggle");
	g_signal_connect (G_OBJECT (button), "button_press_event",
			  G_CALLBACK (cb_trigger_bottom_left_pressed),dialog);
	button = WID ("trigger_bottom_right_corner_toggle");
	g_signal_connect (G_OBJECT (button), "button_press_event",
			  G_CALLBACK (cb_trigger_bottom_right_pressed),dialog);
	button = WID ("trigger_all_corners_button");
	g_signal_connect (G_OBJECT (button), "clicked",
			  G_CALLBACK (cb_trigger_all_clicked),dialog);
	set_circular_scroll_trigger_state (dialog);

	check = WID ("continue_scrolling_check");
	value = g_synaptics_is_coasting_enabled (synaptics);
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (check),
	    			      value);
	g_signal_connect (G_OBJECT (check), "toggled",
			  G_CALLBACK (cb_continue_scrolling_toggled), dialog);

	check = WID ("cursor_movement_check");
	value = g_synaptics_is_edge_motion_enabled (synaptics);
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (check),
	    			      value);
	g_signal_connect (G_OBJECT (check), "toggled",
			  G_CALLBACK (cb_cursor_movement_toggled), dialog);

	/* dialog window */
	widget = WID ("gsynaptics_dialog");
	g_signal_connect (G_OBJECT (widget), "response",
			  G_CALLBACK (dialog_button_clicked_cb), NULL);

	/* gtk_window_set_resizable (GTK_WINDOW (widget), FALSE); */

	gtk_window_set_default_size (GTK_WINDOW (widget), 400, -1);

	path = gnome_pixmap_file ("touchpad.png");
	if (path)
	{
		icon_pixbuf = gdk_pixbuf_new_from_file (path, NULL);
		gtk_window_set_icon (GTK_WINDOW (widget), icon_pixbuf);
		g_free (path);
		if (icon_pixbuf)
			g_object_unref (icon_pixbuf);
	}

	gtk_widget_show_all (widget);
}

static void
setup_user_value (GladeXML *dialog)
{
	/* sensitivity */
}

int
main (int argc, char **argv)
{
	GladeXML       *dialog;
	GSynaptics *synaptics;

	bindtextdomain (GETTEXT_PACKAGE, GSYNAPTICS_LOCALEDIR);
	bind_textdomain_codeset (GETTEXT_PACKAGE, "UTF-8");
	textdomain (GETTEXT_PACKAGE);

	gnome_program_init ("gsynaptics", VERSION,
			LIBGNOMEUI_MODULE, argc, argv,
			GNOME_PARAM_APP_DATADIR, DATADIR,
			NULL);

	synaptics = g_synaptics_new ();
	if (!g_synaptics_is_available(synaptics))
	{
		GtkWidget *dialog;
		g_object_unref (synaptics);
		dialog = gtk_message_dialog_new (NULL, GTK_DIALOG_MODAL, GTK_MESSAGE_ERROR, GTK_BUTTONS_CLOSE, 
					_("GSynaptics couldn't initialize.\n"
			      		"You have to set 'SHMconfig' 'true' in xorg.conf of XF86Config to use GSynaptics\n"));
		gtk_dialog_run (GTK_DIALOG(dialog));
		gtk_widget_destroy (dialog);
		
		exit(1);
	}

	gconf = gconf_client_get_default ();

	dialog = create_dialog ();
	
	synaptics_quark = g_quark_from_string("Touchpad::GSynaptics");

	g_object_set_qdata (G_OBJECT(dialog), synaptics_quark, synaptics);

	setup_dialog (dialog);
	/* set user value */
	setup_user_value (dialog);

	gtk_main ();

	g_object_unref (synaptics);

	g_object_unref (gconf);

	return 0;
}
