/*
copyright (c) 2007 Kazuki IWAMOTO http://www.maid.org/ iwm@maid.org

Permission is hereby granted, free of charge, to any person obtaining
a copy of this software and associated documentation files (the
"Software"), to deal in the Software without restriction, including
without limitation the rights to use, copy, modify, merge, publish,
distribute, sublicense, and/or sell copies of the Software, and to
permit persons to whom the Software is furnished to do so, subject to
the following conditions:

The above copyright notice and this permission notice shall be included
in all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY
CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,
TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
*/
#include <windows.h>
#include <ctype.h>
#include <errno.h>
#include <limits.h>
#include <locale.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <wchar.h>
#include "iconv.h"


#ifndef LC_MESSAGES
# define LC_MESSAGES 1729
#endif /* not LC_MESSAGES */


typedef struct domainlist
{
  char *domainname;
  char *dirname;
  char *codeset;
  char **msgid;
  char **msgstr;
  struct domainlist *next;
} DOMAINLIST;


static char *current = NULL, *nlspath = NULL;
static DOMAINLIST *domainlist = NULL;
const int libintl_version = 0x001000;


static unsigned
libintl_endian (const unsigned s)
{
  return s >> 24 | s >> 8 & 0xff00 | s << 8 & 0xff0000 | s << 24;
}


static char *
libintl_strdup (const char *str)
{
  char *ret = NULL;

  if (str)
    {
      ret = malloc (strlen (str) + 1);
      strcpy (ret, str);
    }
  return ret;
}


static void
libintl_strfreev (char **str)
{
  if (str)
    {
      int i;

      for (i = 0; str[i]; i++)
        free (str[i]);
      free (str);
    }
}


char *
libintl_bind_textdomain_codeset (const char *domainname,
                                 const char *codeset)
{
  DOMAINLIST *d;

  if (!domainname)
    return NULL;
  for (d = domainlist; d; d = d->next)
    if (strcmp (domainname, d->domainname) == 0)
      {
        free (d->codeset);
        if (codeset)
          d->codeset = libintl_strdup (codeset);
        libintl_strfreev (d->msgid);
        libintl_strfreev (d->msgstr);
        d->msgid = d->msgstr = NULL;
        break;
      }
  if (!d)
    {
      d = malloc (sizeof (DOMAINLIST));
      d->domainname = libintl_strdup (domainname);
      d->dirname = NULL;
      d->codeset = libintl_strdup (codeset);
      d->msgid = d->msgstr = NULL;
      d->next = domainlist;
      domainlist = d;
    }
  return d->codeset;
}


char *
libintl_bindtextdomain (const char *domainname,
                        const char *dirname)
{
  DOMAINLIST *d;

  if (!domainname || domainname[0] == '\0')
    return NULL;
  for (d = domainlist; d; d = d->next)
    if (strcmp (domainname, d->domainname) == 0)
      {
        free (d->dirname);
        d->dirname = libintl_strdup (dirname);
        libintl_strfreev (d->msgid);
        libintl_strfreev (d->msgstr);
        d->msgid = d->msgstr = NULL;
        break;
      }
  if (!d)
    {
      d = malloc (sizeof (DOMAINLIST));
      d->domainname = libintl_strdup (domainname);
      d->dirname = libintl_strdup (dirname);
      d->codeset = NULL;
      d->msgid = d->msgstr = NULL;
      d->next = domainlist;
      domainlist = d;
    }
  if (d->dirname)
    {
      int i;

      for (i = 0; d->dirname[i] != '\0';
                                    i += isleadbyte (d->dirname[i]) ? 2 : 1)
        if (d->dirname[i] == '/')
          d->dirname[i] = '\\';
    }
  return d->dirname;
}


char *
libintl_textdomain (const char *domainname)
{
  if (domainname)
    {
      free (current);
      current = libintl_strdup (domainname[0] == '\0' ? "message"
                                                      : domainname);
    }
  return current;
}


static char *
libintl_iconv (iconv_t     cd,
               const char *str)
{
  char *inbuf, *outbuf, *buf = NULL;
  int done = 0, reset = 0;
  size_t inbytes, outbytes, bytes, ret = (size_t)-1;

  if (cd == (iconv_t)-1 || !str)
    return NULL;
  inbytes = strlen (str);
  inbuf = (char *)str;
  bytes = outbytes = inbytes * 3;
  buf = outbuf = malloc (bytes + 1);
  while (!done)
    {
      ret = iconv (cd, &inbuf, &inbytes, &outbuf, &outbytes);
      if (ret == (size_t)-1)
        {
          size_t used;

          if (errno != E2BIG)
            break;
          used = outbuf - buf;
          bytes = (bytes - used) * 2 + used;
          buf = realloc (buf, bytes + 1);
          outbuf = buf + used;
          outbytes = bytes - used;
        }
      else
        {
          if (reset)
            {
              done = 1;
            }
          else
            {
              reset = 1;
              inbytes = 0;
              inbuf = NULL;
            }
        }
    }
  if (ret == (size_t)-1)
    {
      free (buf);
      return NULL;
    }
  *outbuf = '\0';
  return buf;
}


char *
libintl_dcngettext (const char *domainname,
                    const char *msgid1,
                    const char *msgid2,
                    unsigned    n,
                    int         category)
{
  const char *name;
  FILE *fp = NULL;
  DOMAINLIST *d = NULL;

  if (!msgid1)
    return (char *)msgid2;
  name = domainname ? domainname : current;
  for (d = domainlist; d; d = d->next)
    if (strcmp (name, d->domainname) == 0)
      break;
  if (!d)
    {
      d = malloc (sizeof (DOMAINLIST));
      d->domainname = libintl_strdup (name);
      d->dirname = NULL;
      d->codeset = NULL;
      d->msgid = d->msgstr = NULL;
      d->next = domainlist;
      domainlist = d;
    }
  if (!d->msgid || !d->msgstr)
    {
      char lang[8], *ctry[4], ctry0[9], ctry1[9], ctry2[9];
      int i = 0;
      LCID lcid;
      LANGID langid;
      WORD primary, sub;

      libintl_strfreev (d->msgid);
      libintl_strfreev (d->msgstr);
      d->msgid = d->msgstr = NULL;
      lcid = GetThreadLocale ();
      langid = LANGIDFROMLCID (lcid);
      primary = PRIMARYLANGID (langid);
      sub = SUBLANGID (langid);
      if (GetLocaleInfo (lcid, LOCALE_SISO3166CTRYNAME,
                                            ctry0 + 1, sizeof (ctry0) - 1) > 0)
        {
          ctry0[0] = '_';
          ctry[i++] = ctry0;
        }
      if (primary == LANG_CHINESE)
        {
          WORD sortid;
          LCID lcid1 = LOCALE_NEUTRAL, lcid2 = LOCALE_NEUTRAL;

          sortid = SORTIDFROMLCID (lcid);
          switch (sub)
            {
              case SUBLANG_CHINESE_TRADITIONAL:
                lcid1 = MAKELCID (MAKELANGID (LANG_CHINESE,
                                        SUBLANG_CHINESE_HONGKONG), sortid);
                lcid2 = MAKELCID (MAKELANGID (LANG_CHINESE,
                                        SUBLANG_CHINESE_MACAU), sortid);
                break;
              case SUBLANG_CHINESE_SIMPLIFIED:
                lcid1 = MAKELCID (MAKELANGID (LANG_CHINESE,
                                        SUBLANG_CHINESE_SINGAPORE), sortid);
                break;
              case SUBLANG_CHINESE_HONGKONG:
                lcid1 = MAKELCID (MAKELANGID (LANG_CHINESE,
                                        SUBLANG_CHINESE_TRADITIONAL), sortid);
                lcid2 = MAKELCID (MAKELANGID (LANG_CHINESE,
                                        SUBLANG_CHINESE_MACAU), sortid);
                break;
              case SUBLANG_CHINESE_SINGAPORE:
                lcid1 = MAKELCID (MAKELANGID (LANG_CHINESE,
                                        SUBLANG_CHINESE_SIMPLIFIED), sortid);
                break;
              case SUBLANG_CHINESE_MACAU:
                lcid1 = MAKELCID (MAKELANGID (LANG_CHINESE,
                                        SUBLANG_CHINESE_TRADITIONAL), sortid);
                lcid2 = MAKELCID (MAKELANGID (LANG_CHINESE,
                                        SUBLANG_CHINESE_HONGKONG), sortid);
            }
          if (lcid1 != LOCALE_NEUTRAL)
            {
              if (GetLocaleInfo (lcid1, LOCALE_SISO3166CTRYNAME,
                                            ctry1 + 1, sizeof (ctry1) - 1) > 0)
                {
                  ctry1[0] = '_';
                  ctry[i++] = ctry1;
                }
              if (lcid2 != LOCALE_NEUTRAL && GetLocaleInfo (lcid2,
                                            LOCALE_SISO3166CTRYNAME,
                                            ctry2 + 1, sizeof (ctry2) - 1) > 0)
                {
                  ctry2[0] = '_';
                  ctry[i++] = ctry2;
                }
            }
          ctry[i++] = "";
        }
      else 
        {
          ctry[i++] = "";
          if (sub != SUBLANG_DEFAULT
            && GetLocaleInfo (MAKELCID (MAKELANGID (primary, SUBLANG_DEFAULT),
                                                        SORTIDFROMLCID (lcid)),
                LOCALE_SISO3166CTRYNAME, ctry1 + 1, sizeof (ctry1) - 1) > 0)
            {
              ctry1[0] = '_';
              ctry[i++] = ctry1;
            }
        }
      while (i < 4)
        ctry[i++] = NULL;
      if (GetLocaleInfo (lcid, LOCALE_SISO639LANGNAME,
                                                    lang, sizeof (lang)) > 0)
        {
          char *dir, *file, *script = NULL;

          dir = d->dirname ? d->dirname : nlspath;
          file = malloc (strlen (dir) + 8 + 1 + 8 + 5 + 1 + 11 + 1
                                                    + strlen (name) + 3 + 1);
          switch (primary)
            {
              case LANG_AZERI:
                switch (sub)
                  {
                    case SUBLANG_AZERI_LATIN:
                      script = "@Latn";
                      break;
                    case SUBLANG_AZERI_CYRILLIC:
                      script = "@Cyrl";
                  }
                break;
              case LANG_SERBIAN:
                if (sub == SUBLANG_SERBIAN_LATIN)
                  script = "@Latn";
                break;
              case LANG_UZBEK:
                switch (sub)
                  {
                    case SUBLANG_UZBEK_LATIN:
                      script = "@Latn";
                      break;
                    case SUBLANG_UZBEK_CYRILLIC:
                      script = "@Cyrl";
                  }
            }
          for (i = 0; i < (category == LC_MESSAGES ? 8 : 16); i++)
            {
              if (!ctry[(i / 2) % 4] || i % 2 == 0 && !script)
                continue;
              strcpy (file, dir);
              strcat (file, "\\");
              strcat (file, lang);
              strcat (file, ctry[(i / 2) % 4]);
              if (i % 2 == 0)
                strcat (file, script);
              if (i < 8)
                switch (category)
                  {
                    case LC_CTYPE: strcat (file, "\\LC_CTYPE\\"); break;
                    case LC_NUMERIC: strcat (file, "\\LC_NUMERIC\\"); break;
                    case LC_TIME: strcat (file, "\\LC_TIME\\"); break;
                    case LC_COLLATE: strcat (file, "\\LC_COLLATE\\"); break;
                    case LC_MONETARY: strcat (file, "\\LC_MONETARY\\"); break;
                    default: strcat (file, "\\LC_MESSAGES\\");
                  }
              else
                strcat (file, "\\LC_MESSAGES\\");
              strcat (file, name);
              strcat (file, ".mo");
              fp = fopen (file, "rb");
              if (fp)
                break;
            }
          free (file);
        }
    }
  if (fp)
    {
      unsigned magic, revision, offid, offstr;
      size_t len;

      if (fread (&magic, sizeof (unsigned), 1, fp) == 1
            && fread (&revision, sizeof (unsigned), 1, fp) == 1
            && fread (&len, sizeof (size_t), 1, fp) == 1
            && fread (&offid, sizeof (unsigned), 1, fp) == 1
            && fread (&offstr, sizeof (unsigned), 1, fp) == 1
            && (magic == 0x950412de || magic == 0xde120495) && revision == 0)
        {
          unsigned *tblid, *tblstr;

          if (magic == 0xde120495)
            {
              len = libintl_endian (len);
              offid = libintl_endian (offid);
              offstr = libintl_endian (offstr);
            }
          tblid = malloc (len * sizeof (unsigned) * 2);
          tblstr = malloc (len * sizeof (unsigned) * 2);
          d->msgid = calloc (len + 1, sizeof (char *));
          d->msgstr = calloc (len + 1, sizeof (char *));
          if (fseek (fp, offid, SEEK_SET) == 0
                && fread (tblid, sizeof (unsigned), len * 2, fp) == len * 2
                && fseek (fp, offstr, SEEK_SET) == 0
                && fread (tblstr, sizeof (unsigned), len * 2, fp) == len * 2)
            {
              int i;

              if (magic == 0xde120495)
                for (i = 0; i < len * 2; i++)
                  {
                    tblid[i] = libintl_endian (tblid[i]);
                    tblstr[i] = libintl_endian (tblstr[i]);
                  }
              for (i = 0; i < len; i++)
                {
                  d->msgid[i] = calloc (tblid[i * 2] + 1, sizeof (char));
                  d->msgstr[i] = calloc (tblstr[i * 2] + 1, sizeof (char));
                  if (fseek (fp, tblid[i * 2 + 1], SEEK_SET) == 0)
                    fread (d->msgid[i], sizeof (char), tblid[i * 2], fp);
                  if (fseek (fp, tblstr[i * 2 + 1], SEEK_SET) == 0)
                    fread (d->msgstr[i], sizeof (char), tblstr[i * 2], fp);
                }
            }
          free (tblid);
          free (tblstr);
        }
      fclose (fp);
      if (d->msgid && d->msgstr && d->codeset)
        {
          int i;
          iconv_t cd = (iconv_t)-1;

          for (i = 0; d->msgid[i]; i++)
            if ((d->msgid[i])[0] == '\0' && d->msgstr[i])
              {
                char *str, *p;

                str = libintl_strdup (d->msgstr[i]);
                for (p = strtok (str, "\n"); p; p = strtok (NULL, "\n"))
                  if (strnicmp (p, "Content-Type", 12) == 0)
                    {
                      for (p += 12; isspace (*p); p++);
                      if (*p == ':')
                        for (p = strtok (p + 1, ";"); p;
                                                        p = strtok (NULL, ";"))
                          {
                            while (isspace (*p)) p++;
                            if (strnicmp (p, "charset", 7) == 0)
                              {
                                for (p += 7; isspace (*p); p++);
                                if (*p == '=')
                                  {
                                    char *q;

                                    while (isspace (*++p));
                                    for (q = p; *q != '\0' && *q != ';'; q++);
                                    q = '\0';
                                    if (stricmp (d->codeset, p) != 0)
                                      cd = iconv_open (d->codeset, p);
                                  }
                                break;
                              }
                          }
                      break;
                    }
                free (str);
                break;
              }
          if (cd != (iconv_t)-1)
            {
              int i;

              for (i = 0; d->msgid[i]; i++)
                {
                  char *str;

                  str = libintl_iconv (cd, d->msgid[i]);
                  if (str)
                    {
                      free (d->msgid[i]);
                      d->msgid[i] = str;
                    }
                  str = libintl_iconv (cd, d->msgstr[i]);
                  if (str)
                    {
                      free (d->msgstr[i]);
                      d->msgstr[i] = str;
                    }
                }
              iconv_close (cd);
            }
        }
    }
  if (d->msgid && d->msgstr)
    {
      int i;

      for (i = 0; d->msgid[i]; i++)
        if (strcmp (d->msgid[i], msgid1) == 0 && d->msgstr[i])
          return d->msgstr[i];
    }
  return (char *)(n == 1 ? msgid1 : msgid2);
}


char *
libintl_gettext(const char *msgid)
{
  return libintl_dcngettext (NULL, msgid, NULL, 1, LC_MESSAGES);
}


char *
libintl_dgettext (const char *domainname,
                  const char *msgid)
{
  return libintl_dcngettext (domainname, msgid, NULL, 1, LC_MESSAGES);
}


char *
libintl_dcgettext (const char *domainname,
                   const char *msgid,
                   int         category)
{
  return libintl_dcngettext (domainname, msgid, NULL, 1, category);
}


char *
libintl_ngettext (const char *msgid1,
                  const char *msgid2,
                  unsigned    n)
{
  return libintl_dcngettext (NULL, msgid1, msgid2, n, LC_MESSAGES);
}


char *
libintl_dngettext (const char *domainname,
                   const char *msgid1,
                   const char *msgid2,
                   unsigned    n)
{
  return libintl_dcngettext (domainname, msgid1, msgid2, n, LC_MESSAGES);
}


/******************************************************************************
*                                                                             *
******************************************************************************/
int
libintl_vprintf (const char *format,
                 va_list     argptr)
{
  return vprintf (format, argptr);
}


int
libintl_vfprintf (FILE       *stream,
                  const char *format,
                  va_list     argptr)
{
  return vfprintf (stream, format, argptr);
}


int
libintl_vsprintf (char       *buffer,
                  const char *format,
                  va_list     argptr)
{
  return vsprintf (buffer, format, argptr);
}


int
libintl_vsnprintf (char       *buffer,
                   size_t      count,
                   const char *format,
                   va_list     argptr)
{
  return 0;
}


int
libintl_vwprintf (const wchar_t *format,
                  va_list        argptr)
{
  return vwprintf (format, argptr);
}


int
libintl_vfwprintf (FILE          *stream,
                   const wchar_t *format,
                   va_list        argptr)
{
  return vfwprintf (stream, format, argptr);
}


int
libintl_vswprintf (wchar_t       *buffer,
                   const wchar_t *format,
                   va_list        argptr)
{
  return 0;
}


int
libintl_vsnwprintf (wchar_t       *buffer,
                    size_t         count,
                    const wchar_t *format,
                    va_list        argptr)
{
  return 0;
}


int
libintl_printf (const char *format, ...)
{
  int ret;
  va_list arglist;

  va_start (arglist, format);
  ret = libintl_vprintf (format, arglist);
  va_end (arglist);
  return ret;
}


int
libintl_fprintf (FILE       *stream,
                 const char *format, ...)
{
  int ret;
  va_list arglist;

  va_start (arglist, format);
  ret = libintl_vfprintf (stream, format, arglist);
  va_end (arglist);
  return ret;
}

int
libintl_sprintf (char       *buffer,
                 const char *format, ...)
{
  int ret;
  va_list arglist;

  va_start (arglist, format);
  ret = libintl_vsprintf (buffer, format, arglist);
  va_end (arglist);
  return ret;
}


int
libintl_snprintf (char       *buffer,
                  size_t      count,
                  const char *format, ...)
{
  int ret;
  va_list arglist;

  va_start (arglist, format);
  ret = libintl_vsnprintf (buffer, count, format, arglist);
  va_end (arglist);
  return ret;
}


int
libintl_wprintf (const wchar_t *format, ...)
{
  int ret;
  va_list arglist;

  va_start (arglist, format);
  ret = libintl_vwprintf (format, arglist);
  va_end (arglist);
  return ret;
}


int
libintl_fwprintf (FILE          *stream,
                  const wchar_t *format, ...)
{
  int ret;
  va_list arglist;

  va_start (arglist, format);
  ret = libintl_vfwprintf (stream, format, arglist);
  va_end (arglist);
  return ret;
}

int
libintl_swprintf (wchar_t       *buffer,
                  const wchar_t *format, ...)
{
  int ret;
  va_list arglist;

  va_start (arglist, format);
  ret = libintl_vswprintf (buffer, format, arglist);
  va_end (arglist);
  return ret;
}


int
libintl_snwprintf (wchar_t       *buffer,
                   size_t         count,
                   const wchar_t *format, ...)
{
  int ret;
  va_list arglist;

  va_start (arglist, format);
  ret = libintl_vsnwprintf (buffer, count, format, arglist);
  va_end (arglist);
  return ret;
}


/******************************************************************************
*                                                                             *
******************************************************************************/
void
libintl_set_relocation_prefix (const char *orig_prefix,
                               const char *curr_prefix)
{
}


BOOL WINAPI
DllMain (HINSTANCE hinstDLL,
         DWORD     fdwReason,
         LPVOID    lpvReserved)
{
  switch (fdwReason)
    {
      case DLL_PROCESS_ATTACH:
        {
          char *p = NULL;
          int i;
          size_t len;

          current = libintl_strdup ("messages");
          len = GetCurrentDirectoryA (0, NULL);
          nlspath = malloc (len + 13);
          GetCurrentDirectoryA (len, nlspath);
          for (i = 0; nlspath[i] != '\0'; i += isleadbyte (nlspath[i]) ? 2 : 1)
            if (nlspath[i] == '\\')
              p = nlspath + i;
          if (p && stricmp (p, "\\bin") == 0)
            *p = '\0';
          strcat (nlspath, "\\share\\locale");
        }
        break;
      case DLL_PROCESS_DETACH:
        free (current);
        free (nlspath);
        while (domainlist)
          {
            DOMAINLIST *d;

            d = domainlist->next;
            free (domainlist->domainname);
            free (domainlist->dirname);
            free (domainlist->codeset);
            libintl_strfreev (domainlist->msgid);
            libintl_strfreev (domainlist->msgstr);
            free (domainlist);
            domainlist = d;
          }
    }
  return TRUE;
}
