/*
    disw32
    copyright (c) 1998-2011 Kazuki Iwamoto http://www.maid.org/ iwm@maid.org

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
#include "code.h"
#include "data.h"
#include "misc/opcode.h"


/******************************************************************************
*                                                                             *
* ja:コード解析関数                                                           *
*                                                                             *
******************************************************************************/
static gint
compare (gconstpointer a,
         gconstpointer b)
{
  const Dw32Label *la, *lb;

  la = a;
  lb = b;
  if (!(la->type & DW32_TYPE_DATA) && (lb->type & DW32_TYPE_DATA))
    return G_MININT;
  else if ((la->type & DW32_TYPE_DATA) && !(lb->type & DW32_TYPE_DATA))
    return G_MAXINT;
  if (la->type == lb->type)
    return la->index - lb->index;
  return lb->type - la->type;
}


/*  ja:線形に逆アセンブルが可能か調べる
       info,イメージ情報構造体
      index,開始アドレス(イメージベース相対アドレス)
    checked,正当性(TRUEはチェック済み)
        RET,TRUE:逆アセンブルできる,FALSE:逆アセンブルできない              */
static gboolean
dw32_code_line (Dw32Info   *info,
                const gint  index,
                gboolean   *checked)
{
  gboolean *v;
  Opcode op;

  v = g_malloc0 (info->size * sizeof (gboolean));
  op.index = index;
  while (0 <= op.index && op.index < info->size
                            && (info->status[op.index] == DW32_STAT_UNKNOWN
                                || info->status[op.index] == DW32_STAT_CODE))
    {
      gint i, target;

      if (info->status[op.index] == DW32_STAT_CODE)
        {
          g_free (v);
          return TRUE;
        }
      target = op.index;
      if (!opcode_parse (info->image, info->size, info->base, &op))
        goto loop;
      for (i = target + 1; i < op.index; i++)
        if (info->status[i] != DW32_STAT_UNKNOWN)
          goto loop;
      if (op.type & OPCODE_TYPE_EXIT)
        {
          g_free (v);
          return TRUE;
        }
      v[target] = TRUE;
    }
  loop:
  if (checked)
    {
      gint i;

      for (i = index; i < op.index; i++)
        if (v[i])
          checked[i] = TRUE;
    }
  g_free (v);
  return FALSE;
}


static guint
dw32_code_type (guint type)
{
  return (type & OPCODE_TYPE_CALL ? DW32_TYPE_CALL : 0)
                                | (type & OPCODE_TYPE_JMP ? DW32_TYPE_JMP : 0);
}


/*  ja:サブルーチン全体を逆アセンブル
      info,イメージ情報構造体
     index,開始アドレス(イメージベース相対アドレス)
     force,TRUE:エラーでも解析を続ける,FALSE:エラーで中断する
       RET,TRUE:正常終了,FALSE:エラー                                       */
static gboolean
dw32_code_subroutine (Dw32Info       *info,
                      const gint      index,
                      const gboolean  force)
{
  gboolean ret = TRUE;
  Opcode op;
  GList *glist;

  /* ja:既に解析しているかチェックする */
  for (glist = g_list_first (info->subroutine);
                                            glist; glist = g_list_next (glist))
    {
      Dw32Label *label;

      label = glist->data;
      if (label->index == index)
        return TRUE;
    }
  if (!(0 <= index && index < info->size
                                && info->status[index] == DW32_STAT_UNKNOWN))
    return FALSE;
  op.index = index;
  while (0 <= op.index && op.index < info->size
                                && info->status[op.index] == DW32_STAT_UNKNOWN)
    {
      gboolean result;
      gint i, target;

      target = op.index;
      result = opcode_parse (info->image, info->size, info->base, &op);
      if (result)
        for (i = target; i < op.index; i++)
          if (info->status[i] != DW32_STAT_UNKNOWN)
            {
              result = FALSE;
              break;
            }
      if (result)
        {
          info->status[target] = DW32_STAT_CODE;
          for (i = target + 1; i < op.index; i++)
            info->status[i] = DW32_STAT_INTER;
          for (i = target; i < op.index; i++)
            info->belong[i] = index;
          /* ja:相対 */
          if (info->base <= op.relative
                                    && op.relative < info->base + info->size)
            {
              Dw32Label *label;

              label = g_malloc (sizeof (Dw32Label));
              label->name = NULL;
              label->index = op.relative - info->base;
              label->type = DW32_TYPE_RELATIVE | dw32_code_type (op.type);
              label->bit = 32;
              label->size = 0;
              info->analyse = g_list_insert_sorted (info->analyse,
                                                            label, compare);
            }
          /* ja:即値 */
          if (info->base <= op.immediate
                                    && op.immediate < info->base + info->size)
            {
              op.immediate -= info->base;
              for (glist = g_list_first (info->import);
                                            glist; glist = g_list_next (glist))
                {
                  Dw32Label *label;

                  label = glist->data;
                  if (label->index == op.immediate)
                    {
                      info->comment[target] = g_strdup (label->name);
                      break;
                    }
                }
              if (!glist)
                {
                  Dw32Label *label;

                  label = g_malloc (sizeof (Dw32Label));
                  label->name = NULL;
                  label->index = op.immediate;
                  label->type = DW32_TYPE_IMMEDIATE | dw32_code_type (op.type);
                  label->bit = 0;
                  label->size = 0;
                  info->analyse = g_list_insert_sorted (info->analyse,
                                                            label, compare);
                }
            }
          /* ja:オフセット */
          if (info->base <= op.offset && op.offset < info->base + info->size
                                - (op.bit == 32 ? 3 : op.bit == 16 ? 1 : 0))
            {
              op.offset -= info->base;
              for (glist = g_list_first (info->import);
                                            glist; glist = g_list_next (glist))
                {
                  Dw32Label *label;

                  label = glist->data;
                  if (label->index == op.offset)
                    {
                      info->comment[target] = g_strdup (label->name);
                      break;
                    }
                }
              if (!glist)
                {
                  Dw32Label *label;

                  label = g_malloc (sizeof (Dw32Label));
                  label->name = NULL;
                  label->index = op.offset;
                  label->type = DW32_TYPE_OFFSET | dw32_code_type (op.type);
                  label->bit = op.bit;
                  label->size = 0;
                  info->analyse = g_list_insert_sorted (info->analyse,
                                                            label, compare);
                }
            }
          /* ja:配列 */
          if (info->base <= op.array && op.array < info->base + info->size
                                - (op.bit == 32 ? 3 : op.bit == 16 ? 1 : 0))
            {
              Dw32Label *label;

              label = g_malloc (sizeof (Dw32Label));
              label->name = NULL;
              label->index = op.array - info->base;
              label->type = DW32_TYPE_ARRAY | dw32_code_type (op.type);
              label->bit = op.bit;
              label->size = 0;
              info->analyse = g_list_insert_sorted (info->analyse,
                                                            label, compare);
            }
          info->mnemonic[target] = g_strdup (op.mnemonic);
          if (op.type & OPCODE_TYPE_EXIT)
            op.index = -1;
          else if (!(0 <= op.index && op.index < info->size
                                && (info->status[op.index] == DW32_STAT_UNKNOWN
                                || (info->status[op.index] == DW32_STAT_CODE
                                                && info->belong[i] == index))))
            ret = FALSE;
        }
      else
        {
          op.index = -1;
          ret = FALSE;
        }
      if (!force && !ret)
        break;
      while ((info->analyse = g_list_first (info->analyse))
            && (((Dw32Label *)(info->analyse->data))->type & DW32_TYPE_JMP)
                            && !(0 <= op.index && op.index < info->size
                            && info->status[op.index] == DW32_STAT_UNKNOWN))
        {
          Dw32Label *label;

          op.index = -1;
          glist = g_list_first (info->analyse);
          label = glist->data;
          if (label->type & DW32_TYPE_RELATIVE)
            {
              op.index = label->index;
              info->analyse = g_list_delete_link (info->analyse, glist);
              g_free (label);
            }
          else if ((label->type & DW32_TYPE_OFFSET) && label->bit == 32)
            {
              op.index = GUINT32_FROM_LE (*(guint32 *)
                                    (info->image + label->index)) - info->base;
              if (dw32_code_line (info, op.index, NULL)
                        && info->status[label->index] == DW32_STAT_UNKNOWN
                        && info->status[label->index + 1] == DW32_STAT_UNKNOWN
                        && info->status[label->index + 2] == DW32_STAT_UNKNOWN
                        && info->status[label->index + 3] == DW32_STAT_UNKNOWN)
                dw32_data_make_dword (info, label->index, 4, index);
              else
                op.index = -1;
              info->analyse = g_list_delete_link (info->analyse, glist);
              g_free (label);
            }
          else if ((label->type & DW32_TYPE_ARRAY) && label->bit == 32)
            {
              gboolean result = FALSE;

              i = label->index + label->size;
              if (0 <= i && i < info->size - 3)
                {
                  op.index = GUINT32_FROM_LE (*(guint32 *)(info->image + i))
                                                                - info->base;
                  if (dw32_code_line (info, op.index, NULL)
                                && info->status[i] == DW32_STAT_UNKNOWN
                                && info->status[i + 1] == DW32_STAT_UNKNOWN
                                && info->status[i + 2] == DW32_STAT_UNKNOWN
                                && info->status[i + 3] == DW32_STAT_UNKNOWN)
                    {
                      label->size += 4;
                      result = TRUE;
                    }
                }
              if (!result)
                {
                  dw32_data_make_dword (info,
                                            label->index, label->size, index);
                  op.index = -1;
                  info->analyse = g_list_delete_link (info->analyse, glist);
                  g_free (label);
                }
            }
          else
            {
              info->analyse = g_list_delete_link (info->analyse, glist);
              g_free (label);
            }
        }
    }
  while ((info->analyse = g_list_first (info->analyse))
            && (((Dw32Label *)(info->analyse->data))->type & DW32_TYPE_JMP))
    info->analyse = g_list_delete_link (info->analyse, info->analyse);
  return ret;
}


/*  ja:indexに属する逆アセンブルを破棄する
      info,イメージ情報構造体
     index,開始アドレス(イメージベース相対アドレス)                         */
static void
dw32_code_eraser (Dw32Info   *info,
                  const gint  index)
{
  gint i;

  while (info->analyse)
    {
      Dw32Label *label;

      label = info->analyse->data;
      g_free (label->name);
      g_free (label);
      info->analyse = g_list_delete_link (info->analyse, info->analyse);
    }
  for (i = 0; i < info->size; i++)
    if (info->belong[i] == index)
      {
        g_free (info->comment[i]);
        g_free (info->mnemonic[i]);
        info->comment[i] = NULL;
        info->mnemonic[i] = NULL;
        info->status[i] = DW32_STAT_UNKNOWN;
        info->belong[i] = -1;
      }
}


/*  ja:可能ならばサブルーチンを逆アセンブル
     info,イメージ情報構造体
    index,開始アドレス(イメージベース相対アドレス)
      RET,TRUE:正常終了,FALSE:エラー                                        */
static gboolean
dw32_code_procedure (Dw32Info   *info,
                     const gint  index)
{
  gboolean result;
  GList *glist;

  glist = info->analyse;
  info->analyse = NULL;
  result = dw32_code_subroutine (info, index, FALSE);
  if (result)
    while (info->analyse)
      {
        glist = g_list_insert_sorted (glist, info->analyse->data, compare);
        info->analyse = g_list_delete_link (info->analyse, info->analyse);
      }
  else
    dw32_code_eraser (info, index);
  info->analyse = glist;
  return result;
}


/*  ja:すべての候補を逆アセンブル
    info,イメージ情報構造体                                                 */
static void
dw32_code_recursive (Dw32Info *info)
{
  /* ja;データ参照以外を逆アセンブル */
  while ((info->analyse = g_list_first (info->analyse))
            && !(((Dw32Label *)(info->analyse->data))->type & DW32_TYPE_DATA))
    {
      Dw32Label *label;
      GList *glist;

      glist = g_list_first (info->analyse);
      label = glist->data;
      if (label->type & DW32_TYPE_RELATIVE)
        {
          g_fprintf (stderr,
                    "%08x/%08"G_GSIZE_MODIFIER"x Recursive %d Relative\n",
                    label->index, info->size, g_list_length (info->analyse));
          info->analyse = g_list_delete_link (info->analyse, glist);
          dw32_code_subroutine (info, label->index, TRUE);
          info->subroutine = g_list_append (info->subroutine, label);
        }
      else if (label->type & DW32_TYPE_IMMEDIATE)
        {
          g_fprintf (stderr,
                    "%08x/%08"G_GSIZE_MODIFIER"x Recursive %d Immediate\n",
                    label->index, info->size, g_list_length (info->analyse));
          info->analyse = g_list_delete_link (info->analyse, glist);
          if (dw32_code_procedure (info, label->index))
            {
             info->subroutine = g_list_append (info->subroutine, label);
            }
          else
            {
              label->type = (label->type | DW32_TYPE_DATA) & ~DW32_TYPE_CALL;
              info->analyse = g_list_insert_sorted (info->analyse,
                                                            label, compare);
            }
        }
      else if (label->type & DW32_TYPE_OFFSET)
        {
          g_fprintf (stderr,
                    "%08x/%08"G_GSIZE_MODIFIER"x Recursive %d Offset\n",
                    label->index, info->size, g_list_length (info->analyse));
          info->analyse = g_list_delete_link (info->analyse, glist);
          if (label->bit == 32
                        && info->status[label->index] == DW32_STAT_UNKNOWN
                        && info->status[label->index + 1] == DW32_STAT_UNKNOWN
                        && info->status[label->index + 2] == DW32_STAT_UNKNOWN
                        && info->status[label->index + 3] == DW32_STAT_UNKNOWN)
            {
              gboolean result = FALSE;
              gint index;

              dw32_data_make_dword (info, label->index, 4, -1);
              index = GUINT32_FROM_LE (*(guint32 *)
                                    (info->image + label->index)) - info->base;
              if (0 <= index && index < info->size
                            && (info->status[index] == DW32_STAT_UNKNOWN
                                    || info->status[index] == DW32_STAT_CODE))
                {
                  if (label->type & DW32_TYPE_CALL)
                    {
                      dw32_code_subroutine (info, index, TRUE);
                      result = TRUE;
                    }
                  else
                    {
                      result = dw32_code_procedure (info, index);
                    }
                }
              if (result)
                {
                  label->index = index;
                  info->subroutine = g_list_append (info->subroutine, label);
                }
              else
                {
                  label->type = (label->type | DW32_TYPE_DATA)
                                                            & ~DW32_TYPE_CALL;
                  info->analyse = g_list_insert_sorted (info->analyse,
                                                            label, compare);
                }
            }
          else
            {
              g_free (label);
            }
        }
      else if (label->type & DW32_TYPE_ARRAY)
        {
          gboolean result = FALSE;
          gint i;

          g_fprintf (stderr,
                    "%08x/%08"G_GSIZE_MODIFIER"x Recursive %d Array\n",
                    label->index, info->size, g_list_length (info->analyse));
          i = label->index + label->size;
          if (label->bit == 32 && 0 <= i && i < info->size - 3)
            {
              gint index;

              index = GUINT32_FROM_LE (*(guint32 *)(info->image + i))
                                                                - info->base;
              if (info->status[i] == DW32_STAT_UNKNOWN
                                    && info->status[i + 1] == DW32_STAT_UNKNOWN
                                    && info->status[i + 2] == DW32_STAT_UNKNOWN
                                    && info->status[i + 3] == DW32_STAT_UNKNOWN
                                    && dw32_code_procedure (info, index))
                {
                  Dw32Label *l;

                  l = g_malloc (sizeof (Dw32Label));
                  l->name = NULL;
                  l->index = index;
                  l->type = label->type;
                  l->bit = 0;
                  l->size = 0;
                  info->subroutine = g_list_append (info->subroutine, l);
                  label->size += 4;
                  result = TRUE;
                }
            }
          if (!result)
            {
              info->analyse = g_list_delete_link (info->analyse, glist);
              if (label->size <= 0)
                {
                  label->type = (label->type | DW32_TYPE_DATA)
                                                            & ~DW32_TYPE_CALL;
                  info->analyse = g_list_insert_sorted (info->analyse,
                                                            label, compare);
                }
              else
                {
                  dw32_data_make_dword (info, label->index, label->size, -1);
                  g_free (label);
                }
            }
        }
      else
        {
          info->analyse = g_list_delete_link (info->analyse, glist);
          g_free (label);
        }
    }
  /* ja:データ参照 */
  while ((info->analyse = g_list_first (info->analyse)))
    {
      Dw32Label *label;
      GList *glist;

      glist = g_list_first (info->analyse);
      label = glist->data;
      if (label->type & DW32_TYPE_IMMEDIATE)
        {
          g_fprintf (stderr, "%08x/%08"G_GSIZE_MODIFIER"x Data %d Immediate\n",
                    label->index, info->size, g_list_length (info->analyse));
          if (dw32_data_is_stringw (info, label->index) > 0)
            dw32_data_make_stringw (info, label->index, -1);
          else if (dw32_data_is_string (info, label->index) > 0)
            dw32_data_make_string (info, label->index, -1);
        }
      else if (label->type & DW32_TYPE_OFFSET)
        {
          gint i;

          g_fprintf (stderr, "%08x/%08"G_GSIZE_MODIFIER"x Data %d Offset\n",
                    label->index, info->size, g_list_length (info->analyse));
          for (i = label->bit / 4 - 1; i >= 0; i--)
            if (info->status[label->index + i] != DW32_STAT_UNKNOWN)
              break;
          if (i < 0)
            {
              dw32_data_make_dword (info, label->index, label->bit / 4, -1);
              if (label->bit == 32)
                {
                  i = GUINT32_FROM_LE (*(guint32 *)
                                    (info->image + label->index)) - info->base;
                  if (dw32_data_is_stringw (info, i) > 0)
                    dw32_data_make_stringw (info, i, -1);
                  else if (dw32_data_is_string (info, i) > 0)
                    dw32_data_make_string (info, i, -1);
                }
            }
        }
      else if ((label->type & DW32_TYPE_ARRAY) && label->bit == 32)
        {
          gint i;

          g_fprintf (stderr, "%08x/%08"G_GSIZE_MODIFIER"x Data %d Array\n",
                    label->index, info->size, g_list_length (info->analyse));
          i = label->index;
          while (i < info->size - 3 && info->status[i] == DW32_STAT_UNKNOWN
                                  && info->status[i + 1] == DW32_STAT_UNKNOWN
                                  && info->status[i + 2] == DW32_STAT_UNKNOWN
                                  && info->status[i + 3] == DW32_STAT_UNKNOWN)
            {
              gint index;

              index = GUINT32_FROM_LE (*(guint32 *)(info->image + i))
                                                                - info->base;
              if (dw32_data_is_stringw (info, index) > 0)
                dw32_data_make_stringw (info, index, -1);
              else if (dw32_data_is_string (info, index) > 0)
                dw32_data_make_string (info, index, -1);
              else
                break;
              i += 4;
            }
          dw32_data_make_dword (info, label->index, i - label->index, -1);
        }
      info->analyse = g_list_delete_link (info->analyse, glist);
      g_free (label);
    }
}


/*  ja:コード解析
       info,イメージ情報構造体
       proc,TRUE:関数解析を行う
     linear,TRUE:線形解析を行う
    referer,TRUE:データ参照解析を行う
       data,TRUE:データ領域解析を行う                                       */
void
dw32_code_parse (Dw32Info       *info,
                 const gboolean  proc,
                 const gboolean  linear,
                 const gboolean  referer,
                 const gboolean  data)
{
  gboolean *checked;
  gint i, regcall[8];

  g_fprintf (stderr, "Base Analyse\n");
  /* ja:エントリーポイント,エクスポート */
  while (info->analyse)
    {
      Dw32Label *label;
      GList *glist;

      glist = g_list_first (info->analyse);
      label = glist->data;
      if ((label->type & DW32_TYPE_ENTRY) || (label->type & DW32_TYPE_EXPORT))
        {
          g_fprintf (stderr, "%08x/%08"G_GSIZE_MODIFIER"x %s",
                    label->index, info->size,
                    label->type & DW32_TYPE_ENTRY ? "Entry Point" : "Export");
          if (label->name && (label->type & DW32_TYPE_EXPORT))
            g_fprintf (stderr, " %s", label->name);
          g_fprintf (stderr, "\n");
          info->analyse = g_list_delete_link (info->analyse, glist);
          dw32_code_subroutine (info, label->index, TRUE);
          info->subroutine = g_list_append (info->subroutine, label);
        }
      else
        {
          break;
        }
    }
  /* ja:再帰的にコード化 */
  g_fprintf (stderr, "Recursive Analyse\n");
  dw32_code_recursive (info);
  checked = g_malloc0 (info->size * sizeof (gboolean));
  if (proc)
    {
      /* ja:関数の先頭と思われるコードを探す */
      g_fprintf (stderr, "Procedure Analyse\n");
      for (i = 0; i < info->size; i++)
        if (info->image[i] == 0x55 && info->status[i] == DW32_STAT_UNKNOWN
                        && !checked[i] && dw32_code_line (info, i, checked))
          {
            if (dw32_code_subroutine (info, i, FALSE))
              {
                g_fprintf (stderr, "%08x/%08"G_GSIZE_MODIFIER"x Procedure\n",
                                                                i, info->size);
                dw32_code_recursive (info);
              }
            else
              {
                dw32_code_eraser (info, i);
              }
          }
    }
  if (linear)
    {
      /* ja:未定部分のコード化を試みる */
      g_fprintf (stderr, "Linear Analyse\n");
      for (i = 0; i < info->size; i++)
        if (info->status[i] == DW32_STAT_UNKNOWN && !checked[i]
                                        && dw32_code_line (info, i, checked))
          {
            if (dw32_code_subroutine (info, i, FALSE))
              {
                g_fprintf (stderr, "%08x/%08"G_GSIZE_MODIFIER"x Linear\n",
                                                                i, info->size);
                dw32_code_recursive (info);
              }
            else
              {
                dw32_code_eraser (info, i);
              }
          }
    }
  if (referer)
    {
      /* ja:DWORD値を見つけて，その値のアドレスのコード化を試みる */
      g_fprintf (stderr, "Referer Analyse\n");
      for (i = 0; i < info->size - 3; i++)
        if (info->status[i] == DW32_STAT_UNKNOWN
            && info->status[i + 1] == DW32_STAT_UNKNOWN
            && info->status[i + 2] == DW32_STAT_UNKNOWN
            && info->status[i + 3] == DW32_STAT_UNKNOWN)
          {
            gint index;

            index = GUINT32_FROM_LE (*(guint32 *)(info->image + i))
                                                                - info->base;
            if (0 <= index && index < info->size
                                    && info->status[index] == DW32_STAT_UNKNOWN
                                    && !checked[index]
                                    && dw32_code_line (info, index, checked))
              {
                if (dw32_code_subroutine (info, index, FALSE))
                  {
                    g_fprintf (stderr, "%08x/%08"G_GSIZE_MODIFIER"x Referer\n",
                                                                i, info->size);
                    dw32_data_make_dword (info, i, 4, -1);
                    dw32_code_recursive (info);
                  }
                else
                  {
                    dw32_code_eraser (info, index);
                  }
              }
          }
    }
  g_free (checked);
  if (data)
    {
      /* ja:未定部分をデータ化する */
      g_fprintf (stderr, "Data\n");
      i = 0;
      while (i < info->size)
        if (info->status[i] == DW32_STAT_UNKNOWN)
          {
            gint j;

            for (j = i + 1; j < info->size; j++)
              if (info->status[j] != DW32_STAT_UNKNOWN)
                break;
            dw32_data_make_byte (info, i, j - i, -1);
            i = j;
          }
        else
          {
            i++;
          }
    }
  /* ja:同じ関数に挟まれた領域の整理 */
  g_fprintf (stderr, "Merge\n");
  for (i = 0; i < info->size - 2; i++)
    if (info->belong[i] != -1 && info->belong[i] != info->belong[i + 1])
      {
        gint j;

        for (j = i + 2; j < info->size; j++)
          if (info->belong[i + 1] != info->belong[j])
            {
              if (info->belong[i] == info->belong[j])
                while (i < --j)
                  info->belong[j] = info->belong[i];
              break;
            }
      }
  /* ja:サブルーチン */
  g_fprintf (stderr, "Subroutine\n");
  while (info->subroutine)
    {
      Dw32Label *label;

      label = info->subroutine->data;
      info->subroutine = g_list_delete_link (info->subroutine,
                                                            info->subroutine);
      if (0 <= label->index && label->index < info->size
                                            && !info->separate[label->index])
        {
          if (label->name)
            info->separate[label->index] = g_strdup_printf ("\n%s",
                                                                label->name);
          else
            info->separate[label->index] = g_strdup ("");
        }
      g_free (label->name);
      g_free (label);
    }
  /* ja:API呼び出しJMP */
  g_fprintf (stderr, "Comment\n");
  for (i = 0; i < info->size - 5; i++)
    if (info->image[i] == 0xff && info->image[i + 1] == 0x25
                                    && info->comment[i]
                                    && info->status[i] == DW32_STAT_CODE
                                    && info->status[i + 1] == DW32_STAT_INTER
                                    && info->status[i + 2] == DW32_STAT_INTER
                                    && info->status[i + 3] == DW32_STAT_INTER
                                    && info->status[i + 4] == DW32_STAT_INTER
                                    && info->status[i + 5] == DW32_STAT_INTER)
      {
        gint j;

        for (j = 0; j < info->size - 4; j++)
          if (info->image[j] == 0xe8 && !info->comment[j]
            && GUINT32_FROM_LE (*(guint32 *)(info->image + j + 1)) + j + 5 == i
            && info->status[j] == DW32_STAT_CODE
            && info->status[j + 1] == DW32_STAT_INTER
            && info->status[j + 2] == DW32_STAT_INTER
            && info->status[j + 3] == DW32_STAT_INTER
            && info->status[j + 4] == DW32_STAT_INTER)
            info->comment[j] = g_strdup (info->comment[i]);
      }
  /* ja:レジスタによるAPI呼び出し */
  for (i = 0; i < 8; i++)
    regcall[i] = -1;
  for (i = 0; i < info->size; i++)
    if (info->mnemonic[i])
      {
        if (g_strncmp (info->mnemonic[i], "mov ", 4) == 0 && info->comment[i])
          {
            gint j;

            for (j = 0; j < 8; j++)
              if (g_strncmp (info->mnemonic[i] + 4, regname[2][j], 3) == 0)
                {
                  regcall[j] = i;
                  break;
                }
          }
        else if (g_strncmp (info->mnemonic[i], "call ", 5) == 0
                                                        && !info->comment[i])
          {
            gint j;

            for (j = 0; j < 8; j++)
              if (g_strcmp (info->mnemonic[i] + 5, regname[2][j]) == 0
                                && regcall[j] != -1
                                && info->belong[i] == info->belong[regcall[j]])
                info->comment[i] = g_strdup (info->comment[regcall[j]]);
          }
        else
          {
            gchar *p;

            p = g_strchr (info->mnemonic[i], ' ');
            if (p)
              {
                gint j;

                p++;
                for (j = 0; j < 8; j++)
                  if (g_strncmp (p, regname[2][j], 3) == 0 && p[3] == ',')
                    {
                      regcall[j] = -1;
                      break;
                    }
              }
          }
      }
}
