/*
    peimage
    copyright (c) 1998-2011 Kazuki Iwamoto http://www.maid.org/ iwm@maid.org

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
#include "peimage.h"
#include "misc/fileio.h"


/******************************************************************************
*                                                                             *
* ja:PEイメージ関数                                                           *
*                                                                             *
******************************************************************************/
/*  ja:PEイメージの仮想相対アドレスをファイルオフセットに変換する
      image,PEイメージ(ヘッダ)
    address,仮想相対アドレス
        RET,ファイルオフセット,-1:エラー                                    */
gint
peimage_file_address_to_offset (const guint8  *image,
                                const guint32  address)
{
  gint offset = -1;

  if (image)
    {
      gint i;
      guint16 sections;
      ImageSectionHeader *ish;

      sections = pe_ifh_get_number_of_sections (image);
      ish = pe_image_section_header (image);
      for (i = 0; i < sections; i++)
        {
          if (ish_get_virtual_address (ish) <= address
                                && address < ish_get_virtual_address (ish)
                                           + ish_get_virtual_size (ish))
            {
              offset = address - ish_get_virtual_address (ish)
                               + ish_get_pointer_to_raw_data (ish);
              break;
            }
          ish++;
        }
    }
  return offset;
}


/*  ja:PEイメージのファイルオフセットを仮想相対アドレスに変換する
     image,PEイメージ(ヘッダ)
    offset,ファイルオフセット
       RET,仮想相対アドレス,-1:エラー                                       */
guint32
peimage_file_offset_to_address (const guint8 *image,
                                const gint    offset)
{
  guint32 address = (guint32)-1;

  if (image)
    {
      gint i;
      guint16 sections;
      ImageSectionHeader *ish;

      sections = pe_ifh_get_number_of_sections (image);
      ish = pe_image_section_header (image);
      for (i = 0; i < sections; i++)
        {
          if (ish_get_pointer_to_raw_data (ish) <= offset
                                && offset < ish_get_pointer_to_raw_data (ish)
                                          + ish_get_size_of_raw_data (ish))
            {
              address = offset - ish_get_pointer_to_raw_data (ish)
                               + ish_get_virtual_address (ish);
              break;
            }
          ish++;
        }
    }
  return address;
}


/*  ja:PEイメージを読み込む
    file,ファイル名
     RET,PEイメージ,NULL:エラー                                             */
guint8 *
peimage_file_load (const gchar *file)
{
  guint8 *image;
  gint i, sections, alignment;
  gssize length, header_bytes, size_of_image;
  FileIO *fio;
  ImageSectionHeader *ish;

  if (!file)
    return NULL;
  fio = fileio_open (file, FILEIO_ACCESS_READ, FILEIO_SHARE_READ,
                                                    FILEIO_MODE_OPEN_EXISTING);
  if (!fio)
    return NULL;
  image = g_malloc (sizeof (ImageDosHeader));
  if (!image || fileio_read (fio, image, sizeof (ImageDosHeader))
                                                    != sizeof (ImageDosHeader)
                        || pe_idh_get_magic (image) != PEIMAGE_DOS_SIGNATURE)
    {
      g_free (image);
      fileio_close (fio);
      return NULL;
    }
  length = idh_get_lfanew (image) + sizeof (guint32)
                                                    + sizeof (ImageFileHeader);
  if (length > sizeof (ImageDosHeader))
    {
      image = g_realloc (image, length);
      length -= sizeof (ImageDosHeader);
      if (!image || fileio_read (fio, image + sizeof (ImageDosHeader), length)
                                                                    != length)
        {
          g_free (image);
          fileio_close (fio);
          return NULL;
        }
      length += sizeof (ImageDosHeader);
    }
  else
    {
      length = sizeof (ImageDosHeader);
    }
  if (pe_get_signature (image) != PEIMAGE_NT_SIGNATURE
                || pe_ifh_get_machine (image) != PEIMAGE_FILE_MACHINE_I386)
    {
      g_free (image);
      fileio_close (fio);
      return NULL;
    }
  header_bytes = peimage_file_header_bytes (image);
  image = g_realloc (image, header_bytes);
  header_bytes -= length;
  if (!image || fileio_read (fio, image + length, header_bytes) != header_bytes
                || pe_ioh_get_magic (image) != PEIMAGE_NT_OPTIONAL_HDR_MAGIC)
    {
      g_free (image);
      fileio_close (fio);
      return NULL;
    }
  header_bytes += length;
  sections = pe_ifh_get_number_of_sections (image);
  alignment = pe_ioh_get_file_alignment (image);
  length = 0;
  ish = pe_image_section_header (image);
  for (i = 0; i < sections; i++)
    {
      gssize leng, size;

      size = (ish_get_size_of_raw_data (ish) + alignment - 1)
                                                    / alignment * alignment;
      leng = ish_get_virtual_address (ish)
                                    + MAX (ish_get_virtual_size (ish), size);
      if (length < leng)
        length = leng;
      ish++;
    }
  size_of_image = pe_ioh_get_size_of_image (image);
  if (length > size_of_image)
    {
      size_of_image = length;
      pe_ioh_set_size_of_image (image, size_of_image);
    }
  image = g_realloc (image, size_of_image);
  if (!image)
    {
      fileio_close (fio);
      return NULL;
    }
  length = pe_ioh_get_size_of_headers (image);
  if (length < header_bytes)
    {
      length = header_bytes;
      pe_ioh_set_size_of_headers (image, header_bytes);
    }
  g_memset (image + length, 0, size_of_image - length);
  if (length > header_bytes)
    {
      length -= header_bytes;
      if (fileio_read (fio, image + header_bytes, length) != length)
        {
          g_free (image);
          fileio_close (fio);
          return NULL;
        }
    }
  ish = pe_image_section_header (image);
  for (i = 0; i < sections; i++)
    {
      gint pointer;

      pointer = ish_get_pointer_to_raw_data (ish) / alignment * alignment;
      if (fileio_seek (fio, pointer, FILEIO_SEEK_SET) >= 0)
        fileio_read (fio, image + ish_get_virtual_address (ish),
                            (ish_get_size_of_raw_data (ish) + alignment - 1)
                                                    / alignment * alignment);
      ish++;
    }
  if (!fileio_close (fio))
    {
      g_free (image);
      return NULL;
    }
  return image;
}


/*  ja:PEイメージを再配置する
    image,PEイメージ
    delta,差                                                                */
void
peimage_file_relocate (guint8     *image,
                       const gint  delta)
{
  if (image && delta != 0 && pe_ioh_get_data_directory_size (image,
                                            PEIMAGE_DIRECTORY_ENTRY_BASERELOC))
    {
      guint32 address, size;

      size = pe_ioh_get_size_of_image (image);
      address = pe_ioh_get_data_directory_virtual_address (image,
                                            PEIMAGE_DIRECTORY_ENTRY_BASERELOC);
      while (address + sizeof (ImageBaseRelocation) <= size)
        {
          gint i, count;
          guint32 base;
          ImageBaseRelocation *ibr;

          ibr = (ImageBaseRelocation *)(image + address);
          base = ibr_get_virtual_address (ibr);
          if (base == 0 || address + ibr_get_size_of_block (ibr) > size)
            break;
          count = (ibr_get_size_of_block (ibr) - sizeof (guint32) * 2)
                                                            / sizeof (guint16);
          for (i = 0; i < count; i++)
            {
              guint16 typeoffset;

              typeoffset = ibr_get_type_offset (ibr, i);
              if (((typeoffset >> 12) & 0xf) == PEIMAGE_REL_BASED_HIGHLOW
                                        && base + (typeoffset & 0xfff) <= size)
                {
                  guint32 *p;

                  p = (guint32 *)(image + base + (typeoffset & 0xfff));
                  *p = GUINT32_TO_LE (GUINT32_FROM_LE (*p) + delta);
                }
            }
          address += ibr_get_size_of_block (ibr);
        }
    }
}


/*  ja:PEイメージからAPIを取得する
    image,PEイメージ
     name,API名
      RET,アドレス,NULL:エラー                                              */
gconstpointer
peimage_file_get_func (const guint8 *image,
                       const gchar  *name)
{
  gconstpointer func = NULL;

  if (image && pe_ioh_get_data_directory_size (image,
                                            PEIMAGE_DIRECTORY_ENTRY_EXPORT))
    {
      guint32 address, size;
      const ImageExportDirectory *ied;

      size = pe_ioh_get_size_of_image (image);
      address = pe_ioh_get_data_directory_virtual_address (image,
                                            PEIMAGE_DIRECTORY_ENTRY_EXPORT);
      ied = (const ImageExportDirectory *)(image + address);
      if (address + sizeof (ImageExportDirectory) <= size
            && ied_get_address_of_functions (ied)
                + ied_get_number_of_functions (ied) * sizeof (guint32) <= size
            && ied_get_address_of_names (ied)
                + ied_get_number_of_names (ied) * sizeof (guint32) <= size
            && ied_get_address_of_name_ordinals (ied)
                + ied_get_number_of_names (ied) * sizeof (guint16) <= size)
        {
          gint i;
          const guint16 *ordinals;
          const guint32 *functions, *names;

          functions = (const guint32 *)(image
                                    + ied_get_address_of_functions (ied));
          names = (const guint32 *)(image
                                    + ied_get_address_of_names (ied));
          ordinals = (const guint16 *)(image
                                    + ied_get_address_of_name_ordinals (ied));
          for (i = 0; i < ied_get_number_of_names (ied); i++)
            {
              gint j;

              address = GUINT32_FROM_LE (names[i]);
              for (j = 0; address + j < size
                            && image[address + j] != '\0' && name[j] != '\0'
                            && image[address + j] == name[j]; j++);
              if (address + j < size && image[address + j] == name[j])
                {
                  gint index;

                  index = GUINT16_FROM_LE (ordinals[i]);
                  if (index < ied_get_number_of_functions (ied))
                    {
                      address = GUINT32_FROM_LE (functions[index]);
                      if (address < size)
                        func = image + address;
                    }
                  break;
                }
            }
        }
    }
  return func;
}


/*  ja:PEイメージからAPIを取得する
      image,PEイメージ
    ordinal,オーディナル値
        RET,アドレス,NULL:エラー                                            */
gconstpointer
peimage_file_get_proc (const guint8  *image,
                       const guint16  ordinal)
{
  gconstpointer proc = NULL;

  if (image && pe_ioh_get_data_directory_size (image,
                                            PEIMAGE_DIRECTORY_ENTRY_EXPORT))
    {
      guint32 address, size;
      const ImageExportDirectory *ied;

      size = pe_ioh_get_size_of_image (image);
      address = pe_ioh_get_data_directory_virtual_address (image,
                                            PEIMAGE_DIRECTORY_ENTRY_EXPORT);
      ied = (const ImageExportDirectory *)(image + address);
      if (address + sizeof (ImageExportDirectory) <= size
            && ied_get_address_of_functions (ied)
                + ied_get_number_of_functions (ied) * sizeof (guint32) <= size)
        {
          gint index;
          const guint32 *functions;

          functions = (const guint32 *)(image
                                    + ied_get_address_of_functions (ied));
          index = ordinal - ied_get_base (ied);
          if (index < ied_get_number_of_functions (ied))
            {
              address = GUINT32_FROM_LE (functions[index]);
              if (address < size)
                proc = image + address;
            }
        }
    }
  return proc;
}


/*  ja:PEイメージからリソースを取得する
    image,PEイメージ
      key,リソース
      RET,データ,NULL:エラー                                                */
gconstpointer
peimage_file_get_resource (const guint8 *image,
                           const gchar  *key)
{
  gconstpointer resource = NULL;

  if (image && key && pe_ioh_get_data_directory_size (image,
                                            PEIMAGE_DIRECTORY_ENTRY_RESOURCE))
    {
      gchar **element;
      guint32 size;

      size = pe_ioh_get_size_of_image (image);
      element = g_strsplit (key[0] == '/' ? key + 1 : key, "/", G_MAXINT);
      if (element)
        {
          gint i;
          guint32 address, offset = 0;

          address = pe_ioh_get_data_directory_virtual_address (image,
                                            PEIMAGE_DIRECTORY_ENTRY_RESOURCE);
          for (i = 0; element[i]; i++)
            {
              gint j, count;
              const ImageResourceDirectory *ird;
              const ImageResourceDirectoryEntry *irde;

              ird = (const ImageResourceDirectory *)(image + address + offset);
              irde = (const ImageResourceDirectoryEntry *)(ird + 1);
              if (address + offset + sizeof (ImageResourceDirectory) > size)
                break;
              count = ird_get_number_of_named_entries (ird)
                                        + ird_get_number_of_id_entries (ird);
              if (address + offset + sizeof (ImageResourceDirectory)
                        + count * sizeof (ImageResourceDirectoryEntry) > size)
                break;
              for (j = 0; j < count; j++)
                {
                  gboolean result;
                  gchar *name = NULL;

                  if (irde_get_id (irde) & 0x80000000)
                    {
                      guint32 id;

                      id = address + (irde_get_id (irde) & 0x7fffffff);
                      if (id + sizeof (guint16) <= size)
                        {
                          const ImageResourceDirStringU *irdsu;

                          irdsu = (const ImageResourceDirStringU *)
                                                                (image + id);
                          if (sizeof (guint16)
                                + irdsu_get_length (irdsu) * sizeof (gunichar2)
                                                                    <= size)
                            {
                              gchar *utf8str;

                              utf8str = g_utf16_to_utf8
                                (irdsu_get_name_string (irdsu),
                                irdsu_get_length (irdsu), NULL, NULL, NULL);
                              name = g_utf8_strup (utf8str, -1);
                              g_free (utf8str);
                            }
                        }
                    }
                  else
                    {
                      name = g_strdup_printf ("0x%x", irde_get_id (irde));
                    }
                  result = g_strcmp (element[i], name) == 0;
                  g_free (name);
                  if (result)
                    break;
                  irde++;
                }
              if (j < count && irde_get_offset_to_data (irde) & 0x80000000)
                offset = irde_get_offset_to_data (irde) & 0x7fffffff;
              else if (j < count && !element[i + 1])
                resource = image + address + irde_get_offset_to_data (irde);
              else
                break;
            }
          g_strfreev (element);
        }
    }
  return resource;
}


/*  ja:PEイメージのエクスポートテーブルを列挙する
    image,PEイメージ
      RET,リスト,NULL:エラー                                                */
GList *
peimage_file_enum_export (const guint8 *image)
{
  GList *glist = NULL;

  if (image && pe_ioh_get_data_directory_size (image,
                                            PEIMAGE_DIRECTORY_ENTRY_EXPORT))
    {
      guint32 address, size, size_of_image;
      const ImageExportDirectory *ied;

      size_of_image = pe_ioh_get_size_of_image (image);
      address = pe_ioh_get_data_directory_virtual_address (image,
                                            PEIMAGE_DIRECTORY_ENTRY_EXPORT);
      size = pe_ioh_get_data_directory_size (image,
                                            PEIMAGE_DIRECTORY_ENTRY_EXPORT);
      ied = (const ImageExportDirectory *)(image + address);
      if (address + size <= size_of_image
            && size >= sizeof (ImageExportDirectory)
            && ied_get_address_of_functions (ied)
                + ied_get_number_of_functions (ied) * sizeof (guint32)
                                                            <= size_of_image
            && ied_get_address_of_names (ied)
                + ied_get_number_of_names (ied) * sizeof (guint32)
                                                            <= size_of_image
            && ied_get_address_of_name_ordinals (ied)
                + ied_get_number_of_names (ied) * sizeof (guint16)
                                                            <= size_of_image)
        {
          gint i;
          const guint16 *ordinals;
          const guint32 *functions, *names;

          functions = (const guint32 *)(image
                                    + ied_get_address_of_functions (ied));
          names = (const guint32 *)(image
                                    + ied_get_address_of_names (ied));
          ordinals = (const guint16 *)(image
                                    + ied_get_address_of_name_ordinals (ied));
          for (i = 0; i < ied_get_number_of_functions (ied); i++)
            {
              guint32 func;

              func = GUINT32_FROM_LE (functions[i]);
              if (0 < func && func < size_of_image)
                {
                  gint j;
                  PeExport *peexp;

                  peexp = g_malloc0 (sizeof (PeExport));
                  if (address <= func && func < address + size)
                    peexp->forward = (const gchar *)(image + func);
                  else
                    peexp->address = func;
                  peexp->ordinal = i + ied_get_base (ied);
                  for (j = 0; j < ied_get_number_of_names (ied); j++)
                    if (i == GUINT16_FROM_LE (ordinals[j]))
                      {
                        gint k;
                        guint32 name;

                        name = GUINT32_FROM_LE (names[j]);
                        for (k = name; 0 <= k && k < size_of_image; k++)
                          if (image[k] == '\0')
                            {
                              peexp->name = (const gchar *)(image + name);
                              break;
                            }
                      }
                  glist = g_list_append (glist, peexp);
                }
            }
        }
    }
  return glist;
}


/*  ja:PEイメージのインポートテーブルを列挙する
    image,PEイメージ
      RET,リスト,NULL:エラー                                                */
GList *
peimage_file_enum_import (const guint8 *image)
{
  GList *glist = NULL;

  if (image && pe_ioh_get_data_directory_size (image,
                                            PEIMAGE_DIRECTORY_ENTRY_IMPORT))
    {
      guint16 sections;
      guint32 address, size;

      size = pe_ioh_get_size_of_image (image);
      sections = pe_ifh_get_number_of_sections (image);
      for (address = pe_ioh_get_data_directory_virtual_address (image,
                                            PEIMAGE_DIRECTORY_ENTRY_IMPORT);
                        address + sizeof (ImageImportDescriptor) <= size;
                                    address += sizeof (ImageImportDescriptor))
        {
          gint i;
          guint32 lookup;
          const gchar *file;
          const ImageSectionHeader *ish;
          const ImageImportDescriptor *iid;

          iid = (const ImageImportDescriptor *)(image + address);
          for (i = iid_get_name (iid); 0 < i && i < size; i++)
            if (image[i] == '\0')
              break;
          if (i <= 0 || size <= i)
            break;
          file = (const gchar *)(image + iid_get_name (iid));
          ish = pe_image_section_header (image);
          for (i = 0; i < sections; i++)
            {
              if (ish_get_virtual_address (ish)
                                        <= iid_get_original_first_thunk (iid)
                    && iid_get_original_first_thunk (iid)
                                                < ish_get_virtual_address (ish)
                                                + ish_get_virtual_size (ish))
                break;
              ish++;
            }
          lookup = i < sections ? iid_get_original_first_thunk (iid)
                                : iid_get_first_thunk (iid);
          while (lookup + sizeof (guint32) <= size)
            {
              PeImport *peimp;
              const ImageThunkData *ilt;

              ilt = (const ImageThunkData *)(image + lookup);
              if (!itd_get_address_of_data (ilt))
                break;
              peimp = g_malloc0 (sizeof (PeImport));
              peimp->file = file;
              if (itd_get_ordinal (ilt) & PEIMAGE_ORDINAL_FLAG)
                {
                  peimp->ordinal = itd_get_ordinal (ilt)
                                                    & ~PEIMAGE_ORDINAL_FLAG;
                }
              else if (itd_get_address_of_data (ilt)
                                                    + sizeof (guint16) <= size)
                {
                  for (i = itd_get_address_of_data (ilt) + sizeof (guint16);
                                                    0 <= i && i < size; i++)
                    if (image[i] == '\0')
                      {
                        const ImageImportByName *iibn;

                        iibn = (const ImageImportByName *)(image
                                            + itd_get_address_of_data (ilt));
                        peimp->name = iibn_get_name (iibn);
                        break;
                      }
                }
              else
                {
                  g_free (peimp);
                  peimp = NULL;
                }
              if (peimp)
                glist = g_list_append (glist, peimp);
              lookup += sizeof (guint32);
            }
        }
    }
  return glist;
}


static gint
peimage_file_enum_resource_compare (gconstpointer a,
                                    gconstpointer b)
{
  return ((PeResource *)a)->address - ((PeResource *)b)->address;
}


static GList *
peimage_file_enum_resource_recursive (const guint8  *image,
                                      const guint32  address,
                                      const gchar   *dir)
{
  gint count;
  guint32 size;
  GList *glist = NULL;
  const ImageResourceDirectory *ird;

  size = pe_ioh_get_size_of_image (image);
  ird = (ImageResourceDirectory *)(image + address);
  if (address > 0 && size > 0
                && address + sizeof (ImageResourceDirectory) < size
                && address + sizeof (ImageResourceDirectory)
                            + (count = ird_get_number_of_named_entries (ird)
                                     + ird_get_number_of_id_entries (ird))
                                * sizeof (ImageResourceDirectoryEntry) < size)
    {
      gint i;
      const ImageResourceDirectoryEntry *irde;

      irde = (const ImageResourceDirectoryEntry *)(ird + 1);
      for (i = 0; i < count; i++)
        {
          gchar *path = NULL;

          if (irde_get_id (irde) & 0x80000000)
            {
              guint32 id;

              id = address + (irde_get_id (irde) & 0x7fffffff);
              if (id + sizeof (guint16) <= size)
                {
                  const ImageResourceDirStringU *irdsu;

                  irdsu = (const ImageResourceDirStringU *)(image + id);
                  if (sizeof (guint16) + irdsu_get_length (irdsu)
                                                * sizeof (gunichar2) <= size)
                    {
                      gchar *utf8str, *tmp;

                      utf8str = g_utf16_to_utf8 (irdsu_get_name_string (irdsu),
                                irdsu_get_length (irdsu), NULL, NULL, NULL);
                      tmp = g_utf8_strup (utf8str, -1);
                      g_free (utf8str);
                      path = g_strdup_printf ("%s/%s", dir, tmp);
                      g_free (tmp);
                    }
                }
            }
          else
            {
              path = g_strdup_printf ("%s/0x%x", dir, irde_get_id (irde));
            }
          if (path)
            {
              guint32 first;

              first = pe_ioh_get_data_directory_virtual_address (image,
                                            PEIMAGE_DIRECTORY_ENTRY_RESOURCE);
              if (irde_get_offset_to_data (irde) & 0x80000000)
                {
                  GList *gl;

                  gl = peimage_file_enum_resource_recursive (image,
                        first + (irde_get_offset_to_data (irde) & 0x7fffffff),
                        path);
                  while (gl)
                    {
                      glist = g_list_insert_sorted (glist, gl->data,
                                        peimage_file_enum_resource_compare);
                      gl = g_list_delete_link (gl, gl);
                    }
                }
              else if (first + irde_get_offset_to_data (irde)
                                    + sizeof (ImageResourceDataEntry) < size)
                {
                  PeResource *peres;
                  const ImageResourceDataEntry *irda;

                  irda = (ImageResourceDataEntry *)
                            (image + first + irde_get_offset_to_data (irde));
                  peres = g_malloc (sizeof (PeResource));
                  peres->key = g_strdup (path);
                  peres->address = irda_get_offset_to_data (irda);
                  peres->size = irda_get_size (irda);
                  peres->codepage = irda_get_code_page (irda);
                  glist = g_list_insert_sorted (glist, peres,
                                        peimage_file_enum_resource_compare);
                }
              g_free (path);
            }
          irde++;
        }
    }
  return glist;
}


/*  ja:PEイメージのリソースを列挙する
    image,PEイメージ
      RET,リスト,NULL:エラー                                                */
GList *
peimage_file_enum_resource (const guint8 *image)
{
  return image
    && pe_ioh_get_data_directory_size (image, PEIMAGE_DIRECTORY_ENTRY_RESOURCE)
                        ? peimage_file_enum_resource_recursive (image,
                            pe_ioh_get_data_directory_virtual_address (image,
                                PEIMAGE_DIRECTORY_ENTRY_RESOURCE), "") : NULL;
}


static gint
peimage_file_enum_relocate_compare (gconstpointer a,
                                    gconstpointer b)
{
  return GPOINTER_TO_UINT (a) - GPOINTER_TO_UINT (b);
}


/*  ja:PEイメージの再配置を列挙する
    image,PEイメージ
      RET,リスト,NULL:エラー                                                */
GList *
peimage_file_enum_relocate (const guint8 *image)
{
  GList *glist = NULL;

  if (image && pe_ioh_get_data_directory_size (image,
                                            PEIMAGE_DIRECTORY_ENTRY_BASERELOC))
    {
      guint32 address, size;

      size = pe_ioh_get_size_of_image (image);
      address = pe_ioh_get_data_directory_virtual_address (image,
                                            PEIMAGE_DIRECTORY_ENTRY_BASERELOC);
      while (address + sizeof (ImageBaseRelocation) <= size)
        {
          gint i, count;
          guint32 base;
          ImageBaseRelocation *ibr;

          ibr = (ImageBaseRelocation *)(image + address);
          base = ibr_get_virtual_address (ibr);
          if (base == 0 || address + ibr_get_size_of_block (ibr) > size)
            break;
          count = (ibr_get_size_of_block (ibr) - sizeof (guint32) * 2)
                                                            / sizeof (guint16);
          for (i = 0; i < count; i++)
            {
              guint16 typeoffset;

              typeoffset = ibr_get_type_offset (ibr, i);
              if (((typeoffset >> 12) & 0xf) == PEIMAGE_REL_BASED_HIGHLOW
                                        && base + (typeoffset & 0xfff) <= size)
                glist = g_list_insert_sorted (glist,
                                GUINT_TO_POINTER (base + (typeoffset & 0xfff)),
                                        peimage_file_enum_relocate_compare);
            }
          address += ibr_get_size_of_block (ibr);
        }
    }
  return glist;
}


/******************************************************************************
*                                                                             *
* ja:メモリダンプ関数                                                         *
*                                                                             *
******************************************************************************/
/*  ja:PEイメージがメモリダンプか判定する
    image,PEイメージ(ヘッダ)
     size,バイト数
      RET,TRUE:メモリダンプ,FALSE:メモリダンプではない                      */
gboolean
peimage_file_is_memorydump (const guint8 *image,
                            const gsize   size)
{
  gint i, sections;
  ImageSectionHeader *ish;

  if (!peimage_file_is_valid (image, size))
    return FALSE;
  sections = pe_ifh_get_number_of_sections (image);
  ish = pe_image_section_header (image);
  for (i = 0; i < sections; i++)
    {
      if (ish_get_virtual_address (ish) + ish_get_virtual_size (ish) > size)
        return FALSE;
      ish++;
    }
  return TRUE;
}


/*  ja:メモリダンプを正しいPEイメージに変換する
    image,PEイメージ(ヘッダ)
     size,バイト数
      RET,TRUE:メモリダンプ,FALSE:メモリダンプではない                      */
gboolean
peimage_file_canonicalize_memorydump (const guint8 *image,
                                      const gsize   size)
{
  gint i, sections;
  ImageSectionHeader *ish;

  if (!peimage_file_is_memorydump (image, size))
    return FALSE;
  sections = pe_ifh_get_number_of_sections (image);
  ish = pe_image_section_header (image);
  for (i = 0; i < sections; i++)
    {
      ish_set_pointer_to_raw_data (ish, ish_get_virtual_address (ish));
      ish++;
    }
  return TRUE;
}


#define ALIGNMENT 0x00001000
#define IMAGEBASE 0x00400000


/*  ja:バイナリからPEイメージを作る
     data,バイナリ
     size,バイト数
     base,ベースアドレス(-1:デフォルト)
    entry,エントリーポイント(-1:デフォルト)
    align,アライメント(-1:デフォルト)
      RET,PEイメージ,FALSE:エラー                                           */
guint8 *
peimage_file_image_from_binary (const guint8  *data,
                                const gsize    size,
                                const guint32  base,
                                const guint32  entry,
                                const guint32  align)
{
  guint8 *image = NULL;

  if (data && size > 0)
    {
      gsize header, leng;
      guint32 alignment;
      ImageFileHeader *ifh;
      ImageOptionalHeader *ioh;
      ImageSectionHeader *ish;

      alignment = align != (guint32)-1 ? align : ALIGNMENT;
      header = (sizeof (ImageDosHeader)
                + sizeof (guint32)
                + sizeof (ImageFileHeader)
                + sizeof (ImageOptionalHeader)
                + sizeof (ImageSectionHeader)
                + alignment - 1) / alignment * alignment;
      leng = (header + size + alignment - 1) / alignment * alignment;
      image = g_malloc0 (leng);
      g_memmove (image + header, data, size);
      /* ja:設定 */
      pe_idh_set_magic (image, PEIMAGE_DOS_SIGNATURE);
      pe_idh_set_lfanew (image, sizeof (ImageDosHeader));
      pe_set_signature (image, PEIMAGE_NT_SIGNATURE);
      ifh = pe_image_file_header (image);
      ifh_set_machine (ifh, PEIMAGE_FILE_MACHINE_I386);
      ifh_set_number_of_sections (ifh, 1);
      ifh_set_size_of_optional_header (ifh, sizeof (ImageOptionalHeader));
      ifh_set_characteristics (ifh, PEIMAGE_FILE_RELOCS_STRIPPED
                | PEIMAGE_FILE_EXECUTABLE_IMAGE | PEIMAGE_FILE_32BIT_MACHINE);
      ioh = pe_image_optional_header (image);
      ioh_set_magic (ioh, PEIMAGE_NT_OPTIONAL_HDR_MAGIC);
      ioh_set_size_of_code (ioh, size);
      ioh_set_address_of_entry_point (ioh,
                                (entry != (guint32)-1 ? entry : 0) + header);
      ioh_set_base_of_code (ioh, header);
      ioh_set_image_base (ioh, base != (guint32)-1 ? base : IMAGEBASE);
      ioh_set_section_alignment (ioh, alignment);
      ioh_set_file_alignment (ioh, alignment);
      ioh_set_major_operating_system_version (ioh, 4);
      ioh_set_minor_operating_system_version (ioh, 0);
      ioh_set_major_subsystem_version (ioh, 4);
      ioh_set_minor_subsystem_version (ioh, 0);
      ioh_set_size_of_image (ioh, leng);
      ioh_set_size_of_headers (ioh, header);
      ioh_set_subsystem (ioh, PEIMAGE_SUBSYSTEM_WINDOWS_GUI);
      ioh_set_size_of_stack_reserve (ioh, 0x100000);
      ioh_set_size_of_stack_commit (ioh, 0x1000);
      ioh_set_size_of_heap_reserve (ioh, 0x100000);
      ioh_set_size_of_heap_commit (ioh, 0x1000);
      ioh_set_number_of_rva_and_sizes (ioh,
                                        PEIMAGE_NUMBEROF_DIRECTORY_ENTRIES);
      ish = pe_image_section_header (image);
      ish_set_name (ish, ".text");
      ish_set_virtual_size (ish, size);
      ish_set_virtual_address (ish, header);
      ish_set_size_of_raw_data (ish, leng - header);
      ish_set_pointer_to_raw_data (ish, header);
      ish_set_characteristics (ish, PEIMAGE_SCN_CNT_CODE
                                            | PEIMAGE_SCN_CNT_INITIALIZED_DATA
                                            | PEIMAGE_SCN_MEM_EXECUTE
                                            | PEIMAGE_SCN_MEM_READ
                                            | PEIMAGE_SCN_MEM_WRITE);
    }
  return image;
}
