/*
    UnPacker
    copyright (c) 1998-2011 Kazuki Iwamoto http://www.maid.org/ iwm@maid.org

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
#include "ia32/ia32.h"
#include "ia32/cpu.h"
#include "ia32/message.h"
#include "ia32/process.h"
#include "ia32/save.h"
#include "misc/fileio.h"
#include "misc/peimage.h"


/*  ja:ヘルプを表示する
      fp,ファイルポインタ
    file,ファイル名                                                         */
static void
help (FILE        *fp,
      const gchar *file)
{
  g_fprintf (fp,
"UnPacker "VERSION" ("BUILD_ENVIRONMENT")\n"
"copyright (c) 1998-2011 Kazuki Iwamoto http://www.maid.org/ iwm@maid.org\n"
"\n"
"Usage: %s [option...] exefile [argument...]\n"
"\n"
"  -b, --break=ADDRESS    break point\n"
"  -d, --dynamic          include dynamic allocated memory\n"
"  -e, --entry            new entry point\n"
"  -i, --import           import table restructuring\n"
"  -l, --logfile=FILE     log file name\n"
"  -o, --outfile=FILE     output file name\n"
"  -s, --step=STEP        abort step\n"
"  -t, --timer=SECOND     abort timer\n"
"  -m, --trace-mnemonic   trace mnemonic\n"
"  -r, --trace-register   trace register\n"
"  -f, --trace-tail=STEP  trace last steps\n"
"  -v, --verbose          display detail\n"
"\n", file);
}


static GList *bp = NULL;


static GList *
create_bp (const gchar *str)
{
  gchar **s;
  GList *glist = NULL;

  s = g_strsplit (str, ",", 0);
  if (s)
    {
      gint i;

      for (i = 0; s[i]; i++)
        {
          gchar *endptr;
          guint32 start;

          start = g_strtoul (s[i], &endptr, 16);
          if (s[i] != endptr)
            {
              Ia32BreakPoint *b;

              b = g_malloc (sizeof (Ia32BreakPoint));
              b->start = start;
              b->end = start;
              while (*endptr != '\0')
                {
                  gchar *nptr;
                  guint32 end;

                  nptr = endptr + 1;
                  end = g_strtoul (nptr, &endptr, 16);
                  if (nptr != endptr)
                    {
                      b->end = end;
                      break;
                    }
                }
              if (b->start > b->end)
                {
                  guint32 tmp;

                  tmp = b->start;
                  b->start = b->end;
                  b->end = tmp;
                }
              glist = g_list_append (glist, b);
            }
        }
    }
  g_strfreev (s);
  return glist;
}


static void
destroy_bp (void)
{
  while (bp)
    {
      g_free (bp->data);
      bp = g_list_delete_link (bp, bp);
    }
}

int
main (int   argc,
      char *argv[])
{
  gboolean dynamic = FALSE, entry = FALSE, import = FALSE, verbose = FALSE;
  gboolean result = TRUE;
  gsize length;
  gchar *logfile = NULL, *outfile = NULL;
  gint i, timer = 0, step = 0, tail = 0;
  guint flags = 0;
  Ia32Process *process;

  /* ja:初期化 */
  gtk_set_locale ();
  gtk_init (&argc, &argv);

  atexit (destroy_bp);

  /* ja:引数 */
  for (i = 1; i < argc; i++)
    if (g_ascii_strcasecmp (argv[i], "-b") == 0
                            || g_ascii_strcasecmp (argv[i], "--break") == 0)
      {
        bp = g_list_concat (bp, create_bp (argv[++i]));
      }
    else if (g_ascii_strncasecmp (argv[i], "--break=", 8) == 0)
      {
        bp = g_list_concat (bp, create_bp (argv[i] + 8));
      }
    else if (g_ascii_strcasecmp (argv[i], "-d") == 0
                            || g_ascii_strcasecmp (argv[i], "--dynamic") == 0)
      {
        dynamic = TRUE;
      }
    else if (g_ascii_strcasecmp (argv[i], "-e") == 0
                            || g_ascii_strcasecmp (argv[i], "--entry") == 0)
      {
        entry = TRUE;
      }
    else if (g_ascii_strcasecmp (argv[i], "-i") == 0
                            || g_ascii_strcasecmp (argv[i], "--import") == 0)
      {
        import = TRUE;
      }
    else if (g_ascii_strcasecmp (argv[i], "-l") == 0
                            || g_ascii_strcasecmp (argv[i], "--logfile") == 0)
      {
        logfile = argv[++i];
      }
    else if (g_ascii_strncasecmp (argv[i], "--logfile=", 10) == 0)
      {
        logfile = argv[i] + 10;
      }
    else if (g_ascii_strcasecmp (argv[i], "-o") == 0
                            || g_ascii_strcasecmp (argv[i], "--outfile") == 0)
      {
        outfile = argv[++i];
      }
    else if (g_ascii_strncasecmp (argv[i], "--outfile=", 10) == 0)
      {
        outfile = argv[i] + 10;
      }
    else if (g_ascii_strcasecmp (argv[i], "-s") == 0
                            || g_ascii_strcasecmp (argv[i], "--step") == 0)
      {
        step = g_strtol (argv[++i], NULL, 10);
      }
    else if (g_ascii_strncasecmp (argv[i], "--step=", 7) == 0)
      {
        step = g_strtol (argv[i] + 7, NULL, 10);
      }
    else if (g_ascii_strcasecmp (argv[i], "-t") == 0
                            || g_ascii_strcasecmp (argv[i], "--timer") == 0)
      {
        timer = g_strtol (argv[++i], NULL, 10);
      }
    else if (g_ascii_strncasecmp (argv[i], "--timer=", 8) == 0)
      {
        timer = g_strtol (argv[i] + 8, NULL, 10);
      }
    else if (g_ascii_strcasecmp (argv[i], "-m") == 0
                    || g_ascii_strcasecmp (argv[i], "--trace-mnemonic") == 0)
      {
        flags |= IA32_MESSAGE_MNEMONIC;
      }
    else if (g_ascii_strcasecmp (argv[i], "-r") == 0
                    || g_ascii_strcasecmp (argv[i], "--trace-register") == 0)
      {
        flags |= IA32_MESSAGE_REGISTER;
      }
    else if (g_ascii_strcasecmp (argv[i], "-v") == 0
                            || g_ascii_strcasecmp (argv[i], "--verbose") == 0)
      {
        verbose = TRUE;
      }
    else if (g_ascii_strcasecmp (argv[i], "-f") == 0
                        || g_ascii_strcasecmp (argv[i], "--trace-tail") == 0)
      {
        tail = g_strtol (argv[++i], NULL, 10);
      }
    else if (g_ascii_strncasecmp (argv[i], "--trace-tail=", 13) == 0)
      {
        tail = g_strtol (argv[i] + 13, NULL, 10);
      }
    else if (g_ascii_strcasecmp (argv[i], "-h") == 0
                            || g_ascii_strcasecmp (argv[i], "-?") == 0
                            || g_ascii_strcasecmp (argv[i], "--help") == 0)
      {
        help (stdout, argv[0]);
        return 0;
      }
    else
      break;
  if (i >= argc)
    {
      help (stderr, argv[0]);
      return -1;
    }

  /* ja:プロセス実行 */
  process = ia32_create_process_from_argv (argv + i);
  if (!process)
    {
      g_fprintf (stderr, "Create process error.\n");
      return -1;
    }
  if (!process->current->err)
    {
      ia32_cpu_loop (process, bp, step, timer, tail, flags);
      ia32_message_disp_trace (stdout, process->current);
      /* ja:保存 */
      if (outfile && (!entry || process->winmain != IA32_MEMORY_NULL))
        {
          guint8 *image;

          image = ia32_save_process (process, dynamic, import, &length);
          if (image)
            {
              result = fileio_save (outfile, image, length);
              if (!result)
                {
                  gchar *str;

                  str = g_strdup_printf ("\'%s\' save error.\n", outfile);
                  ia32_message_record_error (process->current, str);
                  g_free (str);
                }
              g_free (image);
            }
          else
            {
              result = FALSE;
              ia32_message_record_error (process->current,
                                                "Image construction error.");
            }
        }
    }
  /* ja:情報出力 */
  if (logfile)
    {
      FILE *fp;

      fp = g_strcmp (logfile, "-") == 0 ? stdout : g_fopen (logfile, "w");
      if (fp)
        ia32_message_disp_html (fp, process, "UnPacker");
      if (fp != stdout && (!fp || fclose (fp) != 0))
        result = FALSE;
    }
  ia32_message_disp_info (stderr, process, verbose, "UnPacker");
  if (!ia32_destroy_process (process))
    {
      g_fprintf (stderr, "Destroy process error.\n");
      result = FALSE;
    }
  return result ? 0 : -1;
}
