using System;
using System.Collections;
using System.Collections.Generic;
using System.IO;
using System.Threading;
using System.Drawing;
using GustFront;
using RVII;
using SharpDX.Direct3D9;
using App = RVII.RVIIManager;

public class AreaEngine : RVIIEngine
{
    private class Zone : Element, IBounds
    {
        private Cuboid myBounds = Cuboid.Zero;

        public Zone(string tagName, Cuboid bounds)
            : base(tagName)
        {
            myBounds = bounds;
        }

        public Cuboid Bounds { get { return myBounds; } }

        public override bool GetMember(string name, ScriptValue[] @params, out ScriptValue result)
        {
            if (Util.CaseInsensitiveEquals(name, "X")) {
                result = myBounds.X.Source;
            } else if (Util.CaseInsensitiveEquals(name, "Y")) {
                result = myBounds.Y.Source;
            } else if (Util.CaseInsensitiveEquals(name, "Z")) {
                result = myBounds.Z.Source;
            } else if (Util.CaseInsensitiveEquals(name, "Width")) {
                result = myBounds.X.Length;
            } else if (Util.CaseInsensitiveEquals(name, "Height")) {
                result = myBounds.Y.Length;
            } else if (Util.CaseInsensitiveEquals(name, "Depth")) {
                result = myBounds.Z.Length;
            } else {
                return base.GetMember(name, @params, out result);
            }
            return true;
        }

        public override bool SetMember(string name, ScriptValue[] @params)
        {
            return base.SetMember(name, @params);
        }

        public override Element Duplicate(string newTag)
        {
            throw new NotImplementedException();
        }
    }

    private class AreaInfo
    {
        public Area Area = null;
        public Dictionary<string, Actor> ActorDic = CollectionUtil.CreateCaseInsensitiveDictionary<Actor>(128);
        public List<Actor> Actors = new List<Actor>(128);
        public Dictionary<string, Zone> ZoneDic = CollectionUtil.CreateCaseInsensitiveDictionary<Zone>(16);
        public List<Zone> Zones = new List<Zone>(16);
        public List<IBounds> EnteringBounds = new List<IBounds>(4);
    }

    private IGlobalActor myPrimaryActor = null;

    private Thread myBlockLoadingThread = null;
    private Semaphore myLoadingSignal = new Semaphore(0, 100);
    private Exception myBlockLoadingThreadException = null;
    private Queue<Area.AreaBlock[]> myLoadingBlocks = new Queue<Area.AreaBlock[]>(4);
    private Area.AreaBlock myLoadedBlock = null;

    private Point myViewSift;
    private Area.AreaBlock myCurrentBlock = new Area.AreaBlock(null, null);
    private IList<GeometryData> myPrevVisibleGeometry;
    private Point? myViewCenter = null;
    private List<Rectangle> geo_overwrite = new List<Rectangle>();
    private List<ShaderInfo> myEffects = new List<ShaderInfo>();
    private Cuboid myVisibility = new Cuboid(
        new Vector(0, Int32.MaxValue), new Vector(0, Int32.MaxValue), new Vector(0, Int32.MaxValue));
    private int myBackground = 0;

    private Dictionary<ActionProcessInfo, AreaInfo> myAreaInfoMap = new Dictionary<ActionProcessInfo, AreaInfo>();

    public AreaEngine(UnregisterActionCallback unreg, ModuleInfo script, ScriptValue[] args)
        : base(unreg, script, args)
    {
        myBlockLoadingThread = new Thread(BlockLoadingThreadProc);
        myBlockLoadingThread.IsBackground = true;
        myBlockLoadingThread.Name = "Block Loading Thread";
        myBlockLoadingThread.Start();
    }

    public override void EndOfAction()
    {
        lock (myLoadingBlocks) {
            myLoadingBlocks.Clear();
            myLoadingSignal.Release();
        }

        base.EndOfAction();
    }

    private void AddActor(ActionProcessInfo api, Actor theActor)
    {
        AreaInfo ai = myAreaInfoMap[api];
        ai.ActorDic.Add(theActor.Tag, theActor);
        ai.Actors.Add(theActor);
        if (ai.Actors.Count > 1) {
            theActor.OrderInClass = ai.Actors[ai.Actors.Count - 2].OrderInClass + 1;
        } else {
            theActor.OrderInClass = 1;
        }
    }

    private void RemoveActor(ActionProcessInfo api, Actor theActor)
    {
        AreaInfo aip = myAreaInfoMap[api];
        aip.Actors.Remove(theActor);
        aip.ActorDic.Remove(theActor.Tag);
    }

    private AreaInfo ActorInCurrentProcess
    {
        get { return (myAreaInfoMap.ContainsKey(CurrentThread.Process)) ? myAreaInfoMap[CurrentThread.Process] : null; }
    }

    private AreaInfo GetAreaInfo(Area a)
    {
        foreach (AreaInfo ai in myAreaInfoMap.Values) {
            if (ai.Area == a) return ai;
        }
        return null;
    }

    private Area CurrentArea
    {
        get { return (myPrimaryActor != null) ? myPrimaryActor.Area : null; }
    }

    protected override void EndProcess(ActionProcessInfo api)
    {
        base.EndProcess(api);

        myAreaInfoMap.Remove(api);
    }

    private static int GeometryOrdering(GeometryData g1, GeometryData g2)
    {
        if (g1 == g2) return 0;
        if (g1 == null && g2 != null) return -1;
        if (g1 != null && g2 == null) return 1;

        if (g1.Z != g2.Z) return g1.Z - g2.Z;
        if (g1.Y != g2.Y) return g1.Y - g2.Y;
        if (g1.X != g2.X) return g1.X - g2.X;

        IRenderable r1 = g1.R;
        IRenderable r2 = g2.R;
        if (r1 is Hero) {
            if (r2 is Hero) {
                return ((Hero)g2.R).OrderInClass - ((Hero)g1.R).OrderInClass;
            } else {
                return 1;
            }
        } else if (r1 is Actor) {
            if (r2 is Actor) {
                return ((Actor)r2).OrderInClass - ((Actor)r1).OrderInClass;
            } else {
                return (r2 is Hero) ? -1 : 1;
            }
        } else if (r1 is Vehicle) {
            if (r2 is Vehicle) {
                return ((Vehicle)r2).OrderInClass - ((Vehicle)r1).OrderInClass;
            } else {
                return (r2 is Area) ? 1 : -1;
            }
        } else if (r1 is Area) {
            return (r2 is Area) ? 0 : -1;
        } else {
            return 0;
        }
    }

    private static Rectangle CalcSourceBounds(int tex, int cellsInLine)
    {
        return new Rectangle(
            ((Math.Abs(tex) - 1) % cellsInLine) * Area.CellWidth,
            ((Math.Abs(tex) - 1) / cellsInLine) * Area.CellHeight,
            Area.CellWidth, Area.CellHeight);
    }

    public override void Render(IGraphicsDevice r)
    {
        GraphicsObjects go = (GraphicsObjects)r;
        AreaInfo ai = GetAreaInfo(CurrentArea);

        if (ai != null) {

            Rectangle worldBounds = App.Global.GetViewBounds();
            if (!myViewCenter.HasValue) {
                Point heroLocation = myPrimaryActor.Bounds.Location;
                worldBounds.X = heroLocation.X * Area.CellWidth + (Area.CellWidth - worldBounds.Width) / 2;
                worldBounds.Y = heroLocation.Y * Area.CellHeight + (Area.CellHeight - worldBounds.Height) / 2;
                worldBounds.Offset(0, (myPrimaryActor.Bounds.Z.Source / 5) * (Area.CellHeight / -2));
                worldBounds.Offset(myPrimaryActor.Offset);
            } else {
                worldBounds.X = myViewCenter.Value.X - worldBounds.Width / 2;
                worldBounds.Y = myViewCenter.Value.Y - worldBounds.Height / 2;
            }
            worldBounds.Offset(myViewSift);

            Rectangle visibleBounds = worldBounds;
            if (visibleBounds.X % Area.CellWidth != 0) {
                visibleBounds.Width += Area.CellWidth - visibleBounds.Width % Area.CellWidth;
                visibleBounds.X -= visibleBounds.X % Area.CellWidth;
            }
            if (visibleBounds.Width % Area.CellWidth != 0) {
                visibleBounds.Width += Area.CellWidth - visibleBounds.Width % Area.CellWidth;
            }
            if (visibleBounds.Y % Area.CellHeight != 0) {
                visibleBounds.Height += Area.CellHeight - visibleBounds.Height % Area.CellHeight;
                visibleBounds.Y -= visibleBounds.Y % Area.CellHeight;
            }
            if (visibleBounds.Height % Area.CellHeight != 0) {
                visibleBounds.Height += Area.CellHeight - visibleBounds.Height % Area.CellHeight;
            }
            visibleBounds.Inflate(Area.CellWidth, Area.CellHeight);

            myPrevVisibleGeometry = myCurrentBlock.GetVisibleGeometry(visibleBounds, myBackground, myVisibility);

            List<GeometryData> geometries;
            if (myPrevVisibleGeometry != null) {
                geometries = new List<GeometryData>(myPrevVisibleGeometry.Count * 2);
                geometries.AddRange(myPrevVisibleGeometry);
            } else {
                geometries = new List<GeometryData>(256);
            }

            List<IActor> actors = new List<IActor>(ai.Actors.Count + 8);
            foreach (IActor anActor in ai.Actors) {
                if (anActor.Visible && anActor.Material != null && anActor.TextureInfo != null) {
                    actors.Add(anActor);
                }
            }
            foreach (Vehicle veh in Element.GetElements("Vehicle")) {
                if (veh.Area == CurrentArea && veh.Visible && veh.Material != null && veh.TextureInfo != null) {
                    actors.Add(veh);
                }
            }
            foreach (Hero h in Element.GetElements("Hero")) {
                if (h.Area == CurrentArea && h.Visible && h.Material != null && h.TextureInfo != null) {
                    actors.Add(h);
                }
            }

            foreach (IActor anActor in actors) {
                Cuboid actorBounds = anActor.Bounds;
                if (myVisibility.IntersectsWith(actorBounds)) {
                    foreach (GeometryData matGeo in anActor.GetCurrentGeometry()) {
                        if (visibleBounds.IntersectsWith(new Rectangle(
                            (actorBounds.X.Source + matGeo.X) * Area.CellWidth + anActor.Offset.X,
                            (actorBounds.Y.Source + matGeo.Y) * Area.CellHeight + anActor.Offset.Y -
                                ((actorBounds.Z.Source + matGeo.Z) / 5) * (Area.CellHeight / 2),
                            Area.CellWidth, Area.CellHeight))) {
                            GeometryData geo = new GeometryData(anActor);
                            geo.X = (actorBounds.X.Source + matGeo.X) * Area.CellWidth + anActor.Offset.X;
                            geo.Y = (actorBounds.Y.Source + matGeo.Y) * Area.CellHeight + anActor.Offset.Y;
                            geo.Z = actorBounds.Z.Source + matGeo.Z;
                            geo.T = matGeo.T;
                            geometries.Add(geo);
                        }
                    }
                }
            }
            geometries.Sort(GeometryOrdering);

            TextureInfo currentTex = null;
            int tex = 0;
            Rectangle sb = Rectangle.Empty;
            float hor_rev = 1.0f;
            go.SpriteBatch.Device.SetSamplerState(0, SamplerState.MagFilter, TextureFilter.Point);
            for (int i = 0; i < geometries.Count; i++) {
                GeometryData geo = geometries[i];
                if (geo.R.TextureInfo != null && geo.T.Tex != 0) {
                    if (currentTex != geo.R.TextureInfo || geo.T.Tex != tex) {
                        currentTex = geo.R.TextureInfo;
                        tex = geo.T.Tex;
                        sb = CalcSourceBounds(tex, currentTex.Instance.GetLevelDescription(0).Width / Area.CellWidth);
                        if (tex > 0) {
                            hor_rev = 1.0f;
                        } else {
                            hor_rev = -1.0f;
                        }
                    }
                    SharpDX.Vector2 sps = SharpDX.Vector2.Zero;
                    if (CurrentArea.GetData("Mag").AsBoolean()) {
                        sps = new SharpDX.Vector2(
                            (geo.X - worldBounds.X) * 2 - worldBounds.Width / 2,
                            (geo.Y - (geo.Z / 5) * (Area.CellHeight / 2) - worldBounds.Y) * 2 - worldBounds.Height / 2);
                        if (hor_rev == -1.0f) sps.X += Area.CellWidth;
                        go.SpriteBatch.Transform = SharpDX.Matrix.Scaling(hor_rev * 2.0f, 2.0f, 1.0f) * SharpDX.Matrix.Translation(sps.X, sps.Y, 0.0f);
                    } else {
                        sps = new SharpDX.Vector2(
                            geo.X - worldBounds.X,
                            geo.Y - (geo.Z / 5) * (Area.CellHeight / 2) - worldBounds.Y);
                        if (hor_rev == -1.0f) sps.X += Area.CellWidth;
                        go.SpriteBatch.Transform = SharpDX.Matrix.Scaling(hor_rev, 1.0f, 1.0f) * SharpDX.Matrix.Translation(sps.X, sps.Y, 0.0f);
                    }
                    go.SpriteBatch.Draw(currentTex.Instance,
                        GSConv.Color(Color.White), GSConv.Rectangle(sb), null, null);
                    go.SpriteBatch.Transform = SharpDX.Matrix.Identity;
                }
            }
            //SpriteEffects sprite_fx = SpriteEffects.None;
            //go.SpriteBatch.GraphicsDevice.SamplerStates[0].MagFilter = TextureFilter.None;
            //for (int i = 0; i < geometries.Count; i++) {
            //    GeometryData geo = geometries[i];
            //    if (geo.R.TextureInfo != null && geo.T.Tex != 0) {
            //        if (currentTex != geo.R.TextureInfo || geo.T.Tex != tex) {
            //            currentTex = geo.R.TextureInfo;
            //            tex = geo.T.Tex;
            //            sb = CalcSourceBounds(tex, currentTex.Instance.Width / Area.CellWidth);
            //            sprite_fx = (tex > 0) ? SpriteEffects.None : SpriteEffects.FlipHorizontally;
            //        }
            //        if (CurrentArea.GetData("Mag").AsBoolean()) {
            //            Vector2 sps = new Vector2(
            //                (geo.X - worldBounds.X) * 2 - worldBounds.Width / 2,
            //                (geo.Y - (geo.Z / 5) * (Area.CellHeight / 2) - worldBounds.Y) * 2 - worldBounds.Height / 2);
            //            go.SpriteBatch.Draw(currentTex.Instance,
            //                sps, sb, Color.White, 0.0f, Vector2.Zero, new Vector2(2.0f, 2.0f), sprite_fx, 0.0f);
            //        } else {
            //            Vector2 sps = new Vector2(
            //                geo.X - worldBounds.X,
            //                geo.Y - (geo.Z / 5) * (Area.CellHeight / 2) - worldBounds.Y);
            //            go.SpriteBatch.Draw(currentTex.Instance,
            //                sps, sb, Color.White, 0.0f, Vector2.Zero, Vector2.One, sprite_fx, 0.0f);
            //        }
            //    }
            //}
        }

        foreach (ShaderInfo si in myEffects) {
            ApplyFullScreenEffect(r, si.Instance);
        }

        RenderAllProcess(r);
        base.Render(r);
    }

    private IList<GeometryData> GetActorGeometry(IActor theActor, out int moverFloor, out int moverCeiling)
    {
        moverCeiling = 0;
        moverFloor = 99;

        List<GeometryData> moverGeo = new List<GeometryData>(4);
        Cuboid actorBounds = theActor.Bounds;
        foreach (GeometryData matGeo in theActor.GetCurrentGeometry()) {
            GeometryData geo = new GeometryData(theActor);
            geo.X = actorBounds.X.Source + matGeo.X * Area.CellWidth;
            geo.Y = actorBounds.Y.Source + matGeo.Y * Area.CellWidth;
            geo.Z = ((actorBounds.Z.Source + matGeo.Z) / 5) * 5;
            geo.T = matGeo.T;
            moverCeiling = Math.Max(geo.Z, moverCeiling);
            moverFloor = Math.Min(geo.Z, moverFloor);
            moverGeo.Add(geo);
        }
        return moverGeo;
    }

    private IActor[] GetAroundActors(IActor center)
    {
        List<IActor> actors = new List<IActor>(4);

        int left = Int32.MaxValue;
        int right = Int32.MinValue;
        int top = Int32.MaxValue;
        int bottom = Int32.MinValue;
        int minz = Int32.MaxValue;
        int maxz = Int32.MinValue;
        foreach (GeometryData geo in center.GetCurrentGeometry()) {
            left = Math.Min(geo.X - 1, left);
            right = Math.Max(geo.X + 2, right);
            top = Math.Min(geo.Y - 1, top);
            bottom = Math.Max(geo.Y + 2, bottom);
            minz = Math.Min((geo.Z / 5) * 5, minz);
            maxz = Math.Max((geo.Z / 5 + 1) * 5, maxz);
        }
        Cuboid cl = center.Location;
        Cuboid cc = new Cuboid(
            new Vector(cl.X.Source + left, right - left),
            new Vector(cl.Y.Source + top, bottom - top),
            new Vector((cl.Z.Source / 5) * 5 + minz, maxz - minz));

        foreach (IActor anActor in ActorInCurrentProcess.Actors) {
            if (anActor != center) {
                if (cc.IntersectsWith(anActor.Bounds)) {
                    actors.Add(anActor);
                }
            }
        }

        return actors.ToArray();
    }

    private int GetGroundLevel(Area theArea, IActor theActor)
    {
        int moverCeiling;
        int moverFloor;
        foreach (GeometryData mg in GetActorGeometry(theActor, out moverFloor, out moverCeiling)) {
            if (mg.Z == moverFloor) {
                for (int z = moverFloor; z >= 0; z += -5) {
                    if (myCurrentBlock.GetThere(mg.X, mg.Y, z) != null) return z;
                }
            }
        }
        return 0;
    }

    private Nullable<bool> TestCollisionWithActor(IActor theActor, IList<GeometryData> moverGeo, out IActor thereActor)
    {
        thereActor = null;

        List<IActor> actors = new List<IActor>(100);
        foreach (Hero h in Element.GetElements("Hero")) {
            if (h.OrderInClass == 1 && h.Area == ActorInCurrentProcess.Area && h.Visible) {
                actors.Add(h);
                break;
            }
        }
        foreach (Vehicle veh in Element.GetElements("Vehicle")) {
            if (veh.Area == ActorInCurrentProcess.Area && veh.Visible) {
                actors.Add(veh);
            }
        }
        foreach (Actor anActor in ActorInCurrentProcess.Actors) {
            actors.Add(anActor);
        }
        actors.Remove(theActor);

        foreach (GeometryData movGeo in moverGeo) {
            int x = movGeo.X;
            int y = movGeo.Y;
            int z = movGeo.Z;
            int c = movGeo.T.C;
            foreach (IActor anActor in actors) {
                Cuboid actorBounds = anActor.Bounds;
                foreach (GeometryData matGeo in anActor.GetCurrentGeometry()) {
                    int ax = actorBounds.X.Source + matGeo.X;
                    if (ax == x) {
                        int ay = actorBounds.Y.Source + matGeo.Y;
                        if (ay == y) {
                            if (((actorBounds.Z.Source + matGeo.Z) / 5) * 5 == z) {
                                thereActor = anActor;
                                if ((c & matGeo.T.C) != 0) {
                                    return true;
                                } else {
                                    break;
                                }
                            }
                        }
                    }
                }
            }
        }

        return (thereActor != null) ? false : new Nullable<bool>();
    }

    private bool CheckGroundCollision(Area theArea, IActor theActor, bool ignoreCollision, ref GeometryData thereGeo)
    {
        int moverCeiling = 0;
        int moverFloor = 99;
        IList<GeometryData> moverGeo = GetActorGeometry(theActor, out moverFloor, out moverCeiling);

        int thereZ = 0;
        bool canGo = false;
        foreach (GeometryData mg in moverGeo) {
            if (mg.Z == moverFloor) {
                for (int z = moverFloor; z >= 0; z += -5) {
                    if (z >= thereZ) {
                        GeometryTexture gt = myCurrentBlock.GetThere(mg.X, mg.Y, z);
                        if (gt != null) {
                            GeometryData geo = new GeometryData(myCurrentBlock.Area);
                            geo.X = mg.X;
                            geo.Y = mg.Y;
                            geo.Z = z;
                            geo.T = gt;
                            thereGeo = geo;
                            thereZ = z;
                            canGo = ignoreCollision | (gt.C & mg.T.C) == 0;
                            break;
                        }
                    }
                }
            }
        }

        if (!ignoreCollision) {
            foreach (GeometryData geo in moverGeo) {
                geo.Z += thereZ - moverFloor;
            }

            IActor thereActor;
            Nullable<bool> cfa = TestCollisionWithActor(theActor, moverGeo, out thereActor);
            return (!cfa.HasValue) ? !canGo : cfa.Value;
        } else {
            return !canGo;
        }
    }

    private bool TestCollision(IActor theActor, Point location, out IActor thereActor, out int thereZ)
    {
        int moverCeiling;
        int moverFloor;
        IList<GeometryData> moverGeo = GetActorGeometry(theActor, out moverFloor, out moverCeiling);
        thereActor = null;
        thereZ = 0;
        if (theActor == myPrimaryActor) {
            thereZ = 0;
        }

        Cuboid bounds = theActor.Bounds;
        Point shift = new Point(location.X - bounds.Location.X, location.Y - bounds.Location.Y);

        //Test whole of him is in the world. If not, nobody can go.
        foreach (GeometryData mg in moverGeo) {
            mg.X += shift.X;
            mg.Y += shift.Y;
            if (myCurrentBlock.GetThere(mg.X, mg.Y, 0) == null) return true;
        }

        //Get there floor. If theActor is a Vehicle and a Hover, floor is the same as current z.
        bool canGo = false;
        if ((!(theActor is Vehicle)) || !((Vehicle)theActor).IsHover) {
            foreach (GeometryData mg in moverGeo) {
                if (mg.Z == moverFloor) {
                    foreach (int z in new int[] { mg.Z + 5, mg.Z, mg.Z - 5 }) {
                        if (z >= thereZ) {
                            if (myCurrentBlock.GetThere(mg.X, mg.Y, z) != null) {
                                thereZ = Math.Max(thereZ, z);
                                canGo = true;
                                break;
                            }
                        }
                    }
                }
            }
        } else {
            thereZ = moverFloor;
            canGo = true;
        }

        //Test whole of him can be on the floor.
        if (canGo) {
            foreach (GeometryData mg in moverGeo) {
                //for (int z = mg.Z + thereZ; z <= mg.Z + thereZ + 4; z++) {
                for (int i = 0; i < 5; i++) {
                    int z = (mg.Z - moverFloor) + thereZ + i;
                    GeometryTexture gt = myCurrentBlock.GetThere(mg.X, mg.Y, z);
                    if (gt != null && (gt.C & mg.T.C) != 0) {
                        canGo = false;
                        break;
                    }
                }
                if (!canGo) break;
            }
        }

        //Test for actor...
        foreach (GeometryData geo in moverGeo) {
            geo.Z += thereZ - moverFloor;
        }

        Nullable<bool> cfa = TestCollisionWithActor(theActor, moverGeo, out thereActor);
        return (!cfa.HasValue) ? !canGo : cfa.Value;
    }

    private static Cuboid ParseCuboid(ScriptValue[] @params, int start)
    {
        return new Cuboid(
            new Vector(@params[start].AsInt32(), 1),
            new Vector(@params[start + 1].AsInt32(), 1),
            new Vector(@params[start + 2].AsInt32(), 5));
    }

    private static Vector ParseVectorST(ScriptValue[] @params, int source_index, int target_index)
    {
        int s = @params[source_index].AsInt32();
        return new Vector(s, @params[target_index].AsInt32() - s + 1);
    }

    #region "cmd impl"

    private ScriptValue cmdBeginAreaProcess(ScriptValue[] @params)
    {
        Area obj = (Area)@params[0].Instance;
        foreach (KeyValuePair<ActionProcessInfo, AreaInfo> kv in myAreaInfoMap) {
            if (kv.Value.Area == obj) {
                return new ScriptValue(kv.Key);
            }
        }

        ActionProcessInfo newProcess = BeginProcess(App.Global.ParseModule(@params[1].AsString()), CollectionUtil.ToArray(@params, 2));
        AreaInfo ai = new AreaInfo();
        myAreaInfoMap.Add(newProcess, ai);
        ai.Area = obj;
        return new ScriptValue(newProcess);
    }

    private ScriptValue cmdGetAreaObject(ScriptValue[] @params)
    {
        AreaInfo ai = ActorInCurrentProcess;

        if (ai != null && @params.Length == 0) {
            return new ScriptValue(ai.Area);
        }

        string tagName = @params[0].AsString();

        if (ai != null) {
            if (ai.ActorDic.ContainsKey(tagName)) {
                return new ScriptValue(ai.ActorDic[tagName]);
            } else if (ai.ZoneDic.ContainsKey(tagName)) {
                return new ScriptValue(ai.ZoneDic[tagName]);
            }
        }

        Element elm = Element.GetElement(tagName);
        if (elm is Hero || elm is Vehicle) {
            return new ScriptValue(elm);
        } else {
            return null;
        }
    }

    private void PrimaryActor_Moved(object sender, EventArgs e)
    {
        AreaInfo ai = ActorInCurrentProcess;
        if (ai != null) {
            ai.EnteringBounds.Clear();
            foreach (IBounds anActor in ai.Actors) {
                if (anActor.Bounds.IntersectsWith(myPrimaryActor.Bounds)) {
                    ai.EnteringBounds.Add(anActor);
                }
            }
            foreach (IBounds aZone in ai.Zones) {
                if (aZone.Bounds.IntersectsWith(myPrimaryActor.Bounds)) {
                    ai.EnteringBounds.Add(aZone);
                }
            }
        }
        LoadIt();
    }

    private ScriptValue cmdGetPrimaryActor(ScriptValue[] @params)
    {
        return new ScriptValue(myPrimaryActor);
    }

    private ScriptValue cmdSetPrimaryActor(ScriptValue[] @params)
    {
        Area prevArea = null;
        Cuboid prevLocation = Cuboid.Zero;
        Area currentArea = null;
        Cuboid currentLocation = Cuboid.Zero;
        IGlobalActor oldPM = myPrimaryActor;

        if (myPrimaryActor != null) {
            prevArea = myPrimaryActor.Area;
            prevLocation = myPrimaryActor.Location;
            myPrimaryActor.PositionChanged -= PrimaryActor_Moved;
        }
        myPrimaryActor = (IGlobalActor)@params[0].Instance;
        if (myPrimaryActor != null) {
            currentArea = myPrimaryActor.Area;
            currentLocation = myPrimaryActor.Location;
            myPrimaryActor.PositionChanged += PrimaryActor_Moved;
        }
        if (prevArea != currentArea || prevLocation != currentLocation) {
            LoadIt();
        }

        return new ScriptValue(oldPM);
    }

    private ScriptValue cmdAddActor(ScriptValue[] @params)
    {
        Actor newActor = new Actor(@params[0].AsString(), ParseCuboid(@params, 1));
        if (@params.Length == 6) {
            newActor.Material = (ActorMaterial)@params[4].Instance;
            newActor.Motion = @params[5].AsString();
        }
        AddActor(CurrentThread.Process, newActor);
        return new ScriptValue(newActor);
    }

    private ScriptValue cmdRemoveActor(ScriptValue[] @params)
    {
        Actor theActor = null;
        if (ActorInCurrentProcess.ActorDic.TryGetValue(@params[0].AsString(), out theActor)) {
            RemoveActor(CurrentThread.Process, theActor);
            return true;
        } else {
            return false;
        }
    }

    private ScriptValue cmdGetActor(ScriptValue[] @params)
    {
        Actor theActor = null;
        ActorInCurrentProcess.ActorDic.TryGetValue(@params[0].AsString(), out theActor);
        return new ScriptValue(theActor);
    }

    private ScriptValue cmdGetActors(ScriptValue[] @params)
    {
        return new ScriptValue(CollectionUtil.CreateStdList(ActorInCurrentProcess.Actors));
    }

    private ScriptValue cmdAddZone(ScriptValue[] @params)
    {
        Cuboid p;
        if (@params.Length == 4) {
            p = ParseCuboid(@params, 1);
        } else {
            int x1 = @params[1].AsInt32();
            int y1 = @params[2].AsInt32();
            int z1 = @params[3].AsInt32();
            int x2 = @params[4].AsInt32();
            int y2 = @params[5].AsInt32();
            int z2 = @params[6].AsInt32();
            p = new Cuboid(new Vector(x1, x2 - x1 + 1), new Vector(y1, y2 - y1 + 1), new Vector(z1, z2 - z1 + 5));
        }

        Zone zn = new Zone(@params[0].AsString(), p);
        ActorInCurrentProcess.Zones.Add(zn);
        ActorInCurrentProcess.ZoneDic.Add(zn.Tag, zn);
        return new ScriptValue(zn);
    }

    private ScriptValue cmdGetZone(ScriptValue[] @params)
    {
        Zone z = null;
        ActorInCurrentProcess.ZoneDic.TryGetValue(@params[0].AsString(), out z);
        return new ScriptValue(z);
    }

    private ScriptValue cmdGetZones(ScriptValue[] @params)
    {
        Cuboid c = ParseCuboid(@params, 0);
        IList result = CollectionUtil.CreateStdList();
        foreach (Zone z in ActorInCurrentProcess.Zones) {
            if (z.Bounds.IntersectsWith(c)) result.Add(z);
        }
        return new ScriptValue(result);
    }

    private ScriptValue cmdIsInZone(ScriptValue[] @params)
    {
        return ((Zone)(@params[0].Instance)).Bounds.IntersectsWith(ParseCuboid(@params, 1));
    }

    private ScriptValue cmdTestCollision(ScriptValue[] @params)
    {
        IList results = CollectionUtil.CreateStdList(3);

        IActor thereActor;
        int thereZ;

        results.Add(TestCollision(
            (IActor)@params[0].Instance, new Point(@params[1].AsInt32(), @params[2].AsInt32()),
            out thereActor, out thereZ));
        results.Add(thereActor);
        results.Add(thereZ);

        return new ScriptValue(results);
    }

    private ScriptValue cmdGetAroundActors(ScriptValue[] @params)
    {
        IList result = CollectionUtil.CreateStdList();
        foreach (IActor anActor in GetAroundActors((IActor)@params[0].Instance)) {
            result.Add(anActor);
        }
        return new ScriptValue(result);
    }

    private ScriptValue cmdGetEnteredBoundsList(ScriptValue[] @params)
    {
        IList theList = CollectionUtil.CreateStdList();
        foreach (IBounds b in ActorInCurrentProcess.EnteringBounds) {
            theList.Add(((Element)b).Tag);
        }
        return new ScriptValue(theList);
    }

    private ScriptValue cmdGetGroundMaterial(ScriptValue[] @params)
    {
        GeometryData geo = null;
        CheckGroundCollision(myPrimaryActor.Area, myPrimaryActor, true, ref geo);
        return (geo != null) ? geo.T.M : 0;
    }

    private ScriptValue cmdChangeGeometry(ScriptValue[] @params)
    {
        Rectangle ow = new Rectangle(
            @params[0].AsInt32(), @params[1].AsInt32(), 
            @params[2].AsInt32(), @params[3].AsInt32());
        if (!geo_overwrite.Contains(ow)) {
            geo_overwrite.Add(ow);
        }
        return null;
    }

    private ScriptValue cmdLoadGeometry(ScriptValue[] @params)
    {
        LoadIt();
        return null;
    }

    private ScriptValue cmdAnimateArea(ScriptValue[] @params)
    {
        ActorInCurrentProcess.Area.SetAreaMaterial(@params[0].AsInt32(), @params[1].AsInt32() - 1, @params[2].AsInt32());
        return null;
    }

    private ScriptValue cmdGetActorData(ScriptValue[] @params)
    {
        IActor who = (IActor)@params[0].Instance;
        string name = @params[1].AsString();
        ScriptValue theData = null;
        if (who.GetMember(name, CollectionUtil.ToArray(@params, 2, @params.Length - 2), out theData)) {
            return theData;
        } else {
            throw new Exception("w肳ꂽf[^݂͑܂B");
        }
    }

    private ScriptValue cmdTryGetActorData(ScriptValue[] @params)
    {
        IActor who = (IActor)@params[0].Instance;
        string name = @params[1].AsString();
        ScriptValue theData = null;
        if (who.GetMember(name, CollectionUtil.ToArray(@params, 3, @params.Length - 3), out theData)) {
            if (IsVarRef(@params[2])) SetToVariable(@params[2], theData.Instance);
            return true;
        } else {
            return false;
        }
    }

    private ScriptValue cmdSetActorData(ScriptValue[] @params)
    {
        IActor who = (IActor)@params[0].Instance;
        string name = @params[1].AsString();
        ScriptValue oldData = null;
        if (!who.GetMember(name, CollectionUtil.ToArray(@params, 2, @params.Length - 3), out oldData)) {
            who.DeclareData(name);
        }
        who.SetMember(name, CollectionUtil.ToArray(@params, 2, @params.Length - 2));
        return oldData;
    }

    private ScriptValue cmdAddAreaEffect(ScriptValue[] @params)
    {
        ShaderInfo fx = (ShaderInfo)Element.GetElement(@params[0].AsString());
        myEffects.Add(fx);
        return new ScriptValue(fx);
    }

    private ScriptValue cmdRemoveAreaEffect(ScriptValue[] @params)
    {
        return myEffects.Remove((ShaderInfo)@params[0].Instance);
    }

    private ScriptValue cmdChangeVisibility(ScriptValue[] @params)
    {
        if (@params.Length == 0) {
            myVisibility = new Cuboid(
              new Vector(0, Int32.MaxValue), new Vector(0, Int32.MaxValue), new Vector(0, Int32.MaxValue));
        } else if (@params.Length == 2) {
            myVisibility.Z = ParseVectorST(@params, 0, 1);
        } else if (@params.Length == 4) {
            myVisibility.X = ParseVectorST(@params, 0, 2);
            myVisibility.Y = ParseVectorST(@params, 1, 3);
        } else if (@params.Length == 6) {
            myVisibility.X = ParseVectorST(@params, 0, 3);
            myVisibility.Y = ParseVectorST(@params, 1, 4);
            myVisibility.Z = ParseVectorST(@params, 2, 5);
        }
        return null;
    }

    private ScriptValue cmdChangeBGTex(ScriptValue[] @params)
    {
        if (@params.Length == 0) {
            myBackground = 0;
        } else if (@params.Length == 1) {
            myBackground = @params[0].AsInt32();
        }
        return null;
    }

    private ScriptValue cmdShiftView(ScriptValue[] @params)
    {
        myViewSift.X = @params[0].AsInt32();
        myViewSift.Y = @params[1].AsInt32();
        return null;
    }

    private ScriptValue cmdLinkView(ScriptValue[] @params)
    {
        myViewCenter = null;
        return null;
    }

    private ScriptValue cmdGetViewCenter(ScriptValue[] @params)
    {
        Point view_center;
        if (!myViewCenter.HasValue) {
            Point heroLocation = myPrimaryActor.Bounds.Location;
            view_center = new Point(
                heroLocation.X * Area.CellWidth + Area.CellWidth / 2 + myPrimaryActor.Offset.X,
                heroLocation.Y * Area.CellHeight + Area.CellHeight / 2 + myPrimaryActor.Offset.Y -
                    ((myPrimaryActor.Bounds.Z.Source / 5) * (Area.CellHeight / 2)));
        } else {
            view_center = myViewCenter.Value;
        }
        return new ScriptValue(new PointWrapper(view_center));
    }

    private ScriptValue cmdSetViewCenter(ScriptValue[] @params)
    {
        myViewCenter = new Point(@params[0].AsInt32(), @params[1].AsInt32());
        return null;
    }

    protected override IDictionary<string, CommandImpl> GetCommandMap()
    {
        Dictionary<string, CommandImpl> dic = new Dictionary<string, CommandImpl>(base.GetCommandMap());

        dic.Add("BeginAreaProcess", cmdBeginAreaProcess);
        dic.Add("GetAreaObject", cmdGetAreaObject);
        dic.Add("GetPrimaryActor", cmdGetPrimaryActor);
        dic.Add("SetPrimaryActor", cmdSetPrimaryActor);
        dic.Add("AddActor", cmdAddActor);
        dic.Add("RemoveActor", cmdRemoveActor);
        dic.Add("GetActor", cmdGetActor);
        dic.Add("GetActors", cmdGetActors);
        dic.Add("AddZone", cmdAddZone);
        dic.Add("GetZone", cmdGetZone);
        dic.Add("GetZones", cmdGetZones);
        dic.Add("IsInZone", cmdIsInZone);

        dic.Add("TestCollision", cmdTestCollision);
        dic.Add("GetAroundActors", cmdGetAroundActors);
        dic.Add("GetEnteredBoundsList", cmdGetEnteredBoundsList);

        dic.Add("GetGroundMaterial", cmdGetGroundMaterial);
        dic.Add("ChangeGeometry", cmdChangeGeometry);
        dic.Add("AnimateArea", cmdAnimateArea);
        dic.Add("LoadGeometry", cmdLoadGeometry);

        dic.Add("GetActorData", cmdGetActorData);
        dic.Add("TryGetActorData", cmdTryGetActorData);
        dic.Add("SetActorData", cmdSetActorData);

        dic.Add("AddAreaEffect", cmdAddAreaEffect);
        dic.Add("RemoveAreaEffect", cmdRemoveAreaEffect);
        dic.Add("ChangeVisibility", cmdChangeVisibility);
        dic.Add("ChangeBGTex", cmdChangeBGTex);
        dic.Add("ShiftView", cmdShiftView);
        dic.Add("LinkView", cmdLinkView);
        dic.Add("GetViewCenter", cmdGetViewCenter);
        dic.Add("SetViewCenter", cmdSetViewCenter);

        return dic;
    }

    #endregion

    private void BlockLoadingThreadProc()
    {
        Area targetArea = null;
        Area.AreaBlock[] available_blocks = new Area.AreaBlock[0];

        try {
            do {
                myLoadingSignal.WaitOne();

                Area.AreaBlock[] loaders;
                Area.AreaBlock[] unloaders;

                lock (myLoadingBlocks) {
                    if (myLoadingBlocks.Count > 0) {
                        loaders = myLoadingBlocks.Dequeue();
                        unloaders = myLoadingBlocks.Dequeue();
                    } else {
                        break;
                    }
                }

                foreach (Area.AreaBlock b in unloaders) {
                    if (b.IsLoaded) {
                        b.Unload();
                    }
                }

                foreach (Area.AreaBlock b in loaders) {
                    targetArea = b.Area;
                    if (!b.IsLoaded) {
                        try {
                            b.LoadGeometry(App.Global.GetStreamToRead(
                                Util.CombinePath("Area", b.Area.Tag, "Block", String.Concat(b.Name, ".ini"))));
                        } catch (Exception ex) {
                            throw new Exception(String.Format("ubN {0} ̓ǂݍ݂Ɏs܂B", b.Name), ex);
                        }
                    }
                }

                if (loaders.Length == available_blocks.Length) {
                    foreach (Area.AreaBlock b in loaders) {
                        if (Array.IndexOf(available_blocks, b) == -1) {
                            Interlocked.Exchange(ref myLoadedBlock, Area.AreaBlock.Combine(loaders));
                            break;
                        }
                    }
                } else {
                    Interlocked.Exchange(ref myLoadedBlock, Area.AreaBlock.Combine(loaders));
                }
                available_blocks = loaders;

            } while (true);
        } catch (Exception ex) {
            Interlocked.Exchange(ref myBlockLoadingThreadException, (targetArea != null) ?
                new Exception(String.Format("GA {0} \zɖ肪܂B", targetArea.Tag), ex) : ex);
        }
    }

    private void LoadIt()
    {
        if (CurrentArea != null) {

            Rectangle screenBounds = App.Global.GetViewBounds();
            screenBounds.Width /= Area.CellWidth;
            screenBounds.Height /= Area.CellHeight;
            screenBounds.X = myPrimaryActor.Bounds.Location.X - screenBounds.Width / 2;
            screenBounds.Y = myPrimaryActor.Bounds.Location.Y - screenBounds.Height / 2;

            Rectangle loadBounds = screenBounds; loadBounds.Inflate(screenBounds.Width / 2, screenBounds.Height / 2);
            Rectangle keepBounds = loadBounds; keepBounds.Inflate(keepBounds.Width / 2, keepBounds.Height / 2);

            List<Area.AreaBlock> loaders = new List<Area.AreaBlock>(4);
            List<Area.AreaBlock> unloaders = new List<Area.AreaBlock>(4);
            IList<Area.AreaBlock> blocks = CurrentArea.GetBlocks();
            if (CurrentArea == myCurrentBlock.Area) {
                for (int i = 0; i < blocks.Count; i++) {
                    Area.AreaBlock b = blocks[i];
                    if (!keepBounds.IntersectsWith(b.GeometryBounds)) {
                        unloaders.Add(b);
                    } else if (loadBounds.IntersectsWith(b.GeometryBounds)) {
                        loaders.Add(b);
                    }
                }
            } else {
                geo_overwrite.Clear();
                myCurrentBlock = new Area.AreaBlock(null, CurrentArea);
                for (int i = 0; i < blocks.Count; i++) {
                    Area.AreaBlock b = blocks[i];
                    unloaders.Add(b);
                    if (loadBounds.IntersectsWith(b.GeometryBounds)) {
                        loaders.Add(b);
                    }
                }
            }

            if (loaders.Count > 0 || unloaders.Count > 0) {
                lock (myLoadingBlocks) {
                    myLoadingBlocks.Enqueue(loaders.ToArray());
                    myLoadingBlocks.Enqueue(unloaders.ToArray());
                }
                myLoadingSignal.Release();
            }
        }
    }

    public override void Tick(TimeSpan delta)
    {
        Exception bltex = Interlocked.Exchange(ref myBlockLoadingThreadException, null);
        if (bltex != null) {
            throw new Exception("GA ubN̓ǂݍ݃XbhŖ肪܂B", bltex);
        }

        Area.AreaBlock loadedBlock = Interlocked.Exchange(ref myLoadedBlock, null);
        if (loadedBlock != null) {
            myCurrentBlock = loadedBlock;
        }
        for (int i = 0; i < geo_overwrite.Count; i++) {
            Rectangle ow = geo_overwrite[i];
            myCurrentBlock.SetThere(ow.X, ow.Y, ow.Width, ow.Height);
        }

        TickAllProcess(delta);
        base.Tick(delta);
    }
}

public class AreaEngineElement : IHasElement
{
    private const string TN_AREA = "Area";
    private const string TN_ACTOR = "Actor";
    private const string TN_VEHICLE = "Vehicle";
    private const string TN_HERO = "Hero";
    private const string TN_ENEMY = "Enemy";

    public string[] GetTypeNames()
    {
        return new string[] { TN_AREA, TN_ACTOR, TN_VEHICLE, TN_HERO, TN_ENEMY };
    }

    public Type GetType(string typeName)
    {
        switch (typeName) {
            case TN_AREA:
                return typeof(Area);
            case TN_ACTOR:
                return typeof(ActorMaterial);
            case TN_VEHICLE:
                return typeof(Vehicle);
            case TN_HERO:
                return typeof(Hero);
            case TN_ENEMY:
                return typeof(Enemy);
            default:
                return null;
        }
    }

    public Element[] LoadElement(TextReader reader, string typeName)
    {
        Element[] elms = null;
        switch (typeName) {
            case TN_AREA:
                elms = Element.Load(reader, typeName, typeof(Area));
                foreach (Area a in elms) {
                    a.LoadMaterial(App.Global.GetStreamToRead(Util.CombinePath("Area", a.Tag, "AreaMaterial.ini")));
                    a.LoadBlocks(App.Global.GetStreamToRead(Util.CombinePath("Area", a.Tag, "Block.ini")));
                }
                return elms;
            case TN_ACTOR:
                elms = Element.Load(reader, typeName, typeof(ActorMaterial));
                foreach (ActorMaterial t in elms) {
                    foreach (string motion in (System.Collections.IList)t.GetData("Motion").Instance) {
                        System.Collections.IList frames = (System.Collections.IList)t.GetData(motion).Instance;
                        for (int i = 0; i < frames.Count; i++) {
                            t.LoadMaterial(App.Global.GetStreamToRead(Util.CombinePath("Actor", t.Tag, motion, (string)frames[i])), motion, i);
                        }
                    }
                }
                return elms;
            case TN_VEHICLE:
                return Element.Load(reader, typeName, typeof(Vehicle));
            case TN_HERO:
                return Element.Load(reader, typeName, typeof(Hero));
            case TN_ENEMY:
                return Element.Load(reader, typeName, typeof(Enemy));
            default:
                return null;
        }
    }

    public void SaveElement(TextWriter writer, string typeName)
    {
        switch (typeName) {
            case TN_AREA:
            case TN_ACTOR:
            case TN_VEHICLE:
            case TN_HERO:
            case TN_ENEMY:
                Element.Save(writer, typeName);
                break;
        }
    }
}

public class AreaEngineStarter : IHasCommand
{
    private ScriptEngine myTarget = null;

    public void SetTarget(ScriptEngine obj)
    {
        myTarget = obj;
    }

    private ScriptValue cmdDescendToEarth(ScriptValue[] @params)
    {
        myTarget.RegisterThreadAction(new AreaEngine(
            myTarget.UnregisterThreadAction,
            App.Global.ParseModule(@params[0].AsString()), CollectionUtil.ToArray(@params, 1)));
        return null;
    }

    public IDictionary<string, CommandImpl> GetCommandMap()
    {
        Dictionary<string, CommandImpl> dic = new Dictionary<string, CommandImpl>();

        dic.Add("DescendToEarth", cmdDescendToEarth);

        return dic;

    }
}
