/*
 * Copyright [yyyy] [name of copyright owner]
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */


// Anonymous function start
//
(function( window, undefined )
{

// reference
var Config			= window.h5glib.Config;
var Debug			= window.h5glib.Debug;
var Command			= window.h5glib.Command;
var Task			= window.h5glib.Task;
var SoundTask		= window.h5glib.SoundTask;
var Message			= window.h5glib.Message;
var MessageHandler	= window.h5glib.MessageHandler;
var MessageManager	= window.h5glib.MessageManager;
var Scene			= window.h5glib.Scene;

/**
 * StringRes
 */
var StringRes =
{
	HTML_USAGE	: "<p>[↑][↓]: 選択 [Enter]: 実行</p>",

	TB_OFFENSE	: "戦う",
	TB_ESCAPE	: "逃げる",
	TI_HP		: "HP: ",
	TI_MP		: "MP: ",
	TM_APPEAR	: "があらわれた。",
	TM_READY		: "は様子を見ている。",
	TM_OFFENSE	: "の攻撃。",
	TM_DAMAGE	: "はダメージを受けた。",
	TM_ESCAPE	: "は逃げ出した。",
	TM_WIN		: "に勝利した。",
	TM_LOSE		: "は死んでしまった。"
};

/**
 * ColorRes
 */
var ColorRes =
{
	RECT		: "rgb(32, 178, 170)",
	RECT_FOCUS	: "rgb(255, 20, 147)",
	BORDER		: "rgb(255, 255, 255)",
	TEXT		: "rgb(255, 255, 255)"
};

/**
 * FontRes
 */
var FontRes =
{
	LARGE		: "bold 18px 'ＭＳ Ｐゴシック'",
	SMALL		: "bold 14px 'ＭＳ Ｐゴシック'"
};

/**
 * MessageType
 */
var MessageType =
{
	NONE			: 0,
	RET_PREV_SCENE	: 1
};

/**
 * BattleStatus
 */
var BattleStatus =
{
	NONE	: 0,
	OFFENSE	: 1,
	END		: 2,
	ESCAPE	: 3
};

/**
 * BattleTask
 */
var BattleTask = function( scene )
{
	this.updateFunc	=
	[
		this.updateNone,
		this.updateOffense,
		this.updateEnd,
		this.updateEscape
	];

	this.command	= scene.command;	// input
	this.player		= null;
	this.enemy		= null;

	this.lag		= this.LAG_VAL;
	this.text		= "";
	this.select		= 0;
};
BattleTask.prototype = new Task();

(function( pt )
{
	pt.LAG_VAL		= 8;
	pt.SELECTIONS	= 2;

	/**
	 * 
	 */
	pt.setStatus = function( status )
	{
		this.status	= status;
		this.update	= this.updateFunc[ this.status ];
	};
	/**
	 * 
	 */
	pt.reset = function( scene, player, enemy )
	{
		this.player	= player;
		this.enemy	= enemy;

		this.setStatus( BattleStatus.NONE );
		this.select	= 0;
		this.text	= this.enemy.ad.name + StringRes.TM_APPEAR;
	};

	/**
	 * 
	 */
	pt.updateNone = function( scene )
	{
		var upd = false;

		if ( this.lag > 0 )
		{
			this.lag--;
		}
		else
		{
			if ( this.command.tbl.up )
			{
				this.select--;
				if ( this.select < 0 )
				{
					this.select = this.SELECTIONS - 1;
				}
				this.lag = this.LAG_VAL;
				upd = true;
			}
			else if ( this.command.tbl.down )
			{
				this.select++;
				if ( this.select >= this.SELECTIONS )
				{
					this.select = 0;
				}
				this.lag = this.LAG_VAL;
				upd = true;
			}
			else if ( this.command.tbl.enter )
			{
				switch ( this.select )
				{
				case 0:
					this.setStatus( BattleStatus.OFFENSE );
					this.lag = this.LAG_VAL * 4;
					break;
				case 1:
					this.setStatus( BattleStatus.ESCAPE );
					this.lag = this.LAG_VAL;
					break;
				}
			}
		}
		return upd;
	};
	/**
	 * 
	 */
	pt.updateOffense = function( scene )
	{
		var upd		= false;
		var sound	= scene.soundTask;

		if ( this.lag > 0 )
		{
			// start
			if ( this.lag == this.LAG_VAL * 4 )
			{
				sound.prepare("shoot");
				this.text	= this.player.ad.name + StringRes.TM_OFFENSE;
				upd = true;
			}
			// judge
			if ( this.lag == this.LAG_VAL * 2 )
			{
				var r = Math.random();
				if ( r < 0.5 )
				{
					this.player.ad.hp -= 30;
					this.text	= this.player.ad.name + StringRes.TM_DAMAGE;
				}
				else
				{
					this.enemy.ad.hp -= 30;
					this.text	= this.enemy.ad.name + StringRes.TM_DAMAGE;
				}
				upd = true;
			}
			this.lag--;
		}
		// result
		else
		{
			if ( this.player.ad.hp <= 0 || this.enemy.ad.hp <= 0 )
			{
				this.setStatus( BattleStatus.END );
				this.lag = this.LAG_VAL;
			}
			else
			{
				this.setStatus( BattleStatus.NONE );
				this.text	= this.enemy.ad.name + StringRes.TM_READY;
				upd = true;
			}
		}
		return upd;
	};
	/**
	 * 
	 */
	pt.updateEnd = function( scene )
	{
		var upd			= false;
		var msgManager	= scene.msgManager;
		var sound		= scene.soundTask;

		if ( this.lag > 0 )
		{
			// judge
			if ( this.lag == this.LAG_VAL )
			{
				this.select = 1;
				if ( this.player.ad.hp > 0 )
				{
					this.text	= this.enemy.ad.name + StringRes.TM_WIN;
				}
				else
				{
					this.text	= this.player.ad.name + StringRes.TM_LOSE;
				}
				upd = true;
			}
			this.lag--;
		}
		else
		{
			if ( this.command.tbl.enter )
			{
				sound.cancel("battle");

				var message	= new Message( MessageType.RET_PREV_SCENE );
				msgManager.postBreakMessage( message );
				upd = true;
			}
			upd = true;
		}
		return upd;
	};
	/**
	 * 
	 */
	pt.updateEscape = function( scene )
	{
		var upd			= false;
		var msgManager	= scene.msgManager;
		var sound		= scene.soundTask;

		if ( this.lag > 0 )
		{
			// judge
			if ( this.lag == this.LAG_VAL )
			{
				this.text	= this.player.ad.name + StringRes.TM_ESCAPE;
				upd = true;
			}
			this.lag--;
		}
		else
		{
			sound.cancel("battle");

			var message	= new Message( MessageType.RET_PREV_SCENE );
			msgManager.postBreakMessage( message );
			upd = true;
		}
		return upd;
	};

	/**
	 * 
	 */
	pt.draw = function( scene )
	{
		var canvas		= scene.canvas;
		var context		= scene.context;
		var resource	= scene.data.resource;
		var image;

		context.clearRect( 0, 0, canvas.width, canvas.height );

		// monster
		image = resource.image.monsters[ this.enemy.ad.type ].data;
		var dx = canvas.width  / 2 - image.width  / 2;
		var dy = canvas.height / 4 - image.height / 2;
		context.drawImage(
			image,
			0,				// sx
			0,				// sy
			image.width,	// sw
			image.height,	// sh
			dx,				// dx
			dy,				// dy
			image.width,	// dw
			image.height	// dh
		);

		context.save();

		// text
		var tx	= 10;
		var ty	= canvas.height / 2;
		var tw	= canvas.width - tx * 2;
		var th	= 50;
		this.drawRect( context, tx, ty, tw, th, ColorRes.RECT );
		context.font		= FontRes.LARGE;
		context.fillStyle	= ColorRes.TEXT;
		context.fillText( this.text, tx + 10, ty + 25 );

		// hp, mp
		tx	= 10;
		ty	= canvas.height / 2 + 55;
		tw	= 150;
		th	= canvas.height - ty - 10;
		this.drawRect( context, tx, ty, tw, th, ColorRes.RECT );
		context.font		= FontRes.SMALL;
		context.fillStyle	= ColorRes.TEXT;
		context.fillText( StringRes.TI_HP + this.player.ad.hp, tx + 10, ty + 22 );
		context.fillText( StringRes.TI_MP + this.player.ad.mp, tx + 10, ty + 42 );

		// button
		var color;
		color	= ( this.select == 0 ) ? ColorRes.RECT_FOCUS : ColorRes.RECT;
		tx		= tx + tw + 10;
		tw		= canvas.width - tx - 10;
		th		= 40;
		this.drawRect( context, tx, ty, tw, th, color );
		context.font		= FontRes.SMALL;
		context.fillStyle	= ColorRes.TEXT;
		context.fillText( StringRes.TB_OFFENSE, tx + 10, ty + 22 );

		color	= ( this.select == 1 ) ? ColorRes.RECT_FOCUS : ColorRes.RECT;
		tx		= 170;
		ty		= ty + th + 5;
		th		= 40;
		this.drawRect( context, tx, ty, tw, th, color );
		context.font		= FontRes.SMALL;
		context.fillStyle	= ColorRes.TEXT;
		context.fillText( StringRes.TB_ESCAPE, tx + 10, ty + 22 );

		context.restore();
	};
	/**
	 * 
	 */
	pt.drawRect = function( context, x, y, w, h, color )
	{
		context.fillStyle	= color;
		context.fillRect( x, y, w, h );

		context.strokeStyle	= ColorRes.BORDER;
		context.beginPath();
		context.strokeRect( x, y, w, h );
	};
})( BattleTask.prototype );


/**
 * InputCommand
 */
var InputCommand = function()
{
	this.tbl.left	= 0;
	this.tbl.right	= 0;
	this.tbl.up		= 0;
	this.tbl.down	= 0;
	this.tbl.enter	= 0;
	this.tbl.escape	= 0;
};
InputCommand.prototype = new Command();

(function( pt )
{
	/**
	 * 
	 */
	pt.update = function() {};
	/**
	 * 
	 */
	pt.handleSysEvent = function( event )
	{
		var type = event.type.toLowerCase();
		if ( type.substring( 0, 3 ) == "key" )
		{
			var value = 0;
			if      ( type == "keydown" ) { value = 1; }
			else if ( type == "keyup"   ) { value = 0; }

			switch ( event.keyCode )
			{
				case 37: case 65:	this.tbl.left	= value; break;	// ← | A
				case 39: case 68:	this.tbl.right	= value; break;	// → | D
				case 38: case 87:	this.tbl.up		= value; break;	// ↑ | W
				case 40: case 83:	this.tbl.down	= value; break;	// ↓ | S
				case 13:			this.tbl.enter	= value; break;	// ENTER
				case 27:			this.tbl.escape	= value; break;	// ESC
			}
		}
	};
})( InputCommand.prototype );


/**
 * BattleScene
 */
var BattleScene = function( app, name )
{
	this.app		= app;
	this.name		= name;
	this.data		= null;
	this.command	= new InputCommand();

	this.battleTask	= new BattleTask( this );
	this.soundTask	= new SoundTask( this );
	this.soundTask	.link( this.battleTask );

	this.child		= this.battleTask;	// head of task list

	// message handler
	this.msgManager	= new MessageManager();
	this.msgManager.addHandler( new MessageHandler( MessageType.RET_PREV_SCENE, this, this.handleRetPrevScene ) );
};
BattleScene.prototype = new Scene();

(function( pt )
{
	/**
	 * 
	 */
	pt.handleRetPrevScene = function( scene, message )
	{
		this.app.sceneManager.pop();
		var prevScene = this.app.sceneManager.current;
		prevScene.show();
	};
	/**
	 * 
	 */
	pt.reset = function( data, player, enemy )
	{
		this.data		= data;
		this.battleTask.reset( this, player, enemy );

		this.soundTask.prepareLoop("battle");
	};
	/**
	 * 
	 */
	pt.show = function()
	{
		this.setUsage( StringRes.HTML_USAGE );
		this.command.clear();

		this.holdCanvas();
		this.drawChildren( this );
	};
})( BattleScene.prototype );


// Expose
if ( !window.h5glib ) { window.h5glib = {}; }
window.h5glib.BattleScene	= BattleScene;


// Anonymous function end
//
})( window );
