/*
 * Copyright [yyyy] [name of copyright owner]
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */


// Anonymous function start
//
(function( window, undefined )
{

// reference
var Config			= window.h5glib.Config;
var Debug			= window.h5glib.Debug;
var Command			= window.h5glib.Command;
var Task			= window.h5glib.Task;
var SoundTask		= window.h5glib.SoundTask;
var ReadyTask		= window.h5glib.ReadyTask;
var ImageAnimator	= window.h5glib.ImageAnimator;
var Message			= window.h5glib.Message;
var MessageHandler	= window.h5glib.MessageHandler;
var MessageManager	= window.h5glib.MessageManager;
var SceneStatus		= window.h5glib.SceneStatus;
var Scene			= window.h5glib.Scene;

var PI	= Math.PI;
var PI2	= Math.PI * 2;

/**
 * Priority
 */
/*
var Priority =
{
	NONE	: 0,
	STAGE	: 1,
	ACTOR	: 2,
	EVENT	: 3,
	FRONT	: 4,
	DEBUG	: 5
};
*/
/**
 * RenderInfo
 */
var RenderInfo =
{
	fov			: ( 60 * PI / 180 ),
	rayPixels	: 2,	//  8,   4,   2,   1
	samples		: 200,	// 50, 100, 200, 400
	chipPixels	: 64
};

/**
 * StringRes
 */
var StringRes =
{
	HTML_USAGE	: "<p>" +
				  "[←][→]: 回転 [↑][↓]: 移動 [Z]: 発砲<br>" +
				  "キャラに接近するとイベント発動<br>" +
				  "（馬に接近すると戦闘）<br>" +
				  "（猫に接近するとステージ遷移）" +
				  "</p>",
	TB_EDIT		: "マップエディタに切り替える",
	ID_EDIT		: "edit"
};

/**
 * ColorRes
 */
var ColorRes =
{
	TEXTBOX			: "rgba(192, 80, 77, 0.7)",
	BORDER_TEXTBOX	: "rgb(192, 192, 192)",
	FONT_TEXTBOX	: "rgb(255, 255, 255)"
};

/**
 * FontRes
 */
var FontRes =
{
	LARGE		: "bold 18px 'ＭＳ Ｐゴシック'",
	SMALL		: "bold 14px 'ＭＳ Ｐゴシック'"
};

/**
 * MessageType
 */
var MessageType =
{
	NONE			: 0,
	RET_PREV_SCENE	: 1,
	START_BATTLE	: 2,
	WARP_SCENE		: 3,
	START_EDITOR	: 4
};

/**
 * EventType
 */
var EventType =
{
	NONE	: "none",
	TEXT	: "text",
	BATTLE	: "battle",
	WARP	: "warp"
};

/**
 * StageTask
 */
var StageTask = function( scene )
{
	this.command		= scene.command;	// input

	this.map			= null,
	this.getMapHeight	= function() { return this.map.length; };
	this.getMapWidth	= function() { return this.map[0].length; };

	var RaycastingData	= function()
	{
		this.dist	= 0;
		this.val	= 0;
		this.tx		= 0;
	};
	this.rd	= [];
	for ( var i = 0; i < RenderInfo.samples; i++ )
	{
		this.rd[i]	= new RaycastingData();
	}
};
StageTask.prototype = new Task();

(function( pt )
{
	/**
	 * 
	 */
	pt.setData = function( scene, map )
	{
		this.map	= map;
	};
	/**
	 * 
	 */
	pt.draw = function( scene )
	{
		this.castRay( scene );

		var canvas		= scene.canvas;
		var context		= scene.context;
		var resource	= scene.data.resource;
		var viewPoint	= scene.viewPoint;

		var image;
		// sky
		image		= resource.image.sky.data;
		var skyW2	= image.width - canvas.width;
		context.drawImage(
			image,
			Math.floor( skyW2 - ( viewPoint.ad.dir / PI2 * skyW2 ) ),	// sx
			0,				// sy
			canvas.width,	// sw
			image.height,	// sh
			0,				// dx
			0,				// dy
			canvas.width,	// dw
			image.height	// dh
		);
		// floor
		image = resource.image.floor.data;
		context.drawImage(
			image,
			0,				// sx
			0,				// sy
			canvas.width,	// sw
			image.height,	// sh
			0,				// dx
			canvas.height - image.height,		// dy
			canvas.width,	// dw
			image.height	// dh
		);

		// wall
		var startTheta	= viewPoint.ad.dir + ( RenderInfo.fov / 2 );

		for ( var i = 0; i < RenderInfo.samples; i++ )
		{
			var theta = startTheta - ( i * RenderInfo.fov / RenderInfo.samples );
			theta %= PI2;

			var sx	= this.rd[i].tx * ( RenderInfo.chipPixels - 1 );
			var sy	= ( this.rd[i].val - 1 ) * RenderInfo.chipPixels;
			var sw	= 1;
			var sh	= RenderInfo.chipPixels;

			var dist= this.rd[i].dist * Math.cos( theta - viewPoint.ad.dir );
			var dh	= canvas.height / dist;
			var dx	= i * RenderInfo.rayPixels;
			var dy	= ( canvas.height / 2 ) - ( dh / 2 );
			var dw	= RenderInfo.rayPixels;

			context.drawImage( resource.image.wall.data, sx, sy, sw, sh, dx, dy, dw, dh );
		}
	};
	/**
	 * 
	 */
	pt.castRay = function( scene )
	{
		var viewPoint	= scene.viewPoint;
		var startTheta	= viewPoint.ad.dir + ( RenderInfo.fov / 2 );

		for ( var i = 0; i < RenderInfo.samples; i++ )
		{
			var theta = startTheta - ( i * RenderInfo.fov / RenderInfo.samples );
			theta %= PI2;

			var mx = Math.floor( viewPoint.ad.x );
			var my = Math.floor( viewPoint.ad.y );

			var cosTheta = Math.cos( theta );
			var sinTheta = Math.sin( theta );

			var distX, distY;
			var stepX, stepY;

			if ( cosTheta == 0 )
			{
				distX = 100;
			}
			else if ( cosTheta > 0 )
			{
				stepX = 1;
				distX = ( mx + 1 - viewPoint.ad.x ) / cosTheta;
			}
			else
			{
				stepX = -1;
				cosTheta *= -1.0;
				distX = ( viewPoint.ad.x - mx ) / cosTheta;
			}

			if ( sinTheta == 0 )
			{
				distY = 100;
			}
			else if ( sinTheta > 0 )
			{
				stepY = -1;
				distY = ( viewPoint.ad.y - my ) / sinTheta;
			}
			else
			{
				stepY = 1;
				sinTheta *= -1.0;
				distY = ( my + 1 - viewPoint.ad.y ) / sinTheta;
			}

			while ( true )
			{
				if ( distX < distY )
				{
					mx += stepX;
					var mapVal = ( 0 <= mx || mx < this.getMapWidth() ) ? this.map[ my ][ mx ] : 1;
					if ( mx < 0 || this.getMapWidth() <= mx || mapVal )
					{
						// not actor
						if ( mapVal < Config.actorValBase )
						{
							this.rd[i].dist	= distX;
							this.rd[i].val	= mapVal;
							this.rd[i].tx	= 1 - ( viewPoint.ad.y + distX * sinTheta * stepY ) % 1;
							break;
						}
					}
					distX += ( 1 / cosTheta );	// + dist per grid
				}
				else
				{
					my += stepY;
					var mapVal = this.map[ my ][ mx ];
					var mapVal = ( 0 <= my || my < this.getMapHeight() ) ? this.map[ my ][ mx ] : 1;
					if ( my < 0 || this.getMapHeight() <= my || mapVal )
					{
						// not actor
						if ( mapVal < Config.actorValBase )
						{
							this.rd[i].dist	= distY;
							this.rd[i].val	= mapVal;
							this.rd[i].tx	= ( viewPoint.ad.x + distY * cosTheta * stepX ) % 1;
							break;
						}
					}
					distY += ( 1 / sinTheta );	// + dist per grid
				}
			}
		}
	};
})( StageTask.prototype );

/**
 * ActorListTask
 */
var ActorListTask = function( scene )
{
	this.command	= scene.command;	// input
	this.visible	= [];
};
ActorListTask.prototype = new Task();

(function( pt )
{
	/**
	 * 
	 */
	pt.setData = function( scene, actors )
	{
		this.child	= null;

		// load actorList
		var prev	= null;
		for ( var i = 0; i < actors.length; i++ )
		{
			var task	= new ActorTask();
			task.init( scene, i, actors[i] );
			// add
			if ( prev == null )
			{
				this.child = task;
			}
			else
			{
				task.link( prev );
			}
			// for next step
			prev = task;
		}
	};

	/**
	 * 
	 */
	pt.update = function( scene )
	{
		return this.updateChildren( scene );
	};

	/**
	 * 
	 */
	pt.draw = function( scene )
	{
		this.pickVisible( scene );

		var canvas		= scene.canvas;
		var context		= scene.context;
		var resource	= scene.data.resource;
		var stage		= scene.stageTask;
		var viewPoint	= scene.viewPoint;

		for ( var i = 0; i < this.visible.length; i++ )
		{
			var actor = this.visible[i];
			// if behind wall
			if ( stage.rd[ actor.rd.index ].dist <= actor.rd.dist )
			{
				continue;
			}

			var offSx = actor.animator.fx;
			var offSy = 0;
			// angle
			var sdir = viewPoint.ad.dir - actor.ad.dir;
			if ( sdir < 0 ) { sdir = PI2 + sdir; }
			sdir = sdir / ( PI / 4 );
			if ( sdir < 1 || sdir >= 7 )
			{
				sdir = 0;
			}
			else
			{
				sdir = Math.floor( ( sdir + 1 ) / 2 );
			}
			offSy = sdir * actor.animator.fh;

			var dh	= canvas.height / actor.rd.dist;
			var dw	= dh;
			var dx	= actor.rd.index * RenderInfo.rayPixels - dw / 2;
			var dy	= canvas.height / 2 - dh / 2;

			var rx	= dx / RenderInfo.rayPixels;
			var rw	= dw / RenderInfo.rayPixels;

			for ( var n = 0; n < rw; n++ )
			{
				var rid = Math.floor( rx + n );
				if ( rid < 0 ) { continue; }
				if ( rid >= RenderInfo.samples ) { break; }
				// draw
				if ( stage.rd[ rid ].dist > actor.rd.dist )
				{
					var image = actor.animator.image;
					var sx = offSx + ( n / rw ) * ( RenderInfo.chipPixels - 1 );
					var sy = offSy;
					var sw = RenderInfo.chipPixels / rw;
					var sh = RenderInfo.chipPixels;
					context.drawImage(
						image, sx, sy, sw, sh, dx + ( n * RenderInfo.rayPixels ), dy, RenderInfo.rayPixels, dh );
				}
			}
		}
	};
	/**
	 * 
	 */
	pt.pickVisible = function( scene )
	{
		var viewPoint	= scene.viewPoint;
		var stage		= scene.stageTask;

		var startTheta	= viewPoint.ad.dir + ( RenderInfo.fov / 2 );

		// search visible actor
		this.visible.length = 0;
		for ( var actor = this.child.next; actor != null; actor = actor.next )
		{
			var x		= actor.ad.x - viewPoint.ad.x;
			var y		= viewPoint.ad.y - actor.ad.y;
			var thetaV	= Math.atan2( y, x );	// PI ... PI * -1
			if ( thetaV < 0 ) { thetaV += PI2; }

			var diff	= startTheta - thetaV;
			if ( 0 <= diff && diff <= RenderInfo.fov )
			{
				actor.rd.index	= Math.floor( diff / ( RenderInfo.fov / RenderInfo.samples ) );
				actor.rd.dist	= Math.sqrt( x * x + y * y );
				this.visible.push( actor );
			}
		}
		this.visible.sort( function( a, b ) { return b.rd.dist - a.rd.dist;  } );
	};
})( ActorListTask.prototype );

/**
 * ActorStatus
 */
var ActorStatus =
{
	STATIC	: 0,
	ACTIVE	: 1,
	MORTAL	: 2
};

/**
 * ActorTask
 */
var ActorTask = function()
{
	this.statusObjs		= null;
	this.animator		= null;
	this.id		= 0;
	// actor data
	this.ad		= null;
	// raycasting data
	this.rd			=
	{
		index	: 0,
		dist	: 0
	};
	this.velocity	= 0;
};
ActorTask.prototype = new Task();

(function( pt )
{
	pt.ROTATE_VAL	= 0.08;
	pt.STEP_VAL		= 0.20;

	/**
	 * 
	 */
	pt.init = function( scene, id, actor )
	{
		this.id		= id;
		this.ad		= actor,

		this.moveTo( scene.stageTask, this.ad.x, this.ad.y );
		this.command= ( id == 0 ) ? scene.command : new AICommand();

		var image		= scene.data.resource.image.actors[ this.ad.type ].data;
		var animation	= scene.data.animation;
		this.statusObjs	=
		[
			{ update : this.updateStatic,	animator : new ImageAnimator( image, animation.actorAction ) },
			{ update : this.updateActive,	animator : new ImageAnimator( image, animation.actorAction ) },
			{ update : this.updateMortal,	animator : new ImageAnimator( image, animation.actorMotal ) }
		];
		this.setStatus( this.ad.status );
	};
	/**
	 * 
	 */
	pt.setStatus = function( status )
	{
		this.status		= status;

		var sobj		= this.statusObjs[ this.status ];
		this.update		= sobj.update;
		this.animator	= sobj.animator;
	};
	/**
	 * 
	 */
	pt.moveTo = function( stage, x, y )
	{
		// restore map value
		stage.map[ Math.floor( this.ad.y ) ][ Math.floor( this.ad.x ) ] = 0;
		this.ad.x = x;
		this.ad.y = y;
		// update map value
		stage.map[ Math.floor( this.ad.y ) ][ Math.floor( this.ad.x ) ] = Config.actorValBase + this.id;
	};
	/**
	 * 
	 */
	pt.rotate = function( val )
	{
		this.ad.dir	+= val;
		this.ad.dir	%= PI2;
		if ( this.ad.dir < 0 ) { this.ad.dir += PI2; }
	};
	/**
	 * 
	 */
	pt.isInnerWall = function( stage, x, y )
	{
		var arr = [ -0.3, 0.3 ];

		for ( var i = 0; i < arr.length; i++ )
		{
			var my = Math.floor( y + arr[i] );
			if ( my < 0 || stage.getMapHeight() <= my ) { return true; }

			for ( var j = 0; j < arr.length; j++ )
			{
				var mx = Math.floor( x + arr[j] );
				if ( mx < 0 || stage.getMapWidth() <= mx ) { return true; }

				var value = stage.map[ my ][ mx ];
				if ( value != 0 && value != Config.actorValBase + this.id )
				{
					return true;
				}
			}
		}
		return false;
	};
	/**
	 * 
	 */
	pt.updateStatic = function( scene )
	{
		var upd = false;
		// animation
		if ( this.animator.isActive() )
		{
			upd = this.animator.proceed();
		}
		else
		{
			var r = Math.floor( Math.random() * 15 );
			if ( r == 1 )
			{
				this.animator.start();
				upd = true;
			}
		}
		return upd;
	};
	/**
	 * 
	 */
	pt.updateActive = function( scene )
	{
		var upd = false;

		this.command.update();

		// left right
		if ( this.command.tbl.left )
		{
			this.rotate( this.ROTATE_VAL );
			upd = true;
		}
		else if ( this.command.tbl.right )
		{
			this.rotate( this.ROTATE_VAL * -1 );
			upd = true;
		}

		// go back
		if ( this.command.tbl.go )
		{
			this.velocity = this.STEP_VAL;
			upd = true;
		}
		else if ( this.command.tbl.back )
		{
			this.velocity = this.STEP_VAL * -1;
			upd = true;
		}
		else
		{
			if ( this.velocity != 0 )
			{
				this.velocity /= 2;
				if ( this.velocity <= this.STEP_VAL / 8 )
				{
					this.velocity = 0;
				}
				upd = true;
			}
		}

		if ( this.velocity != 0 )
		{
			var newX	= this.ad.x + Math.cos( this.ad.dir ) * this.velocity;
			var newY	= this.ad.y - Math.sin( this.ad.dir ) * this.velocity;

			if ( !this.isInnerWall( scene.stageTask, newX, this.ad.y ) )
			{
				this.moveTo( scene.stageTask, newX, this.ad.y );
				upd = true;
			}
			if ( !this.isInnerWall( scene.stageTask, this.ad.x, newY ) )
			{
				this.moveTo( scene.stageTask, this.ad.x, newY );
				upd = true;
			}
		}
		// animation
		if ( this.animator.isActive() )
		{
			upd = this.animator.proceed();
		}
		else
		{
			var r = Math.floor( Math.random() * 10 );
			if ( r == 1 )
			{
				this.animator.start();
				upd = true;
			}
		}
		return upd;
	};
	/**
	 * 
	 */
	pt.updateMortal = function( scene )
	{
		var upd = false;
		// animation
		if ( this.animator.isActive() )
		{
			upd = this.animator.proceed();
		}
		return upd;
	};
})( ActorTask.prototype );

/**
 * EventTask
 */
var EventTask = function( scene )
{
	this.command	= scene.command;	// input
	this.prevDist	= [];
};
EventTask.prototype = new Task();

(function( pt )
{
	pt.EVENT_DIST	= 1;
	pt.FIRE_DIST	= 2;

	/**
	 * 
	 */
	pt.update = function( scene )
	{
		var upd			= false;
		var actorList	= scene.actorListTask;
		var msgManager	= scene.msgManager;
		var sound		= scene.soundTask;

		// check distance
		for ( var i = 0; i < actorList.visible.length; i++ )
		{
			var actor = actorList.visible[i];
			// check event
			if ( actor.rd.dist < this.EVENT_DIST )
			{
				var dist = this.prevDist[ actor.id ];
				if ( dist != undefined && dist > this.EVENT_DIST )
				{
					// text box
					if ( actor.ad.event.type == EventType.TEXT )
					{
						sound.prepare("notice");

						// add task
						var task = new TextBoxTask( scene );
						task.link( this );
						task.eventActor	= actor;
						// setFocus
						scene.setFocus( task );

						upd = true;
					}
					// battle
					else if ( actor.ad.event.type == EventType.BATTLE )
					{
						var message = new Message( MessageType.START_BATTLE );
						message.data.enemy = actor;
						msgManager.postBreakMessage( message );

						upd = true;
					}
					// warp
					else if ( actor.ad.event.type == EventType.WARP )
					{
						sound.prepare("warp");

						var message = new Message( MessageType.WARP_SCENE );
						message.data.baseUrl = actor.ad.event.baseUrl;
						msgManager.postBreakMessage( message );

						upd = true;
					}
					break;
				}
			}
			// check fire
			if ( this.command.tbl.fire )
			{
				if ( actor.rd.dist < this.FIRE_DIST && actor.status == ActorStatus.ACTIVE )
				{
					actor.setStatus( ActorStatus.MORTAL );
					actor.animator.start();
					upd = true;
					break;
				}
			}
		}
		// copy dist
		this.prevDist.length = 0;
		for ( var i = 0; i < actorList.visible.length; i++ )
		{
			var actor = actorList.visible[i];
			this.prevDist[ actor.id ] = actor.rd.dist;
		}

		//...Debug
		if ( this.command.tbl.escape )
		{
			var message	= new Message( MessageType.RET_PREV_SCENE );
			msgManager.postBreakMessage( message );
		}
		return upd;
	};
})( EventTask.prototype );

/**
 * TextBoxTask
 */
var TextBoxTask = function( scene )
{
	this.command	= scene.command;	// input
	this.eventActor	= null;

	this.lag		= this.LAG_VAL;
};
TextBoxTask.prototype = new Task();

(function( pt )
{
	pt.LAG_VAL	= 10;

	/**
	 * 
	 */
	pt.update = function( scene )
	{
		var upd = false;

		if ( scene.getFocus() == this )
		{
			if ( this.lag > 0 )
			{
				this.lag--;
			}
			else if ( this.command.tbl.go || this.command.tbl.back || this.command.tbl.left || this.command.tbl.right )
			{
				scene.setFocus( null );
				this.unlink();
				upd = true;
			}
		}
		return upd;
	};

	/**
	 * 
	 */
	pt.draw = function( scene )
	{
		var canvas		= scene.canvas;
		var context		= scene.context;

		var tx = 10;
		var ty = 10;
		var th = 50;

		context.save();

		// rect
		context.fillStyle	= ColorRes.TEXTBOX;
		context.fillRect( tx, ty, canvas.width - tx * 2, th );
		// border
		context.strokeStyle	= ColorRes.BORDER_TEXTBOX;
		context.beginPath();
		context.strokeRect( tx, ty, canvas.width - tx * 2, th );
		// text
		context.font		= FontRes.LARGE;
		context.fillStyle	= ColorRes.FONT_TEXTBOX;
		context.fillText( this.eventActor.ad.event.text, tx + 10, ty + 25 );

		context.restore();
	};
})( TextBoxTask.prototype );

/**
 * FireTask
 */
var FireTask = function( scene, cx, cy )
{
	this.animator	= new ImageAnimator( scene.data.resource.image.fire.data, scene.data.animation.fire );
	this.animator.start();
	this.trans	= { x : cx, y : cy };	// center position
};
FireTask.prototype = new Task();

(function( pt )
{
	/**
	 * 
	 */
	pt.update = function( scene )
	{
		var upd = false;

		if ( this.animator.isActive() )
		{
			upd = this.animator.proceed();
		}
		else
		{
			this.unlink();
			upd = true;
		}
		return upd;
	};

	/**
	 * 
	 */
	pt.draw = function( scene )
	{
		var canvas		= scene.canvas;
		var context		= scene.context;

		context.save();

		context.drawImage(
			this.animator.image,
			this.animator.fx,							// sx
			this.animator.fy,							// sy
			this.animator.fw,							// sw
			this.animator.fh,							// sh
			this.trans.x - ( this.animator.fw / 2 ),	// dx
			this.trans.y - ( this.animator.fh / 2 ),	// dy
			this.animator.fw,							// dw
			this.animator.fh							// dh
		);

		context.restore();
	};
})( FireTask.prototype );

/**
 * FrontViewTask
 */
var FrontViewTask = function( scene )
{
	this.command	= scene.command;	// input
	this.gx			= 0;
	this.gy			= 0;
	this.lag		= 0;
};
FrontViewTask.prototype = new Task();

(function( pt )
{
	pt.LAG_VAL	= 5;

	/**
	 * 
	 */
	pt.update = function( scene )
	{
		var upd		= false;
		var sound	= scene.soundTask;

		// lag
		if ( this.lag )
		{
			this.lag--;
			upd = true;
		}
		else
		{
			if ( this.command.tbl.fire )
			{
				sound.prepare("shoot");

				var task	= new FireTask( scene, this.gx, this.gy );
				task.link( this.prev );
				this.lagF	= this.LAG_VAL;
				upd = true;
			}
		}
		return upd;
	};

	/**
	 * 
	 */
	pt.draw = function( scene )
	{
		var canvas		= scene.canvas;
		var context		= scene.context;
		var resource	= scene.data.resource;
		var stage		= scene.stageTask;
		var viewPoint	= scene.viewPoint;
		var image;

		// gun
		this.gx = canvas.width  - resource.image.gun.data.width;
		this.gy = canvas.height - resource.image.gun.data.height;
		this.gx += 20 + Math.cos( scene.ticks / 6 ) * viewPoint.velocity * 70;
		this.gy += 20 + Math.cos( scene.ticks / 5 ) * viewPoint.velocity * 70;

		image = resource.image.gun.data;
		context.drawImage(
			image,
			0,				// sx
			0,				// sy
			image.width,	// sw
			image.height,	// sh
			this.gx,		// dx
			this.gy,		// dy
			image.width,	// dw
			image.height	// dh
		);
	};
})( FrontViewTask.prototype );

/**
 * DebugTask
 */
var DebugTask = function( scene )
{
	this.command	= scene.command;	// input
	this.info		= null;
	this.lag		= 0;
};
DebugTask.prototype = new Task();

(function( pt )
{
	pt.LAG_VAL	= 10;

	/**
	 * 
	 */
	pt.update = function( scene )
	{
		var upd = false;
		// lag
		if ( this.lag )
		{
			this.lag--;
		}
		else
		{
			if ( this.command.tbl.debug )
			{
				this.info	= "viewPoint.ad.x=" + scene.viewPoint.ad.x + ", viewPoint.ad.y=" + scene.viewPoint.ad.y;
				this.lag	= this.LAG_VAL;
				upd = true;
			}
		}
		return upd;
	};

	/**
	 * 
	 */
	pt.draw = function( scene )
	{
		// output debug info
		if ( this.info )
		{
			Debug.print( this.info );
			this.info = null;
		}
	};
})( DebugTask.prototype );


/**
 * AICommand
 */
var AICommand = function()
{
	this.tbl.left	= 0;
	this.tbl.right	= 0;
	this.tbl.go		= 0;
	this.tbl.back	= 0;
};
AICommand.prototype = new Command();

(function( pt )
{
	/**
	 * 
	 */
	pt.update = function()
	{
		this.clear();

		var r = Math.floor( Math.random() * 18 );
		if      ( 0 <= r && r <= 3 )	{ this.tbl.left		= 1; }
		else if ( 4 <= r && r <= 8 )	{ this.tbl.right	= 1; }
		else if ( r == 9 )				{ this.tbl.go		= 1; }
		else if ( r == 10 )				{ this.tbl.back		= 1; }
	};
})( AICommand.prototype );

/**
 * InputCommand
 */
var InputCommand = function()
{
	this.tbl.left	= 0;
	this.tbl.right	= 0;
	this.tbl.go		= 0;
	this.tbl.back	= 0;
	this.tbl.fire	= 0;
	this.tbl.seek	= 0;
	this.tbl.escape	= 0;
	this.tbl.debug	= 0;
};
InputCommand.prototype = new Command();

(function( pt )
{
	/**
	 * 
	 */
	pt.update = function() {};
	/**
	 * 
	 */
	pt.handleSysEvent = function( event )
	{
		var type = event.type.toLowerCase();
		if ( type.substring( 0, 3 ) == "key" )
		{
			var value = 0;
			if      ( type == "keydown" ) { value = 1; }
			else if ( type == "keyup"   ) { value = 0; }

			switch ( event.keyCode )
			{
				case 37: case 65:	this.tbl.left	= value; break;	// ← | A
				case 39: case 68:	this.tbl.right	= value; break;	// → | D
				case 38: case 87:	this.tbl.go		= value; break;	// ↑ | W
				case 40: case 83:	this.tbl.back	= value; break;	// ↓ | S
				case 90:			this.tbl.fire	= value; break;	// Z
				case 32:			this.tbl.seek	= value; break;	// space;
				case 27:			this.tbl.escape	= value; break;	// ESC
				case 81:			this.tbl.debug	= value; break;	// Q
			}
		}
	};
})( InputCommand.prototype );

/**
 * StageScene
 */
var StageScene = function( app, name )
{
	this.app			= app;
	this.name			= name;
	this.data			= null;
	this.battleName		= null;
	this.editorName		= null;

	this.command		= new InputCommand();
	this.viewPoint		= null;

	// create task
	this.readyTask		= new ReadyTask( this );
	this.stageTask		= new StageTask( this );
	this.actorListTask	= new ActorListTask( this );
	this.eventTask		= new EventTask( this );
	this.frontViewTask	= new FrontViewTask( this );
	this.soundTask		= new SoundTask( this );
	this.debugTask		= new DebugTask( this );
	// create list
	this.actorListTask	.link( this.stageTask );
	this.eventTask		.link( this.actorListTask );
	this.frontViewTask	.link( this.eventTask );
	this.soundTask		.link( this.frontViewTask );
	this.debugTask		.link( this.soundTask );

	this.headTasks		=
	[
		this.readyTask,
		this.stageTask,
		this.readyTask
	];
	this.setStatus( SceneStatus.READY );

	// message handler
	this.msgManager		= new MessageManager();
	this.msgManager.addHandler( new MessageHandler( MessageType.RET_PREV_SCENE,	this, this.handleRetPrevScene ) );
	this.msgManager.addHandler( new MessageHandler( MessageType.START_BATTLE,	this, this.handleStartBattle ) );
	this.msgManager.addHandler( new MessageHandler( MessageType.WARP_SCENE,		this, this.handleWarpScene ) );
	this.msgManager.addHandler( new MessageHandler( MessageType.START_EDITOR,	this, this.handleStartEditor ) );
};
StageScene.prototype = new Scene();

(function( pt )
{
	/**
	 * 
	 */
	pt.handleRetPrevScene = function( scene, message )
	{
		this.app.sceneManager.pop();
		var prevScene = this.app.sceneManager.current;
		prevScene.stageName = this.name;	// for MenuScene
		prevScene.show();
	};
	/**
	 * 
	 */
	pt.handleStartBattle = function( scene, message )
	{
		var battleScene	= this.app.sceneManager.push( this.battleName );
		battleScene.reset( this.data, this.viewPoint, message.data.enemy );
		battleScene.show();
	};
	/**
	 * 
	 */
	pt.handleWarpScene = function( scene, message )
	{
		// set status
		this.setStatus( SceneStatus.READY );
		// pop this
		this.app.sceneManager.pop();

		var baseUrl	= message.data.baseUrl;
		var stageScene	= this.app.sceneManager.push( baseUrl );
		if ( stageScene )
		{
			// set status
			stageScene.setStatus( SceneStatus.RUNNING );
		}
		else
		{
			// create scene
			stageScene	= new StageScene( this.app, baseUrl );
			// scene name
			stageScene.battleName	= this.battleName;
			stageScene.editorName	= this.editorName;

			this.app.sceneManager.add( stageScene );
			this.app.sceneManager.push( stageScene.name );

			stageScene.init();
			stageScene.show();

			// load data
			this.app.loadScript( baseUrl + Config.dataJSName );
		}
	};
	/**
	 * 
	 */
	pt.handleStartEditor = function( scene, message )
	{
		var editorScene	= this.app.sceneManager.push( this.editorName );
		editorScene.setData( this.data );
		// set status
		editorScene.setStatus( SceneStatus.RUNNING );
		editorScene.show();
	};

	/**
	 * 
	 */
	pt.setStatus = function( status )
	{
		this.status	= status;
		this.child	= this.headTasks[ this.status ];
	};

	/**
	 * 
	 */
	pt.init = function()
	{
		// window.onload のタイミング
		var self	= this;
		var button	= this.app.getDomElement( StringRes.ID_EDIT );
		if ( button )
		{
			this.app.addSysEventHandler( button, "click", function()
			{
				try
				{
					if ( self.app.sceneManager.current == self )
					{
						var message = new Message( MessageType.START_EDITOR );
						self.msgManager.postBreakMessage( message );
					}
				}
				catch ( e )
				{
					self.app.kill();
					Debug.alertError( e );
				}
			});
		}
	};
	/**
	 * 
	 */
	pt.show = function()
	{
		var button	= this.app.getDomElement( StringRes.ID_EDIT );
		if ( button )
		{
			button.value = StringRes.TB_EDIT;
		}

		this.setUsage( StringRes.HTML_USAGE );

		this.command.clear();
		this.holdCanvas();
		this.drawChildren( this );
	};

	/**
	 * 
	 */
	pt.setData = function( data )
	{
		this.data	= data;

		// load stage
		this.stageTask.setData( this, data.map );
		// load actorList
		this.actorListTask.setData( this, data.actors );
		// init viewPoint
		this.viewPoint	= this.actorListTask.child;
	};
	/**
	 * 
	 */
	pt.getData = function()
	{
		return this.data;
	};
	/**
	 * 
	 */
	pt.loadData = function( data )
	{
		try
		{
			// load resource
			this.app.loadResource( data.baseUrl, data.resource );
			// set data
			this.setData( data );
			// set status
			var self = this;
			window.setTimeout( function() { self.setStatus( SceneStatus.RUNNING ); self.show(); }, 500 );
		}
		catch ( e )
		{
			this.app.kill();
			Debug.alertError( e );
		}
	};
})( StageScene.prototype );


// Expose
if ( !window.h5glib ) { window.h5glib = {}; }
window.h5glib.StageScene	= StageScene;

// Anonymous function end
//
})( window );
