/*
 * Copyright [yyyy] [name of copyright owner]
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */


// Anonymous function start
//
(function( window, undefined )
{

// reference
var Config			= window.h5glib.Config;
var Debug			= window.h5glib.Debug;
var Command			= window.h5glib.Command;
var Task			= window.h5glib.Task;
var SoundTask		= window.h5glib.SoundTask;
var ReadyTask		= window.h5glib.ReadyTask;
var ImageAnimator	= window.h5glib.ImageAnimator;
var Message			= window.h5glib.Message;
var MessageHandler	= window.h5glib.MessageHandler;
var MessageManager	= window.h5glib.MessageManager;
var SceneStatus		= window.h5glib.SceneStatus;
var Scene			= window.h5glib.Scene;

/**
 * RenderInfo
 */
var RenderInfo =
{
	chipPixels	: 32
};

/**
 * Res
 */
var Res =
{
	String :
	{
		HTML_USAGE		: "<p>" +
						  "＜キー入力ができない場合はゲームの画面をクリックしてみてください＞<br>" +
						  "<br>" +
						  "</p>"
	},
	Color :
	{
		BACKGROUND		: "black",
		TEXTBOX			: "rgba(192, 80, 77, 0.7)",
		BORDER_TEXTBOX	: "rgb(192, 192, 192)",
		FONT_TEXTBOX	: "rgb(255, 255, 255)"
	},
	Font :
	{
		LARGE			: "bold 18px 'ＭＳ Ｐゴシック'",
		SMALL			: "bold 14px 'ＭＳ Ｐゴシック'"
	}
};

/**
 * MessageType
 */
var MessageType =
{
	NONE			: 0,
	RET_PREV_SCENE	: 1,
	START_STAGE		: 2
};

/**
 * EventType
 */
var EventType =
{
	NONE	: "none"
};

/**
 * EventTask
 */
var EventTask = function( scene )
{
	this.command		= scene.command;	// input

	this.script			= null;

	this.currentExp		= -1;
	this.currentTask	= null;

	this.bgTask			= null;
	this.bgmTask		= null;

	this.expTaskFunc	=
	{
		"bg"		: BgExpTask,
		"bgm"		: BgmExpTask,
		"message"	: MessageExpTask,
		"wait"		: WaitExpTask,
		"stage"		: StageExpTask
	};
};
EventTask.prototype = new Task();

(function( pt )
{
	/**
	 * 
	 */
	pt.setData = function( scene, script )
	{
		this.script		= script;
		this.proceed( scene );
	};
	/**
	 * 
	 */
	pt.proceed = function( scene )
	{
		this.currentExp++;
		// loop ?
		if ( this.currentExp >= this.script.expressions.length )
		{
			this.currentExp	= 0;
		}
		var expression		= this.script.expressions[ this.currentExp ];
		var func			= this.expTaskFunc[ expression.type ];
		this.currentTask	= new func( scene );
		this.currentTask.setExpression( expression );

		// set bg, bgm
		if ( expression.type == "bg" )
		{
			this.bgTask		= this.currentTask;
		}
		else if ( expression.type == "bgm" )
		{
			if ( this.bgmTask )
			{
				this.bgmTask.cancel( scene );
			}
			this.bgmTask	= this.currentTask;
		}
	};

	/**
	 * 
	 */
	pt.update = function( scene )
	{
		var upd = false;

		if ( this.currentTask.terminated )
		{
			this.proceed( scene );
		}
		upd = this.currentTask.update( scene );

		return upd;
	};
	/**
	 * 
	 */
	pt.draw = function( scene )
	{
		if ( this.bgTask && this.bgTask != this.currentTask )
		{
			this.bgTask.draw( scene );
		}
		this.currentTask.draw( scene );
	};
})( EventTask.prototype );

/**
 * ExpTask
 */
var ExpTask = function( scene )
{
	this.expression	= null;
	this.terminated	= false;
};
ExpTask.prototype = new Task();

(function( pt )
{
	/**
	 * 
	 */
	pt.setExpression = function( expression )
	{
		this.expression	= expression;
	};
})( ExpTask.prototype );

/**
 * BgExpTask
 */
var BgExpTask = function( scene )
{
	this.command		= scene.command;	// input
};
BgExpTask.prototype = new ExpTask();

(function( pt )
{
	/**
	 * 
	 */
	pt.update = function( scene )
	{
		var upd = !this.terminated;
		this.terminated	= true;
		return upd;
	};
	/**
	 * 
	 */
	pt.draw = function( scene )
	{
		var context		= scene.context;
		var canvas		= context.canvas;
		var resource	= scene.data.resource;

		context.save();

		if ( this.expression.image )
		{
			var image	= resource.image[ this.expression.image ].data;

			context.drawImage(
				image,
				0,				// sx
				0,				// sy
				image.width,	// sw
				image.height,	// sh
				0,				// dx
				0,				// dy
				canvas.width,	// dw
				canvas.height	// dh
			);
		}
		else
		{
			context.fillStyle	= Res.Color.BACKGROUND;
			context.fillRect( 0, 0, canvas.width, canvas.height );
		}
		context.restore();
	};
})( BgExpTask.prototype );

/**
 * BgmExpTask
 */
var BgmExpTask = function( scene )
{
	this.command		= scene.command;	// input
};
BgmExpTask.prototype = new ExpTask();

(function( pt )
{
	/**
	 * 
	 */
	pt.cancel = function( scene )
	{
		var resource	= scene.data.resource;
		var sound		= scene.soundTask;
		sound.cancel( resource.sound[ this.expression.sound ].data );
	};
	/**
	 * 
	 */
	pt.update = function( scene )
	{
		var upd = !this.terminated;

		if ( !this.terminated )
		{
			var resource	= scene.data.resource;
			var sound		= scene.soundTask;
			sound.prepareLoop( resource.sound[ this.expression.sound ].data );
		}
		this.terminated	= true;

		return upd;
	};
})( BgmExpTask.prototype );

/**
 * MessageExpTask
 */
var MessageExpTask = function( scene )
{
	this.command	= scene.command;	// input

	this.lag		= 0;
	this.count		= 0;
};
MessageExpTask.prototype = new ExpTask();

(function( pt )
{
	pt.LAG_VAL	= 30;

	/**
	 * 
	 */
	pt.update = function( scene )
	{
		var upd = false;

		var resource	= scene.data.resource;
		var sound		= scene.soundTask;

		if ( this.lag > 0 )
		{
			this.lag--;
		}
		else
		{
			this.lag	= this.LAG_VAL;

			this.count++;
			if ( this.count <= this.expression.texts.length )
			{
				sound.prepare( resource.sound[ this.expression.tsound ].data );
				upd	= true;
			}
			else
			{
				this.terminated	= true;
			}
		}
		return upd;
	};
	/**
	 * 
	 */
	pt.draw = function( scene )
	{
		var context		= scene.context;
		var canvas		= context.canvas;
		var resource	= scene.data.resource;

		context.save();

		// box
		if ( this.expression.box )
		{
			var box = this.expression.box;
			// rect
			context.fillStyle	= Res.Color.TEXTBOX;
			context.fillRect( box.x, box.y, box.w, box.h );
			// border
			context.strokeStyle	= Res.Color.BORDER_TEXTBOX;
			context.beginPath();
			context.strokeRect( box.x, box.y, box.w, box.h );
		}
		// image
		if ( this.expression.image )
		{
			var image	= resource.image[ this.expression.image ].data;

			context.drawImage(
				image,
				0,							// sx
				0,							// sy
				image.width,				// sw
				image.height,				// sh
				this.expression.imageDst.x,	// dx
				this.expression.imageDst.y,	// dy
				image.width,				// dw
				image.height				// dh
			);
		}
		// text
		var fheight			= 18;
		context.font		= Res.Font.SMALL;
		context.fillStyle	= Res.Color.FONT_TEXTBOX;
		for ( var i = 0; i < this.count; i++ )
		{
			context.fillText( this.expression.texts[i], this.expression.textDst.x, this.expression.textDst.y + ( i + 1 ) * fheight );
		}

		context.restore();
	};
})( MessageExpTask.prototype );

/**
 * WaitExpTask
 */
var WaitExpTask = function( scene )
{
	this.command	= scene.command;	// input
	this.count		= 0;
};
WaitExpTask.prototype = new ExpTask();

(function( pt )
{
	/**
	 * 
	 */
	pt.update = function( scene )
	{
		var upd = false;
		var msgManager	= scene.msgManager;

		if ( this.expression.key )
		{
			if ( this.command.tbl[ this.expression.key ] )
			{
				// terminate
				this.terminated	= true;
			}
		}
		else if ( this.expression.frames )
		{
			this.count++;
			if ( this.count >= this.expression.frames )
			{
				// terminate
				this.terminated	= true;
			}
		}
		return upd;
	};
})( WaitExpTask.prototype );

/**
 * StageExpTask
 */
var StageExpTask = function( scene )
{
	this.command	= scene.command;	// input
};
StageExpTask.prototype = new ExpTask();

(function( pt )
{
	/**
	 * 
	 */
	pt.update = function( scene )
	{
		var upd = false;
		var msgManager	= scene.msgManager;

		var message = new Message( MessageType.START_STAGE );
		message.data.baseUrl = this.expression.baseUrl;
		msgManager.postBreakMessage( message );
		// stop bgm
		if ( scene.eventTask.bgmTask )
		{
			scene.eventTask.bgmTask.cancel( scene );
		}
		// terminate
		this.terminated	= true;

		return upd;
	};
})( StageExpTask.prototype );

/**
 * DebugTask
 */
var DebugTask = function( scene )
{
	this.command	= scene.command;	// input
	this.info		= null;
	this.lag		= 0;
};
DebugTask.prototype = new Task();

(function( pt )
{
	pt.LAG_VAL	= 10;

	/**
	 * 
	 */
	pt.update = function( scene )
	{
		var upd = false;
		// lag
		if ( this.lag )
		{
			this.lag--;
		}
		else
		{
			if ( this.command.tbl.debug )
			{
				this.info	= "viewPoint.ad.x=" + scene.viewPoint.ad.x + ", viewPoint.ad.y=" + scene.viewPoint.ad.y;
				this.lag	= this.LAG_VAL;
				upd = true;
			}
		}
		return upd;
	};

	/**
	 * 
	 */
	pt.draw = function( scene )
	{
		// output debug info
		if ( this.info )
		{
			Debug.print( this.info );
			this.info = null;
		}
	};
})( DebugTask.prototype );

/**
 * InputCommand
 */
var InputCommand = function()
{
	this.tbl.seek	= 0;
	this.tbl.escape	= 0;
	this.tbl.debug	= 0;

	this.tbl.any	= 0;
};
InputCommand.prototype = new Command();

(function( pt )
{
	/**
	 * 
	 */
	pt.update = function() {};
	/**
	 * 
	 */
	pt.handleSysEvent = function( event )
	{
		var type = event.type.toLowerCase();
		if ( type.substring( 0, 3 ) == "key" )
		{
			var value = 0;
			if      ( type == "keydown" ) { value = 1; }
			else if ( type == "keyup"   ) { value = 0; }

			switch ( event.keyCode )
			{
				case 32: this.tbl.seek	= value; break;	// space;
				case 27: this.tbl.escape= value; break;	// ESC
				case 81: this.tbl.debug	= value; break;	// Q
			}
			this.tbl.any	= value;	// any key
		}
	};
})( InputCommand.prototype );

/**
 * EventScene
 */
var EventScene = function( app, name, stageFunc )
{
	this.app			= app;
	this.name			= name;
	this.data			= null;

	this.stageFunc		= stageFunc;

	this.command		= new InputCommand();

	// create task
	this.eventTask		= new EventTask( this );
	this.soundTask		= new SoundTask( this );
	this.debugTask		= new DebugTask( this );
	// create list
	this.eventTask.append( this.soundTask );
	this.eventTask.append( this.debugTask );
	// head of task list
	this.child		= this.eventTask;
	this.setStatus( SceneStatus.READY );

	// message handler
	this.msgManager		= new MessageManager();
	this.msgManager.addHandler( new MessageHandler( MessageType.RET_PREV_SCENE,	this, this.handleRetPrevScene ) );
	this.msgManager.addHandler( new MessageHandler( MessageType.START_STAGE,	this, this.handleStartStage ) );
};
EventScene.prototype = new Scene();

(function( pt )
{
	/**
	 * 
	 */
	pt.handleRetPrevScene = function( scene, message )
	{
		this.app.sceneManager.pop();
		var prevScene = this.app.sceneManager.current;
		prevScene.show();
	};
	/**
	 * 
	 */
	pt.handleStartStage = function( scene, message )
	{
		// set status
		this.setStatus( SceneStatus.READY );
		// pop this
		this.app.sceneManager.pop();

		var baseUrl	= message.data.baseUrl;
		var stageScene	= this.app.sceneManager.push( baseUrl );
		if ( stageScene )
		{
			// set status
			stageScene.setStatus( SceneStatus.RUNNING );
			stageScene.show();	// clear input
		}
		else
		{
			// create scene
			stageScene	= new this.stageFunc( this.app, baseUrl );

			this.app.sceneManager.add( stageScene );
			this.app.sceneManager.push( stageScene.name );

			stageScene.init();
			stageScene.show();

			// load data
			var base = ( baseUrl.charAt( baseUrl.length - 1 ) == '/' ) ? baseUrl : baseUrl + '/';
			this.app.loadScript( base + Config.dataJSName );
		}
	};

	/**
	 * 
	 */
	pt.init = function()
	{
	};
	/**
	 * 
	 */
	pt.show = function()
	{
		this.setUsage( Res.String.HTML_USAGE );

		this.command.clear();
		this.holdContext();
		this.draw( this );
	};

	/**
	 * 
	 */
	pt.setData = function( data )
	{
		this.data	= data;

		// load stage
		this.eventTask.setData( this, data.script );
	};
	/**
	 * 
	 */
	pt.getData = function()
	{
		return this.data;
	};
	/**
	 * 
	 */
	pt.loadData = function( data )
	{
		try
		{
			var self = this;

			// load resource
			this.app.loadResource( this.name, data.resource );
			// set data
			this.setData( data );
			// set status
			window.setTimeout( function() { self.setStatus( SceneStatus.RUNNING ); self.show(); }, Config.loadInterval + 300 );
		}
		catch ( e )
		{
			this.app.kill();
			Debug.alertError( e );
		}
	};
})( EventScene.prototype );


// Expose
if ( !window.h5glib ) { window.h5glib = {}; }
window.h5glib.EventScene	= EventScene;

// Anonymous function end
//
})( window );
