//  sine_wave.hpp: sine wave generator

//  Copyright Takeshi Mouri 2006.
//  Use, modification, and distribution are subject to the
//  Boost Software License, Version 1.0. (See accompanying file
//  LICENSE_1_0.txt or copy at http://www.boost.org/LICENSE_1_0.txt)

#ifndef HAMIGAKI_AUDIO_SINE_WAVE_HPP
#define HAMIGAKI_AUDIO_SINE_WAVE_HPP

#include <hamigaki/audio/pcm/format.hpp>
#include <hamigaki/iostreams/catable.hpp>
#include <boost/iostreams/categories.hpp>
#include <boost/iostreams/positioning.hpp>
#include <cmath>

namespace hamigaki { namespace audio {

class sine_wave_source
{
public:
    typedef char char_type;

    struct category
        : public boost::iostreams::input
        , public boost::iostreams::device_tag
    {};

    sine_wave_source(unsigned rate, unsigned bits, double freq)
        : rate_(rate), bits_(bits), freq_(freq), sum_(0.0)
    {
    }

    std::size_t block_size() const
    {
        return (bits_ + 7) / 8;
    }

    std::streamsize read(char* s, std::streamsize n)
    {
        if (n <= 0)
            return -1;

        const double pi = 3.14159265358979323846;
        const double twice_pi = 2.0*pi;
        const double w = twice_pi * freq_ / static_cast<double>(rate_);

        const std::streamsize smp_sz = block_size();
        if (n % smp_sz != 0)
            throw BOOST_IOSTREAMS_FAILURE("invalid read size");
        const std::streamsize count = n / smp_sz;

        if (smp_sz == 1)
        {
            for (std::streamsize i = 0; i < count; ++i)
            {
                boost::int32_t val =
                    128 + static_cast<boost::int32_t>(64 * std::sin(sum_));

                *(s++) = static_cast<char>(static_cast<boost::uint8_t>(val));

                sum_ += w;
                if (sum_ >= twice_pi)
                    sum_ -= twice_pi;
            }
        }
        else
        {
            for (std::streamsize i = 0; i < count; ++i)
            {
                boost::int32_t val =
                    static_cast<boost::int32_t>(16384 * std::sin(sum_));

                // int32_t may not be two's complement representation
                boost::uint32_t tmp =
                    val >= 0
                    ? static_cast<boost::uint32_t>(val)
                    : 0x10000 - static_cast<boost::uint32_t>(-val);

                // little endian output
                *(s++) = static_cast<char>(
                    static_cast<boost::uint8_t>(tmp & 0xFF));
                *(s++) = static_cast<char>(
                    static_cast<boost::uint8_t>((tmp >> 8) & 0xFF));

                sum_ += w;
                if (sum_ >= twice_pi)
                    sum_ -= twice_pi;
            }
        }

        return n;
    }

private:
    unsigned rate_;
    unsigned bits_;
    double freq_;
    double sum_;
};

} } // End namespaces audio, hamigaki.

HAMIGAKI_IOSTREAMS_CATABLE(hamigaki::audio::sine_wave_source, 0)

#endif // HAMIGAKI_AUDIO_SINE_WAVE_HPP
