//  vorbis_encoder.hpp: vorbisenc device adapter

//  Copyright Takeshi Mouri 2006.
//  Use, modification, and distribution are subject to the
//  Boost Software License, Version 1.0. (See accompanying file
//  LICENSE_1_0.txt or copy at http://www.boost.org/LICENSE_1_0.txt)

#ifndef HAMIGAKI_AUDIO_VORBIS_ENCODER_HPP
#define HAMIGAKI_AUDIO_VORBIS_ENCODER_HPP

#include <hamigaki/audio/pcm/format.hpp>
#include <hamigaki/iostreams/device/file.hpp>
#include <boost/iostreams/close.hpp>
#include <boost/iostreams/write.hpp>
#include <boost/noncopyable.hpp>

#include <boost/config/abi_prefix.hpp>

namespace hamigaki { namespace audio {

namespace detail
{

class vorbis_encoder_base : boost::noncopyable
{
public:
    typedef std::streamsize (*write_func)(void*, const char*, std::streamsize);
    typedef void (*close_func)(void*);

    vorbis_encoder_base();
    ~vorbis_encoder_base();

    const pcm_format& format() const;

    void open(void* self, const pcm_format& fmt,
        write_func write, close_func close);

    void close();

    std::streamsize write(const char* s, std::streamsize n);

private:
    void* ptr_;
    bool is_open_;
};

template<typename Sink>
class vorbis_file_sink_impl : vorbis_encoder_base
{
public:
    typedef char char_type;

    using vorbis_encoder_base::format;
    using vorbis_encoder_base::write;
    using vorbis_encoder_base::close;

    vorbis_file_sink_impl(const Sink& sink, const pcm_format& fmt)
        : sink_(sink)
    {
        vorbis_encoder_base::open(&sink_, fmt,
            &vorbis_file_sink_impl::write_func,
            &vorbis_file_sink_impl::close_func);
    }

    ~vorbis_file_sink_impl()
    {
        try
        {
            close();
        }
        catch (...)
        {
        }
    }

    std::size_t block_size() const
    {
        return format().block_size();
    }

private:
    Sink sink_;

    static std::streamsize write_func(
        void* datasink, const char* s, std::streamsize n)
    {
        Sink& sink = *static_cast<Sink*>(datasink);
        return boost::iostreams::write(sink, s, n);
    }

    static void close_func(void* datasink)
    {
        Sink& sink = *static_cast<Sink*>(datasink);
        boost::iostreams::close(sink, BOOST_IOS::out);
    }
};

} // namespace detail

template<typename Sink>
struct basic_vorbis_file_sink
{
    typedef detail::vorbis_file_sink_impl<Sink> impl_type;

public:
    typedef char char_type;

    struct category :
        boost::iostreams::optimally_buffered_tag,
        boost::iostreams::output,
        boost::iostreams::device_tag,
        boost::iostreams::closable_tag {};

    basic_vorbis_file_sink(const Sink& sink, const pcm_format& fmt)
        : pimpl_(new impl_type(sink, fmt))
    {
    }

    std::streamsize optimal_buffer_size() const
    {
        return pimpl_->block_size() * (pimpl_->format().rate / 5);
    }

    std::size_t block_size() const
    {
        return pimpl_->block_size();
    }

    const pcm_format& format() const
    {
        return pimpl_->format();
    }

    std::streamsize write(const char_type* s, std::streamsize n)
    {
        return pimpl_->write(s, n);
    }

    void close()
    {
        pimpl_->close();
    }

private:
    boost::shared_ptr<impl_type> pimpl_;
};

class vorbis_file_sink
    : public basic_vorbis_file_sink<hamigaki::iostreams::file_sink>
{
private:
    typedef basic_vorbis_file_sink<hamigaki::iostreams::file_sink> base_type;

public:
    vorbis_file_sink(const std::string& path, const pcm_format& fmt)
        : base_type(hamigaki::iostreams::file_sink(
            path, BOOST_IOS::out|BOOST_IOS::binary), fmt)
    {
    }
};

template<typename Sink>
inline basic_vorbis_file_sink<Sink>
make_vorbis_file_sink(const Sink& sink, const pcm_format& fmt)
{
    return basic_vorbis_file_sink<Sink>(sink, fmt);
}

} } // End namespaces audio, hamigaki.

#include <boost/config/abi_suffix.hpp>

#endif // HAMIGAKI_AUDIO_VORBIS_ENCODER_HPP
