//  vorbis_encoder.cpp: vorbisenc device adapter

//  Copyright Takeshi Mouri 2006.
//  Use, modification, and distribution are subject to the
//  Boost Software License, Version 1.0. (See accompanying file
//  LICENSE_1_0.txt or copy at http://www.boost.org/LICENSE_1_0.txt)

#include <hamigaki/audio/vorbis_encoder.hpp>
#include <hamigaki/audio/detail/endian.hpp>
#include <vorbis/vorbisenc.h>
#include <cstdlib>
#include <ctime>

namespace hamigaki { namespace audio {

namespace detail
{

class vorbisenc
{
public:
    const pcm_format& format() const
    {
        return format_;
    }

    void open(void* self, const pcm_format& fmt,
        vorbis_encoder_base::write_func write,
        vorbis_encoder_base::close_func close)
    {
        file_ = self;
        format_ = fmt;
        write_ = write;
        close_ = close;

        ::vorbis_info_init(&vi_);

        ::vorbis_encode_init_vbr(&vi_, fmt.channels, fmt.rate, 0.1);

        ::vorbis_comment_init(&vc_);

        ::vorbis_analysis_init(&vd_, &vi_);
        ::vorbis_block_init(&vd_, &vb_);

        std::srand(std::time(0));
        ::ogg_stream_init(&os_, std::rand());

        ::ogg_packet header;
        ::ogg_packet header_comm;
        ::ogg_packet header_code;

        ::vorbis_analysis_headerout(
            &vd_, &vc_, &header, &header_comm, &header_code);
        ::ogg_stream_packetin(&os_, &header);
        ::ogg_stream_packetin(&os_, &header_comm);
        ::ogg_stream_packetin(&os_, &header_code);

        while (::ogg_stream_flush(&os_, &og_) != 0)
        {
            write_helper(og_.header, og_.header_len);
            write_helper(og_.body, og_.body_len);
        }
    }

    void close()
    {
        try
        {
            write_to_downstream(0);
            (*close_)(file_);
        }
        catch (...)
        {
        }
        ::ogg_stream_clear(&os_);
        ::vorbis_block_clear(&vb_);
        ::vorbis_dsp_clear(&vd_);
        ::vorbis_comment_clear(&vc_);
        ::vorbis_info_clear(&vi_);
    }

    std::streamsize write(const char* s, std::streamsize n)
    {
        std::size_t block = format_.block_size();

        if (n % block != 0)
            throw BOOST_IOSTREAMS_FAILURE("invalid write size");

        const std::size_t count = n / block;
        float** buffer = ::vorbis_analysis_buffer(&vd_, count);

        if (format_.bits == 8)
        {
            for (std::size_t i = 0; i < count; ++i)
            {
                for (std::size_t j = 0; j < format_.channels; ++j)
                {
                    boost::int16_t tmp = static_cast<unsigned char>(s[0]);
                    buffer[j][i] = (tmp - 128) / 128.f;
                    ++s;
                }
            }
        }
        else
        {
            for (std::size_t i = 0; i < count; ++i)
            {
                for (std::size_t j = 0; j < format_.channels; ++j)
                {
                    boost::uint16_t tmp = decode_little_endian16(s);

                    // int16_t may not be two's complement representation
                    if (tmp & 0x8000)
                        buffer[j][i] = ((~tmp & 0xFFFF) + 1) / (-32768.f);
                    else
                        buffer[j][i] = tmp / 32768.f;

                    s += 2;
                }
            }
        }

        write_to_downstream(count);

        return n;
    }

private:
    ::ogg_stream_state  os_;
    ::ogg_page          og_;
    ::ogg_packet        op_;
    ::vorbis_info       vi_;
    ::vorbis_comment    vc_;
    ::vorbis_dsp_state  vd_;
    ::vorbis_block      vb_;

    void* file_;
    pcm_format format_;
    vorbis_encoder_base::write_func write_;
    vorbis_encoder_base::close_func close_;

    void write_helper(const unsigned char* s, long n)
    {
        (*write_)(file_, reinterpret_cast<const char*>(s), n);
    }

    void write_to_downstream(int count)
    {
        ::vorbis_analysis_wrote(&vd_, count);

        bool eos = false;
        while (::vorbis_analysis_blockout(&vd_, &vb_) == 1)
        {
            ::vorbis_analysis(&vb_, 0);
            ::vorbis_bitrate_addblock(&vb_);

            while (::vorbis_bitrate_flushpacket(&vd_, &op_))
            {
                ::ogg_stream_packetin(&os_, &op_);

                while (!eos)
                {
                    int result = ::ogg_stream_pageout(&os_, &og_);
                    if (result == 0)
                        break;

                    write_helper(og_.header, og_.header_len);
                    write_helper(og_.body, og_.body_len);

                    if (::ogg_page_eos(&og_))
                        eos = true;
                }
            }
        }
    }
};

vorbis_encoder_base::vorbis_encoder_base()
    : ptr_(new vorbisenc), is_open_(false)
{
}

vorbis_encoder_base::~vorbis_encoder_base()
{
    close();
    delete static_cast<vorbisenc*>(ptr_);
}

const pcm_format& vorbis_encoder_base::format() const
{
    vorbisenc& enc = *static_cast<vorbisenc*>(ptr_);
    return enc.format();
}

void vorbis_encoder_base::open(void* self, const pcm_format& fmt,
    vorbis_encoder_base::write_func write,
    vorbis_encoder_base::close_func close)
{
    if (is_open_)
        this->close();

    vorbisenc& enc = *static_cast<vorbisenc*>(ptr_);
    enc.open(self, fmt, write, close);
    is_open_ = true;
}

void vorbis_encoder_base::close()
{
    if (is_open_)
    {
        vorbisenc& enc = *static_cast<vorbisenc*>(ptr_);
        enc.close();
        is_open_ = false;
    }
}

std::streamsize vorbis_encoder_base::write(const char* s, std::streamsize n)
{
    vorbisenc& enc = *static_cast<vorbisenc*>(ptr_);
    return enc.write(s, n);
}

} // namespace detail

} } // End namespaces audio, hamigaki.
