//  vorbis_file_test.cpp: test case for vorbis_file/vorbis_file_sink

//  Copyright Takeshi Mouri 2006.
//  Use, modification, and distribution are subject to the
//  Boost Software License, Version 1.0. (See accompanying file
//  LICENSE_1_0.txt or copy at http://www.boost.org/LICENSE_1_0.txt)

#include <hamigaki/audio/pcm_device.hpp>
#include <hamigaki/audio/sine_wave.hpp>
#include <hamigaki/audio/stereo.hpp>
#include <hamigaki/audio/vorbis_encoder.hpp>
#include <hamigaki/audio/vorbis_file.hpp>
#include <hamigaki/iostreams/device/tmp_file.hpp>
#include <hamigaki/iostreams/dont_close.hpp>
#include <boost/iostreams/copy.hpp>
#include <boost/iostreams/restrict.hpp>
#include <boost/test/unit_test.hpp>
#include "detail/utility.hpp"

namespace audio = hamigaki::audio;
namespace io_ex = hamigaki::iostreams;
namespace io = boost::iostreams;
namespace ut = boost::unit_test;

using namespace hamigaki::audio::test;

void vorbis_file_test_aux(const audio::pcm_format& fmt, double freq)
{
    std::ostringstream os;
    os
        << "rate=" << fmt.rate
        << ", bits=" << fmt.bits
        << ", channels=" << fmt.channels;
    BOOST_CHECKPOINT(os.str());

    io_ex::tmp_file tmp;
    io::copy(
        io::restrict(
            audio::stereo(
                audio::sine_wave_source(fmt.rate, fmt.bits, freq),
                fmt.channels
            ),
            0, fmt.block_size()*calc_samples_per_note(fmt.rate,100)
        ),
        audio::make_vorbis_file_sink(io_ex::dont_close(tmp), fmt)
    );

    audio::vorbis_sample_params params;
    params.endian = audio::vorbis::little_endian;
    params.size = (fmt.bits + 7) / 8;
    params.is_signed = (fmt.bits > 8);

    io::seek(tmp, 0, BOOST_IOS::beg);
    io::copy(
        audio::make_vorbis_file_source(tmp, params),
        audio::pcm_sink(fmt)
    );
}

void vorbis_file_test()
{
    const unsigned rates[] = { 11025, 22050, 44100 };
    const unsigned bits[] = { 8, 16 };

    unsigned short note = 60;
    for (std::size_t i = 0; i < sizeof(rates)/sizeof(rates[0]); ++i)
    {
        for (std::size_t j = 0; j < sizeof(bits)/sizeof(bits[0]); ++j)
        {
            for (unsigned channels = 1; channels <= 2; ++channels)
            {
                audio::pcm_format fmt;
                fmt.rate = rates[i];
                fmt.bits = bits[j];
                fmt.channels = channels;
                vorbis_file_test_aux(fmt, calc_frequency(note));
            }
            note = next_note(note);
        }
    }
}

ut::test_suite* init_unit_test_suite(int, char* [])
{
    ut::test_suite* test = BOOST_TEST_SUITE("vorbis file test");
    test->add(BOOST_TEST_CASE(&vorbis_file_test));
    return test;
}
