/* $Id: abyss.c,v 1.5 2002/12/17 08:09:51 tkubo Exp $ */

/* abyss.c - output XML data of abyss.
 *
 * Copyright (C) 2002 Hardmeter Project <http://hardmeter.sourceforge.jp>
 *
 * This project is supported by IPA(Information-technology Promotion
 * Agency, Japan).
 *
 * "xhardmeter" is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include "xhardmeter.h"

struct builderdata {
  FILE *fp;
  int state;
  char *experiment_name;
  char *event_name;
  GSList *set_flags;
  GSList *clr_flags;
  GSList *other_flags;
};
typedef struct builderdata builderdata_t;

struct intel_to_abyss {
  const char *intel;
  const char *abyss;
};

struct intel_to_abyss intel_to_abyss_event[] = {
  {"64bit_MMX_uop", "mmx_64bit_uop"},
  {"128bit_MMX_uop", "mmx_128bit_uop"},
  {"memory_loads", "loads_retired"},
  {"memory_stores", "store_retired"},
  {"memory_loads & memory_stores", "mem_retired"},
  {"128_bit_MMX_retired", "mmx_128bit_retired"},
  {"64_bit_MMX_retired", "mmx_64bit_retired"},
  {"1stL_cache_load_miss_retired", "ld_miss_1L_retired"},
  {"2ndL_cache_load_miss_retired", "ld_miss_2L_retired"},
  {"DTLB_all_miss_retired", "dtlb_miss_retired"},
  {"MOB_load_replay_retired", "unaligned_ld_retired"},
  {NULL, NULL},
};

struct intel_to_abyss intel_to_abyss_flag[] = {
  /* tc_deliver_mode */
  {"DD", NULL},
  {"DB", NULL},
  {"DI", "deliver"},
  {"BD", NULL},
  {"BB", NULL},
  {"BI", "build"},
  {"ID", NULL},
  {"IB", NULL},
  /* memory_cancel */
  {"64K_CONF", "conf_64k"},
  /* bsq_cache_reference */
  {"RD_2ndL_HITS", "rd_2ndL_hits"},
  {"RD_2ndL_HITE", "rd_2ndL_hite"},
  {"RD_2ndL_HITM", "rd_2ndL_hitm"},
  {"RD_3ndL_HITS", "rd_3ndL_hits"},
  {"RD_3ndL_HITE", "rd_3ndL_hite"},
  {"RD_3ndL_HITM", "rd_3ndL_hitm"},
  {"RD_2ndL_MISS", "rd_2ndL_miss"},
  {"RD_3ndL_MISS", "rd_3ndL_miss"},
  {"WR_2ndL_MISS", "wr_2ndL_miss"},
  /* ioq_allocation & ioq_active_entries */
  {"Bit 0-4 (single field)", "req"},
  {NULL, NULL},
};

static const char *intel_to_abyss(const char *in, struct intel_to_abyss *table);
static const char *to_xml_name(const char *in);
static void cleanup_data(builderdata_t *bd);
static void cleanup_flags(builderdata_t *bd);
static void store_flag(builderdata_t *bd, const char *name, guint64 val);
static void print_flags(builderdata_t *bd, const char *prefix);

/* members of hardmeter_builder_t */
static int builder_start(void **top_datap, void *userdata);
static int builder_end(void *top_data);
static int builder_program(void *top_data, const char *name, const char *path);
static int builder_default_flag(void *top_data, const char *name, guint64 val);
static int builder_experiment_start(void **experiment_datap, void *top_data, const char *name);
static int builder_experiment_end(void *experiment_data);
static int builder_event_start(void **event_datap, void *experiment_data, const char *name, hardmeter_event_t *base, hardmeter_ebs_t *ebs);
static int builder_event_end(void *event_data);
static int builder_flag(void *event_data, const char *name, guint64 val);
static int builder_error_cleanup(void *top_data);

static hardmeter_builder_t builder = {
  builder_start,
  builder_end,
  builder_program,
  builder_default_flag,
  builder_experiment_start,
  builder_experiment_end,
  builder_event_start,
  builder_event_end,
  builder_flag,
  builder_error_cleanup
};

hardmeter_builder_t *hardmeter_abyss_output_builder = &builder;

static const char *intel_to_abyss(const char *in, struct intel_to_abyss *table)
{
  static char buf[128];
  int i;
  while (table->intel != NULL) {
    if (strcasecmp(table->intel, in) == 0) {
      return table->abyss;
    }
    table++;
  }
  for (i = 0; i < sizeof(buf) - 1; i++) {
    buf[i] = ('A' <= in[i] && in[i] <= 'Z') ? in[i] - 'A' + 'a' : in[i];
    if (in[i] == '\0')
      return buf;
  }
  buf[i] = '\0';
  return buf;
}

static const char *to_xml_name(const char *in)
{
  static char buf[128];
  int i, j;
  if (('a' <= in[0] && in[0] <= 'z') || ('A' <= in[0] && in[0] <= 'Z')
      || (in[0] == '_')) {
    j = 0;
  } else {
    buf[0] = '_';
    j = 1;
  }
  for (i = 0;j < sizeof(buf) - 1 && in[i] != '\0'; i++) {
    if (('a' <= in[i] && in[i] <= 'z') || ('A' <= in[i] && in[i] <= 'Z')
	|| ('0' <= in[i] && in[i] <= '9') || (in[i] == '.')
	|| (in[i] == '-') || (in[i] == '_') || (in[i] == ':')) {
      buf[j++] = in[i];
    }
  }
  buf[j] = '\0';
  return buf;
}

static void cleanup_data(builderdata_t *bd)
{
  if (bd->fp)
    fclose(bd->fp);
  if (bd->experiment_name)
    free(bd->experiment_name);
  if (bd->event_name)
    free(bd->event_name);
  free(bd);
}

static void cleanup_flags(builderdata_t *bd)
{
  g_slist_foreach(bd->set_flags, (GFunc)g_free, NULL);
  g_slist_foreach(bd->clr_flags, (GFunc)g_free, NULL);
  g_slist_foreach(bd->other_flags, (GFunc)g_free, NULL);
  g_slist_free(bd->set_flags);
  g_slist_free(bd->clr_flags);
  g_slist_free(bd->other_flags);
  bd->set_flags = NULL;
  bd->clr_flags = NULL;
  bd->other_flags = NULL;
}

static void store_flag(builderdata_t *bd, const char *name, guint64 val)
{
  char buf[41];
  char *str;
  int size;
  int i;

  name = intel_to_abyss(name, intel_to_abyss_flag);
  if (strcmp(name, "threshold") == 0) {
    size = 4;
  } else if (strcmp(name, "req") == 0) {
    size = 5;
  } else if (strcmp(name, "counter") == 0) {
    size = 40;
  } else {
    if (val) {
      str = g_strdup_printf("<%s/>", name);
      bd->set_flags = g_slist_append(bd->set_flags, str);
    } else {
      str = g_strdup_printf("<%s/>", name);
      bd->clr_flags = g_slist_append(bd->clr_flags, str);
    }
    return;
  }
  for (i = 0; i < size; i++) {
    buf[i] = (val & (((guint64)1) << (size - i - 1))) ? '1' : '0';
  }
  buf[i] = '\0';
  str = g_strdup_printf("<%s>%s</%s>", name, buf, name);
  bd->other_flags = g_slist_append(bd->other_flags, str);
}

static void print_flags(builderdata_t *bd, const char *prefix)
{
  GSList *l;
  if (bd->set_flags != NULL) {
    fprintf(bd->fp, "%s<set>\n", prefix);
    for (l = bd->set_flags; l != NULL; l = l->next) {
      fprintf(bd->fp, "%s    %s\n", prefix, (char *)l->data);
    }
    fprintf(bd->fp, "%s</set>\n", prefix);
  }
  if (bd->clr_flags != NULL) {
    fprintf(bd->fp, "%s<clr>\n", prefix);
    for (l = bd->set_flags; l != NULL; l = l->next) {
      fprintf(bd->fp, "%s    %s\n", prefix, (char *)l->data);
    }
    fprintf(bd->fp, "%s</clr>\n", prefix);
  }
  if (bd->other_flags != NULL) {
    for (l = bd->other_flags; l != NULL; l = l->next) {
      fprintf(bd->fp, "%s%s\n", prefix, (char *)l->data);
    }
  }
}

static int builder_start(void **top_datap, void *userdata)
{
  builderdata_t *bd = g_new0(builderdata_t, 1);
  time_t timep;

  bd->fp = fopen(userdata, "w");
  bd->state = 0;
  if (bd->fp == NULL) {
    hardmeter_error("cannot open file %s", (char *)userdata);
    free(bd);
    return -1;
  }
  time(&timep);
  fprintf(bd->fp,
	  "<?xml version='1.0'?>\n"
	  "<!-- generated by xhardmeter at %s-->\n"
	  "\n"
	  "<exp_config>\n"
	  "\n"
	  "\t<!-- Identify the processor here. -->\n"
	  "\t<processor type=\"pentium4\"/>\n", ctime(&timep));
  *top_datap = bd;
  return 0;
}

static int builder_end(void *top_data)
{
  builderdata_t *bd = (builderdata_t *)top_data;
  if (bd->state == 1) {
    fprintf(bd->fp,
	    "\t</programs>\n"
	    "\n");
  } else if (bd->state == 2) {
    print_flags(bd, "\t\t\t");
    cleanup_flags(bd);
    fprintf(bd->fp,
	    "\t\t</default>\n"
	    "\t</experiments>\n"
	    "\n");
  } else if (bd->state == 3) {
    fprintf(bd->fp,
	    "\t</experiments>\n"
	    "\n");
  }

  fprintf(bd->fp,
	  "</exp_config>\n");
  cleanup_data(bd);
  return 0;
}

static int builder_program(void *top_data, const char *name, const char *path)
{
  builderdata_t *bd = (builderdata_t *)top_data;
  if (bd->state == 0) {
    fprintf(bd->fp,
	    "\n"
	    "\t<!-- List the programs to be run here, giving a name and a command for each program. -->\n"
	    "\t<programs>\n");
    bd->state = 1;
  }
  name = to_xml_name(name);
  fprintf(bd->fp, "\t\t<%s command=\"%s\"/>\n", name, path);
  return 0;
}

static int builder_default_flag(void *top_data, const char *name, guint64 val)
{
  builderdata_t *bd = (builderdata_t *)top_data;
  if (bd->state == 0) {
    fprintf(bd->fp,
	    "\n"
	    "\t<!-- List the experiments to be run for each program here. -->\n"
	    "\t<experiments>\n"
	    "\t\t<default>\n");
    bd->state = 2;
  } else if (bd->state == 1) {
    fprintf(bd->fp,
	    "\t</programs>\n"
	    "\n"
	    "\t<!-- List the experiments to be run for each program here. -->\n"
	    "\t<experiments>\n"
	    "\t\t<default>\n");
    bd->state = 2;
  }
  store_flag(bd, name, val);
  return 0;
}

static int builder_experiment_start(void **experiment_datap, void *top_data, const char *name)
{
  builderdata_t *bd = (builderdata_t *)top_data;
  if (bd->state == 0) {
    fprintf(bd->fp,
	    "\n"
	    "\t<!-- List the experiments to be run for each program here. -->\n"
	    "\t<experiments>\n"
	    "\n");
    bd->state = 3;
  } else if (bd->state == 1) {
    fprintf(bd->fp,
	    "\t</programs>\n"
	    "\n"
	    "\t<!-- List the experiments to be run for each program here. -->\n"
	    "\t<experiments>\n"
	    "\n");
    bd->state = 3;
  } else if (bd->state == 2) {
    print_flags(bd, "\t\t\t");
    cleanup_flags(bd);
    fprintf(bd->fp,
	    "\t\t</default>\n"
	    "\n");
    bd->state = 3;
  }
  name = to_xml_name(name);
  fprintf(bd->fp, "\t\t<%s>\n", name);
  if (bd->experiment_name)
    free(bd->experiment_name);
  bd->experiment_name = strdup(name);
  *experiment_datap = bd;
  return 0;
}

static int builder_experiment_end(void *experiment_data)
{
  builderdata_t *bd = (builderdata_t *)experiment_data;
  fprintf(bd->fp, "\t\t</%s>\n", bd->experiment_name);
  return 0;
}

static int builder_event_start(void **event_datap, void *experiment_data, const char *name, hardmeter_event_t *base, hardmeter_ebs_t *ebs)
{
  builderdata_t *bd = (builderdata_t *)experiment_data;
  const char *basename;

  name = to_xml_name(name);
  basename = intel_to_abyss(base->name, intel_to_abyss_event);
  if (ebs->enabled) {
    fprintf(bd->fp, "\t\t\t<%s_ebs base=\"%s\">\n", name, basename);
    fprintf(bd->fp, "\t\t\t    <ebs type=\"%s\" interval=\"%lld\" buffer=\"%d\" sample=\"%c\" max=\"%d\"/>\n",
	    ebs->type ? "precise" : "imprecise", ebs->interval, ebs->buffer,
	    ebs->sample ? 'A' : 'E', ebs->max);
    fprintf(bd->fp, "\t\t\t</%s_ebs>\n\n", name);
    basename = intel_to_abyss(base->tag_base->name, intel_to_abyss_event);
    fprintf(bd->fp, "\t\t\t<%s base=\"%s\">\n", name, basename);
  } else {
    fprintf(bd->fp, "\t\t\t<%s base=\"%s\">\n", name, basename);
  }
  if (bd->event_name)
    free(bd->event_name);
  bd->event_name = strdup(name);
  *event_datap = bd;
  return 0;
}

static int builder_event_end(void *event_data)
{
  builderdata_t *bd = (builderdata_t *)event_data;
  print_flags(bd, "\t\t\t\t");
  cleanup_flags(bd);
  fprintf(bd->fp, "\t\t\t</%s>\n\n", bd->event_name);
  return 0;
}

static int builder_flag(void *event_data, const char *name, guint64 val)
{
  builderdata_t *bd = (builderdata_t *)event_data;
  store_flag(bd, name, val);
  return 0;
}

static int builder_error_cleanup(void *top_data)
{
  builderdata_t *bd = (builderdata_t *)top_data;
  cleanup_data(bd);
  return 0;
}
