/* $Id: xprogram.c,v 1.5 2002/12/17 08:09:51 tkubo Exp $ */

/* xprogram.c - A part of Gtk module.
 *
 * Copyright (C) 2002 Hardmeter Project <http://hardmeter.sourceforge.jp>
 *
 * This project is supported by IPA(Information-technology Promotion
 * Agency, Japan).
 *
 * "xhardmeter" is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <gtk/gtk.h>
#include "xhardmeter.h"

struct cb_data {
  GtkWidget *dialog;
  GtkWidget *name;
  GtkWidget *path;
  GtkWidget *message;
  int rc;
};

/* xdialog */
static int xdialog(GtkWindow *window, char **name, char **path);
static void xdialog_ok_cb(GtkWidget *button, struct cb_data *cbdata);
static void xdialog_ref_cb(GtkWidget *button, struct cb_data *cbdata);

/*
 * implement xdialog
 */
static int xdialog(GtkWindow *window, char **name, char **path)
{
  static struct cb_data cbdata;
  GtkWidget *frame;
  GtkWidget *button;
  GtkWidget *hbox;

  if (cbdata.dialog == NULL) {
    /* get toplevel window */

    cbdata.dialog = gtk_dialog_new();
    gtk_window_set_modal(GTK_WINDOW(cbdata.dialog), TRUE);
    gtk_window_set_transient_for(GTK_WINDOW(cbdata.dialog), window);

    /* prepare frame for Name */
    frame = gtk_frame_new("Name");
    gtk_box_pack_start(GTK_BOX(GTK_DIALOG(cbdata.dialog)->vbox), frame, TRUE, TRUE, 0);
    cbdata.name = gtk_entry_new();
    gtk_container_add(GTK_CONTAINER(frame), cbdata.name);

    /* prepare frame for Program Path */
    frame = gtk_frame_new("Program Path");
    gtk_box_pack_start(GTK_BOX(GTK_DIALOG(cbdata.dialog)->vbox), frame, TRUE, TRUE, 0);
    hbox = gtk_hbox_new(FALSE, 0);
    gtk_container_add(GTK_CONTAINER(frame), hbox);
    cbdata.path = gtk_entry_new();
    gtk_box_pack_start(GTK_BOX(hbox), cbdata.path, TRUE, TRUE, 0);
    button = gtk_button_new_with_label("Ref");
    gtk_box_pack_start(GTK_BOX(hbox), button, FALSE, TRUE, 0);
    gtk_signal_connect(GTK_OBJECT(button), "clicked", xdialog_ref_cb, &cbdata);

    /* prepare frame for Message */
    frame = gtk_frame_new("Message");
    gtk_box_pack_start(GTK_BOX(GTK_DIALOG(cbdata.dialog)->vbox), frame, TRUE, TRUE, 0);
    cbdata.message = gtk_label_new("");
    gtk_container_add(GTK_CONTAINER(frame), cbdata.message);

    button = gtk_button_new_with_label("OK");
    gtk_signal_connect(GTK_OBJECT(button), "clicked", xdialog_ok_cb, &cbdata);
    gtk_box_pack_start(GTK_BOX(GTK_DIALOG(cbdata.dialog)->action_area), button, TRUE, TRUE, 0);
  
    button = gtk_button_new_with_label("Cancel");
    gtk_signal_connect(GTK_OBJECT(button), "clicked", gtk_main_quit, NULL);
    gtk_box_pack_start(GTK_BOX(GTK_DIALOG(cbdata.dialog)->action_area), button, TRUE, TRUE, 0);

    gtk_signal_connect(GTK_OBJECT(cbdata.dialog), "delete_event", gtk_main_quit, NULL);
  }

  gtk_entry_set_text(GTK_ENTRY(cbdata.name), *name != NULL ? *name : "");
  gtk_entry_set_text(GTK_ENTRY(cbdata.path), *path != NULL ? *path : "");
  gtk_label_set_text(GTK_LABEL(cbdata.message), "");
  cbdata.rc = -1;

  gtk_widget_show_all(cbdata.dialog);
  gtk_main();
  if (cbdata.rc == 0) {
    *name = g_strdup(gtk_entry_get_text(GTK_ENTRY(cbdata.name)));
    *path = g_strdup(gtk_entry_get_text(GTK_ENTRY(cbdata.path)));
  }
  gtk_widget_hide_all(cbdata.dialog);
  return cbdata.rc;
}

static void xdialog_ok_cb(GtkWidget *button, struct cb_data *cbdata)
{
  struct stat sbuf;
  char *name;
  char *path;

  name = gtk_entry_get_text(GTK_ENTRY(cbdata->name));
  path = gtk_entry_get_text(GTK_ENTRY(cbdata->path));
  if (path == NULL || path[0] == '\0') {
    gtk_label_set_text(GTK_LABEL(cbdata->message), "path is empty.");
    return;
  }
  if (stat(path, &sbuf) != 0) {
    gtk_label_set_text(GTK_LABEL(cbdata->message), "no such file or directory.");
    return;
  }
  if (!S_ISREG(sbuf.st_mode)) {
    gtk_label_set_text(GTK_LABEL(cbdata->message), "not a file.");
    return;
  }
  if (name == NULL || name[0] == '\0') {
    gtk_entry_set_text(GTK_ENTRY(cbdata->name), g_basename(path));
  }
  cbdata->rc = 0;
  gtk_main_quit();
}

static void xdialog_ref_cb(GtkWidget *button, struct cb_data *cbdata)
{
  const char *filename = xhardmeter_xfileselection(GTK_WINDOW(cbdata->dialog));
  if (filename != NULL)
    gtk_entry_set_text(GTK_ENTRY(cbdata->path), filename);
}

/*
 * exported functions
 */
GtkWidget *xhardmeter_program_new(xhardmeter_window_t *w)
{
  GtkWidget *frame;
  GtkWidget *hbox;
  GtkWidget *scrolled_window;
  GtkWidget *clist;
  GtkWidget *vbox;
  GtkWidget *button;
  gchar *titles[] = {"name", "program path"};

  frame = gtk_frame_new("Programs");
  gtk_widget_set_usize(GTK_WIDGET(frame), 300, 100);
  hbox = gtk_hbox_new(FALSE, 5);
  gtk_container_add(GTK_CONTAINER(frame), hbox);

  scrolled_window = gtk_scrolled_window_new(NULL, NULL);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled_window),
				 GTK_POLICY_AUTOMATIC, GTK_POLICY_ALWAYS);
  gtk_box_pack_start(GTK_BOX(hbox), scrolled_window, TRUE, TRUE, 0);

  clist = gtk_clist_new_with_titles(2, titles);
  gtk_clist_set_column_auto_resize(GTK_CLIST(clist), 0, TRUE);
  gtk_clist_set_column_auto_resize(GTK_CLIST(clist), 1, TRUE);
  gtk_container_add(GTK_CONTAINER(scrolled_window), clist);

  vbox = gtk_vbox_new(FALSE, 3);
  gtk_box_pack_start(GTK_BOX(hbox), vbox, FALSE, TRUE, 0);

  button = gtk_button_new_with_label("Add");
  gtk_signal_connect_object(GTK_OBJECT(button), "clicked",
			    xhardmeter_program_add_cb, (void *)w);
  gtk_box_pack_start(GTK_BOX(vbox), button, FALSE, TRUE, 0);

  button = gtk_button_new_with_label("Edit");
  gtk_signal_connect_object(GTK_OBJECT(button), "clicked",
			    xhardmeter_program_edit_cb, (void *)w);
  gtk_box_pack_start(GTK_BOX(vbox), button, FALSE, TRUE, 0);

  button = gtk_button_new_with_label("Delete");
  gtk_signal_connect_object(GTK_OBJECT(button), "clicked",
			    xhardmeter_program_delete_cb, (void *)w);
  gtk_box_pack_start(GTK_BOX(vbox), button, FALSE, TRUE, 0);

  w->clist = GTK_CLIST(clist);

  return frame;
}


void xhardmeter_program_add_cb(xhardmeter_window_t *w)
{
  gchar *text[2] = {NULL, NULL};

  if (xdialog(w->window, &(text[0]), &(text[1])) != 0)
    return;
  gtk_clist_append(w->clist, text);
  g_free(text[0]);
  g_free(text[1]);
}

void xhardmeter_program_edit_cb(xhardmeter_window_t *w)
{
  gint row;
  gchar *name = NULL;
  gchar *path = NULL;

  if (w->clist->selection == NULL)
    return;

  row = GPOINTER_TO_INT(w->clist->selection->data);
  gtk_clist_get_text(w->clist, row, 0, &name);
  gtk_clist_get_text(w->clist, row, 1, &path);
  if (xdialog(w->window, &name, &path) != 0)
    return;
  gtk_clist_set_text(w->clist, row, 0, name);
  gtk_clist_set_text(w->clist, row, 1, path);
  g_free(name);
  g_free(path);
}

void xhardmeter_program_delete_cb(xhardmeter_window_t *w)
{
  gint row;
  
  if (w->clist->selection == NULL)
    return;
  row = GPOINTER_TO_INT(w->clist->selection->data);
  gtk_clist_remove(w->clist, row);
  if (!w->clist->selection)
    gtk_clist_select_row(w->clist, row == w->clist->rows ? w->clist->rows - 1 : row, -1);
}
