#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <signal.h>
#include "hardmeter.h"

static hardmeter_t *hardmeter_handle;

static void cleanup(int signo)
{
	if (hardmeter_handle != NULL)
		hardmeter_terminate(hardmeter_handle, NULL);
	exit(1);
}

void show_usage(const char *argv0, int status)
{
	const hardmeter_template_t *templates = hardmeter_get_templates(NULL);
	int i;

	fprintf(stderr,
		"Usage: %s (-u | -k) [-o OUTFILE] [-i INTERVAL] [-c COUNT] -t TYPE EXE_OR_PID\n"
		"  option\n"
		"    -u - sample user-mode events\n"
		"    -k - sample kernel-mode events\n"
		"    -o - store sampled data to file\n"
		"    -i - sampling interval(default: %d)\n"
		"    -c - max sampling count(default: %d)\n"
		"    -t - event type\n"
		"  event type\n", argv0, HARDMETER_DEFAULT_INTERVAL, HARDMETER_DEFAULT_COUNT);
	for (i = 0; templates[i].name != NULL; i++) {
		if (templates[i].control == NULL) {
			/* comment */
			fprintf(stderr, "    (%5s)\n", templates[i].name);
		} else {
			fprintf(stderr, "      %-15s - %s\n", templates[i].name, templates[i].description);
		}
	}
	exit(status);
}

static int parse_option(struct hardmeter_option_t *opt, char **filenamep, int *argc, char ***argv)
{
	int ac = *argc;
	char **av = *argv;
	const hardmeter_template_t *template = NULL;
	int user = 0;
	int kernel = 0;
	int interval = HARDMETER_DEFAULT_INTERVAL;
	int count = HARDMETER_DEFAULT_COUNT;
	char *filename = NULL;
	int i;
	int skip_next = 0;

	ac--; av++;
	while (ac > 0 && av[0][0] == '-') {
		for (i = 1; av[0][i] != '\0'; i++) {
			switch (av[0][i]) {
			case 'h':
				show_usage(*argv[0], 0);
				break;
			case 't':
				if (ac == 1)
					break;
				template = hardmeter_search_template(av[1], NULL);
				if (template == NULL) {
					fprintf(stderr, "no such event type: %s\n", av[1]);
					show_usage(*argv[0], 1);
				}
				skip_next = 1;
				break;
			case 'i':
				if (ac == 1)
					break;
				interval = atoi(av[1]);
				if (interval <= 0) {
					fprintf(stderr, "Please set positive value for -i.\n");
					exit(1);
				}
				skip_next = 1;
				break;
			case 'c':
				if (ac == 1)
					break;
				count = atoi(av[1]);
				if (count <= 0) {
					fprintf(stderr, "Please set positive value for -c.\n");
					exit(1);
				}
				skip_next = 1;
				break;
			case 'u':
				user = 1;
				break;
			case 'k':
				kernel = 1;
				break;
			case 'o':
				if (ac == 1)
					break;
				filename = av[1];
				skip_next = 1;
				break;
			default:
				fprintf(stderr, "invalid option %s\n", av[0]);
				show_usage(*argv[0], 1);
			}
		}
		if (skip_next) {
			ac--; av++;
			skip_next = 0;
		}
		ac--; av++;
	}
	if (template == NULL) {
		fprintf(stderr, "Please set event type.\n");
		show_usage(*argv[0], 1);
	}
	if (kernel == 0 && user == 0) {
		fprintf(stderr, "Please set -k or -u.\n");
		show_usage(*argv[0], 1);
	}
	if (ac == 0) {
		fprintf(stderr, "Please set executable or pid.\n");
		show_usage(*argv[0], 1);
	}
	opt->template = template;
	opt->kernel = kernel;
	opt->user = user;
	opt->interval = interval;
	opt->count = count;
	*filenamep = filename;
	*argc = ac;
	*argv = av;
	return 0;
}

int main(int argc, char **argv)
{
	pid_t pid;
	const char *err;
	struct hardmeter_option_t opt;
	char *filename;
	char *endptr;

	if (hardmeter_init(&err)) {
		printf("%s\n", err);
		exit(1);
	}
	memset(&opt, 0, sizeof(opt));
	parse_option(&opt, &filename, &argc, &argv);
	if (argc == 1 && ((pid = strtol(argv[0], &endptr, 10)), (*endptr == '\0'))) {
		hardmeter_handle = hardmeter_attach_process(&opt, pid, &err);
	} else {
		hardmeter_handle = hardmeter_start_process(&opt, argv[0], argv, &err);
	}
	if (hardmeter_handle == NULL) {
		printf("%s\n", err);
		exit(1);
	}
	signal(SIGINT, cleanup);
	signal(SIGQUIT, cleanup);
	signal(SIGTERM, cleanup);
	signal(SIGABRT, cleanup);
	if (hardmeter_dump(hardmeter_handle, filename, 0, &err) == -1) {
		printf("%s\n", err);
	}
	hardmeter_terminate(hardmeter_handle, NULL);
	return 0;
}
