#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <time.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <sys/ioctl.h>
#include <sys/ptrace.h>
#include <fcntl.h>
#include <errno.h>
#include "hardmeter.h"

struct hardmeter_t {
	int fd;
	pid_t pid;
	int user;
	int kernel;
	int interval;
	int is_pebs;
	time_t start_time;
};

static hardmeter_template_t *hardmeter_template;
static int (*hardmeter_fill_control)(struct vperfctr_control *dest, const hardmeter_option_t *opt);

int hardmeter_init(const char **err)
{
	static int is_initialized = 0;
	static const char *errmsg = NULL; /* if NULL, no error. */
	struct perfctr_info info;
	int fd = -1;
	int need_unlink = 0;

	if (is_initialized) {
		if (err != NULL)
			*err = errmsg;
		return errmsg ? -1 : 0;
	}
	fd = open("/proc/self/perfctr", O_RDONLY | O_CREAT);
	if (fd != -1) {
		need_unlink = 1;
	} else {
		fd = open("/proc/self/perfctr", O_RDONLY);
		if (fd == -1) {
			errmsg = "Could not open \"/proc/self/perfctr\"";
			goto out;
		}

	}
	ioctl(fd, PERFCTR_INFO, &info);
	switch (info.cpu_type) {
	case PERFCTR_X86_GENERIC:
		errmsg = "Unsupported CPU: Generic x86 with TSC";
		goto out;
	case PERFCTR_X86_INTEL_P5:
        	errmsg = "Unsupported CPU: Intel Pentium";
		goto out;
	case PERFCTR_X86_INTEL_P5MMX:
		errmsg = "Unsupported CPU: Intel Pentium MMX";
		goto out;
	case PERFCTR_X86_INTEL_P6:
        	errmsg = "Unsupported CPU: Intel Pentium Pro";
		goto out;
	case PERFCTR_X86_INTEL_PII:
        	errmsg = "Unsupported CPU: Intel Pentium II";
		goto out;
	case PERFCTR_X86_INTEL_PIII:
		errmsg = "Unsupported CPU: Intel Pentium III";
		goto out;
	case PERFCTR_X86_CYRIX_MII:
		errmsg = "Unsupported CPU: Cyrix 6x86MX/MII/III";
		goto out;
	case PERFCTR_X86_WINCHIP_C6:
		errmsg = "Unsupported CPU: WinChip C6";
		goto out;
	case PERFCTR_X86_WINCHIP_2:
		errmsg = "Unsupported CPU: WinChip 2/3";
		goto out;
	case PERFCTR_X86_AMD_K7:
		errmsg = "Unsupported CPU: AMD K7";
		goto out;
	case PERFCTR_X86_VIA_C3:
		errmsg = "Unsupported CPU: VIA C3";
		goto out;
	case PERFCTR_X86_INTEL_P4:
	case PERFCTR_X86_INTEL_P4M2:
		hardmeter_template = hardmeter_p4_template;
		hardmeter_fill_control = hardmeter_p4_fill_control;
		break;
	default:
		errmsg = "Unknown CPU: ?";
		goto out;
	}
	if ((info.cpu_features & PERFCTR_FEATURE_PCINT) == 0) {
		errmsg = "Interrupt support is not available.";
		goto out;
	}
out:
	if (fd != -1) {
		if (need_unlink) {
			ioctl(fd, VPERFCTR_UNLINK, NULL);
		}
		close(fd);
	}
	is_initialized = 1;
	if (err != NULL)
		*err = errmsg;
	return errmsg ? -1 : 0;
}

const hardmeter_template_t *hardmeter_get_templates(const char **err)
{
	if (hardmeter_init(err) != 0)
		return NULL;
	return hardmeter_template;
}

const hardmeter_template_t *hardmeter_search_template(const char *name, const char **err)
{
	int i;

	if (hardmeter_init(err) != 0)
		return NULL;
	for (i = 0; hardmeter_template[i].name != NULL; i++) {
		if (hardmeter_template[i].control == NULL) {
			continue;
		}
		if (strcmp(name, hardmeter_template[i].name) == 0) {
			return &(hardmeter_template[i]);
		}
	}
	if (err != NULL)
		*err = "No such type";
	return NULL;
}

static hardmeter_t *hardmeter_do_open(const hardmeter_option_t *opt, pid_t pid, const char **err)
{
	const char *errmsg;
	hardmeter_t *h;
	char filename[64];
	struct vperfctr_control control;

	if (hardmeter_init(err) != 0)
		return NULL;
	hardmeter_fill_control(&control, opt);

	h = malloc(sizeof(*h));
	h->pid = pid;
	h->user = opt->user;
	h->kernel = opt->kernel;
	h->interval = opt->interval;
	h->is_pebs = opt->template->is_pebs;
	h->start_time = time(0);

	sprintf(filename, "/proc/%d/perfctr", pid);
	h->fd = open(filename, O_RDONLY | O_CREAT);
	if (h->fd == -1) {
		errmsg = "Could not open \"/proc/PID/perfctr\"";
		goto err;
	}
	if (ioctl(h->fd, VPERFCTR_CONTROL, &control) == -1) {
		switch (errno) {
		case EFAULT:
			errmsg = "vperfctr control: bad address";
			goto err;
		case EINVAL:
			errmsg = "vperfctr control: invalid argument";
			goto err;
		case EPERM:
			errmsg = "vperfctr control: permittion denied";
			goto err;
		default:
			errmsg = "vperfctr control: unkown error";
			goto err;
		}
	}
	if (err != NULL)
		*err = NULL;
	return h;
err:
	free(h);
	if (err != NULL)
		*err = errmsg;
	return NULL;
}

hardmeter_t *hardmeter_open(const hardmeter_option_t *opt, const char **err)
{
	return hardmeter_do_open(opt, getpid(), err);
}

hardmeter_t *hardmeter_attach_process(const hardmeter_option_t *opt, pid_t pid, const char **err)
{
	const char *errmsg = NULL;
	hardmeter_t *h = NULL;
	int status;

	if (ptrace(PTRACE_ATTACH, pid, 0, 0) == -1) {
		switch (errno) {
		case EPERM:
			errmsg = "Permission denied to attach process.";
			goto out;
		case ESRCH:
			errmsg  = "No such process to attach.";
			goto out;
		default:
			errmsg  = "Unknown error while attaching to process";
			goto out;
		}
	}
	while (waitpid(pid, &status, 0) == -1) {
		switch (errno) {
		case EINTR:
			continue;
		case ECHILD:
			errmsg  = "No such process to attach.";
			goto out2;
		default:
			errmsg  = "Unknown error while attaching to process";
			goto out2;
		}
	}
	if (!WIFSTOPPED(status)) {
		errmsg  = "process exited unexpectedly.";
		goto out2;
	}
	h = hardmeter_do_open(opt, pid, err);
	ptrace(PTRACE_DETACH, pid, 0, 0);
	return h;
out2:
	ptrace(PTRACE_DETACH, pid, 0, 0);
out:
	if (err != NULL)
		*err = errmsg;
	return NULL;
}

hardmeter_t *hardmeter_start_process(const hardmeter_option_t *opt, const char *file, char *const argv[], const char **err)
{
	const char *errmsg = NULL;
	hardmeter_t *h = NULL;
	int status;
	pid_t pid;

	pid = fork();
	if (pid == -1) {
		errmsg = "fork error";
		goto out;
	} else if (pid == 0) {
		/* child */
		if (ptrace(PTRACE_TRACEME, 0, 0, 0) == -1) {
			perror("ptrace");
			exit(1);
		}
		execvp(argv[0], argv);
		perror("execvp");
		exit(1);
	}
	while (waitpid(pid, &status, 0) == -1) {
		switch (errno) {
		case EINTR:
			continue;
		case ECHILD:
			errmsg  = "No such process to attach.";
			goto out;
		default:
			errmsg  = "Unknown error while attaching to process";
			goto out;
		}
	}
	if (!WIFSTOPPED(status)) {
		errmsg  = "child process exited unexpectedly.";
		goto out;
	}
	h = hardmeter_do_open(opt, pid, err);
	ptrace(PTRACE_DETACH, pid, 0, 0);
	return h;
out:
	if (err != NULL)
		*err = errmsg;
	return NULL;
}

int hardmeter_dump(hardmeter_t *h, const char *filename, int non_blocking, const char **err)
{
	const char *errmsg = NULL;
	FILE *fp;
	unsigned long eips[1000];
	ssize_t ssize;
	int i, num, sum;

	if (h == NULL || h->pid <= 0 || h->fd == -1) {
		errmsg = "Invalid argument";
		goto out;
	}
	if (filename == NULL) {
		fp = stdout;
	} else {
		fp = fopen(filename, "w");
		if (fp == NULL) {
			errmsg = "Could not open dumpfile.";
			goto out;
		}
	}
	if (non_blocking) {
		fcntl(h->fd, F_SETFL, fcntl(h->fd, F_GETFL) | O_NONBLOCK);
	} else {
		fcntl(h->fd, F_SETFL, fcntl(h->fd, F_GETFL) & ~O_NONBLOCK);
	}
	sum = 0;
	if (h->is_pebs)
		fputs("#eflags  liner_ip   eax      ebx      ecx      edx      esi      edi      ebp      esp\n", fp);
	while ((ssize = read(h->fd, eips, sizeof(eips))) > 0) {
		num = ssize / sizeof(eips[i]);
		for (i = 0; i < num; i++) {
			fprintf(fp, "%08lx", eips[i]);
			fputc((++sum % 10) == 0 ? '\n' : ' ', fp);
		}
	}
	if (sum % 10 != 0)
		fputc('\n', fp);
	if (h->is_pebs)
		sum /= 10;
	fprintf(fp,"#\n# start time: %s# user      : %s\n# kernel    : %s\n# interval  : %d\n# count     : %d\n#\n",
		ctime(&h->start_time), h->user ? "yes" : "no", h->kernel ? "yes" : "no", h->interval, sum);
	if (filename != NULL)
		fclose(fp);
	if (ssize == -1) {
		switch (errno) {
		case EINTR:
			errmsg = "interrupted by a signal.";
			goto out;
		case EAGAIN:
			errmsg = "eagain";
			goto out;
		default:
			errmsg = "Unknown error while reading data.";
			goto out;
		}
	}
out:
	if (err != NULL)
		*err = errmsg;
	return errmsg ? -1 : 0;
}

int hardmeter_terminate(hardmeter_t *h, const char **err)
{
	const char *errmsg = NULL;
	pid_t mypid = getpid();
	int status;

	if (h == NULL || h->pid <= 0 || h->fd == -1) {
		errmsg = "Invalid argument";
		goto out;
	}
       	if (h->pid != mypid) {
		if (ptrace(PTRACE_ATTACH, h->pid, 0, 0) == -1) {
			switch (errno) {
			case EPERM:
				errmsg = "Permission denied to attach process.";
				goto out;
			case ESRCH:
				errmsg  = "No such process to attach.";
				goto out;
			default:
				errmsg  = "Unknown error while attaching to process";
				goto out;
			}
		}
		while (waitpid(h->pid, &status, 0) == -1) {
			switch (errno) {
			case EINTR:
				continue;
			case ECHILD:
				errmsg  = "No such process to attach.";
				goto out2;
			default:
				errmsg  = "Unknown error while attaching to process";
				goto out2;
			}
		}
		if (!WIFSTOPPED(status)) {
			errmsg  = "process exited unexpectedly.";
			goto out2;
		}
	}
	if (ioctl(h->fd, VPERFCTR_UNLINK, NULL) == -1) {
		errmsg  = "Unknown error while unlinking vperfctr.";
	}
out2:
	if (h->pid != mypid) {
		ptrace(PTRACE_DETACH, h->pid, 0, 0);
	}
out:
	if (err != NULL)
		*err = errmsg;
	return errmsg ? -1 : 0;
}


int hardmeter_close(hardmeter_t *h, const char **err)
{
	const char *errmsg = NULL;

	if (h == NULL || h->pid <= 0 || h->fd == -1) {
		errmsg = "Invalid argument";
		goto out;
	}
	close(h->fd);
	free(h);
out:
	if (err != NULL)
		*err = errmsg;
	return errmsg ? -1 : 0;
}
