#include "main.h"

struct tasks *tss;	/* main.c, int.c, console.c  extern */
struct timer *tts;	/* ^XNXCb` */

struct task *task_init(void)
{
	struct sdesc *sd = (struct sdesc *) GDT_ADDR;
	struct memory *mem = (struct memory *) MEMORY_ADDR;
	struct task *tsys, *tidle;
	int i;

	tss = (struct tasks *) memory_alloc(mem, sizeof (struct tasks));
	for (i = 0; i < TASKS; i++) {
		tss->task[i].flag = 0;
		tss->task[i].sel = (3 + i) * 8;
		tss->task[i].tss.ldtr = (3 + TASKS + i) * 8;
		gdt_set(sd + 3 + i, (int) &tss->task[i].tss, 103, TSS);
		gdt_set(sd + 3 + TASKS + i, (int) &tss->task[i].ldt, 15, LDT);
	}
	for (i = 0; i < TLEVELS; i++) {
		tss->tl[i].running = 0;
		tss->tl[i].now = 0;
	}

	tsys = task_alloc();
	tsys->flag = TASK_RUNNING;
	tsys->pri = 2;
	tsys->lv = 1;
	task_add(tsys);
	task_switch0();
	load_tr(tsys->sel);

	tidle = task_alloc();
	tidle->tss.esp = memory_alloc(mem, 65536) + 65536;
	tidle->tss.eip = (int) &task_idle;
	tidle->tss.es = 1 * 8;
	tidle->tss.cs = 2 * 8;
	tidle->tss.ss = 1 * 8;
	tidle->tss.ds = 1 * 8;
	tidle->tss.fs = 1 * 8;
	tidle->tss.gs = 1 * 8;
	task_run(tidle, TLEVELS - 1, 1);

	tts = timer_alloc();
	timer_sett(tts, tsys->pri, 0);
	tss->tfpu = 0;
	return tsys;
}

void task_switch(void)
{
	struct tlevel *tl = &tss->tl[tss->now];
	struct task *nt, *t = tl->t[tl->now];

	tl->now++;
	if (tl->now == tl->running) {
		tl->now = 0;
	}
	if (tss->chlv != 0) {
		task_switch0();
		tl = &tss->tl[tss->now];
	}
	nt = tl->t[tl->now];
	timer_sett(tts, nt->pri, 0);
	if (nt != t) {
		farjmp(0, nt->sel);
	}
	return;
}

struct task *task_alloc(void)
{
	struct task *t;
	int i;

	for (i = 0; i < TASKS; i++) {
		if (tss->task[i].flag == 0) {
			t = &tss->task[i];
			t->flag = TASK_USING;
			t->tss.eflags = 0x00000202;	/* IF = 1 */
			t->tss.eax = 0;
			t->tss.ecx = 0;
			t->tss.edx = 0;
			t->tss.ebx = 0;
			t->tss.ebp = 0;
			t->tss.esi = 0;
			t->tss.edi = 0;
			t->tss.es = 0;
			t->tss.ds = 0;
			t->tss.fs = 0;
			t->tss.gs = 0;
			t->tss.iomap = 0x40000000;
			t->tss.ss0 = 0;
			t->fpu[0] = 0x037f;		/* CW(control word) */
			t->fpu[1] = 0x0000;		/* SW(status word) */
			t->fpu[2] = 0xffff;		/* TW(tag word) */
			for (i = 3; i < 27; i++) {
				t->fpu[i] = 0;
			}
			t->lbyte = 0;
			return t;
		}
	}
	return 0;
}

void task_run(struct task *t, int lv, int pri)
{
	if (lv < 0) {
		lv = t->lv;
	}
	if (pri > 0) {
		t->pri = pri;
	}

	if (t->flag == TASK_RUNNING && t->lv != lv) {
		task_remove(t);
	}
	if (t->flag != TASK_RUNNING) {
		t->lv = lv;
		task_add(t);
	}
	tss->chlv = 1;
	return;
}

void task_sleep(struct task *t)
{
	struct task *nt;

	if (t->flag == TASK_RUNNING) {
		nt = task_now();
		task_remove(t);
		if (t == nt) {
			timer_cancel(tts);
			task_switch0();
			nt = task_now();
			timer_sett(tts, nt->pri, 0);
			farjmp(0, nt->sel);
		}
	}
	return;
}

struct task *task_now(void)
{
	struct tlevel *tl = &tss->tl[tss->now];

	return tl->t[tl->now];
}

void task_add(struct task *t)
{
	struct tlevel *tl = &tss->tl[t->lv];

	tl->t[tl->running] = t;
	tl->running++;
	t->flag = TASK_RUNNING;
	return;
}

void task_remove(struct task *t)
{
	struct tlevel *tl = &tss->tl[t->lv];
	int i;

	for (i = 0; i < tl->running; i++) {
		if (tl->t[i] == t) {
			break;
		}
	}
	tl->running--;
	if (i < tl->now) {
		tl->now--;
	}
	if (tl->now >= tl->running) {
		tl->now = 0;
	}
	t->flag = TASK_USING;
	for (; i < tl->running; i++) {
		tl->t[i] = tl->t[i + 1];
	}
	return;
}

void task_switch0(void)
{
	int i;

	for (i = 0; i < TLEVELS; i++) {
		if (tss->tl[i].running > 0) {
			break;
		}
	}
	tss->now = i;
	tss->chlv = 0;
	return;
}

void task_idle(void)
{
	for (;;) {
		io_hlt();
	}
}
