/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.accesscontrol.ejb.session;

import java.rmi.RemoteException;
import java.security.Principal;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.Locale;

import javax.ejb.CreateException;
import javax.ejb.EJBException;
import javax.ejb.FinderException;
import javax.ejb.RemoveException;
import javax.ejb.SessionContext;
import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.swt.SWT;

import com.clustercontrol.accesscontrol.bean.UserListTableDefine;
import com.clustercontrol.accesscontrol.factory.ModifyUser;
import com.clustercontrol.accesscontrol.factory.SelectUser;
import com.clustercontrol.accesscontrol.factory.UserFilterProperty;
import com.clustercontrol.accesscontrol.factory.UserProperty;
import com.clustercontrol.bean.Property;
import com.clustercontrol.bean.TableColumnInfo;
import com.clustercontrol.util.Messages;

/**
 * アクセス機能の管理を行う Session Bean です。<BR>
 * クライアントからの Entity Bean へのアクセスは、Session Bean を介して行います。
 * 
 * @ejb.bean name="AccessController"	
 *     jndi-name="AccessController"
 *     type="Stateless" 
 *     transaction-type="Container"
 *           
 * @jboss.depends name="jboss.jca:service=DataSourceBinding,name=HinemosDS"
 * 
 * @ejb.permission
 *     unchecked="true"
 *     method-intf="Home"
 *     
 * @ejb.permission
 *     unchecked="true"
 *     method-intf="LocalHome"
 *     
 * @ejb.permission
 *     unchecked="true"
 *     method-intf="Local"
 */
public abstract class AccessControllerBean implements javax.ejb.SessionBean {
	/** ログ出力のインスタンス */
	protected static Log m_log = LogFactory.getLog( AccessControllerBean.class ); 
	
	/** コンテキスト情報 */
	@SuppressWarnings("unused")
	private SessionContext m_context;
	
	/**
	 * コンテキスト情報を設定します。<BR>
	 * Session Bean がインスタンスプールに格納される際に行う処理を実装します。
	 * @see javax.ejb.SessionBean#setSessionContext(javax.ejb.SessionContext)
	 */
	public void setSessionContext(SessionContext ctx) throws EJBException, RemoteException {
		m_context = ctx;
	}

	/**
	 * アクセス[ユーザ]ビューのテーブル定義情報を返します。<BR>
	 * リストに、カラム毎にテーブルカラム情報をセットします。
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     unchecked="true"
     *     method-intf="Remote"
	 * 
	 * @param locale ロケール情報
	 * @return テーブル定義情報（{@link com.clustercontrol.bean.TableColumnInfo}のリスト）
	 */
	@SuppressWarnings("unchecked")
	public ArrayList getUserListTableDefine(Locale locale) {
		
		//テーブル定義配列
		ArrayList tableDefine = new ArrayList();

		tableDefine.add(UserListTableDefine.UID, 
				new TableColumnInfo(Messages.getString("user.id", locale), TableColumnInfo.NONE, 100, SWT.LEFT));
		tableDefine.add(UserListTableDefine.NAME, 
				new TableColumnInfo(Messages.getString("name", locale), TableColumnInfo.NONE, 100, SWT.LEFT));
		tableDefine.add(UserListTableDefine.DESCRIPTION, 
				new TableColumnInfo(Messages.getString("description", locale), TableColumnInfo.NONE, 150, SWT.LEFT));
		tableDefine.add(UserListTableDefine.CREATOR_NAME, 
				new TableColumnInfo(Messages.getString("creator.name", locale), TableColumnInfo.NONE, 100, SWT.LEFT));
		tableDefine.add(UserListTableDefine.CREATE_TIME, 
				new TableColumnInfo(Messages.getString("create.time", locale), TableColumnInfo.NONE, 100, SWT.LEFT));
		tableDefine.add(UserListTableDefine.MODIFIER_NAME, 
				new TableColumnInfo(Messages.getString("modifier.name", locale), TableColumnInfo.NONE, 100, SWT.LEFT));
		tableDefine.add(UserListTableDefine.MODIFY_TIME, 
				new TableColumnInfo(Messages.getString("update.time", locale), TableColumnInfo.NONE, 100, SWT.LEFT));

		return tableDefine;
	}
	
	/**
	 * ユーザ一覧情報を取得します。
	 * 
	 * @ejb.interface-method
     *     
	 * @ejb.permission
     *     role-name="AccessControlRead"
     *     method-intf="Remote"
     *     
	 * @return ユーザ一覧情報（Objectの2次元配列）
	 * @throws FinderException
	 * @throws NamingException
	 * 
	 * @see com.clustercontrol.accesscontrol.factory.SelectUser#getUserList()
	 */
	public ArrayList getUserList() throws FinderException, NamingException {

		ArrayList list = null;
		try {
			//ファシリティツリーを取得
			SelectUser select = new SelectUser();
			list = select.getUserList();
		} catch (FinderException e) {
			m_log.debug("getUserList() -> " + e.getMessage());
			throw e;
		} catch (NamingException e) {
			throw e;
		}

		return list;
	}
	
	/**
	 * ユーザ一覧情報を取得します。
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="AccessControlRead"
     *     method-intf="Remote"
     *     
	 * @param property ユーザフィルタ用プロパティ
	 * @return ユーザ一覧情報（Objectの2次元配列）
	 * @throws FinderException
	 * @throws NamingException
	 * 
	 * @see com.clustercontrol.accesscontrol.factory.SelectUser#getUserList(Property)
	 */
	public ArrayList getUserList(Property property) throws FinderException, NamingException {
		
		ArrayList list = null;
		try {
			//ファシリティツリーを取得
			SelectUser select = new SelectUser();
			list = select.getUserList(property);
		} catch (FinderException e) {
			m_log.debug("getUserList() -> " + e.getMessage());
			throw e;
		} catch (NamingException e) {
			throw e;
		}

		return list;
	}
	
	/**
	 * ユーザフィルタ用プロパティを返します。
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="AccessControlRead"
     *     method-intf="Remote"
     *     
	 * @param locale ロケール情報
	 * @return ユーザフィルタ用プロパティ
	 * 
	 * @see com.clustercontrol.accesscontrol.factory.UserFilterProperty#getProperty(Locale)
	 */
	public Property getUserFilterProperty(Locale locale) {
		UserFilterProperty user = new UserFilterProperty();
		return user.getProperty(locale);
	}
	
	/**
	 * ユーザ用プロパティを返します。
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="AccessControlRead"
     *     method-intf="Remote"
	 * 
	 * @param uid ユーザID
	 * @param mode プロパティモード
	 * @param locale ロケール情報
	 * @return ユーザ用プロパティ
	 * @throws FinderException
	 * @throws NamingException
	 * 
	 * @see com.clustercontrol.bean.PropertyConstant
	 * @see com.clustercontrol.accesscontrol.factory.UserProperty#getProperty(String, int, Locale)
	 */
	public Property getUserProperty(String uid, int mode, Locale locale) throws FinderException, NamingException {
		try {
			UserProperty user = new UserProperty();
			return user.getProperty(uid, mode, locale);
		} catch (FinderException e) {
			m_log.debug("getUserProperty() -> " + e.getMessage());
			throw e;
		} catch (NamingException e) {
			throw e;
		}
	}
	
	/**
	 * ユーザを追加します。
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="AccessControlWrite"
     *     method-intf="Remote"
	 * 
	 * @param property ユーザ用プロパティ
	 * @throws NamingException
	 * @throws CreateException
	 * @throws FinderException
	 * @throws ParseException
	 * 
	 * @see com.clustercontrol.accesscontrol.factory.ModifyUser#addUser(Property, String)
	 */
	public void addUser(Property property) throws NamingException, CreateException, FinderException, ParseException {
		Principal pri = m_context.getCallerPrincipal();
		
		try {
			ModifyUser modify = new ModifyUser();
			modify.addUser(property, pri.getName());
		} catch (NamingException e) {
			throw e;
		} catch (CreateException e) {
			throw e;
		} catch (FinderException e) {
			m_log.debug("addUser() -> " + e.getMessage());
			throw e;
		} catch (ParseException e) {
			throw e;
		}
	}
	
	/**
	 * ユーザを変更します。
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="AccessControlWrite"
     *     method-intf="Remote"
	 * 
	 * @param property ユーザ用プロパティ
	 * @throws NamingException
	 * @throws FinderException
	 * 
	 * @see com.clustercontrol.accesscontrol.factory.ModifyUser#modifyUser(Property, String)
	 */
	public void modifyUser(Property property) throws NamingException, FinderException {
		Principal pri = m_context.getCallerPrincipal();
		
		try {
			ModifyUser modify = new ModifyUser();
			modify.modifyUser(property, pri.getName());
		} catch (NamingException e) {
			throw e;
		} catch (FinderException e) {
			m_log.debug("modifyUser() -> " + e.getMessage());
			throw e;
		}
	}
	
	/**
	 * ユーザを削除します。
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="AccessControlWrite"
     *     method-intf="Remote"
	 * 
	 * @param uid ユーザID
	 * @throws NamingException
	 * @throws RemoveException
	 * @throws FinderException
	 * 
	 * @see com.clustercontrol.accesscontrol.factory.ModifyUser#deleteUser(String)
	 */
	public void deleteUser(String uid) throws NamingException, RemoveException, FinderException  {
		try {
			ModifyUser modify = new ModifyUser();
			modify.deleteUser(uid);
		} catch (NamingException e) {
			throw e;
		} catch (RemoveException e) {
			throw e;
		} catch (FinderException e) {
			m_log.debug("deleteUser() -> " + e.getMessage());
			throw e;
		}
	}
	
	/**
	 * パスワードを変更します。
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="AccessControlWrite"
     *     method-intf="Remote"
	 * 
	 * @param uid ユーザID
	 * @param password パスワード
	 * @throws NamingException
	 * @throws RemoveException
	 * @throws FinderException
	 * 
	 * @see com.clustercontrol.accesscontrol.factory.ModifyUser#modifyPassword(String, String)
	 */
	public void changePassword(String uid, String password) throws NamingException, RemoveException, FinderException  {
		try {
			ModifyUser modify = new ModifyUser();
			modify.modifyPassword(uid, password);
		} catch (FinderException e) {
			m_log.debug("changePassword() -> " + e.getMessage());
			throw e;
		} catch (NamingException e) {
			throw e;
		}
	}
	
	/**
	 * ログインユーザが指定したRoleの権限を持っているかチェックします。
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     unchecked="true"
     *     method-intf="Remote"
	 * 
	 * @param roleName Role名
	 * @return 権限の有無（true：権限あり、false：権限なし）
	 * @throws NamingException
	 */
	public boolean isPermission(String roleName) throws NamingException {
		return m_context.isCallerInRole(roleName);
	}
	
	/**
	 * ログインユーザのユーザ名を返します。
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     unchecked="true"
     *     method-intf="Remote"
	 * 
	 * @return ユーザ名
	 * @throws NamingException
	 * @throws FinderException
	 * 
	 * @see com.clustercontrol.accesscontrol.factory.SelectUser#getUserName(String)
	 */
	public String getUserName() throws NamingException, FinderException {
		
		Principal pri = m_context.getCallerPrincipal();
		
		SelectUser select = new SelectUser();
		return select.getUserName(pri.getName());
	}
}
