/*
                                                                                                                                                                 
Copyright (C) 2006 NTT DATA Corporation
                                                                                                                                                                 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.
                                                                                                                                                                 
This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.
                                                                                                                                                                 
*/

package com.clustercontrol.performanceMGR.bean;

import java.io.Serializable;
import java.util.Date;

/**
 * 収集した性能値を格納するDTOクラス
 * 
 * 収集日時と性能値をペアで保持します。
 * 
 * @version 1.0
 * @since 1.0
 *
 */
public class CollectedDataInfo implements Serializable {
	private String m_facilityID;
	private String m_itemCode;
	private int m_deviceIndex;
	private int m_collectorItemID;
	private Date m_date;
	private double m_value;

	/**
	 * 指定された、ファシリティID、収集項目IDを格納したCollectedDataInfoオブジェクトを生成します。
	 * 
	 * @param facilityID ファシリティID
	 * @param deviceIndex 収集項目ID
	 */
	public CollectedDataInfo(final String facilityID, final String itemCode, final int deviceIndex){
		this.m_facilityID = facilityID;
		this.m_itemCode = itemCode;
		this.m_deviceIndex = deviceIndex;
	}
	
	/**
	 * 指定された、ファシリティID、収集項目IDを格納したCollectedDataInfoオブジェクトを生成します。
	 * 
	 * @param facilityID ファシリティID
	 * @param collectorItemID 収集項目ID
	 */
	public CollectedDataInfo(final String facilityID, final int collectorItemID){
		m_facilityID = facilityID;
		m_collectorItemID = collectorItemID;
	}
	
	/**
	 * 指定されたCollectedDataInfoインスタンスをもとにCollectedDataInfoオブジェクトを生成します。
	 * 
	 * @param collectorItemInfo 収集項目設定情報
	 */
	public CollectedDataInfo(final CollectorItemInfo collectorItemInfo){
		m_collectorItemID = collectorItemInfo.getCollectorItemID();
	}
	
	/**
	 * 指定された、ファシリティID、収集項目ID、収集日時、性能値を格納したCollectedDataInfoオブジェクトを生成します。
	 * 
	 * @param facilityID ファシリティID
	 * @param collectorItemID 収集項目ID
	 * @param date 日時
	 * @param value 性能値
	 */
	public CollectedDataInfo(final String facilityID, final int collectorItemID, final Date date, final double value){
		m_facilityID = facilityID;
		m_collectorItemID = collectorItemID;
		m_date = date;
		m_value = value;
	}
	
	/**
	 * 指定された、CollectedDataInfoと同じ値を保持するオブジェクトを生成します。
	 * 
	 * @param dataInfo 収集したデータセット
	 */
	public CollectedDataInfo(final CollectedDataInfo dataInfo){
		m_facilityID = dataInfo.getFacilityID();
		m_collectorItemID = dataInfo.getCollectorItemID();
		m_date = dataInfo.getDate();
		m_value = dataInfo.getValue();
	}
	
	/**
	 * 収集項目ID を取得します。
	 * @return 収集項目ID 
	 */
	public int getCollectorItemID() {
		return m_collectorItemID;
	}
	
	/**
	 * ファシリティID を取得します。
	 * @return ファシリティID
	 */
	public String getFacilityID() {
		return m_facilityID;
	}
	
	/**
	 * 収集日時 を取得します。
	 * @return 収集日時
	 */
	public Date getDate() {
		return m_date;
	}
	
	/**
	 * 性能値 を取得します。
	 * @return  性能値
	 */
	public double getValue() {
		return m_value;
	}
	
	/**
	 * 日時 を設定します。
	 * @param date 日時 
	 */
	public void setDate(Date date) {
		m_date = date;
	}
	
	/**
	 * 性能値 を設定します。
	 * @param value 性能値
	 */
	public void setValue(double value) {
		m_value = value;
	}
	/**
	 * facilityID を設定します。
	 * @param facilityid ファシリティID
	 */
	public void setFacilityID(String facilityid) {
		m_facilityID = facilityid;
	}

	/**
	 * デバイス番号を取得します。
	 * @return
	 */
	public int getDeviceIndex() {
		return m_deviceIndex;
	}

	/**
	 * デバイス番号を設定します。
	 * @param deviceIndex デバイス番号
	 */
	public void setDeviceIndex(int deviceIndex) {
		this.m_deviceIndex = deviceIndex;
	}

	/**
	 * 収集項目コードを取得します。
	 * @return 収集項目コード
	 */
	public String getItemCode() {
		return m_itemCode;
	}

	/**
	 * 収集項目コードを設定します。
	 * @param itemCode 収集項目コード
	 */
	public void setItemCode(String itemCode) {
		this.m_itemCode = itemCode;
	}
	
	/**
	 * データの内容を文字列として返します。
	 */
	public String toString(){
		String str =
			"FID:" + m_facilityID + 
			"  Item:" +  m_itemCode +
			"." + m_deviceIndex +
			"  " + m_date + 
			" : " + m_value;
		return str;
	}
}
