/*
 
 Copyright (C) 2006 NTT DATA Corporation
 
 This program is free software; you can redistribute it and/or
 Modify it under the terms of the GNU General Public License
 as published by the Free Software Foundation, version 2.
 
 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.
 
 */

package com.clustercontrol.snmppoller;

import java.net.InetAddress;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;
import java.util.TreeSet;

import javax.ejb.EJBException;
import javax.naming.InitialContext;
import javax.naming.NamingException;
import javax.rmi.PortableRemoteObject;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.quartz.CronTrigger;
import org.quartz.JobDetail;
import org.quartz.Scheduler;
import org.quartz.SchedulerException;
import org.quartz.impl.StdScheduler;

import com.clustercontrol.repository.bean.FacilityAttributeConstant;
import com.clustercontrol.repository.ejb.session.RepositoryControllerLocal;
import com.clustercontrol.repository.ejb.session.RepositoryControllerUtil;
import com.clustercontrol.sharedtable.bean.DataTable;
import com.clustercontrol.sharedtable.bean.ValueObject;
import com.clustercontrol.snmppoller.bean.PollingConfig;
import com.clustercontrol.snmppoller.job.CheckJob;
import com.clustercontrol.snmppoller.job.PollingThread;
import com.clustercontrol.snmppoller.job.SnmpPollingJob;
import com.clustercontrol.snmppoller.util.JNDIConnectionManager;

/**
 * SNMPポーリングで取得した情報をノード単位で保持するクラス
 * 
 * @version 2.0.0
 * @since 2.0.0
 */
public class SnmpSharedTable {
	protected static Log m_log = LogFactory.getLog( SnmpSharedTable.class );
	
	private final String m_checkJobGroup = "SNMP_CHECK_JOB";  // チェック用ジョブグループの名前
	private String m_quartzJndiName = null;  // Quartzをlookupする際のJNDI名
	
	private int m_defaultPort = -1;
	private int m_defaultVersion = 1;
	private String m_defaultCommunity = "public";
	private int m_defaultRetries = -1;
	private int m_defaultTimeout = 1000;  // milliseconds
	
	private String m_jndiName;  // JNDIに登録する際の名前
	private String m_groupName; // Quartzに登録する際の名前
	
	private String m_checkJobName; // チェック用ジョブの名前
	
	private String m_checkInterval = "34 */10 * * * ? *"; // チェックジョブの起動条件
	private List m_oidList;  // 収集するOIDのセット
	
	// 猶予期間(秒)
	// チェック時に収集間隔よりこの期間以上参照されていないテーブルは削除される
	private int m_gracePeriod = 5;
	
	// スケジューリングを実行するか否かのフラグ
	private boolean m_scheduleFlag = false;
	
	// このテーブルで動作しているポーラー(複数ある)の中で最も収集間隔が小さいものの値
	// Quartzのジョブとして登録されているもので最小の収集間隔のもの
	private int m_minInterval = Integer.MAX_VALUE;
	
	private HashMap<String, DataTable> m_dataTable;   // ノード毎のテーブルのマップ
	private HashMap<String, PollingConfig>  m_pollerConf;  // ノード毎のタイマーテーブル
	
	public SnmpSharedTable(String jndiName, String groupName){
		this.m_jndiName = jndiName;
		this.m_groupName = groupName;
		this.m_dataTable = new HashMap<String, DataTable>();
		this.m_pollerConf = new HashMap<String, PollingConfig>();
		this.m_checkJobName = groupName;
	}
	
	/**
	 * 周期チェックジョブをQuartzに登録します。
	 */
	public void scheduleCheckJob(){
		// 新規にジョブを生成
		JobDetail job = new JobDetail(m_checkJobName, m_checkJobGroup, CheckJob.class);
		//ジョブ完了時に削除されないようにする
		job.setDurability(true);
		
		//JobDetailに呼び出すメソッドの引数を設定
		job.getJobDataMap().put("jndiName", m_jndiName);
		
		scheduleJob(m_checkJobName, m_checkJobGroup, job, m_checkInterval);
	}
	
	/**
	 * 周期チェックジョブを削除します
	 */
	public void deleteCheckJob(){
		m_log.debug("deleteCheckJob()  JobName : " + m_checkJobName + "  JobGroup : " + m_checkJobGroup);
		
		// ジョブスケジューラを取得
		Scheduler scheduler = getScheduler();
		
		// 登録されているジョブを削除する
		try {
			scheduler.deleteJob(m_checkJobName, m_checkJobGroup);
		} catch (SchedulerException e) {
			m_log.error(e);
		}
	}
	
	/**
	 * 収集間隔以上参照されていないテーブルを調べ、ポーラーの収集設定を変更します。
	 *
	 */
	public void checkAll() {
		m_log.debug("checkAll()  Group : " + m_groupName + "  参照されていないテーブルを探します");
		
		// 全てのノードに対してチェックを行う
		Set<String> facilityIdSet = m_dataTable.keySet();		
		
		// 下で呼び出すschedule()の中でm_dataTableの要素を削除するため、
		// 静的なStringの配列にコピーしてからループを回す
		String[] facilityIds = 
			(String[])facilityIdSet.toArray(new String[facilityIdSet.size()]);
		
		for(int i=0; i<facilityIds.length; i++){
			check(facilityIds[i]);
		}
		
		// Quartzの設定を変更する
		if(m_scheduleFlag){
			schedule();
		}
		
		m_log.debug("Group : " + m_groupName + "  参照されていないテーブルのチェック完了");
	}
	
	/**
	 * 収集間隔以上参照されていないテーブルを調べ、ポーラーの収集設定を変更します。
	 *
	 */
	private void check(String facilityId) {
		m_log.debug("check()  Group : " + m_groupName + "  " + facilityId
				+ "  参照されていないテーブルを探します");
		
		synchronized (m_pollerConf) {			
			// 収集間隔の変更を行うか否かを示すフラグ
			boolean changeFlag = false;
			
			// 現在時刻を取得します
			long now = System.currentTimeMillis();
			
			// 最終更新時刻からの経過時間を算出
			int elapsedTime;
			try {
				elapsedTime = (int)((now - getLastReferenceTime(facilityId))/1000);
				
				// 収集間隔を取得
				int interval = this.getInterval(facilityId);
				
				// デバッグログを出力
				m_log.debug("経過時間 " + elapsedTime + "s   収集間隔 " + interval + "s");
				
				// 経過時間よりも小さな収集間隔の設定は削除
				while(interval != 0 && (interval + m_gracePeriod) < elapsedTime){
					// デバッグログを出力
					m_log.debug("Group : " + m_groupName + "  " + facilityId + "  削除 " + interval);
					
					// 収集間隔を削除
					removePollingInterval(facilityId, interval);
					changeFlag = true;
					
					// 収集間隔を再度取得
					interval = this.getInterval(facilityId);
				}
				
				if(interval == 0){
					// 収集間隔が全て削除された場合は収集を停止する
					this.deleteTable(facilityId);
				} else if(changeFlag){
					// 収集間隔の設定に変更がある場合はQuartzの収集設定を変更する
					m_log.debug("Quartzの収集設定を変更  " + facilityId);
					m_scheduleFlag = true;
				}
				
			} catch (FacilityNotFoundExciption e) {
				m_log.error(e);
			}
		}
		
		m_log.debug("Group : " + m_groupName + "  " + facilityId
				+ "  参照されていないテーブルのチェック完了");
	}
	
	/**
	 * 新規にテーブルを生成する。
	 */
	private void createTable(String facilityId){
		m_log.debug("createTable()  Group : " + m_groupName + "  " + facilityId);
		
		PollingConfig pConfig = new PollingConfig(
				getIpAddress(facilityId),
				this.m_defaultPort,
				this.m_defaultVersion,
				this.m_defaultCommunity,
				this.m_defaultRetries,
				this.m_defaultTimeout
		);
		
		synchronized (m_pollerConf) {
			// 値格納用のテーブルを生成
			this.m_pollerConf.put(facilityId, pConfig);
			this.m_dataTable.put(facilityId, new DataTable());
		}
		
		// スレッドをたててポーリングを実行する
		PollingThread pt = new PollingThread(
				pConfig.getSnmpConfig(),
				this.m_oidList,
				this.m_jndiName,
				facilityId
		);
		Thread th = new Thread(pt);
		th.start();
	}
	
	/**
	 * テーブルを削除し、
	 * テーブルに値を格納するために動作していたポーラーを停止させる。
	 * @param facilityId
	 */
	public void deleteTable(String facilityId){
		m_log.debug("deleteTable()  Group : " + facilityId);
		
		synchronized (m_pollerConf) {
			this.m_dataTable.remove(facilityId);
			this.m_pollerConf.remove(facilityId);
			m_scheduleFlag = true;
		}
	}
	
	/**
	 * 全てのテーブルを削除し、
	 * テーブルに値を格納するために動作していたポーラーを停止させる。
	 */
	public void deleteTableAll(){
		m_log.debug("deleteTableAll()  Group : "  + m_groupName);
		
		Iterator itr = getTargetNodeList().iterator();
		
		synchronized (m_pollerConf) {
			while(itr.hasNext()){
				String facilityId = (String)itr.next();
				
				// テーブルを削除する
				this.m_dataTable.remove(facilityId);
				this.m_pollerConf.remove(facilityId);
			}
		}
		
		// 登録されているジョブを削除する
		try {
			// ジョブスケジューラを取得
			Scheduler scheduler = getScheduler();
			
			// 指定のグループ名で登録されているジョブを全て削除する
			String[] jobNames = scheduler.getJobNames(m_groupName);
			
			for(int i=0; i<jobNames.length; i++){
				// デバッグ情報
				m_log.debug("deleteJob  JobName : " + jobNames[i] + "  JobGroup : " + m_groupName);
				
				scheduler.deleteJob(jobNames[i], m_groupName);
			}
		} catch (SchedulerException e) {
			e.printStackTrace();
		}
	}
	
	/**
	 * 現在のリポジトリのノード情報をテーブル情報に反映させる。
	 */
	public void refreshTable(){
		m_log.debug("refreshTable()  Group : " + m_groupName + "  リポジトリ情報とIPアドレスの異なるテーブルを探します");
		
		// 全てのノードに対してチェックを行う
		Set<String> facilityIdSet = m_dataTable.keySet();		
		
		// 下で呼び出すschedule()の中でm_dataTableの要素を削除するため、
		// 静的なStringの配列にコピーしてからループを回す
		String[] facilityIds = 
			(String[])facilityIdSet.toArray(new String[facilityIdSet.size()]);
		
		for(int i=0; i<facilityIds.length; i++){
			String facilityId = facilityIds[i];
			
			// テーブルに登録されているIPアドレスを取得します
			PollingConfig config = getConfig(facilityId);
			String tableIpString = config.getSnmpConfig().getAddress().getHostAddress();
			
			// リポジトリに登録されているIPアドレスを取得します
			InetAddress repIpAddress = getIpAddress(facilityId);

			if(m_log.isDebugEnabled()){
				m_log.debug("Table IP Address : " + tableIpString);
				m_log.debug("Repository IP Address : " + repIpAddress.getHostAddress());
			}
			
			// テーブルに登録されているIPアドレスとリポジトリの内容が異なる場合
			if(!tableIpString.equals(repIpAddress.getHostAddress())){
				m_log.debug("Change IP Address " + facilityId + " : "
						+ tableIpString + " -> " + repIpAddress.getHostAddress());
				
				// テーブルの内容を削除する
				try {
					getTable(facilityId).setValueAll(new DataTable());
				} catch (FacilityNotFoundExciption e) {
					m_log.debug(e);
				}

				// IPアドレスを再設定
				config.getSnmpConfig().setAddress(repIpAddress);
				
				m_log.debug("Quartzの収集設定を変更  " + facilityId);
				m_scheduleFlag = true;
			}
		}
		
		// Quartzの設定を変更する
		if(m_scheduleFlag){
			schedule();
		}
		
		m_log.debug("Group : " + m_groupName + "  リポジトリ情報とIPアドレスの異なるテーブルのチェック完了");
	}
	
	/**
	 * ポーリングのスケジューリングを設定する
	 */
	public void schedule() {
		m_log.debug("schedule()  Group : "  + m_groupName);
		
		if(m_scheduleFlag == false){
			return;
		}
		
		synchronized (m_pollerConf) {
			m_log.debug("スケジュール変更  Group : "  + m_groupName);
			
			// 指定可能な全ての収集間隔を配列で取得
			int[] intervalList = PollingConfig.getAvailableIntervals();
			
			// 最小収集設定の定義のために初期値として最大値を設定する
			m_minInterval = Integer.MAX_VALUE;
			
			// Quartzには、収集間隔毎にジョブが登録される
			// 各ジョブに収集対象のノード(複数)のIPアドレス等の属性を設定する
			for(int i=0; i<intervalList.length; i++){
				int interval = intervalList[i];
				
				// ジョブの名前を定義
				String jobName = "Int" + interval + "Sec";
				
				// 新規にジョブを生成
				JobDetail job = new JobDetail(jobName, m_groupName, SnmpPollingJob.class);
				//ジョブ完了時に削除されないようにする
				job.setDurability(true);
				
				// 保持しているテーブル全てのポーリング情報に対して収集間隔をチェックし、
				// 指定の収集間隔のものをリストとしてQuartzに設定
				Iterator itr = m_pollerConf.keySet().iterator();
				
				HashMap<String, SnmpConfig> confMap = new HashMap<String, SnmpConfig>();
				while(itr.hasNext()){
					String facilityId = (String)itr.next();
					PollingConfig conf = (PollingConfig)m_pollerConf.get(facilityId);
					// チェック対象のファシリティ(ノード)の最小収集間隔設定が指定の収集間隔と一致した場合
					// 指定の収集間隔での収集対象ノードとして、Quartに登録する。
					if(conf.getMinInterval() == interval){  
						confMap.put(facilityId, conf.getSnmpConfig());
					}
				}
				
				// 収集対象のノードがひとつもない場合は、ジョブをスケジューリングしない
				if(confMap.size() == 0){
					// ジョブスケジューラを取得
					Scheduler scheduler = getScheduler();
					
					// 登録されているジョブを削除する
					try {
						scheduler.deleteJob(jobName, m_groupName);
					} catch (SchedulerException e) {
						e.printStackTrace();
					}
				} else {
					// 収集対象ノードの収集情報をジョブに登録しスケジューリングする
					
					m_minInterval = Math.min(m_minInterval, interval);
					
					job.getJobDataMap().put("jndiName", m_jndiName);  // テーブルをlookupするためのJNDI名
					job.getJobDataMap().put("interval", interval);    // 収集間隔
					job.getJobDataMap().put("oidList", m_oidList);     // 収集するOIDのリスト
					// 収集設定のリストをジョブに登録
					job.getJobDataMap().put("snmpConfMap", confMap);
					
					//スケジュールを設定
					String cronString = PollingConfig.parseCronExpression(interval);
					scheduleJob(jobName, m_groupName, job, cronString);
				}
			}
		}
		
		m_scheduleFlag = false;
	}
	
	private void scheduleJob(String jobName, String groupName, JobDetail job, String cronString) {
		m_log.debug("scheduleJob()  Group : " + m_groupName + " JobName : " + jobName + " " +  groupName + " : " + cronString);
		
		//CronTriggerを作成
		CronTrigger cronTrigger = new CronTrigger(jobName, groupName);
		
		try {
			cronTrigger.setCronExpression(cronString);
		} catch (ParseException e) {
			// エラー処理
			m_log.error(e);
		}
		
		// Schedulerにジョブを登録する
		try {
			// ジョブスケジューラを取得
			Scheduler scheduler = getScheduler();
			
			// 既に登録されているジョブを削除する(登録されていない場合は何もおこらない)
			scheduler.deleteJob(jobName, groupName);
			
			// ジョブを登録する
			scheduler.scheduleJob(job, cronTrigger);
		} catch (SchedulerException e) {
			// エラー処理
			m_log.error(e);
		}
	}
	
	private Scheduler getScheduler() {
		Scheduler scheduler = null;
		try {
			//QuartzのSchedulerをルックアップ
			InitialContext iniCtx = JNDIConnectionManager.getInitialContext();
			Object obj = iniCtx.lookup(m_quartzJndiName);
			scheduler = (Scheduler)PortableRemoteObject.narrow(obj, StdScheduler.class);
		} catch (NamingException e) {
			// エラー処理
			m_log.error(e);
		}
		return scheduler;
	}
	
	/**
	 * 指定ファシリティIDのノードのIPアドレスを取得します。
	 * 
	 * @param facilityId 対象のファシリティID
	 */
	private InetAddress getIpAddress(String facilityId) {
		// IPアドレスを取得
		try {
			RepositoryControllerLocal bean = RepositoryControllerUtil.getLocalHome().create();
			ArrayList<String> attributes = new ArrayList<String>();
			attributes.add(FacilityAttributeConstant.IPNETWORKNUMBER);
			HashMap nodeInfo = bean.getNodeDetail(facilityId, attributes);
			bean.remove();
			
			String ipAddressStr = (String)nodeInfo.get(FacilityAttributeConstant.IPNETWORKNUMBER);
			return InetAddress.getByName(ipAddressStr);
		} 
		catch (Exception e) {			
			String message= "Couldn't get an IP Address : " + facilityId;
			throw new EJBException(message, e) ;
		}
	}
	
	public PollingConfig getConfig(String facilityId) {
		PollingConfig conf = (PollingConfig)m_pollerConf.get(facilityId);
		
		if(conf == null){
			// エラー処理
			m_log.debug("登録されていないファシリティIDで収集設定の取得が実行されました : " + facilityId);
		}
		
		return conf;
	}
	
	public void setPollingInterval(String facilityId, int interval) {
		synchronized (m_pollerConf) {
			// ノードのファシリティIDに対応するテーブルがない場合は新規に作成
			if(this.m_dataTable.containsKey(facilityId) == false){
				this.createTable(facilityId);
			}
			
			// 対象ノードの設定を取得
			PollingConfig conf = this.getConfig(facilityId);
			
			// 収集間隔を設定
			// 設定の前後でポーリングの設定に変更が必要な場合にはQuartzのスケジュールを変更
			if(conf.addInterval(interval)){  // 収集間隔を追加することで収集間隔に変更が生じる場合はtrueが返る
				m_scheduleFlag = true;
				
				// 現在設定されている最小の収集間隔よりも小さな収集間隔が設定された場合はリスケジューリング
				if(interval < m_minInterval){
					schedule();
				}
			}
		}
	}
	
	private void removePollingInterval(String facilityId, int interval) throws FacilityNotFoundExciption{
		synchronized (m_pollerConf) {
			// 対象ノードの設定を取得
			PollingConfig conf = this.getConfig(facilityId);
			
			// 収集間隔を削除
			conf.removeInterval(interval);
			m_scheduleFlag = true;
		}
	}
	
	/**
	 * ポーラーの収集間隔を返します。
	 * @param facilityId ノードのファシリティID
	 * @return 収集間隔
	 * @throws FacilityNotFoundExciption 
	 */
	public int getInterval(String facilityId) throws FacilityNotFoundExciption{
		return this.getConfig(facilityId).getMinInterval();
	}
	
	/**
	 * 指定のファシリティIDのテーブルに対する最終参照時刻を返します。
	 * @param facilityId ファシリティID
	 * @return 最終参照時刻
	 * @throws FacilityNotFoundExciption 
	 */
	public long getLastReferenceTime(String facilityId) throws FacilityNotFoundExciption{
		return getTable(facilityId).getLastReference();
	}
	
	private DataTable getTable(String facilityId) throws FacilityNotFoundExciption{
		DataTable table = (DataTable)m_dataTable.get(facilityId);
		
		if(table == null){
			// エラー処理
			m_log.debug("登録されていないファシリティIDでテーブルが参照されました : " + facilityId);
			throw new FacilityNotFoundExciption();
		}
		
		return table;
	}
	
	/**
	 * 指定ファシリティIDのノードのMIB値を共有テーブルに格納します。
	 * @param facilityId ノードのファシリティID
	 * @param oid 収集値のOID
	 * @param date 収集時刻
	 * @param value 収集値
	 * @throws FacilityNotFoundExciption 
	 */
	public void putValue(String facilityId, String oid, long date, Object value) throws FacilityNotFoundExciption {
		m_log.debug("putValue()   Group : " + m_groupName + "  " + facilityId + " : " + value);
		
		// ノードのファシリティIDに対応するテーブルがない場合は新規に作成
		if(this.m_dataTable.containsKey(facilityId) == false){
			this.createTable(facilityId);			
		}
		
		getTable(facilityId).putValue(oid, date, value);
	}
	
	/**
	 * 指定ファシリティIDのノードの共有テーブルに複数の収集値を一括で格納します。
	 * 同じOIDのものの値を上書きします。
	 * 
	 * @param facilityId ファシリティID
	 * @param table 収集値が格納されているテーブル
	 * @throws FacilityNotFoundExciption 
	 */
	public void putValueAll(String facilityId, DataTable table) throws FacilityNotFoundExciption {
		m_log.debug("putValueAll()   Group : " + m_groupName + "  " + facilityId);
		
		// ノードのファシリティIDに対応するテーブルがない場合は新規に作成
		if(this.m_dataTable.containsKey(facilityId) == false){
			this.createTable(facilityId);
		}
		
		getTable(facilityId).putValueAll(table);
	}
	
	/**
	 * 指定ファシリティIDのノードの共有テーブルを指定のテーブルと置き換えます。
	 * 前回の収集値はクリアされます。
	 * 
	 * @param facilityId ファシリティID
	 * @param table 収集値が格納されているテーブル
	 * @throws FacilityNotFoundExciption 
	 * @throws FacilityNotFoundExciption 
	 */
	public void setValueAll(String facilityId, DataTable table) throws FacilityNotFoundExciption {
		m_log.debug("setValueAll()   Group : " + m_groupName + "  " + facilityId);
		
		// ノードのファシリティIDに対応するテーブルがない場合は新規に作成
		if(this.m_dataTable.containsKey(facilityId) == false){
			this.createTable(facilityId);
		}
		
		getTable(facilityId).setValueAll(table);
	}
	
	/**
	 * 指定ファシリティの指定OIDの収集値を返します。
	 * 同時に、指定された問い合わせ間隔でOIDの値が更新されるようにポーラーを設定します。
	 * 
	 * @param facilityId ファシリティID
	 * @param interval 問い合わせ間隔(秒)
	 * @param oid SNMPのOID
	 * @return 指定のOIDの収集値
	 * @throws FacilityNotFoundExciption 
	 */
	public ValueObject getValue(String facilityId, int interval, String oid) {
		// ノードのファシリティIDに対応するテーブルがない場合は新規に作成
		if(this.m_dataTable.containsKey(facilityId) == false){
			this.createTable(facilityId);
		}
		
		// 収集条件を設定しポーラーのスケジュールに変更が生じる場合は変更する
		setPollingInterval(facilityId, interval);
		try {
			return getTable(facilityId).getValue(oid);
		} catch (FacilityNotFoundExciption e) {
			m_log.debug(e);
		}
		return null;
	}
	
	/**
	 * 指定ファシリティの指定OID配下のMIBツリーの収集値を返します。
	 * 同時に、指定された問い合わせ間隔でOIDの値が更新されるようにポーラーを設定します。
	 * 
	 * @param facilityId ファシリティID
	 * @param interval 問い合わせ間隔(秒)
	 * @param oid SNMPのOID
	 * @return 指定OID以下のツリーに含まれる収集値のセット(ない場合はnullを返す)
	 * @throws FacilityNotFoundExciption 
	 */
	public Set getValueSet(String facilityId, int interval, String oid) {
		// ノードのファシリティIDに対応するテーブルがない場合は新規に作成
		if(this.m_dataTable.containsKey(facilityId) == false){
			this.createTable(facilityId);
		}
		
		// 収集条件を設定しポーラーのスケジュールに変更が生じる場合は変更する
		setPollingInterval(facilityId, interval);
		
		// 戻りの値を格納するセット
		HashSet set = new HashSet();
		
		DataTable dt;
		try {
			dt = getTable(facilityId).getValueAll();
			
			// キーとされているフルのOIDを取得しそのフルOIDの文字列の先頭部分が、
			// 引数指定のOIDである場合は、その値を戻りのセットに格納する
			Iterator itr = dt.keySet().iterator();
			while(itr.hasNext()){
				String fullOid = (String)itr.next();
				
				// もし先頭の部分が一致するならそのツリーの子孫と判定
				if(fullOid.startsWith(oid)){
					set.add(getTable(facilityId).getValue(fullOid));
				}
			}
		} catch (FacilityNotFoundExciption e) {
			m_log.debug(e);
		}
		// サブツリーに含まれる収集値がない場合はnullを返す
		if(set.size() == 0){
			return null;
		} else {
			return set;
		}
	}
	
	/**
	 * 指定ファシリティのノードに対し指定の収集間隔でSNMPポーリングを行うように
	 * ポーラーの設定を変更します。
	 * @param facilityId 収集対象ノードのファシリティID
	 * @param interval 収集間隔(秒)
	 * @throws FacilityNotFoundExciption 
	 */
	public void setPollingSchedule(String facilityId, int interval) {
		// ノードのファシリティIDに対応するテーブルがない場合は新規に作成
		if(this.m_dataTable.containsKey(facilityId) == false){
			this.createTable(facilityId);
		}
		
		// 収集設定を更新
		this.setPollingInterval(facilityId, interval);
	}
	
	/**
	 * 指定ファシリティのノードに対し指定の収集間隔でSNMPポーリングを行うように
	 * ポーラーの設定を変更します。
	 * @param facilityIds 収集対象ノードのファシリティIDのリスト
	 * @param interval 収集間隔(秒)
	 * @param waitTime ポーラ初回起動時の場合の待ち時間
	 * @throws FacilityNotFoundExciption 
	 */
	public void setPollingSchedule(String[] facilityIds, int interval, long waitTime) {
		for(int i=0; i<facilityIds.length; i++){
			String facilityId = facilityIds[i];
			
			// ノードのファシリティIDに対応するテーブルがない場合は新規に作成
			if(this.m_dataTable.containsKey(facilityId) == false){
				this.createTable(facilityId);
			}
			
			// 収集設定を更新
			this.setPollingInterval(facilityId, interval);
		}
		
		// Quartzのジョブスケジュールを設定
		this.m_scheduleFlag = true;
		this.schedule();
		
		// ポーリング結果が返ってくるのを指定の時間待つ
		try {
			Thread.sleep(waitTime);
		} catch (InterruptedException e) {
			m_log.error(e);
		}
	}

	/**
	 * 指定ファシリティのノードに対し指定の収集間隔でSNMPポーリングを行うように
	 * ポーラーの設定を変更します。
	 * @param facilityIds 収集対象ノードのファシリティIDのリスト
	 * @param interval 収集間隔(秒)
	 * @throws FacilityNotFoundExciption 
	 */
	public void setPollingSchedule(String[] facilityIds, int interval) {
		setPollingSchedule(facilityIds, interval, 0);
	}

	
	public String getCheckInterval() {
		return m_checkInterval;
	}
	
	public void setCheckInterval(String interval) {
		m_checkInterval = interval;
	}
	
	public List getOidList() {
		return m_oidList;
	}
	
	public void setOidList(List oidList) {
		m_oidList = oidList;
	}
	
	public List getTargetNodeList(){
		TreeSet set = new TreeSet(this.m_dataTable.keySet());
		return new ArrayList(set);
	}
	
	public int getDefaultPort() {
		return m_defaultPort;
	}
	
	public void setDefaultPort(int port) {
		m_defaultPort = port;
	}
	
	public String getDefaultCommunity() {
		return m_defaultCommunity;
	}
	
	public void setDefaultCommunity(String community) {
		m_defaultCommunity = community;
	}
	
	public int getDefaultVersion() {
		return m_defaultVersion;
	}
	
	public void setDefaultVersion(int version) {
		m_defaultVersion = version;
	}
	
	public int getDefaultRetries() {
		return m_defaultRetries;
	}
	
	public void setDefaultRetries(int retries) {
		m_defaultRetries = retries;
	}
	
	public int getDefaultTimeout() {
		return m_defaultTimeout;
	}
	
	public void setDdefaultTimeout(int timeout) {
		m_defaultTimeout = timeout;
	}
	
	public String getQuartzJndiName() {
		return m_quartzJndiName;
	}
	
	public void setQuartzJndiName(String jndiName) {
		m_quartzJndiName = jndiName;
	}
}
