/*
 
Copyright (C) 2006 NTT DATA Corporation
 
 This program is free software; you can redistribute it and/or
 Modify it under the terms of the GNU General Public License 
 as published by the Free Software Foundation, version 2.
 
 This program is distributed in the hope that it will be
 useful, but WITHOUT ANY WARRANTY; without even the implied 
 warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.composite;

import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.DisposeEvent;
import org.eclipse.swt.events.DisposeListener;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Tree;

import com.clustercontrol.action.GetFacilityTree;
import com.clustercontrol.bean.FacilityTreeItem;
import com.clustercontrol.etc.action.UpdateFacilityTreeTask;
import com.clustercontrol.util.UpdateRepositoryInfoReceiveTopic;

/**
 * スコープツリーを表示するコンポジットクラス<BR>
 * 
 * @version 1.0.0
 * @since 1.0.0
 */
public class FacilityTreeComposite extends Composite {
	
	//	 ----- instance フィールド ----- //
	
	/** ツリービューア */
	protected TreeViewer treeViewer = null;
	
	/** 選択アイテム */
	protected FacilityTreeItem selectItem = null;
	
	/** 選択アイテム数 */
	protected int subScopeNumber;
	
	/**ノードをツリーに含めるか？ */
	protected boolean scopeOnly = false;
	
	/**未登録ノード　スコープをツリーに含めるか？**/
	protected boolean unregistered = true;
	
	/**内部イベント　スコープをツリーに含めるか？**/
	protected boolean internal = true;
	
	/** リポジトリ情報更新により、表示をリフレッシュするかどうか **/
	protected boolean topicRefresh = true;
	
	protected UpdateRepositoryInfoReceiveTopic updateRepository = null;
	
	// ----- コンストラクタ ----- //
	
	public FacilityTreeComposite(Composite parent, int style) {
		super(parent, style);
		
		this.scopeOnly = false;
		this.unregistered = true;
		this.internal = true;
		
		this.createContents();
	}
	
	public FacilityTreeComposite(Composite parent, int style,
			boolean scopeOnly ,
			boolean unregistered,
			boolean internal) {
		super(parent, style);
		
		this.scopeOnly = scopeOnly;
		this.unregistered = unregistered;
		this.internal = internal;
		this.createContents();
	}
	
	/**
	 * コンストラクタ
	 * 
	 * @param parent 親のコンポジット
	 * @param style スタイル
	 * @param scopeOnly スコープのみのスコープツリーとするかどうか
	 * @param unregistered UNREGISTEREDをスコープツリーに含めるかどうか
	 * @param internal INTERNALをスコープツリーに含めるかどうか
	 * @param topicRefresh リポジトリ情報が更新された際に画面リフレッシュするかどうか
	 */
	public FacilityTreeComposite(Composite parent, int style,
			boolean scopeOnly ,
			boolean unregistered,
			boolean internal, 
			boolean topicRefresh) {
		super(parent, style);
		
		this.scopeOnly = scopeOnly;
		this.unregistered = unregistered;
		this.internal = internal;
		this.topicRefresh = topicRefresh;
		this.createContents();
	}
	
	// ----- instance メソッド ----- //
	
	
	/**
	 * このコンポジットが利用するツリービューアを返します。
	 * 
	 * @return ツリービューア
	 */
	public TreeViewer getTreeViewer() {
		return this.treeViewer;
	}
	
	/**
	 * このコンポジットが利用するツリーを返します。
	 * 
	 * @return ツリー
	 */
	public Tree getTree() {
		return this.treeViewer.getTree();
	}
	
	/**
	 * 現在選択されているツリーアイテムを返します。
	 * 
	 * @return ツリーアイテム
	 */
	public FacilityTreeItem getSelectItem() {
		return this.selectItem;
	}
	
	/**
	 * 現在選択されているツリーのサブスコープ数を返します。
	 * 
	 * @return サブスコープ数
	 */
	public int getSubScopeNumber() {
		return subScopeNumber;
	}
	
	/**
	 * コンポジットを生成します。
	 */
	protected void createContents() {
		
		// コンポジットのレイアウト定義
		GridLayout layout = new GridLayout(1, true);
		layout.marginHeight = 0;
		layout.marginWidth = 0;
		this.setLayout(layout);
		
		// ツリー作成
		Tree tree = new Tree(this, SWT.SINGLE | SWT.BORDER);
		
		// ツリーのレイアウトデータ定義
		GridData layoutData = new GridData();
		layoutData.horizontalAlignment = GridData.FILL;
		layoutData.verticalAlignment = GridData.FILL;
		layoutData.grabExcessHorizontalSpace = true;
		layoutData.grabExcessVerticalSpace = true;
		tree.setLayoutData(layoutData);
		
		// ツリービューア作成
		this.treeViewer = new TreeViewer(tree);
		
		// ツリービューア設定
		this.treeViewer.setContentProvider(new FacilityTreeContentProvider());
		this.treeViewer.setLabelProvider(new FacilityTreeLabelProvider());
		this.treeViewer.setSorter(new FacilityTreeViewerSorter());
		
		// 選択アイテム取得イベント定義
		this.treeViewer
		.addSelectionChangedListener(new ISelectionChangedListener() {
			public void selectionChanged(SelectionChangedEvent event) {
				StructuredSelection selection = (StructuredSelection) event
				.getSelection();
				
				selectItem = (FacilityTreeItem) selection
				.getFirstElement();
				
				if (selectItem instanceof FacilityTreeItem) {
					subScopeNumber = selectItem.getChildren().length;
				}
			}
		});
		
		//マネージャからのファシリティツリー更新
		if (topicRefresh) {
			this.updateRepository = new UpdateRepositoryInfoReceiveTopic(this.getDisplay(), new UpdateFacilityTreeTask(this));
		}
		
		this.addDisposeListener(new DisposeListener(){
			public void widgetDisposed(DisposeEvent e) {
				if(updateRepository != null){
					updateRepository.terminate();
				}
			}
		});
		
		// 表示します。
		this.update();
	}
	
	/**
	 * ビューの表示内容を更新します。
	 */
	public void update() {
		GetFacilityTree getFacilityTree = new GetFacilityTree();
		FacilityTreeItem treeItem = getFacilityTree.getTreeItem(scopeOnly);
		
		if(treeItem != null){
			
			//"スコープ"というツリーから直下のスコープを削除する。
			FacilityTreeItem scope = (treeItem.getChildren())[0];	
			
			//ファシリティツリーから特定のスコープを取り外す。 
			//FIXME ここでは組込みスコープがFacilityIDで決めうちになっている。
			//後々は直したい。
			
			String Internal     = "INTERNAL";
			String Unregistered = "UNREGISTERED"; 
			
			if(!internal){
				if(scope.removeChild(Internal)){	   
				}else{
					//まずい。"INTERNAL"スコープがない！
				}
			}
			if(!unregistered){
				if(scope.removeChild(Unregistered)){	   
				}else{
					//まずい。"UNREGISTERED"スコープがない！
				}  
			}			
			
			this.treeViewer.setInput(treeItem);
			FacilityTreeItem selectItem[] = treeItem.getChildren();
			this.treeViewer.setSelection(new StructuredSelection(selectItem[0]), true);
		}
		
		//スコープのレベルまで展開
		this.treeViewer.expandToLevel(2);
	}
	
	/**
	 * ツリーを展開して表示するかを指定します。
	 *  
	 */
	public void setExpand(boolean isExpand) {
		if (isExpand) {
			this.treeViewer.setAutoExpandLevel(TreeViewer.ALL_LEVELS);
		}
	}
	
	/**
	 * ツリーの表示内容を更新します。
	 * 
	 * @param treeItem
	 */
	public void setScopeTree(FacilityTreeItem treeItem) {
		try {
			this.treeViewer.setInput(treeItem);
			this.treeViewer.setAutoExpandLevel(TreeViewer.ALL_LEVELS);
			
		} catch (Exception e) {
		}
	}
	
	/**
	 * TreeをセットしながらfacilityIDに対応する要素を選択状態にします。
	 * 
	 * @param treeItem
	 * @param facilityID
	 */
	public void setScopeTreeWithSelection(FacilityTreeItem treeItem,
			String facilityID) {
		this.setScopeTree(treeItem);
		
		FacilityTreeItem[] tmpItem = treeItem.getChildren();
		
		//引数のFaiclityIDに対応するTreeItemがあるか探します。
		for (int i = 0; i < tmpItem.length; i++) {
			setScopeTreeWithSelectionSub(tmpItem[i], facilityID);
			if (facilityID.equals(tmpItem[i].getData().getFacilityId())) {
				this.treeViewer.setSelection(
						new StructuredSelection(tmpItem[i]), true);
			}
		}
	}
	
	/**
	 * setScopeTreeWithSelectionから呼ばれること前提とした再帰用のメソッド
	 * 
	 * @param treeItem
	 * @param facilityID
	 */
	public void setScopeTreeWithSelectionSub(FacilityTreeItem treeItem,
			String facilityID) {
		FacilityTreeItem[] tmpItem = treeItem.getChildren();
		
		for (int i = 0; i < tmpItem.length; i++) {
			setScopeTreeWithSelectionSub(tmpItem[i], facilityID);
			
			if (facilityID.equals(tmpItem[i].getData().getFacilityId())) {
				this.treeViewer.setSelection(
						new StructuredSelection(tmpItem[i]), true);
				
			}
		}
		
	}
}