/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.notify.dialog;

import java.util.ArrayList;

import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Text;

import com.clustercontrol.ClusterControlPlugin;
import com.clustercontrol.bean.ExecFacilityConstant;
import com.clustercontrol.bean.FacilityConstant;
import com.clustercontrol.bean.FacilityPath;
import com.clustercontrol.bean.FacilityTreeItem;
import com.clustercontrol.bean.NotifyTypeConstant;
import com.clustercontrol.bean.PriorityColorConstant;
import com.clustercontrol.bean.PriorityConstant;
import com.clustercontrol.bean.ValidConstant;
import com.clustercontrol.dialog.ScopeTreeDialog;
import com.clustercontrol.dialog.ValidateResult;
import com.clustercontrol.jobmanagement.common.composite.JobIdSelectionListener;
import com.clustercontrol.notify.action.AddNotify;
import com.clustercontrol.notify.action.GetNotify;
import com.clustercontrol.notify.action.ModifyNotify;
import com.clustercontrol.notify.bean.NotifyInfo;
import com.clustercontrol.notify.bean.NotifyInfoDetail;
import com.clustercontrol.notify.bean.NotifyJobInfo;
import com.clustercontrol.util.Messages;

/**
 * 通知（ジョブ実行）作成・変更ダイアログクラス<BR>
 * 
 * @version 3.0.0
 * @since 3.0.0
 */
public class NotifyJobCreateDialog extends NotifyBasicCreateDialog {
	
	/** カラム数（重要度）。 */
	private static final int WIDTH_PRIORITY 		= 1;
	
	/** カラム数（ジョブ実行）。 */
	private static final int WIDTH_JOB_RUN	 		= 1;
	
	/** カラム数（ジョブID）。 */
	private static final int WIDTH_JOB_ID	 		= 4;
	
	/** カラム数（参照ボタン）。 */
	private static final int WIDTH_REF_BTN 			= 2;
	
	/** カラム数（通知抑制と連動）。 */
	private static final int WIDTH_INHIBITION_FLG 	= 2;
	
	/** カラム数（呼出失敗時）。 */
	private static final int WIDTH_FAILURE_PRIORITY = 2;
	
	/** カラム数（空欄）。 */
	private static final int WIDTH_BLANK 			= 3;
	
    // ----- instance フィールド ----- //
	
	/** 通知タイプ 
	 * @see com.clustercontrol.bean.NotifyTypeConstant
	 */
	private final Integer m_notifyType = NotifyTypeConstant.TYPE_JOB;
	
	/** 入力値を保持するオブジェクト。 */
	private NotifyInfo inputData = null;
	
	/** 入力値の正当性を保持するオブジェクト。 */
	private ValidateResult validateResult = null;
	
	/** スコープ用テキスト */
    private Text m_textScope = null;
    
    /** ファシリティID */
    private String m_facilityId = null;
    
    /** スコープ */
    private String m_facilityPath = null;
	
    /** ジョブ変数用ラジオボタン */
    private Button m_radioGenerationNodeValue = null;
    
    /** 固定値用ラジオボタン */
    private Button m_radioFixedValue = null;
	
    /** スコープ参照用ボタン */
    private Button m_scopeSelect = null;
    
	/** 実行（通知） チェックボックス。 */ 
	private Button m_checkJobRunInfo = null;
	/** 実行（警告） チェックボックス。 */
	private Button m_checkJobRunWarning = null;
	/** 実行（異常） チェックボックス。 */
	private Button m_checkJobRunCritical = null;
	/** 実行（不明） チェックボックス。 */
	private Button m_checkJobRunUnknown = null;
	
	/** ジョブID（通知） テキストボックス。 */
	private Text m_textJobIdInfo = null;
	/** ジョブID（警告） テキストボックス。 */
	private Text m_textJobIdWarning = null;
	/** ジョブID（異常） テキストボックス。 */
	private Text m_textJobIdCritical = null;
	/** ジョブID（不明） テキストボックス。 */
	private Text m_textJobIdUnknown = null;
	
	/** 参照（通知） ボタン。 */
	private Button m_buttonReferInfo = null;
	/** 参照（警告） ボタン。 */
	private Button m_buttonReferWarning = null;
	/** 参照（異常） ボタン。 */
	private Button m_buttonReferCritical = null;
	/** 参照（不明） ボタン。 */
	private Button m_buttonReferUnknown = null;
	
	/** 抑制と連動（通知） チェックボックス。 */ 
	private Button m_checkInhibitionInfo = null;
	/** 抑制と連動（警告） チェックボックス。 */
	private Button m_checkInhibitionWarning = null;
	/** 抑制と連動（異常） チェックボックス。 */
	private Button m_checkInhibitionCritical = null;
	/** 抑制と連動（不明） チェックボックス。 */
	private Button m_checkInhibitionUnknown = null;
	
	/** 呼出失敗時の重要度（通知） コンボボックス。 */ 
	private Combo m_comboFailurePriorityInfo = null;
	/** 呼出失敗時の重要度（警告） コンボボックス。 */
	private Combo m_comboFailurePriorityWarning = null;
	/** 呼出失敗時の重要度（異常） コンボボックス。 */
	private Combo m_comboFailurePriorityCritical = null;
	/** 呼出失敗時の重要度（不明） コンボボックス。 */
	private Combo m_comboFailurePriorityUnknown = null;
	
	// ----- コンストラクタ ----- //
	
	/**
	 * 作成用ダイアログのインスタンスを返します。
	 * 
	 * @param parent 親のシェルオブジェクト
	 */
	public NotifyJobCreateDialog(Shell parent) {
		super(parent);
	}
	
	/**
	 * 変更用ダイアログのインスタンスを返します。
	 * 
	 * @param parent 親のシェルオブジェクト
	 * @param notifyId 変更する通知情報の通知ID
	 * @param referenceFlg 参照フラグ（参照のみで変更不可の場合、<code> true </code>）
	 */
	public NotifyJobCreateDialog(Shell parent, String notifyId, boolean referenceFlg) {
		super(parent, notifyId, referenceFlg);
	}
	
    // ----- instance メソッド ----- //
	
	/**
	 * ダイアログエリアを生成します。
	 * 
	 * @param parent 親のコンポジット
	 * 
	 * @see com.clustercontrol.notify.dialog.NotifyBasicCreateDialog#customizeDialog(Composite)
	 * @see com.clustercontrol.notify.action.GetNotify#getNotify(String)
	 * @see #setInputData(NotifyInfo)
	 */
    protected void customizeDialog(Composite parent) {
    	
    	super.customizeDialog(parent);
    	
        // 通知IDが指定されている場合、その情報を初期表示する。
        NotifyInfo info = null; 
        if(this.notifyId != null){
        	info = new GetNotify().getNotify(this.notifyId);        	
        }
        else{
        	info = new NotifyInfo();
        }
        this.setInputData(info);
        
    }
    
	/**
	 * 親のクラスから呼ばれ、各通知用のダイアログエリアを生成します。
	 * 
	 * @param parent 親のコンポジット
	 * 
	 * @see com.clustercontrol.notify.dialog.NotifyBasicCreateDialog#customizeDialog(Composite)
	 */
    protected void customizeSettingDialog(Composite parent) {
    	final Shell shell = this.getShell();

        // タイトル
        shell.setText(Messages.getString("dialog.notify.job.create.modify"));

        // 変数として利用されるラベル
        Label label = null;
        // 変数として利用されるグリッドデータ
        GridData gridData = null;

        // レイアウト
        GridLayout layout = new GridLayout(1, true);
        layout.marginWidth = 10;
        layout.marginHeight = 10;
        layout.numColumns = 15;
        parent.setLayout(layout);     
        
        /*
         * ジョブ
         */
		label = new Label(parent, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = 15;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);
		label.setText(Messages.getString("notifies.job") + " : ");
		
        /*
         * 大グループ
         */
        // ジョブグループ
        Group groupJob = new Group(parent, SWT.NONE);
        layout = new GridLayout(1, true);
        layout.marginWidth = 5;
        layout.marginHeight = 5;
        layout.numColumns = 15;
        groupJob.setLayout(layout);
        gridData = new GridData();
        gridData.horizontalSpan = 15;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        groupJob.setLayoutData(gridData);

        /*
         * スコープグループ
         */
        Group groupScope = new Group(groupJob, SWT.NONE);
        groupScope.setText(Messages.getString("notify.job.scope"));
        layout = new GridLayout(1, true);
        layout.marginWidth = 5;
        layout.marginHeight = 5;
        layout.numColumns = 15;
        groupScope.setLayout(layout);
        gridData = new GridData();
        gridData.horizontalSpan = 15;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        groupScope.setLayoutData(gridData);
        
        // ジョブ変数 ラジオボタン
        this.m_radioGenerationNodeValue = new Button(groupScope, SWT.RADIO);
        gridData = new GridData();
        gridData.horizontalSpan = 15;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        this.m_radioGenerationNodeValue.setLayoutData(gridData);
        this.m_radioGenerationNodeValue.setText(Messages.getString("notify.node.generation") + " : ");
        this.m_radioGenerationNodeValue.addSelectionListener(new SelectionListener() {
			public void widgetSelected(SelectionEvent e) {
				Button check = (Button) e.getSource();
				if (check.getSelection()) {
					m_radioFixedValue.setSelection(false);
					m_scopeSelect.setEnabled(false);
				}
			}
			
			public void widgetDefaultSelected(SelectionEvent e) {
				
			}
		});
        
        // 固定値 ラジオボタン
        this.m_radioFixedValue = new Button(groupScope, SWT.RADIO);
        gridData = new GridData();
        gridData.horizontalSpan = 3;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        this.m_radioFixedValue.setLayoutData(gridData);
        this.m_radioFixedValue.setText(Messages.getString("notify.node.fix") + " : ");
        this.m_radioFixedValue.addSelectionListener(new SelectionListener() {
			public void widgetSelected(SelectionEvent e) {
				Button check = (Button) e.getSource();
				if (check.getSelection()) {
					m_radioGenerationNodeValue.setSelection(false);
					m_scopeSelect.setEnabled(true);
				}
			}

			public void widgetDefaultSelected(SelectionEvent e) {

			}
		});
        
        this.m_textScope = new Text(groupScope, SWT.BORDER | SWT.READ_ONLY);
		gridData = new GridData();
		gridData.horizontalSpan = 5;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		this.m_textScope.setLayoutData(gridData);
		this.m_textScope.setText("");
		
		this.m_scopeSelect = new Button(groupScope, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = 2;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		this.m_scopeSelect.setLayoutData(gridData);
		this.m_scopeSelect.setText(Messages.getString("refer"));
		this.m_scopeSelect.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				ScopeTreeDialog dialog = new ScopeTreeDialog(shell);
				if (dialog.open() == IDialogConstants.OK_ID) {
					FacilityTreeItem selectItem = dialog.getSelectItem();
					if (selectItem.getData().getType() == FacilityConstant.TYPE_COMPOSITE) {
						m_facilityPath = "";
						m_facilityId = "";
					} else {
						FacilityPath path = new FacilityPath(
								ClusterControlPlugin.getDefault()
										.getSeparator());
						m_facilityPath = path.getPath(selectItem);
						m_facilityId = selectItem.getData().getFacilityId();
					}

					m_textScope.setText(m_facilityPath);
				}
			}
		});
        
        label = new Label(groupScope, SWT.NONE);
        gridData = new GridData();
        gridData.horizontalSpan = 5;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        label.setLayoutData(gridData);  
        

        
        // 空行
        label = new Label(groupJob, SWT.NONE);
        gridData = new GridData();
        gridData.horizontalSpan = 15;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        label.setLayoutData(gridData);  
        
        /*
         * 重要度 ごとの設定
         */
        // ラベル（重要度）
        label = new Label(groupJob, SWT.NONE);
        gridData = new GridData();
        gridData.horizontalSpan = WIDTH_PRIORITY;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        label.setLayoutData(gridData);
        label.setText(Messages.getString("priority"));
        
        // ラベル（実行する）
        label = new Label(groupJob, SWT.NONE);
        gridData = new GridData();
        gridData.horizontalSpan = WIDTH_JOB_RUN;
        gridData.horizontalAlignment = GridData.CENTER;
        gridData.grabExcessHorizontalSpace = true;
        label.setLayoutData(gridData);
        label.setText(Messages.getString("run"));
        
        // ラベル（ジョブID）
        label = new Label(groupJob, SWT.NONE);
        gridData = new GridData();
        gridData.horizontalSpan = WIDTH_JOB_ID;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        label.setLayoutData(gridData);
        label.setText(Messages.getString("job.id"));
        
        // ラベル（参照）
        label = new Label(groupJob, SWT.NONE);
        gridData = new GridData();
        gridData.horizontalSpan = WIDTH_REF_BTN;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        label.setLayoutData(gridData);
        
        // ラベル（抑制と連動）
        label = new Label(groupJob, SWT.NONE);
        gridData = new GridData();
        gridData.horizontalSpan = WIDTH_INHIBITION_FLG;
        gridData.horizontalAlignment = GridData.CENTER;
        gridData.grabExcessHorizontalSpace = true;
        label.setLayoutData(gridData);
        label.setText(Messages.getString("notify.event.treatment.inhibition"));
        
        // ラベル（呼出失敗時）
        label = new Label(groupJob, SWT.NONE);
        gridData = new GridData();
        gridData.horizontalSpan = WIDTH_FAILURE_PRIORITY;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        label.setLayoutData(gridData);
        label.setText(Messages.getString("failure.call.value"));
        
        // 空欄
        label = new Label(groupJob, SWT.NONE);
        gridData = new GridData();
        gridData.horizontalSpan = WIDTH_BLANK;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        label.setLayoutData(gridData);  
        
        //　ジョブ　重要度：通知
        label = this.getLabelPriority(groupJob, Messages.getString("info"),PriorityColorConstant.COLOR_INFO);
        this.m_checkJobRunInfo = this.getCheckJobRun(groupJob);
        this.m_textJobIdInfo = this.getTextJobId(groupJob);
        this.m_buttonReferInfo = this.getButtonJobRefer(groupJob);
        this.m_checkInhibitionInfo = this.getCheckJobInhibition(groupJob);
        this.m_comboFailurePriorityInfo = this.getComboPriority(groupJob);
        
        this.m_checkJobRunInfo.addSelectionListener(new SelectionAdapter() {
        	public void widgetSelected(SelectionEvent e) {
				setEnabledForJob(m_checkJobRunInfo.getSelection(), m_textJobIdInfo, m_buttonReferInfo, m_checkInhibitionInfo, m_comboFailurePriorityInfo);
        	}
        });
        this.m_buttonReferInfo.addSelectionListener(new JobIdSelectionListener(m_textJobIdInfo));
        
        // 空欄
        label = new Label(groupJob, SWT.NONE);
        gridData = new GridData();
        gridData.horizontalSpan = WIDTH_BLANK;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        label.setLayoutData(gridData);  
        
        //　ジョブ　重要度：警告
        label = this.getLabelPriority(groupJob, Messages.getString("warning"),PriorityColorConstant.COLOR_WARNING);
        this.m_checkJobRunWarning = this.getCheckJobRun(groupJob);
        this.m_textJobIdWarning = this.getTextJobId(groupJob);
        this.m_buttonReferWarning = this.getButtonJobRefer(groupJob);
        this.m_checkInhibitionWarning = this.getCheckJobInhibition(groupJob);
        this.m_comboFailurePriorityWarning = this.getComboPriority(groupJob);
        
        this.m_checkJobRunWarning.addSelectionListener(new SelectionAdapter() {
        	public void widgetSelected(SelectionEvent e) {
				setEnabledForJob(m_checkJobRunWarning.getSelection(), m_textJobIdWarning, m_buttonReferWarning, m_checkInhibitionWarning, m_comboFailurePriorityWarning);
        	}
        });
        this.m_buttonReferWarning.addSelectionListener(new JobIdSelectionListener(m_textJobIdWarning));
        
        // 空欄
        label = new Label(groupJob, SWT.NONE);
        gridData = new GridData();
        gridData.horizontalSpan = WIDTH_BLANK;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        label.setLayoutData(gridData);  
        
        //　ジョブ　重要度：危険
        label = this.getLabelPriority(groupJob, Messages.getString("critical"),PriorityColorConstant.COLOR_CRITICAL);
        this.m_checkJobRunCritical = this.getCheckJobRun(groupJob);
        this.m_textJobIdCritical = this.getTextJobId(groupJob);
        this.m_buttonReferCritical = this.getButtonJobRefer(groupJob);
        this.m_checkInhibitionCritical = this.getCheckJobInhibition(groupJob);
        this.m_comboFailurePriorityCritical = this.getComboPriority(groupJob);
        
        this.m_checkJobRunCritical.addSelectionListener(new SelectionAdapter() {
        	public void widgetSelected(SelectionEvent e) {
				setEnabledForJob(m_checkJobRunCritical.getSelection(), m_textJobIdCritical, m_buttonReferCritical, m_checkInhibitionCritical, m_comboFailurePriorityCritical);
        	}
        });
        this.m_buttonReferCritical.addSelectionListener(new JobIdSelectionListener(m_textJobIdCritical));
        
        // 空欄
        label = new Label(groupJob, SWT.NONE);
        gridData = new GridData();
        gridData.horizontalSpan = WIDTH_BLANK;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        label.setLayoutData(gridData);  
        
        //　ジョブ　重要度：不明
        label = this.getLabelPriority(groupJob, Messages.getString("unknown"),PriorityColorConstant.COLOR_UNKNOWN);
        this.m_checkJobRunUnknown = this.getCheckJobRun(groupJob);
        this.m_textJobIdUnknown = this.getTextJobId(groupJob);
        this.m_buttonReferUnknown = this.getButtonJobRefer(groupJob);
        this.m_checkInhibitionUnknown = this.getCheckJobInhibition(groupJob);
        this.m_comboFailurePriorityUnknown = this.getComboPriority(groupJob);
        
        this.m_checkJobRunUnknown.addSelectionListener(new SelectionAdapter() {
        	public void widgetSelected(SelectionEvent e) {
				setEnabledForJob(m_checkJobRunUnknown.getSelection(), m_textJobIdUnknown, m_buttonReferUnknown, m_checkInhibitionUnknown, m_comboFailurePriorityUnknown);
        	}
        }); 
        this.m_buttonReferUnknown.addSelectionListener(new JobIdSelectionListener(m_textJobIdUnknown));
        
        // 空欄
        label = new Label(groupJob, SWT.NONE);
        gridData = new GridData();
        gridData.horizontalSpan = WIDTH_BLANK;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        label.setLayoutData(gridData);  

    }

	/**
	 * 入力値を保持した通知情報を返します。
	 * 
	 * @return 通知情報
	 */
    public NotifyInfo getInputData() {
        return this.inputData;
    }

	/**
	 * 引数で指定された通知情報の値を、各項目に設定します。
	 * 
	 * @param notify 設定値として用いる通知情報
	 */
    protected void setInputData(NotifyInfo notify) {

        this.inputData = notify;
        
        // 通知基本情報
        super.m_notifyBasic.setInputData(notify);
        
        // 抑制情報
        super.m_notifyInhibition.setInputData(notify);
      
        // ジョブ実行情報
        if (notify.getNotifyType() != null && notify.getNotifyType().intValue() == this.m_notifyType.intValue() && notify.getNotifyInfoDetail() != null) {
        	ArrayList<NotifyInfoDetail> list = notify.getNotifyInfoDetail();
        	if (list != null) {
        		for (int index=0; index<list.size(); index++){
        			NotifyJobInfo job = (NotifyJobInfo)list.get(index);
        			if(job != null){
        				// 通知の場合
        				if(PriorityConstant.TYPE_INFO == job.getPriority()){
        					this.setInputDataForJob(job, m_checkJobRunInfo, m_textJobIdInfo, m_checkInhibitionInfo, m_comboFailurePriorityInfo);
        					// 通知のみジョブ通知の共通部分の設定を行う
        					if (job.getJobExecFacility() != null) {
        						this.m_facilityId = job.getJobExecFacility();
        						this.m_textScope.setText(job.getJobExecScope());
        					}
        					if (job.getJobExecFacilityFlg() != null && job.getJobExecFacilityFlg() == ExecFacilityConstant.TYPE_GENERATION) {
        						this.m_radioGenerationNodeValue.setSelection(true);
        						this.m_scopeSelect.setEnabled(false);
        					}
        					else {
        						this.m_radioFixedValue.setSelection(true);
        						this.m_scopeSelect.setEnabled(true);
        					}
        				}
        				// 警告の場合
        				else if(PriorityConstant.TYPE_WARNING == job.getPriority()){
        					this.setInputDataForJob(job, m_checkJobRunWarning, m_textJobIdWarning, m_checkInhibitionWarning, m_comboFailurePriorityWarning);
                    	}
        				// 危険の場合
        				else if(PriorityConstant.TYPE_CRITICAL == job.getPriority()){
        					this.setInputDataForJob(job, m_checkJobRunCritical, m_textJobIdCritical, m_checkInhibitionCritical, m_comboFailurePriorityCritical);
        				}
        				// 不明の場合
        				else if(PriorityConstant.TYPE_UNKNOWN == job.getPriority()){
        					this.setInputDataForJob(job, m_checkJobRunUnknown, m_textJobIdUnknown, m_checkInhibitionUnknown, m_comboFailurePriorityUnknown);
        				}
        			}
        		}
        	}
        }
        // 新規追加の場合
        else {
        	this.m_radioGenerationNodeValue.setSelection(true);
        	this.m_scopeSelect.setEnabled(false);
        }
        
        // 有効／無効
        if (notify.getValidFlg() != null && notify.getValidFlg() == ValidConstant.TYPE_INVALID) {
            this.m_confirmValid.setSelection(false);
        }
        else {
        	this.m_confirmValid.setSelection(true);
        }
        
        // 入力制御
        if(this.referenceFlg){
        	this.setEnabled(false);
        }
        else{
        	this.setEnabled(true);
        }
        
    }

    /**
	 * 引数で指定された通知ジョブ情報の値を、各項目に設定します。
	 * 
	 * @param info 設定値として用いる通知ジョブ情報
	 * @param checkJobRun 通知チェックボックス
	 * @param textJobId ジョブIDテキストボックス
	 * @param checkInhibition 抑制チェックボックス
	 * @param comboFailurePriority 呼出失敗時の重要度
	 */
    private void setInputDataForJob(NotifyJobInfo info, 
    		Button checkJobRun, 
    		Text textJobId,
    		Button checkInhibition, 
    		Combo comboFailurePriority
    		) {
    	// ジョブ実行
        if (info.getValidFlg() != null && info.getValidFlg().intValue() == ValidConstant.TYPE_VALID) {
        	checkJobRun.setSelection(true);
        } 
        // ジョブID
        if (info.getJobId() != null) {
        	textJobId.setText(info.getJobId());
        }
        // 抑制
        if (info.getInhibitionFlg() != null && info.getInhibitionFlg().intValue() == ValidConstant.TYPE_VALID) {
        	checkInhibition.setSelection(true);
        } 
        // ジョブ失敗時の重要度
        if (info.getJobFailurePriority() != null) {
        	comboFailurePriority.setText(PriorityConstant.typeToString(info.getJobFailurePriority()));
        }
    }
    
	/**
	 * 入力値を設定した通知情報を返します。<BR>
	 * 入力値チェックを行い、不正な場合は<code>null</code>を返します。
	 * 
	 * @return 通知情報
	 * 
	 * @see #createInputDataForJob(ArrayList, int, Button, Text, Button, Combo)
	 * @see #setValidateResult(String, String)
	 */
    protected NotifyInfo createInputData() {
        NotifyInfo info = new NotifyInfo();

        // 通知タイプの設定
    	info.setNotifyType(m_notifyType);
        
        // 通知基本情報
        validateResult = super.m_notifyBasic.createInputData(info, this.notifyId);
        if(validateResult != null){
        	return null;
        }
        
        // 抑制情報
        validateResult = super.m_notifyInhibition.createInputData(info);
        if(validateResult != null){
        	return null;
        }
        
        //　ジョブ情報
        // 各重要度で１つも通知が選択されていない場合はエラー
    	if(!this.m_checkJobRunInfo.getSelection() && 
    			!this.m_checkJobRunWarning.getSelection() && 
    			!this.m_checkJobRunCritical.getSelection() && 
    			!this.m_checkJobRunUnknown.getSelection()){
        	this.setValidateResult(Messages.getString("message.hinemos.1"),Messages.getString("message.notify.13"));
                return null;
    	}
    	
        ArrayList<NotifyInfoDetail> list = new ArrayList<NotifyInfoDetail>();
                
        list = this.createInputDataForJob(list, PriorityConstant.TYPE_INFO,			this.m_checkJobRunInfo,			this.m_textJobIdInfo,			this.m_checkInhibitionInfo,			this.m_comboFailurePriorityInfo);
        list = this.createInputDataForJob(list, PriorityConstant.TYPE_WARNING,		this.m_checkJobRunWarning,		this.m_textJobIdWarning,		this.m_checkInhibitionWarning,		this.m_comboFailurePriorityWarning);
        list = this.createInputDataForJob(list, PriorityConstant.TYPE_CRITICAL,		this.m_checkJobRunCritical,		this.m_textJobIdCritical,		this.m_checkInhibitionCritical,		this.m_comboFailurePriorityCritical);
        list = this.createInputDataForJob(list, PriorityConstant.TYPE_UNKNOWN,		this.m_checkJobRunUnknown,		this.m_textJobIdUnknown,		this.m_checkInhibitionUnknown,		this.m_comboFailurePriorityUnknown);
        
        if(list == null){
        	return null;
        }
        info.setNotifyInfoDetail(list);
        
        // 有効/無効
        if (this.m_confirmValid.getSelection()) {
        	info.setValidFlg(ValidConstant.TYPE_VALID);
        } else {
        	info.setValidFlg(ValidConstant.TYPE_INVALID);
        }
        
        return info;
    }

    /**
	 * 入力値を設定した通知ジョブ情報を返します。<BR>
	 * 入力値チェックを行い、不正な場合は<code>null</code>を返します。
	 * 
	 * @return 通知ジョブ情報
	 * 
	 * @see #setValidateResult(String, String)
	 */
    protected ArrayList<NotifyInfoDetail> createInputDataForJob(ArrayList<NotifyInfoDetail> list,
    		int priority,
    		Button checkJobRun, 
    		Text textJobId,
    		Button checkInhibition, 
    		Combo comboFailurePriority
    		) {
 
    	if(list == null)
    		return null;
    	
    	NotifyJobInfo job = new NotifyJobInfo();
    	
    	// 重要度
    	job.setPriority(priority);
    	
    	//　実行
        if (checkJobRun.getSelection()) {
        	job.setValidFlg(new Integer(ValidConstant.TYPE_VALID));
        } else {
        	job.setValidFlg(new Integer(ValidConstant.TYPE_INVALID));
        }

        // jobId
        if (textJobId.getText() != null
        		&& !"".equals(textJobId.getText().trim())) {
        	job.setJobId(textJobId.getText());
        }
        else if (checkJobRun.getSelection()) {
        	String[] args = { "(" + PriorityConstant.typeToString(priority) + ")"};
			this.setValidateResult(Messages.getString("message.hinemos.1"),Messages.getString("message.notify.20", args));
			return null;
        }
        
        // 抑制
        if (checkInhibition.getSelection()) {
        	// 抑制がチェックされているが、抑制条件が指定されていない場合
        	if (checkJobRun.getSelection() && this.m_notifyInhibition.m_radioNoInhibition.getSelection()) {
        		this.setValidateResult(Messages.getString("message.hinemos.1"),Messages.getString("message.notify.16"));
                return null;
        	}
        	job.setInhibitionFlg(new Integer(ValidConstant.TYPE_VALID));
        } else {
        	job.setInhibitionFlg(new Integer(ValidConstant.TYPE_INVALID));
        }
        
        //　呼出失敗時
        if (comboFailurePriority.getText() != null
        		&& !"".equals(comboFailurePriority.getText())) {
        	job.setJobFailurePriority(PriorityConstant.stringToType(comboFailurePriority.getText()));
        }

        // 共通部分登録
        // 実行ファシリティID
        if (this.m_textScope.getText() != null
        		&& !"".equals(this.m_textScope.getText())) {
        	job.setJobExecFacility(this.m_facilityId);
        	job.setJobExecScope(this.m_textScope.getText());
        }
        // 実行ファシリティ
        if (this.m_radioGenerationNodeValue.getSelection()) {
        	job.setJobExecFacilityFlg(ExecFacilityConstant.TYPE_GENERATION);
        }
        else if (this.m_radioFixedValue.getSelection()){
        	job.setJobExecFacilityFlg(ExecFacilityConstant.TYPE_FIX);
        }
        
        list.add(job);
        return list;
    }
    
	/**
	 * 入力値チェックをします。
	 * 
	 * @return 検証結果
	 */
    protected ValidateResult validate() {
        // 入力値生成
        this.inputData = this.createInputData();

        if (this.inputData != null) {
            return super.validate();
        } else {
            return validateResult;
        }
    }
    
    /**
     * 入力値をマネージャに登録します。
     * 
     * @return true：正常、false：異常
     * 
     * @see com.clustercontrol.dialog.CommonDialog#action()
     */
    protected boolean action() {
    	boolean result = false;
    	
    	NotifyInfo info = this.getInputData();
    	if(info != null){
            if(this.notifyId == null){
            	// 作成の場合
            	result = new AddNotify().add(info);
            }
            else{
            	// 変更の場合
            	result = new ModifyNotify().modify(info);
            }
    	}
    	
    	return result;
    }

	/**
	 * ＯＫボタンのテキストを返します。
	 * 
	 * @return ＯＫボタンのテキスト
	 */
    protected String getOkButtonText() {
        return Messages.getString("ok");
    }

	/**
	 * キャンセルボタンのテキストを返します。
	 * 
	 * @return キャンセルボタンのテキスト
	 */
    protected String getCancelButtonText() {
        return Messages.getString("cancel");
    }
    
	/**
	 * コンポジットの選択可/不可を設定します。
	 * 
	 * @param enable 選択可の場合、<code> true </code>
	 */
    protected void setEnabled(boolean enable) {

    	super.m_notifyBasic.setEnabled(enable);
    	super.m_notifyInhibition.setEnabled(enable);
    	
    	this.setEnabledForJob(enable);
    }    
    
	/**
	 * ジョブ通知の通知関連コンポジットの選択可/不可を設定します。
	 * 
	 * @param enable 選択可の場合、<code> true </code>
	 * 
	 * @see #setEnabledForJobs(boolean, Button, Text, Button, Button, Combo)
	 */
    private void setEnabledForJob(boolean enable) {

    	// 通知関連
        this.setEnabledForJobs(enable, this.m_checkJobRunInfo, 		this.m_textJobIdInfo, 		this.m_buttonReferInfo, 	this.m_checkInhibitionInfo, 	this.m_comboFailurePriorityInfo);
        this.setEnabledForJobs(enable, this.m_checkJobRunWarning, 	this.m_textJobIdWarning, 	this.m_buttonReferWarning, 	this.m_checkInhibitionWarning, 	this.m_comboFailurePriorityWarning);
        this.setEnabledForJobs(enable, this.m_checkJobRunCritical, 	this.m_textJobIdCritical, 	this.m_buttonReferCritical, this.m_checkInhibitionCritical, this.m_comboFailurePriorityCritical);
        this.setEnabledForJobs(enable, this.m_checkJobRunUnknown, 	this.m_textJobIdUnknown, 	this.m_buttonReferUnknown, 	this.m_checkInhibitionUnknown, 	this.m_comboFailurePriorityUnknown);
    } 
    
    /**
	 * ジョブ通知のコンポジットの選択可/不可を設定します。
	 *
	 * @param enable 選択可の場合、<code> true </code>
	 * @param checkJobRun ジョブ通知チェックボックス
	 * @param textJobId ジョブIDテキストボックス
	 * @param buttonRefer 参照ボタン
	 * @param checkInhibition　抑制チェックボックス
	 * @param comboFailure　呼出失敗時コンボボックス
	 */
    private void setEnabledForJobs(boolean enable, 
    		Button checkJobRun,
    		Text textJobId,
    		Button buttonRefer,
    		Button checkInhibition,
    		Combo comboFailure
    		) {

        if (enable) {
        	checkJobRun.setEnabled(true);
        	
        	textJobId.setEnabled(checkJobRun.getSelection());
        	buttonRefer.setEnabled(checkJobRun.getSelection());
        	checkInhibition.setEnabled(checkJobRun.getSelection());
        	comboFailure.setEnabled(checkJobRun.getSelection());
        	
        }
        else{
        	checkJobRun.setEnabled(false);
        	
        	textJobId.setEnabled(false);
        	buttonRefer.setEnabled(false);
        	checkInhibition.setEnabled(false);
        	comboFailure.setEnabled(false);
        }
    }
    
    /**
	 * 引数で指定されたコンポジットの選択可/不可を設定します。
	 * 
	 * @param enabled 選択可の場合、</code> true </code>
	 * @param textJobId 設定対象のジョブIDテキストボックス
	 * @param buttonRefer 設定対象の参照ボタン
	 * @param checkInhibitionFlg 設定対象の通知抑制と連動チェックボックス
	 * @param comboFailurePriority 設定対象の呼出失敗時コンボボックス
	 */
	public void setEnabledForJob(boolean enabled,
			Text textJobId,
			Button buttonRefer,
			Button checkInhibitionFlg,
			Combo comboFailurePriority
	) {
		
		textJobId.setEnabled(enabled);
		buttonRefer.setEnabled(enabled);
		checkInhibitionFlg.setEnabled(enabled);
		comboFailurePriority.setEnabled(enabled);
	}
    
    
	/**
	 * 無効な入力値の情報を設定します。
	 * 
	 * @param id ID
	 * @param message メッセージ
	 */
    protected void setValidateResult(String id, String message) {

        this.validateResult = new ValidateResult();
        this.validateResult.setValid(false);
        this.validateResult.setID(id);
        this.validateResult.setMessage(message);
    }
    
	/**
	 * ボタンを生成します。<BR>
	 * 参照フラグが<code> true </code>の場合は閉じるボタンを生成し、<code> false </code>の場合は、デフォルトのボタンを生成します。
	 * 
	 * @param parent ボタンバーコンポジット
	 * 
	 * @see #createButtonsForButtonBar(Composite)
	 */
    protected void createButtonsForButtonBar(Composite parent) {
    	
        if(!this.referenceFlg){
        	super.createButtonsForButtonBar(parent);
        }
        else{
        	// 閉じるボタン
            this.createButton(parent, IDialogConstants.CANCEL_ID, IDialogConstants.CLOSE_LABEL, false);
        }
    }    
    
	/**
	 * 重要度のラベルを返します。
	 * 
	 * @param parent 親のコンポジット
	 * @param text ラベルに表示するテキスト
	 * @param background ラベルの背景色
	 * @return 生成されたラベル
	 */
    protected Label getLabelPriority(Composite parent,
    		String text,
    		Color background
    		) {
    	
    	// ラベル（重要度）
    	Label label = new Label(parent, SWT.NONE);
    	GridData gridData = new GridData();
    	gridData.horizontalSpan = WIDTH_PRIORITY;
    	gridData.horizontalAlignment = GridData.FILL;
    	gridData.grabExcessHorizontalSpace = true;
    	label.setLayoutData(gridData);
    	label.setText(text + " : ");
    	label.setBackground(background);
        
        return label;
    }
    
	/**
	 * ジョブの実行チェックボックスを返します。
	 * 
	 * @param parent 親のコンポジット
	 * @return 生成されたチェックボックス
	 */
    protected Button getCheckJobRun(Composite parent) {
    	
        // チェックボックス（実行）
    	Button button = new Button(parent, SWT.CHECK);
        GridData gridData = new GridData();
        gridData.horizontalSpan = WIDTH_JOB_RUN;
        gridData.horizontalAlignment = GridData.CENTER;
        gridData.grabExcessHorizontalSpace = true;
        button.setLayoutData(gridData);
        
        return button;
    }
    
	/**
	 * ジョブジョブIDテキストボックスを返します。
	 * 
	 * @param parent 親のコンポジット
	 * @return 生成されたチェックボックス
	 */
    protected Text getTextJobId(Composite parent) {
    	
        // チェックボックス（実行）
    	Text text = new Text(parent, SWT.BORDER | SWT.READ_ONLY);
        GridData gridData = new GridData();
        gridData.horizontalSpan = WIDTH_JOB_ID;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        text.setLayoutData(gridData);
        
        return text;
    }
    
	/**
	 * ジョブの参照ボタンを返します。
	 * 
	 * @param parent 親のコンポジット
	 * @return 生成されたチェックボックス
	 */
    protected Button getButtonJobRefer(Composite parent) {
    	
        // チェックボックス（参照）
    	Button button = new Button(parent, SWT.NONE);
        GridData gridData = new GridData();
        gridData.horizontalSpan = WIDTH_REF_BTN;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        button.setLayoutData(gridData);
        button.setText(Messages.getString("refer"));
        
        return button;
    }
    
	/**
	 * ジョブの抑制チェックボックスを返します。
	 * 
	 * @param parent 親のコンポジット
	 * @return 生成されたチェックボックス
	 */
    protected Button getCheckJobInhibition(Composite parent) {
    	
        // チェックボックス（実行）
    	Button button = new Button(parent, SWT.CHECK);
        GridData gridData = new GridData();
        gridData.horizontalSpan = WIDTH_INHIBITION_FLG;
        gridData.horizontalAlignment = GridData.CENTER;
        gridData.grabExcessHorizontalSpace = true;
        button.setLayoutData(gridData);
        
        return button;
    }
    
	/**
	 * ジョブの重要度のコンボボックスを返します。
	 * 
	 * @param parent 親のコンポジット
	 * @param horizontalSpan コンボボックスのカラム数
	 * @return コンボボックス
	 */
	protected Combo getComboPriority(Composite parent) {
		
		Combo combo = new Combo(parent, SWT.DROP_DOWN | SWT.READ_ONLY);
		GridData gridData = new GridData();
		gridData.horizontalSpan = WIDTH_FAILURE_PRIORITY;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		combo.setLayoutData(gridData);
		combo.add(PriorityConstant.STRING_CRITICAL);
		combo.add(PriorityConstant.STRING_WARNING);
		combo.add(PriorityConstant.STRING_INFO);
		combo.add(PriorityConstant.STRING_UNKNOWN);
		combo.setText(PriorityConstant.STRING_UNKNOWN);
		
		return combo;
	}
        
}