/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.jobmanagement.composite;

import org.eclipse.jface.viewers.TableTreeViewer;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.TableTree;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Table;

import com.clustercontrol.bean.CommonTableTreeItem;
import com.clustercontrol.composite.CommonTableTreeViewer;
import com.clustercontrol.jobmanagement.action.GetRunScheduleDetailList;
import com.clustercontrol.jobmanagement.action.GetRunScheduleDetailListTableDefine;
import com.clustercontrol.jobmanagement.bean.RunScheduleDetailListTableDefine;
import com.clustercontrol.jobmanagement.composite.action.VerticalBarSelectionListener;
import com.clustercontrol.util.Messages;

/**
 * ジョブ[実行予定詳細]ビューの左側用のコンポジットクラスです。
 * 
 * @version 2.0.0
 * @since 2.0.0
 */
public class RunScheduleDetailListComposite extends Composite {
	/** テーブルビューア */
	protected CommonTableTreeViewer m_viewer = null;
    /** ID用ラベル */
    protected Label m_labelId = null;
    /** 件数用ラベル */
    protected Label m_labelCount = null;
    /** スケジュールID */
    protected String m_scheduleId = null;

    /**
     * コンストラクタ
     * 
     * @param parent 親のコンポジット
     * @param style スタイル
     * 
     * @see org.eclipse.swt.SWT
     * @see org.eclipse.swt.widgets.Composite#Composite(Composite parent, int style)
     * @see #initialize()
     */
    public RunScheduleDetailListComposite(Composite parent, int style) {
        super(parent, style);
        initialize();
    }

    /**
     * コンポジットを構築します。
     */
    private void initialize() {
        GridLayout layout = new GridLayout(1, true);
        this.setLayout(layout);
        layout.marginHeight = 0;
        layout.marginWidth = 0;
        
        //ジョブIDラベル作成
        m_labelId = new Label(this, SWT.LEFT);
        GridData gridData = new GridData();
        gridData.horizontalAlignment = GridData.FILL;
        m_labelId.setLayoutData(gridData);
        
        //予定詳細一覧テーブル作成
        TableTree tree = new TableTree(this, SWT.H_SCROLL | SWT.V_SCROLL
                | SWT.FULL_SELECTION | SWT.SINGLE | SWT.BORDER);

        Table table = tree.getTable();
        table.setHeaderVisible(true);
        table.setLinesVisible(true);
        table.getHorizontalBar().setVisible(true);
        table.getHorizontalBar().setEnabled(true);
        table.getVerticalBar().setVisible(true);
        table.getVerticalBar().setEnabled(true);
        
        gridData = new GridData();
        gridData.horizontalAlignment = GridData.FILL;
        gridData.verticalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        gridData.grabExcessVerticalSpace = true;
        tree.setLayoutData(gridData);
        
        m_labelCount = new Label(this, SWT.RIGHT);
        gridData = new GridData();
        gridData.horizontalAlignment = GridData.FILL;
        gridData.verticalAlignment = GridData.FILL;
        m_labelCount.setLayoutData(gridData);

        m_viewer = new CommonTableTreeViewer(tree);
        m_viewer.createTableColumn(GetRunScheduleDetailListTableDefine.get(),
        		RunScheduleDetailListTableDefine.SORT_COLUMN_INDEX,
        		RunScheduleDetailListTableDefine.SORT_ORDER);
        
        m_viewer.getTableTree().getTable().getVerticalBar().addSelectionListener(
        		new VerticalBarSelectionListener(this));
    }

    /**
     * テーブルビューアーを更新します。<BR>
     * 引数にしてされたスケジュールIDに該当する基準日からの実行予定詳細一覧情報を取得し、テーブルビューアーにセットします。
     * <p>
	 * <ol>
	 * <li>引数にしてされたスケジュールIDに該当する基準日からの実行予定詳細一覧情報を取得します。</li>
	 * <li>テーブルビューアーに実行予定詳細一覧情報をセットします。</li>
	 * </ol>
	 * 
     * @see com.clustercontrol.jobmanagement.action.GetRunScheduleDetailList#get(String)
     */
    public void update(String scheduleId) {
    	
        //予定詳細一覧情報取得
        GetRunScheduleDetailList getList = new GetRunScheduleDetailList();
        CommonTableTreeItem item = null;
        if (scheduleId != null && scheduleId.length() > 0) {
            item = getList.get(scheduleId);
        }

        m_viewer.setInput(item);
        m_viewer.expandAll();

//        Object[] args = { new Integer(info.size()) };
//        m_labelCount.setText(Messages.getString("records", args));
        
        m_scheduleId = scheduleId;

        //スケジュールIDを表示
        if (m_scheduleId != null) {
        	m_labelId.setText(Messages.getString("schedule.id") + " : " + m_scheduleId);
        } else {
        	m_labelId.setText(Messages.getString("schedule.id") + " : ");
        }
    }
    
    /**
     * このコンポジットが利用するテーブルツリービューアを返します。
     * 
     * @return テーブルツリービューア
     */
    public TableTreeViewer getTableTreeViewer() {
        return m_viewer;
    }

    /**
     * このコンポジットが利用するテーブルを返します。
     * 
     * @return テーブル
     */
    public Table getTable() {
        return m_viewer.getTableTree().getTable();
    }

    /**
     * このコンポジットが利用するテーブルツリーを返します。
     * 
     * @return テーブルツリー
     */
    public TableTree getTableTree() {
        return m_viewer.getTableTree();
    }
    
	/**
	 * スケジュールIDを返します。
	 * 
	 * @return スケジュールID
	 */
	public String getScheduleId() {
		return m_scheduleId;
	}
	
	/**
	 * スケジュールIDを設定します。
	 * 
	 * @param scheduleId スケジュールID
	 */
	public void setScheduleId(String scheduleId) {
		m_scheduleId = scheduleId;
	}
}