/*
 
 Copyright (C) 2006 NTT DATA Corporation
 
 This program is free software; you can redistribute it and/or
 Modify it under the terms of the GNU General Public License
 as published by the Free Software Foundation, version 2.
 
 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.
 
 */

package com.clustercontrol.performance.util.code;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

import javax.ejb.FinderException;
import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.bean.FacilityConstant;
import com.clustercontrol.bean.YesNoConstant;
import com.clustercontrol.performance.bean.CollectMethodConstant;
import com.clustercontrol.performance.monitor.ejb.entity.CollectorCategoryMstData;
import com.clustercontrol.performance.monitor.ejb.entity.CollectorCategoryMstLocal;
import com.clustercontrol.performance.monitor.ejb.entity.CollectorCategoryMstUtil;
import com.clustercontrol.performance.monitor.ejb.entity.CollectorDeviceInfoData;
import com.clustercontrol.performance.monitor.ejb.entity.CollectorDeviceInfoLocal;
import com.clustercontrol.performance.monitor.ejb.entity.CollectorDeviceInfoUtil;
import com.clustercontrol.performance.monitor.ejb.entity.CollectorItemCalcMethodMstLocal;
import com.clustercontrol.performance.monitor.ejb.entity.CollectorItemCalcMethodMstUtil;
import com.clustercontrol.performance.monitor.ejb.entity.CollectorItemCodeMstData;
import com.clustercontrol.performance.monitor.ejb.entity.CollectorItemCodeMstLocal;
import com.clustercontrol.performance.monitor.ejb.entity.CollectorItemCodeMstUtil;
import com.clustercontrol.performance.monitor.ejb.entity.CollectorScopeSnapLocal;
import com.clustercontrol.performance.monitor.ejb.entity.CollectorScopeSnapUtil;
import com.clustercontrol.performance.util.Repository;

/**
 * 収集項目コードの情報を生成するファクトリクラス
 * 
 * @version 3.0
 * @since 1.0
 */
public class CollectorItemCodeTable {
	private static Log m_log = LogFactory.getLog( CollectorItemCodeTable.class );
	private static final HashMap<String, CollectorItemTreeItem> m_codeTable;
	/**
	 * コンストラクター
	 *
	 */
	public CollectorItemCodeTable() {
	}
	
	static {
		m_codeTable = new HashMap<String, CollectorItemTreeItem>(); // 収集項目コードがキー

		// カテゴリコードからカテゴリ情報を参照するためのテーブル
		HashMap<String, CollectorItemTreeItem> categoryTable = 
			new HashMap<String, CollectorItemTreeItem>();
		
		// カテゴリ情報の読み込み
		Collection<CollectorCategoryMstLocal> cate = null;
		try {
			cate = CollectorCategoryMstUtil.getLocalHome().findAll();
		} catch (FinderException e) {
			m_log.error(e.getMessage(), e);
		} catch (NamingException e) {
			m_log.error(e.getMessage(), e);
		}
		
		Iterator<CollectorCategoryMstLocal> cateItr = cate.iterator();
		while(cateItr.hasNext()){
			CollectorCategoryMstLocal bean = cateItr.next();
			
			CollectorCategoryMstData category  = 
				new CollectorCategoryMstData(bean.getCategoryCode(), bean.getCategoryName());
			
			CollectorItemTreeItem categoryItem = 
				new CollectorItemTreeItem(null, category, null, null);  // 親の要素はないためnull
			
			// カテゴリコードにマッピングするようにカテゴリ情報を登録
			categoryTable.put(bean.getCategoryCode(), categoryItem);
		}

		
		
		try {			
			// 収集項目コードの読み込み
			Collection<CollectorItemCodeMstLocal> ctItemCodeMst = CollectorItemCodeMstUtil.getLocalHome().findAll();			
			
			Iterator<CollectorItemCodeMstLocal> itrItemCode = ctItemCodeMst.iterator();
			
			while(itrItemCode.hasNext()) {
				CollectorItemCodeMstLocal codeBean = itrItemCode.next();
				String itemCode = codeBean.getItemCode();
				
				if(itemCode == null){
					continue;
				}
				
				if(itemCode != null && codeBean.getCategoryCode() != null
						&& codeBean.getDeviceSupport() != null && codeBean.getGraphRange() != null){
					
					// 収集項目コードマスタ情報のDTOを作成
					CollectorItemCodeMstData codeData = new CollectorItemCodeMstData(
							codeBean.getItemCode(),
							codeBean.getCategoryCode(),
							codeBean.getParentItemCode(),
							codeBean.getItemName(),
							codeBean.getMeasure(),
							codeBean.getDeviceSupport(),
							codeBean.getDeviceType(),
							codeBean.getGraphRange()
					);
				
					// カテゴリ名を調べます
					CollectorItemTreeItem categoryTreeItem = categoryTable.get(codeBean.getCategoryCode());
				
					// 親のオブジェクトを取得（存在しない場合はnull）
					CollectorItemTreeItem parentItem = 
						(CollectorItemTreeItem) m_codeTable.get(codeBean.getParentItemCode());
					
					// 親のコードが存在しない場合はカテゴリの直下の要素とする
					if(parentItem == null){
						parentItem = categoryTreeItem;
					}
					
					CollectorItemTreeItem ctItem = 
						new CollectorItemTreeItem(parentItem, codeData, CollectMethodConstant.SNMP, null);
					
					// 収集項目コードをキーとして登録
					m_codeTable.put(itemCode, ctItem);
				}
			}
			
		} catch (Exception e) {
			m_codeTable.clear();
			// エラー処理
			e.printStackTrace();
		}
		
	}
	
	
	
	/**
	 * 与えられたコードのセットで収集項目ツリーを生成する
	 * @param codeSet 有効なコードのセット
	 * @param disableCodeSet 無効なコードのセット（現バージョンでは未使用）
	 * @param deviceSet デバイス情報のセット
	 * @return
	 */
	public static CollectorItemTreeItem[] createTree(
			Set<CollectorItemCodeMstData> codeSet, 
			Set<CollectorItemCodeMstData> disableCodeSet,
			String collectMethod,
			Set<CollectorDeviceInfoData> deviceSet){
		ArrayList<CollectorItemTreeItem> itemTree = new ArrayList<CollectorItemTreeItem>();
		
		// 参照しやすいようにカテゴリコードをマップで管理
		HashMap<String, CollectorItemTreeItem> categoryMap = 
			new HashMap<String, CollectorItemTreeItem>();

		// カテゴリ情報の読み込み
		Collection<CollectorCategoryMstLocal> cate = null;
		try {
			cate = CollectorCategoryMstUtil.getLocalHome().findAll();
		} catch (FinderException e) {
			m_log.error(e.getMessage(), e);
		} catch (NamingException e) {
			m_log.error(e.getMessage(), e);
		}
		
		Iterator<CollectorCategoryMstLocal> cateItr = cate.iterator();
		while(cateItr.hasNext()){
			CollectorCategoryMstLocal bean = cateItr.next();
			
			CollectorCategoryMstData category  = 
				new CollectorCategoryMstData(bean.getCategoryCode(), bean.getCategoryName());
			
			CollectorItemTreeItem categoryItem = 
				new CollectorItemTreeItem(null, category, null, null);  // 親の要素はないためnull
			 
			// カテゴリコードにマッピングするようにカテゴリ情報を登録
			categoryMap.put(bean.getCategoryCode(), categoryItem);
			 
			// リターン用のリストに追加
			itemTree.add(categoryItem);
		}

		// カテゴリコード順にソート
		CollectorItemTreeItem[] list = itemTree.toArray(new CollectorItemTreeItem[0]);
        Arrays.sort(list, new Comparator<CollectorItemTreeItem>(){
            public int compare(CollectorItemTreeItem o1, CollectorItemTreeItem o2) {
                //比較
            	return o1.getCategory().getCategoryCode().compareTo(o2.getCategory().getCategoryCode());
            }
        });
		
		// 処理済の収集項目コードのマップ
		//　キー ： 収集項目コードとデバイス名をセットにして管理するクラスのインスタンス
		// 値　：　収集項目要素
		HashMap<ItemCodeAndDeviceName, CollectorItemTreeItem> createdCodeItemMap = 
			new HashMap<ItemCodeAndDeviceName, CollectorItemTreeItem>();
		
		// 有効な収集項目コードを設定
		Iterator<CollectorItemCodeMstData> itr = codeSet.iterator();
		
		// 生成済みのコードのセット（重複して生成するのを防ぐ）
		HashSet<String> createdCodeSet = new HashSet<String>();
		while(itr.hasNext()){
			CollectorItemCodeMstData itemCodeData = itr.next();
			createItemTreeItem(createdCodeSet, itemCodeData, codeSet, collectMethod, deviceSet, categoryMap, createdCodeItemMap);
		}
		
		// 配列を返す
//		return itemTree.toArray(new CollectorItemTreeItem[0]);
		return list;
	}
	
	/**
	 * 
	 * @param createdCodeSet 既に生成されたコードを保持（重複して生成するのを防ぐため）
	 * @param itemCodeData
	 * @param codeSet
	 * @param collectMethod
	 * @param deviceSet
	 * @param categoryMap
	 * @param createdCodeMap 既に要素のインスタンスが生成されたもののマップ
	 */
	private static void createItemTreeItem(
			HashSet<String> createdCodeSet,
			final CollectorItemCodeMstData itemCodeData,
			final Set<CollectorItemCodeMstData> codeSet, 
			final String collectMethod,
			final Set<CollectorDeviceInfoData> deviceSet,
			final HashMap<String, CollectorItemTreeItem> categoryMap,
			HashMap<ItemCodeAndDeviceName, CollectorItemTreeItem> createdCodeMap){
		// 既に生成済みのコードの場合は抜ける
		if(createdCodeSet.contains(itemCodeData.getItemCode())){
			return;
		}
		
		// 親の項目コード
		String parentItemCode = itemCodeData.getParentItemCode();
		
		// 親のコードがない場合は再起呼び出し終了
		if(parentItemCode == null){
			// カテゴリ直下のコードとして生成するため、親要素はカテゴリ要素とする
			CollectorItemTreeItem parentItem = categoryMap.get(itemCodeData.getCategoryCode());
			
			// カテゴリが存在しない場合
			if(parentItem == null){
				return;
			}
			
			// デバイスサポートの有無をチェック
			if(itemCodeData.getDeviceSupport() == YesNoConstant.TYPE_NO){
				CollectorItemTreeItem item = new CollectorItemTreeItem(parentItem, itemCodeData, collectMethod, null);
				// 作成済みマップに登録する
				createdCodeMap.put(new ItemCodeAndDeviceName(itemCodeData.getItemCode(), null), item);
			} else {
				// デバイスのリスト全てをチェックする
				Iterator<CollectorDeviceInfoData> dItr = deviceSet.iterator();
				while(dItr.hasNext()){
					CollectorDeviceInfoData dData = dItr.next();
					// 収集項目で定義されているデバイスタイプと一致する場合は要素として追加
					if(itemCodeData.getDeviceType().equals(dData.getDeviceType())){
						CollectorItemTreeItem item = new CollectorItemTreeItem(parentItem, itemCodeData, collectMethod, dData);
						// 作成済みマップに登録する
						createdCodeMap.put(new ItemCodeAndDeviceName(itemCodeData.getItemCode(), dData.getDeviceName()), item);
					}
				}
			}
			// 復帰する
		} else {
			// 親のコードがある場合
			// デバイスサポートの有無をチェック
			if(itemCodeData.getDeviceSupport() == YesNoConstant.TYPE_NO){
				// 生成済みの要素から親の要素を取得する
				CollectorItemTreeItem parentItem = 
					createdCodeMap.get(new ItemCodeAndDeviceName(itemCodeData.getParentItemCode(), null));
				
				// まだ生成されていない場合
				if(parentItem == null){
					// 全ての収集項目データセットから親の項目を特定する
					Iterator<CollectorItemCodeMstData> itr = codeSet.iterator();
					while(itr.hasNext()){
						CollectorItemCodeMstData data = itr.next();
						// コードが一致するものが見つかった場合はその要素を生成する（再起呼び出し）
						if(parentItemCode.equals(data.getItemCode())){
							createItemTreeItem(createdCodeSet, data, codeSet, collectMethod, deviceSet, categoryMap, createdCodeMap);
						}
					}
					
					// 生成した親の要素を取得する
					parentItem = createdCodeMap.get(new ItemCodeAndDeviceName(itemCodeData.getParentItemCode(), null));
					
					if(parentItem == null){
						// 親が存在しないため、カテゴリを親要素とする
						parentItem = categoryMap.get(itemCodeData.getCategoryCode());
					}
				}
				
				CollectorItemTreeItem item = new CollectorItemTreeItem(parentItem, itemCodeData, collectMethod, null);
				// 作成済みマップに登録する
				createdCodeMap.put(new ItemCodeAndDeviceName(itemCodeData.getItemCode(), null), item);				
			} else {
				// デバイスをサポートしている場合
				// デバイスのリスト全てをチェックする
				Iterator<CollectorDeviceInfoData> dItr = deviceSet.iterator();
				while(dItr.hasNext()){
					CollectorDeviceInfoData dData = dItr.next();
					// 収集項目で定義されているデバイスタイプと一致する場合は要素として追加
					if(itemCodeData.getDeviceType().equals(dData.getDeviceType())){
						// 生成済みの要素から親の要素を取得する
						CollectorItemTreeItem parentItem = 
							createdCodeMap.get(new ItemCodeAndDeviceName(itemCodeData.getParentItemCode(), dData.getDeviceName()));
						
						// まだ生成されていない場合
						if(parentItem == null){
							// 全ての収集項目データセットから親の項目を特定する
							Iterator<CollectorItemCodeMstData> itr = codeSet.iterator();
							while(itr.hasNext()){
								CollectorItemCodeMstData data = itr.next();
								// コードが一致するものが見つかった場合はその要素を生成する（再起呼び出し）
								if(parentItemCode.equals(data.getItemCode())){
									createItemTreeItem(createdCodeSet, data, codeSet, collectMethod, deviceSet, categoryMap, createdCodeMap);
								}
							}
							
							// 生成した親の要素を取得する
							parentItem = createdCodeMap.get(new ItemCodeAndDeviceName(itemCodeData.getParentItemCode(), dData.getDeviceName()));
							
							if(parentItem == null){
								// 親が存在しないため、カテゴリを親要素とする
								parentItem = categoryMap.get(itemCodeData.getCategoryCode());
							}
						}
						
						CollectorItemTreeItem item = new CollectorItemTreeItem(parentItem, itemCodeData, collectMethod, dData);
						// 作成済みマップに登録する
						createdCodeMap.put(new ItemCodeAndDeviceName(itemCodeData.getItemCode(), dData.getDeviceName()), item);				
					}
				}
			}
		}
		createdCodeSet.add(itemCodeData.getItemCode());
	}

	private static class ItemCodeAndDeviceName{
		private String itemCode;
		private String deviceName;
		
		public ItemCodeAndDeviceName(String itemCode, String deviceName){
			this.itemCode = itemCode;
			this.deviceName = deviceName;
		}
		
		public boolean equals(Object other) {		
			if (other instanceof ItemCodeAndDeviceName) {
				ItemCodeAndDeviceName info = (ItemCodeAndDeviceName)other;
				if (this.itemCode == null && this.deviceName == null){
					if (info.itemCode == null && info.deviceName == null){
						return true;
					}
				} else if (this.itemCode == null && this.deviceName != null){
					if (info.itemCode == null && this.deviceName.equals(info.deviceName)){
						return true;
					}
				} else if (this.itemCode != null && this.deviceName == null){
					if (this.itemCode.equals(info.itemCode) && info.deviceName == null){
						return true;
					}
				} else {
					if (this.itemCode.equals(info.itemCode)){
						return this.deviceName.equals(info.deviceName);
					}
				}
				return false;
			} else {
				return false;
			}
		}

		public int hashCode() {
			int result = 17;

			result = 37 * result + ((this.itemCode != null) ? this.itemCode.hashCode() : 0);

			result = 37 * result + ((this.deviceName != null) ? this.deviceName.hashCode() : 0);

			return result;
		}
	}	
	
	public static class DeviceTypeAndDeviceName{
		private String deviceType;
		private String deviceName;
		
		public DeviceTypeAndDeviceName(String deviceType, String deviceName){
			this.deviceType = deviceType;
			if(deviceName != null){
				this.deviceName = deviceName;
			}
			else {
				this.deviceName = "";
			}
		}
		
		public boolean equals(Object other) {
			if (other instanceof DeviceTypeAndDeviceName) {
				DeviceTypeAndDeviceName info = (DeviceTypeAndDeviceName)other;
				
				if (this.deviceType == null && this.deviceName == null){
					if (info.deviceType == null && info.deviceName == null){
						return true;
					}
				} else if (this.deviceType == null && this.deviceName != null){
					if (info.deviceType == null && this.deviceName.equals(info.deviceName)){
						return true;
					}
				} else if (this.deviceType != null && this.deviceName == null){
					if (this.deviceType.equals(info.deviceType) && info.deviceName == null){
						return true;
					}
				} else {
					if (this.deviceType.equals(info.deviceType)){
						return this.deviceName.equals(info.deviceName);
					}
				}
				return false;
			} else {
				return false;
			}
		}

		public int hashCode() {
			int result = 17;

			result = 37 * result + ((this.deviceType != null) ? this.deviceType.hashCode() : 0);

			result = 37 * result + ((this.deviceName != null) ? this.deviceName.hashCode() : 0);

			return result;
		}

		public String getDeviceType() {
			return deviceType;
		}

		public String getDeviceName() {
			return deviceName;
		}
	}	
	
	/**
	 * 指定の収集IDの収集のデバイス情報をDBから取得し収集項目ツリーを生成して返します。
	 * 指定の収集IDがnullの場合はリポジトリからデバイス情報を取得する。
	 * 
	 * @param facilityId ファシリティID
	 * @param collectorId 収集ID
	 * @return
	 */
	public static CollectorItemTreeItem[] getCollectorTree(String facilityId, String collectorId){
		return getCollectorTree(facilityId, collectorId, CollectMethodConstant.SNMP);
	}

	private static CollectorItemTreeItem[] getCollectorTree(String facilityId, String collectorId, String collectMethod){
		Set<CollectorDeviceInfoData> deviceSet = new HashSet<CollectorDeviceInfoData>();

		// 全てのノードのプラットフォームIDのセット
		Set<String> platformSet = new HashSet<String>();
		
		// いずれかのノードに含まれる（OR条件）デバイス名の情報
		Set<DeviceTypeAndDeviceName> deviceSetAdd = new HashSet<DeviceTypeAndDeviceName>();
		
		// 全てのノードに含まれる（AND条件）デバイス名の情報
		Set<DeviceTypeAndDeviceName> deviceSetRetain = null;
		
		if(collectorId == null){
			if(facilityId != null){
				// リポジトリからデバイス情報を取得
				Repository repository = new Repository();
				
				// ファシリティがノードの場合
				if(repository.isNode(facilityId)){
					// プラットフォームIDを取得
					String platformId = Repository.getPlatformId(facilityId);
					platformSet.add(platformId);
					
					List<CollectorDeviceInfoData> deviceList = repository.getDeviceList(facilityId);		
					Iterator<CollectorDeviceInfoData> itr = deviceList.iterator();
					
					Set<DeviceTypeAndDeviceName> tmp = new HashSet<DeviceTypeAndDeviceName>();
					while(itr.hasNext()){
						CollectorDeviceInfoData data = itr.next();
						
						tmp.add(new DeviceTypeAndDeviceName(data.getDeviceType(), data.getDeviceName()));
					}
					
					deviceSetAdd.addAll(tmp);
					deviceSetRetain = new HashSet<DeviceTypeAndDeviceName>(tmp);
				}
				// ファシリティがスコープの場合
				else {
					// スコープ配下に存在する全てのノードを取得する
					List<ArrayList<String>> nodeList = repository.getNodeList(facilityId);
					
					// デバイス情報をノード単位で取得
					Iterator<ArrayList<String>> nodeItr = nodeList.iterator();
					
					// スコープ内に存在するノードでループ
					while(nodeItr.hasNext()){
						String nodeFacilityId = (nodeItr.next()).get(0).toString();
						
						// プラットフォームIDを取得
						String platformId = Repository.getPlatformId(nodeFacilityId);
						platformSet.add(platformId);
						
						// デバイス情報を取得
						List<CollectorDeviceInfoData> deviceList = repository.getDeviceList(nodeFacilityId);		
						Iterator<CollectorDeviceInfoData> itr = deviceList.iterator();
						
						// 現在着目しているノードに含まれる全てのデバイスを格納
						Set<DeviceTypeAndDeviceName> tmp = new HashSet<DeviceTypeAndDeviceName>();
						while(itr.hasNext()){
							CollectorDeviceInfoData data = itr.next();
							tmp.add(new DeviceTypeAndDeviceName(data.getDeviceType(), data.getDeviceName()));
						}
						
						// 全て追加する
						deviceSetAdd.addAll(tmp);
						
						// 同じものだけ追加する
						if(deviceSetRetain == null){
							deviceSetRetain = new HashSet<DeviceTypeAndDeviceName>(tmp);
						} else {
							deviceSetRetain.retainAll(tmp);
						}
					}		
				}
			}
		} else {
			// DBからノード情報を取得
			Collection<CollectorScopeSnapLocal> facilitys;
			try {
				facilitys = CollectorScopeSnapUtil.getLocalHome().findByCollectorId(collectorId);
			} catch (FinderException e) {
				m_log.error(e.getMessage(), e);
				return null;
			} catch (NamingException e) {
				m_log.error(e.getMessage(), e);
				return null;
			}
			
			// DBから全てのデバイス情報を取得（複数ノードのデバイスが混在する）
			Collection<CollectorDeviceInfoLocal> col;
			try {
				col = CollectorDeviceInfoUtil.getLocalHome().findByCollectorId(collectorId);
			} catch (FinderException e) {
				m_log.error(e.getMessage(), e);
				return null;
			} catch (NamingException e) {
				m_log.error(e.getMessage(), e);
				return null;
			}
			
			Iterator<CollectorScopeSnapLocal> facilityItr = facilitys.iterator();
			while(facilityItr.hasNext()){
				CollectorScopeSnapLocal bean = facilityItr.next();
				
				// ファシリティ種別がノードの場合
				if(FacilityConstant.TYPE_NODE_STRING.equals(bean.getFacilityType())){
					platformSet.add(bean.getPlatformId());

					// ノード毎のデバイス情報を取得し、
					// どれかに含まれるデバイスセット（deviceSetAdd）、
					// どれにも含まれるデバイスセット（deviceSetRetain）を生成する。
					String targetFacilityId = bean.getFacilityId();

					// デバイス情報をノード単位で取得
					Iterator<CollectorDeviceInfoLocal> nodeItr = col.iterator();

					// ターゲットのノードが保持するデバイスの情報を格納するセット
					Set<DeviceTypeAndDeviceName> tmp = new HashSet<DeviceTypeAndDeviceName>();
					while(nodeItr.hasNext()){
						CollectorDeviceInfoLocal deviceInfo = nodeItr.next();

						// ターゲットのノードのデバイス情報のみ抽出する
						if(targetFacilityId.equals(deviceInfo.getFacilityId())){
							tmp.add(new DeviceTypeAndDeviceName(
									deviceInfo.getDeviceType(), deviceInfo.getDeviceName()));
						}
					}

					// 全て追加する
					deviceSetAdd.addAll(tmp);

					// 同じものだけ追加する
					if(deviceSetRetain == null){
						deviceSetRetain = new HashSet<DeviceTypeAndDeviceName>(tmp);
					} else {
						deviceSetRetain.retainAll(tmp);
					}
				}
			}
		}
		
		// 選択可能な収集項目コード情報のセット
		Set<CollectorItemCodeMstData> codeSet = null;
		
		// 選択不可な収集項目コード情報のセット
		Set<CollectorItemCodeMstData> disableCodeSet = new HashSet<CollectorItemCodeMstData>();

		// 計算方法管理テーブルから指定の収集方法、プラットフォームで収集可能な収集項目コードを取得する
		Collection<CollectorItemCalcMethodMstLocal> calsMethods;
		try {
			calsMethods = CollectorItemCalcMethodMstUtil.getLocalHome().findAll();
		} catch (FinderException e) {
			m_log.error(e.getMessage(), e);
			return null;
		} catch (NamingException e) {
			m_log.error(e.getMessage(), e);
			return null;
		}
		
		Iterator<String> platformItr = platformSet.iterator();
		// プラットフォームごとにループをまわす
		while(platformItr.hasNext()){
			String platformId = platformItr.next();

			// プラットフォームごとに収集可能な収集項目コードのリストを求める
			HashSet<CollectorItemCodeMstData> tmpByPlatform = new HashSet<CollectorItemCodeMstData>();
			
			// 計算方法管理テーブルの要素ごとにループをまわす
			Iterator<CollectorItemCalcMethodMstLocal> itr = calsMethods.iterator();
			while(itr.hasNext()){
				CollectorItemCalcMethodMstLocal bean = itr.next();

				// 収集方法が一致する場合
				if (collectMethod.equals(bean.getCollectMethod())){
					CollectorItemCodeMstLocal codeBean;
					try {
						codeBean = CollectorItemCodeMstUtil.getLocalHome().findByPrimaryKey(bean.getItemCode());
					} catch (FinderException e) {
						m_log.error(e.getMessage(), e);
						return null;
					} catch (NamingException e) {
						m_log.error(e.getMessage(), e);
						return null;
					}

					// 収集項目コードマスタ情報のDTOを作成
					CollectorItemCodeMstData codeData = new CollectorItemCodeMstData(
							codeBean.getItemCode(),
							codeBean.getCategoryCode(),
							codeBean.getParentItemCode(),
							codeBean.getItemName(),
							codeBean.getMeasure(),
							codeBean.getDeviceSupport(),
							codeBean.getDeviceType(),
							codeBean.getGraphRange()
					);

					// 収集方法およびプラットフォームIDが一致する場合
					if(platformId.equals(bean.getPlatformId())){
						tmpByPlatform.add(codeData);
					}
				}
			}
			// プラットフォームごとに収集可能な項目をANDもしくはOR条件でセットに格納する
			// 同じものだけ追加する(AND条件)
			if(codeSet == null){
				codeSet = new HashSet<CollectorItemCodeMstData>(tmpByPlatform);
			} else {
				codeSet.retainAll(tmpByPlatform);
			}
			
			// まずは全て追加する（あとでcodeSetに含まれる要素は削除する）
			disableCodeSet.addAll(tmpByPlatform);
		}

		// デバイスがひとつもない場合はここで空のセットを初期化する
		if(deviceSetRetain == null){
			deviceSetRetain = new HashSet<DeviceTypeAndDeviceName>();
		}
		
		// デバイス情報のセットを生成
		// デバイス情報を設定
		Iterator<DeviceTypeAndDeviceName> deviceItr = deviceSetRetain.iterator();
		while(deviceItr.hasNext()){
			DeviceTypeAndDeviceName info = deviceItr.next();

			// デバイス名とデバイス種別のみのインスタンスを生成
			CollectorDeviceInfoData deviceData = new CollectorDeviceInfoData(
					null,  // 収集ID
					null,  // ファシリティID
					info.getDeviceName(),
					null,
					null,
					info.getDeviceType());
			
			deviceSet.add(deviceData);
		}

		// コードがひとつもない場合はここで空のセットを初期化する
		if(codeSet == null){
			codeSet = new HashSet<CollectorItemCodeMstData>();
		}
		
		// 全ての項目からcodeSetに含まれる要素は削除する
		disableCodeSet.removeAll(codeSet);
		
		return createTree(codeSet, disableCodeSet, collectMethod, deviceSet);
	}
	
	/**
	 * 収集項目コードとデバイス名から収集項目表示用文字列を生成し返します。
	 * 形式 ：　収集名[デバイス名]
	 * 
	 * @param itemCode 収集項目コード
	 * @param deviceName デバイス名
	 * @return 収集項目表示
	 */
	public static String getFullItemName(String itemCode, String deviceName){
		try {
			CollectorItemCodeMstLocal bean = CollectorItemCodeMstUtil.getLocalHome().findByPrimaryKey(itemCode);
			String itemName = bean.getItemName();
			if(bean.getDeviceSupport().intValue() == YesNoConstant.TYPE_YES){
				itemName = itemName + "[" + deviceName + "]";
			}
			return itemName;
		} catch (FinderException e) {
			m_log.error(e.getMessage(), e);
		} catch (NamingException e) {
			m_log.error(e.getMessage(), e);
		}
		return itemCode + " is not found.";
	}
	
	/**
	 * 収集項目コードをキーにした、CollectorItemTreeItemのHashMapを返します。
	 * @return 
	 */
	public static HashMap<String, CollectorItemTreeItem> getItemCodeMap(){
		return m_codeTable;
	}
}