/*

Copyright (C) 2006 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.accesscontrol.dialog;

import java.util.ArrayList;
import java.util.List;
import java.util.Locale;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.TableTree;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;

import com.clustercontrol.accesscontrol.action.CheckPermission;
import com.clustercontrol.accesscontrol.bean.RoleConstant;
import com.clustercontrol.accesscontrol.bean.UserConstant;
import com.clustercontrol.accesscontrol.bean.UserRoleList;
import com.clustercontrol.accesscontrol.util.AccessEndpointWrapper;
import com.clustercontrol.accesscontrol.util.UserPropertyUtil;
import com.clustercontrol.accesscontrol.view.UserListView;
import com.clustercontrol.bean.Property;
import com.clustercontrol.bean.PropertyConstant;
import com.clustercontrol.bean.PropertyTreeItem;
import com.clustercontrol.bean.RequiredFieldColorConstant;
import com.clustercontrol.composite.PropertySheet;
import com.clustercontrol.dialog.CommonDialog;
import com.clustercontrol.dialog.ValidateResult;
import com.clustercontrol.util.Messages;
import com.clustercontrol.util.PropertyUtil;
import com.clustercontrol.ws.access.InvalidRole_Exception;
import com.clustercontrol.ws.access.UserDuplicate_Exception;
import com.clustercontrol.ws.access.UserInfo;

/**
 * アクセス[ユーザの作成・変更]ダイアログクラスです。
 * 
 * @version 4.0.0
 * @since 2.0.0
 */
public class UserDialog extends CommonDialog {

	// ログ
	private static Log m_log = LogFactory.getLog( UserDialog.class );

	/** ユーザID */
	private String uid = ""; //$NON-NLS-1$

	/** 変更用ダイアログ判別フラグ */
	private boolean isModifyDialog = false;

	/** ユーザ用プロパティシート */
	private PropertySheet propertySheet = null;

	/** ユーザ用テーブル **/
	private TableTree table = null;

	private Button buttonRead = null;
	private Button buttonWrite = null;
	private Button buttonExec = null;
	private Button buttonAll = null;
	private Button buttonClear = null;

	private final UserRoleList _userRoleList;
	private boolean permission = false;		// 現在のユーザが変更権限をもつか否か
	
	/** 呼び出し元ビュー **/
	private UserListView view = null;

	/**
	 * コンストラクタ
	 * 
	 * @param parent 親シェル
	 * @param uid ユーザID
	 * @param isModifyDialog 変更用ダイアログとして利用する場合は、true
	 */
	public UserDialog(Shell parent, String uid, boolean isModifyDialog, UserListView view) {
		super(parent);

		this.uid = uid;
		this.isModifyDialog = isModifyDialog;
		this.view = view;

		_userRoleList = UserPropertyUtil.getRoleList();

		// 現在のユーザがアクセス管理機能の設定権限をもつか否かを取得する
		permission = new CheckPermission().check("AccessControlWrite");

	}

	/**
	 * ダイアログの初期サイズを返します。
	 * 
	 * @return 初期サイズ
	 * 
	 * @see org.eclipse.jface.window.Window#getInitialSize()
	 */
	@Override
	protected Point getInitialSize() {
		return new Point(600, 600);
	}

	/**
	 * ダイアログエリアを生成します。
	 * 
	 * @param parent 親コンポジット
	 * 
	 * @see com.clustercontrol.dialog.CommonDialog#customizeDialog(org.eclipse.swt.widgets.Composite)
	 */
	@Override
	protected void customizeDialog(Composite parent) {
		Shell shell = this.getShell();

		// タイトル
		shell.setText(Messages
				.getString("dialog.accesscontrol.user.create.modify")); //$NON-NLS-1$

		// レイアウト
		GridLayout layout = new GridLayout(1, true);
		layout.marginWidth = 10;
		layout.marginHeight = 10;
		layout.numColumns = 8;
		parent.setLayout(layout);

		/*
		 * 属性プロパティシート
		 */

		// ラベル
		Label label = new Label(parent, SWT.LEFT);
		GridData gridData = new GridData();
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		gridData.horizontalSpan = 6;
		label.setLayoutData(gridData);
		label.setText(Messages.getString("attribute") + " : "); //$NON-NLS-1$ //$NON-NLS-2$


		// プロパティシート
		table = new TableTree(parent, SWT.H_SCROLL | SWT.V_SCROLL
				| SWT.FULL_SELECTION | SWT.MULTI | SWT.BORDER);
		gridData = new GridData();
		gridData.horizontalAlignment = GridData.FILL;
		gridData.verticalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		gridData.grabExcessVerticalSpace = true;
		gridData.horizontalSpan = 6;
		table.setLayoutData(gridData);
		if(permission){
			table.addSelectionListener(new SelectionAdapter(){
				@Override
				public void widgetSelected(SelectionEvent e) {
					update();
				}
			});
		}

		this.propertySheet = new PropertySheet(table);

		// プロパティ取得及び設定
		Property property = null;
		int mode = 0;
		if(permission){
			if (this.isModifyDialog) {
				mode = PropertyConstant.MODE_MODIFY;
			} else {
				mode = PropertyConstant.MODE_ADD;
			}
		}else{
			mode = PropertyConstant.MODE_SHOW;
		}

		try {
			UserInfo info = AccessEndpointWrapper.getUserInfo(this.uid);
			property = UserPropertyUtil.dto2property(info, mode, Locale.getDefault());
		} catch (InvalidRole_Exception e) {
			MessageDialog.openInformation(null, Messages.getString("message"),
					Messages.getString("message.accesscontrol.16"));
		} catch (Exception e) {
			m_log.warn("customizeDialog(), " + e.getMessage(), e);
			MessageDialog.openError(
					null,
					Messages.getString("failed"),
					Messages.getString("message.hinemos.failure.unexpected") + ", " + e.getMessage());
		}

		this.propertySheet.setInput(property);
		this.propertySheet.expandAll();

		/*
		 * 操作ボタン用コンポジット
		 */
		Composite composite = new Composite(parent, SWT.NONE);
		layout = new GridLayout(1, true);
		layout.numColumns = 1;
		composite.setLayout(layout);
		gridData = new GridData();
		gridData.horizontalAlignment = GridData.FILL;
		gridData.verticalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		gridData.grabExcessVerticalSpace = true;
		gridData.horizontalSpan = 2;
		composite.setLayoutData(gridData);

		// すべての権限ボタン
		@SuppressWarnings("unused")
		Label dummy = new Label(composite, SWT.NONE);
		this.buttonAll = this.createButton(composite, Messages.getString("UserDialog.all_prerogative")); //$NON-NLS-1$
		this.buttonAll.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				Property prop = getInputData();
				//データを取得します。
				Object[] items =  prop.getChildren();
				for(int j = 0 ; j < items.length; j++){
					//アクセス権のプロパティを探します。
					if(((PropertyTreeItem)items[j]).getID().equals("access")){ //$NON-NLS-1$

						//アクセス権の部分を取り出します。
						items  = (((Property)items[j]).getChildren());
						//各権限がにチェックを入れます。
						for (int k = 0 ; k <items.length ;  k++){
							((Property)items[k]).setValue(true);
						}
						//アクセス権を探す処理をExit
						break;
					}
				}
				propertySheet.setInput(prop);
				propertySheet.expandAll();

			}
		});

		// すべての参照権限ボタン
		dummy = new Label(composite, SWT.NONE);
		this.buttonRead = this
		.createButton(composite, Messages.getString("UserDialog.read_prerogative")); //$NON-NLS-1$
		this.buttonRead.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {

				Property prop = getInputData();
				//データを取得します。(一番上位の項目)
				Object[] items =  prop.getChildren();


				List<String> readRoles = _userRoleList.getReadRoles();


				for(int j = 0 ; j < items.length; j++){
					//アクセス権のプロパティを探します。
					if(((PropertyTreeItem)items[j]).getID().equals("access")){ //$NON-NLS-1$
						//アクセス権の部分を取り出します。
						items  = (((Property)items[j]).getChildren());
						//各権限がにチェックを入れます。
						for (int k = 0 ; k <items.length ;  k++){
							for (int i= 0 ; i < readRoles.size(); i++){
								if(((PropertyTreeItem)items[k]).getID().equals(readRoles.get(i))){
									((Property)items[k]).setValue(true);
									break;
								}
							}
						}
						//アクセス権を探す処理をExit
						break;
					}
				}
				propertySheet.setInput(prop);
				propertySheet.expandAll();


			}
		});

		// すべての設定権限ボタン
		dummy = new Label(composite, SWT.NONE);
		this.buttonWrite = this.createButton(composite, Messages.getString("UserDialog.write_prerogative")); //$NON-NLS-1$
		this.buttonWrite.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {

				Property prop = getInputData();
				//データを取得します。(一番上位の項目)
				Object[] items =  prop.getChildren();

				List<String> writeRoles= _userRoleList.getWriteRoles();

				for(int j = 0 ; j < items.length; j++){
					//アクセス権のプロパティを探します。
					if(((PropertyTreeItem)items[j]).getID().equals("access")){ //$NON-NLS-1$
						//アクセス権の部分を取り出します。
						items  = (((Property)items[j]).getChildren());
						//各権限がにチェックを入れます。
						for (int k = 0 ; k <items.length ;  k++){
							for (int i= 0 ; i < writeRoles.size(); i++){
								if(((PropertyTreeItem)items[k]).getID().equals(writeRoles.get(i))){
									((Property)items[k]).setValue(true);
									break;
								}
							}
						}
						//アクセス権を探す処理をExit
						break;
					}
				}
				propertySheet.setInput(prop);
				propertySheet.expandAll();


			}
		});

		// すべての実行権限ボタン
		dummy = new Label(composite, SWT.NONE);
		this.buttonExec = this.createButton(composite, Messages.getString("UserDialog.exec_prerogative")); //$NON-NLS-1$
		this.buttonExec.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {

				Property prop = getInputData();
				//データを取得します。(一番上位の項目)
				Object[] items =  prop.getChildren();

				List<String> execRoles= _userRoleList.getExecRoles();
				for(int j = 0 ; j < items.length; j++){
					//アクセス権のプロパティを探します。
					if(((PropertyTreeItem)items[j]).getID().equals(UserConstant.ACCESS)){
						//アクセス権の部分を取り出します。
						items  = (((Property)items[j]).getChildren());
						//各権限にチェックを入れます。
						for (int k = 0 ; k <items.length ;  k++){
							for (int i= 0 ; i < execRoles.size(); i++){
								if(((PropertyTreeItem)items[k]).getID().equals(execRoles.get(i))){
									((Property)items[k]).setValue(true);
									break;
								}
							}
						}
						//アクセス権を探す処理をExit
						break;
					}
				}
				propertySheet.setInput(prop);
				propertySheet.expandAll();
			}
		});

		// クリアボタン
		dummy = new Label(composite, SWT.NONE);
		this.buttonClear = this.createButton(composite, Messages.getString("UserDialog.clear")); //$NON-NLS-1$
		this.buttonClear.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				Property prop = getInputData();
				//データを取得します。
				Object[] items =  prop.getChildren();
				for(int j = 0 ; j < items.length; j++){
					//アクセス権のプロパティを探します。
					if(((PropertyTreeItem)items[j]).getID().equals("access")){ //$NON-NLS-1$
						//アクセス権の部分を取り出します。
						items  = (((Property)items[j]).getChildren());
						//各権限がにチェックを入れます。
						for (int k = 0 ; k <items.length ;  k++){
							if(!((PropertyTreeItem)items[k]).getID().equals(RoleConstant.REPOSITORY_READ)){
								((Property)items[k]).setValue(false);
							}
						}
						//アクセス権を探す処理をExit
						break;
					}
				}
				propertySheet.setInput(prop);
				propertySheet.expandAll();
			}
		});

		// 権限に応じてボタンを有効/無効化する
		this.buttonAll.setEnabled(permission);
		this.buttonRead.setEnabled(permission);
		this.buttonWrite.setEnabled(permission);
		this.buttonExec.setEnabled(permission);
		this.buttonClear.setEnabled(permission);

		// ラインを引く
		Label line = new Label(parent, SWT.SEPARATOR | SWT.HORIZONTAL);
		gridData = new GridData();
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		gridData.horizontalSpan = 8;
		line.setLayoutData(gridData);

		// 画面中央に
		Display display = shell.getDisplay();
		shell.setLocation((display.getBounds().width - shell.getSize().x) / 2,
				(display.getBounds().height - shell.getSize().y) / 2);

		// 必須入力項目を可視化
		this.update();
	}

	/**
	 * 更新処理
	 * 
	 */
	public void update(){
		// 必須項目を明示

		// ユーザIDのインデックス：0
		if("".equals(table.getItem(0).getText(1))){
			table.getItem(0).setBackground(RequiredFieldColorConstant.COLOR_REQUIRED);
		}else{
			table.getItem(0).setBackground(RequiredFieldColorConstant.COLOR_UNREQUIRED);
		}
		// ユーザ名のインデックス：1
		if("".equals(table.getItem(1).getText(1))){
			table.getItem(1).setBackground(RequiredFieldColorConstant.COLOR_REQUIRED);
		}else{
			table.getItem(1).setBackground(RequiredFieldColorConstant.COLOR_UNREQUIRED);
		}
	}

	/**
	 * 入力値チェックをします。
	 * 
	 * @return 検証結果
	 * 
	 * @see com.clustercontrol.dialog.CommonDialog#validate()
	 */
	@Override
	protected ValidateResult validate() {
		return null;
	}

	/**
	 * 入力値をマネージャに登録します。
	 * 
	 * @return true：正常、false：異常
	 * 
	 * @see com.clustercontrol.dialog.CommonDialog#action()
	 */
	@Override
	protected boolean action() {
		boolean result = false;

		Property property = this.getInputData();
		if(property == null){
			return result;
		}

		Property copy = PropertyUtil.copy(property);
		PropertyUtil.deletePropertyDefine(copy);
		UserInfo info = UserPropertyUtil.property2dto(copy);
		if(!this.isModifyDialog){
			// 作成の場合
			try {
				AccessEndpointWrapper.addUserInfo(info);
				result = true;

				// 完了メッセージ
				MessageDialog.openInformation(
						null,
						Messages.getString("successful"),
						Messages.getString("message.accesscontrol.7"));

			} catch (UserDuplicate_Exception e) {
				//ユーザID取得
				ArrayList values = PropertyUtil.getPropertyValue(copy, UserConstant.UID);
				String args[] = { (String)values.get(0) };

				// ユーザIDが重複している場合、エラーダイアログを表示する
				MessageDialog.openInformation(
						null,
						Messages.getString("message"),
						Messages.getString("message.monitor.53", args));

			} catch (Exception e) {
				String errMessage = "";
				if (e instanceof InvalidRole_Exception) {
					// 権限なし
					MessageDialog.openInformation(null, Messages.getString("message"),
							Messages.getString("message.accesscontrol.16"));
				} else {
					errMessage = ", " + e.getMessage();
				}
				MessageDialog.openError(
						null,
						Messages.getString("failed"),
						Messages.getString("message.accesscontrol.8") + errMessage);

			}
		} else{
			// 変更の場合
			try {
				AccessEndpointWrapper.modifyUserInfo(info);
				result = true;

				// 完了メッセージ
				MessageDialog.openInformation(
						null,
						Messages.getString("successful"),
						Messages.getString("message.accesscontrol.9"));

			} catch (Exception e) {
				String errMessage = "";
				if (e instanceof InvalidRole_Exception) {
					// 権限なし
					MessageDialog.openInformation(null, Messages.getString("message"),
							Messages.getString("message.accesscontrol.16"));
				} else {
					errMessage = ", " + e.getMessage();
				}
				MessageDialog.openError(
						null,
						Messages.getString("failed"),
						Messages.getString("message.accesscontrol.10") + errMessage);

			}
		}

		return result;
	}

	/**
	 * 変更用ダイアログなのかを返します。
	 * 
	 * @return 変更用ダイアログの場合、true
	 */
	public boolean isModifyDialog() {
		return this.isModifyDialog;
	}

	/**
	 * ユーザ用プロパティを返します。
	 * 
	 * @return ユーザ用プロパティ
	 */
	public Property getInputData() {
		return (Property) this.propertySheet.getInput();
	}

	/**
	 * ユーザ用プロパティを設定します。
	 * 
	 * @param property ユーザ用プロパティ
	 */
	public void setInputData(Property property) {
		propertySheet.setInput(property);
		this.update();
	}

	/**
	 * ユーザIDを返します。
	 * 
	 * @return ユーザID
	 */
	public String getUid() {
		return this.uid;
	}

	/**
	 * ＯＫボタンのテキストを返します。
	 * 
	 * @return ＯＫボタンのテキスト
	 */
	@Override
	protected String getOkButtonText() {
		if (isModifyDialog()) {
			return Messages.getString("modify"); //$NON-NLS-1$
		} else {
			return Messages.getString("register"); //$NON-NLS-1$
		}
	}

	/**
	 * キャンセルボタンのテキストを返します。
	 * 
	 * @return キャンセルボタンのテキスト
	 */
	@Override
	protected String getCancelButtonText() {
		return Messages.getString("cancel"); //$NON-NLS-1$
	}

	/**
	 * OKボタンを押されたときの処理
	 * 
	 * @see org.eclipse.jface.dialogs.Dialog#okPressed()
	 */
	@Override
	protected void okPressed() {
		ValidateResult result = this.validate();

		if (result == null || result.isValid()) {

			//変更時
			if(this.isModifyDialog){
				super.okPressed();
			}
			//作成時
			else{
				this.action();
			}
			
			this.view.update();

		} else {
			this.displayError(result);
		}
	}


	/**
	 * 共通のボタンを生成します。
	 * 
	 * @param parent
	 *            親のコンポジット
	 * @param label
	 *            ボタンのラベル
	 * @return 生成されたボタン
	 */
	private Button createButton(Composite parent, String label) {
		Button button = new Button(parent, SWT.NONE);

		GridData gridData = new GridData(GridData.FILL_HORIZONTAL);
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		button.setLayoutData(gridData);

		button.setText(label);

		return button;
	}

}