/*

Copyright (C) since 2009 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
 useful, but WITHOUT ANY WARRANTY; without even the implied
 warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
 PURPOSE.  See the GNU General Public License for more details.

 */
package com.clustercontrol.calendar.util;

import java.sql.Time;
import java.util.ArrayList;
import java.util.Date;
import java.util.Locale;

import com.clustercontrol.bean.DataRangeConstant;
import com.clustercontrol.bean.DayOfWeekConstant;
import com.clustercontrol.bean.Property;
import com.clustercontrol.bean.PropertyConstant;
import com.clustercontrol.calendar.bean.WeekdayConstant;
import com.clustercontrol.util.Messages;
import com.clustercontrol.util.PropertyUtil;
import com.clustercontrol.ws.calendar.WeekdayInfo;

/**
 * カレンダ曜日別情報のDTOとプロパティを相互変換するためのユーティリティクラスです。
 * @since 0.1
 */
public class WeekdayPropertyUtil {

	/**
	 * プロパティをカレンダ曜日別情報に変換するメソッドです。
	 * 
	 * @param property
	 * @return カレンダ曜日別情報
	 */
	public static WeekdayInfo property2dto(Property property){
		WeekdayInfo info = new WeekdayInfo();
		info.setWeekday(0);
		info.setRegDate(0L);
		info.setUpdateDate(0L);

		ArrayList object = null;

		//曜日
		object = PropertyUtil.getPropertyValue(property, WeekdayConstant.WEEKDAY);
		if (object.size() > 0) {
			info.setWeekday(Integer.valueOf(DayOfWeekConstant.stringToType((String) object.get(0))));
		}

		//時間帯(From)
		object = PropertyUtil.getPropertyValue(property, WeekdayConstant.TIME_FROM);
		if (object.size() > 0 && object.get(0) != null && !object.get(0).toString().equals("")) {
			info.setTimeFrom(((Date) (object.get(0))).getTime());
		}
		//時間帯(To)
		object = PropertyUtil.getPropertyValue(property, WeekdayConstant.TIME_TO);
		if (object.size() > 0 && object.get(0) != null && !object.get(0).toString().equals("")) {
			info.setTimeTo(((Date) (object.get(0))).getTime());
		}
		//説明
		object = PropertyUtil.getPropertyValue(property, WeekdayConstant.DESCRIPTION);
		if (object.size() > 0) {
			info.setDescription((String)object.get(0));
		}
		//登録者
		object = PropertyUtil.getPropertyValue(property, WeekdayConstant.CREATOR_NAME);
		if (object.size() > 0) {
			info.setRegUser((String)object.get(0));
		}
		//登録日時
		object = PropertyUtil.getPropertyValue(property, WeekdayConstant.CREATE_TIME);
		if (object.size() > 0 && object.get(0) != null && !object.get(0).toString().equals("")) {
			info.setRegDate(((Date) (object.get(0))).getTime());
		}
		//更新者
		object = PropertyUtil.getPropertyValue(property, WeekdayConstant.MODIFIER_NAME);
		if (object.size() > 0) {
			info.setUpdateUser((String)object.get(0));
		}
		//更新日時
		object = PropertyUtil.getPropertyValue(property, WeekdayConstant.MODIFY_TIME);
		if (object.size() > 0 && object.get(0) != null && !object.get(0).toString().equals("")) {
			info.setUpdateDate(((Date) (object.get(0))).getTime());
		}


		return info;
	}

	/**
	 * カレンダ曜日別情報をプロパティに変換するメソッドです。
	 * 
	 * @param info
	 * @param mode
	 * @param locale
	 * @return プロパティ
	 */
	public static Property dto2property(WeekdayInfo info, int mode, Locale locale){
		/** ローカル変数 */
		Property property = null;
		ArrayList propertyList = null;

		/** メイン処理 */
		property = getProperty(mode, locale);

		if(info != null){
			//曜日
			propertyList = PropertyUtil.getProperty(property, WeekdayConstant.WEEKDAY);
			if(info.getTimeFrom() != null){
				((Property)propertyList.get(0)).setValue(DayOfWeekConstant.typeToString(info.getWeekday().intValue()));
			}
			//時間帯(From)
			propertyList = PropertyUtil.getProperty(property, WeekdayConstant.TIME_FROM);
			if(info.getTimeFrom() != null){
				((Property)propertyList.get(0)).setValue(new Time(info.getTimeFrom()));
			}
			//時間帯(To)
			propertyList = PropertyUtil.getProperty(property, WeekdayConstant.TIME_TO);
			if(info.getTimeTo() != null){
				((Property)propertyList.get(0)).setValue(new Time(info.getTimeTo()));
			}
			//説明
			propertyList = PropertyUtil.getProperty(property, WeekdayConstant.DESCRIPTION);
			if(info.getDescription() != null && info.getDescription().compareTo("") != 0){
				((Property)propertyList.get(0)).setValue(info.getDescription());
			}
			//登録者
			propertyList = PropertyUtil.getProperty(property, WeekdayConstant.CREATOR_NAME);
			if(info.getRegUser() != null && info.getRegUser().compareTo("") != 0){
				((Property)propertyList.get(0)).setValue(info.getRegUser());
			}
			//登録日時
			propertyList = PropertyUtil.getProperty(property, WeekdayConstant.CREATE_TIME);
			if(info.getRegDate() != null){
				((Property)propertyList.get(0)).setValue(new Date(info.getRegDate()));
			}
			//更新者
			propertyList = PropertyUtil.getProperty(property, WeekdayConstant.MODIFIER_NAME);
			if(info.getUpdateUser() != null && info.getUpdateUser().compareTo("") != 0){
				((Property)propertyList.get(0)).setValue(info.getUpdateUser());
			}
			//更新日時
			propertyList = PropertyUtil.getProperty(property, WeekdayConstant.MODIFY_TIME);
			if(info.getUpdateDate() != null){
				((Property)propertyList.get(0)).setValue(new Date(info.getUpdateDate()));
			}
		}

		return property;
	}


	/**
	 * カレンダ曜日別情報用プロパティを返します。
	 * 
	 * @param mode
	 * @param locale
	 * @return カレンダ曜日別情報用プロパティ
	 */
	private static Property getProperty(int mode, Locale locale) {
		//曜日
		Property weekday =
			new Property(WeekdayConstant.WEEKDAY, Messages.getString("weekday", locale), PropertyConstant.EDITOR_SELECT);
		//時間帯(From)
		Property timeFrom =
			new Property(WeekdayConstant.TIME_FROM, Messages.getString("timestamp", locale) + "(" + Messages.getString("start", locale) + ") " + Messages.getString("timestamp.example", locale),
					PropertyConstant.EDITOR_TIME);
		//時間帯(To)
		Property timeTo =
			new Property(WeekdayConstant.TIME_TO, Messages.getString("timestamp", locale) + "(" + Messages.getString("end", locale) + ") " + Messages.getString("timestamp.example", locale),
					PropertyConstant.EDITOR_TIME);
		//説明
		Property description =
			new Property(WeekdayConstant.DESCRIPTION, Messages.getString("description", locale), PropertyConstant.EDITOR_TEXT, DataRangeConstant.VARCHAR_256);
		//登録日時
		Property createTime =
			new Property(WeekdayConstant.CREATE_TIME, Messages.getString("create.time", locale), PropertyConstant.EDITOR_DATETIME);
		//登録者
		Property creatorName =
			new Property(WeekdayConstant.CREATOR_NAME, Messages.getString("creator.name", locale), PropertyConstant.EDITOR_TEXT);
		//更新日時
		Property modifyTime =
			new Property(WeekdayConstant.MODIFY_TIME, Messages.getString("update.time", locale), PropertyConstant.EDITOR_DATETIME);
		//更新者
		Property modifierName =
			new Property(WeekdayConstant.MODIFIER_NAME, Messages.getString("modifier.name", locale), PropertyConstant.EDITOR_TEXT);

		//値を初期化
		Object values[][] = {
				{ DayOfWeekConstant.STRING_SUNDAY, DayOfWeekConstant.STRING_MONDAY, DayOfWeekConstant.STRING_TUESDAY, DayOfWeekConstant.STRING_WEDNESDAY, DayOfWeekConstant.STRING_THURSDAY, DayOfWeekConstant.STRING_FRIDAY, DayOfWeekConstant.STRING_SATURDAY },
				{ DayOfWeekConstant.STRING_SUNDAY, DayOfWeekConstant.STRING_MONDAY, DayOfWeekConstant.STRING_TUESDAY, DayOfWeekConstant.STRING_WEDNESDAY, DayOfWeekConstant.STRING_THURSDAY, DayOfWeekConstant.STRING_FRIDAY, DayOfWeekConstant.STRING_SATURDAY }};

		weekday.setSelectValues(values);
		weekday.setValue(DayOfWeekConstant.STRING_SUNDAY);
		timeFrom.setValue("");
		timeTo.setValue("");
		description.setValue("");
		createTime.setValue("");
		creatorName.setValue("");
		modifyTime.setValue("");
		modifierName.setValue("");

		//モードにより、変更可及びコピー可を設定
		if(mode == PropertyConstant.MODE_ADD){
			weekday.setModify(PropertyConstant.MODIFY_OK);
			timeFrom.setModify(PropertyConstant.MODIFY_OK);
			timeTo.setModify(PropertyConstant.MODIFY_OK);
			description.setModify(PropertyConstant.MODIFY_OK);
		}
		else if(mode == PropertyConstant.MODE_MODIFY){
			weekday.setModify(PropertyConstant.MODIFY_OK);
			timeFrom.setModify(PropertyConstant.MODIFY_OK);
			timeTo.setModify(PropertyConstant.MODIFY_OK);
			description.setModify(PropertyConstant.MODIFY_OK);
		}

		Property property = new Property(null, null, "");

		// 初期表示ツリーを構成。
		property.removeChildren();
		property.addChildren(weekday);
		property.addChildren(timeFrom);
		property.addChildren(timeTo);
		property.addChildren(description);
		property.addChildren(createTime);
		property.addChildren(creatorName);
		property.addChildren(modifyTime);
		property.addChildren(modifierName);

		return property;
	}
}
