/*

Copyright (C) 2006 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.jobmanagement.dialog;

import java.text.DecimalFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.layout.RowData;
import org.eclipse.swt.layout.RowLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Text;

import com.clustercontrol.bean.DataRangeConstant;
import com.clustercontrol.bean.DayOfWeekConstant;
import com.clustercontrol.bean.Property;
import com.clustercontrol.bean.RequiredFieldColorConstant;
import com.clustercontrol.bean.ScheduleConstant;
import com.clustercontrol.bean.SizeConstant;
import com.clustercontrol.bean.ValidConstant;
import com.clustercontrol.composite.ComboModifyListener;
import com.clustercontrol.composite.NumberKeyListener;
import com.clustercontrol.composite.StringVerifyListener;
import com.clustercontrol.dialog.CommonDialog;
import com.clustercontrol.dialog.ValidateResult;
import com.clustercontrol.jobmanagement.action.GetCalendarIdList;
import com.clustercontrol.jobmanagement.action.GetScheduleTableDefine;
import com.clustercontrol.jobmanagement.bean.JobConstant;
import com.clustercontrol.jobmanagement.util.JobEndpointWrapper;
import com.clustercontrol.util.Messages;
import com.clustercontrol.ws.common.Schedule;
import com.clustercontrol.ws.jobmanagement.InvalidRole_Exception;
import com.clustercontrol.ws.jobmanagement.InvalidSetting_Exception;
import com.clustercontrol.ws.jobmanagement.InvalidUserPass_Exception;
import com.clustercontrol.ws.jobmanagement.JobSchedule;
import com.clustercontrol.ws.jobmanagement.JobTreeItem;

/**
 * ジョブ[スケジュールの作成・変更]ダイアログクラスです。
 * 
 * @version 4.0.0
 * @since 1.0.0
 */
public class ScheduleDialog extends CommonDialog {

	// ログ
	private static Log m_log = LogFactory.getLog( ScheduleDialog.class );

	protected JobSchedule m_jobSchedule ;

	/** スケジュールID用テキスト */
	protected Text m_scheduleId = null;
	/** スケジュール名用テキスト */
	protected Text m_scheduleName = null;
	/** ジョブID用テキスト */
	protected Text m_jobId = null;
	/** ジョブ名用テキスト */
	protected Text m_jobName = null;
	/** カレンダID用テキスト */
	protected Combo m_calendarId = null;
	/** ジョブ参照用ボタン */
	protected Button m_jobSelect = null;
	/** スケジュール日時指定用ラジオボタン */
	protected Button m_type1 = null;
	/** スケジュール曜日指定用ラジオボタン */
	protected Button m_type2 = null;
	/** 有効用ラジオボタン */
	protected Button m_valid = null;
	/** 無効用ラジオボタン */
	protected Button m_invalid = null;
	/** 月指定用コンボボックス */
	protected Combo m_comboMonth = null;
	/** 日指定用コンボボックス */
	protected Combo m_comboDay = null;
	/** 時指定用コンボボックス */
	protected Combo m_comboHours1 = null;
	/** 分指定用コンボボックス */
	protected Combo m_comboMinutes1 = null;
	/** 曜日指定用コンボボックス */
	protected Combo m_comboDayOfWeek = null;
	/** 時指定用コンボボックス */
	protected Combo m_comboHours2 = null;
	/** 分指定用コンボボックス */
	protected Combo m_comboMinutes2 = null;
	/** シェル */
	protected Shell m_shell = null;
	/** スケジュール情報 */
	protected ArrayList m_schedule = null;

	/** 所属ジョブユニットのジョブID */
	protected String m_jobunitId = null;

	/**
	 * コンストラクタ
	 * 
	 * @param parent 親シェル
	 */
	public ScheduleDialog(Shell parent) {
		super(parent);
	}

	/**
	 * ダイアログエリアを生成します。
	 * 
	 * @param parent 親コンポジット
	 */
	@Override
	protected void customizeDialog(Composite parent) {
		m_shell = this.getShell();

		parent.getShell().setText(
				Messages.getString("dialog.job.add.modify.schedule"));

		RowLayout layout = new RowLayout();
		layout.type = SWT.VERTICAL;
		layout.spacing = 10;
		layout.marginWidth = 10;
		layout.marginHeight = 10;
		layout.fill = true;
		parent.setLayout(layout);

		Composite composite = null;
		RowLayout rowLayout = null;

		composite = new Composite(parent, SWT.NONE);
		rowLayout = new RowLayout();
		rowLayout.type = SWT.HORIZONTAL;
		rowLayout.spacing = 10;
		composite.setLayout(rowLayout);
		Label labelScheduleId = new Label(composite, SWT.NONE);
		labelScheduleId.setText(Messages.getString("schedule.id") + " : ");
		labelScheduleId.setLayoutData(new RowData(120,
				SizeConstant.SIZE_LABEL_HEIGHT));
		if (m_schedule instanceof ArrayList) {
			//スケジュール変更時、スケジュールIDは変更不可とする
			m_scheduleId = new Text(composite, SWT.BORDER | SWT.READ_ONLY);
		} else {
			m_scheduleId = new Text(composite, SWT.BORDER);
			m_scheduleId.addVerifyListener(
					new StringVerifyListener(DataRangeConstant.VARCHAR_64));
		}
		this.m_scheduleId.setLayoutData(new RowData(150,
				SizeConstant.SIZE_LABEL_HEIGHT));
		this.m_scheduleId.addModifyListener(new ModifyListener(){
			@Override
			public void modifyText(ModifyEvent arg0) {
				update();
			}
		});

		composite = new Composite(parent, SWT.NONE);
		rowLayout = new RowLayout();
		rowLayout.type = SWT.HORIZONTAL;
		rowLayout.spacing = 10;
		composite.setLayout(rowLayout);
		Label labelScheduleName = new Label(composite, SWT.NONE);
		labelScheduleName.setText(Messages.getString("schedule.name") + " : ");
		labelScheduleName.setLayoutData(new RowData(120,
				SizeConstant.SIZE_LABEL_HEIGHT));
		this.m_scheduleName = new Text(composite, SWT.BORDER);
		this.m_scheduleName.setLayoutData(new RowData(150,
				SizeConstant.SIZE_LABEL_HEIGHT));
		this.m_scheduleName.addVerifyListener(
				new StringVerifyListener(DataRangeConstant.VARCHAR_64));
		this.m_scheduleName.addModifyListener(new ModifyListener(){
			@Override
			public void modifyText(ModifyEvent arg0) {
				update();
			}
		});

		composite = new Composite(parent, SWT.NONE);
		rowLayout = new RowLayout();
		rowLayout.type = SWT.HORIZONTAL;
		rowLayout.spacing = 10;
		composite.setLayout(rowLayout);
		Label labelJobId = new Label(composite, SWT.NONE);
		labelJobId.setText(Messages.getString("job.id") + " : ");
		labelJobId.setLayoutData(new RowData(120,
				SizeConstant.SIZE_LABEL_HEIGHT));
		this.m_jobId = new Text(composite, SWT.READ_ONLY | SWT.BORDER);
		this.m_jobId.setLayoutData(new RowData(150, SizeConstant.SIZE_LABEL_HEIGHT));
		this.m_jobId.addModifyListener(new ModifyListener(){
			@Override
			public void modifyText(ModifyEvent arg0) {
				update();
			}
		});

		m_jobSelect = new Button(composite, SWT.NONE);
		m_jobSelect.setText(Messages.getString("refer"));
		m_jobSelect.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				JobTreeDialog dialog = new JobTreeDialog(m_shell, true);
				if (dialog.open() == IDialogConstants.OK_ID) {
					JobTreeItem selectItem = dialog.getSelectItem();
					if (selectItem.getData().getType() != JobConstant.TYPE_COMPOSITE) {
						m_jobId.setText(selectItem.getData().getId());
						m_jobName.setText(selectItem.getData().getName());
						setJobunitId(selectItem.getData().getJobunitId());
					} else {
						m_jobId.setText("");
						m_jobName.setText("");
						setJobunitId(null);
					}
				}
			}
		});

		composite = new Composite(parent, SWT.NONE);
		rowLayout = new RowLayout();
		rowLayout.type = SWT.HORIZONTAL;
		rowLayout.spacing = 10;
		composite.setLayout(rowLayout);
		Label labelJobName = new Label(composite, SWT.NONE);
		labelJobName.setText(Messages.getString("job.name") + " : ");
		labelJobName.setLayoutData(new RowData(120,
				SizeConstant.SIZE_LABEL_HEIGHT));
		m_jobName = new Text(composite, SWT.BORDER | SWT.READ_ONLY);
		m_jobName
		.setLayoutData(new RowData(150, SizeConstant.SIZE_LABEL_HEIGHT));

		composite = new Composite(parent, SWT.NONE);
		rowLayout = new RowLayout();
		rowLayout.type = SWT.HORIZONTAL;
		rowLayout.spacing = 10;
		composite.setLayout(rowLayout);
		Label labelCalendarId = new Label(composite, SWT.NONE);
		labelCalendarId.setText(Messages.getString("calendar.id") + " : ");
		labelCalendarId.setLayoutData(new RowData(120,
				SizeConstant.SIZE_LABEL_HEIGHT));
		m_calendarId = new Combo(composite, SWT.LEFT | SWT.READ_ONLY);
		m_calendarId.setLayoutData(new RowData(150, SizeConstant.SIZE_COMBO_HEIGHT));
		List<String> idList = null;
		try {
			idList = JobEndpointWrapper.getCalendarIdList();
		} catch (InvalidRole_Exception e) {
			// アクセス権なしの場合、エラーダイアログを表示する
			MessageDialog.openInformation(
					null,
					Messages.getString("message"),
					Messages.getString("message.accesscontrol.16"));
			m_calendarId.setEnabled(false);
			
		} catch (Exception e) {
			m_log.warn("getCalendarIdList(), " + e.getMessage(), e);
			MessageDialog.openError(
					null,
					Messages.getString("failed"),
					Messages.getString("message.hinemos.failure.unexpected") + ", " + e.getMessage());
		}

		if(idList != null){
			m_calendarId.add("");
			for(int i = 0; i < idList.size(); i++){
				m_calendarId.add(idList.get(i));
			}
		}

		Group group1 = new Group(parent, SWT.NONE);
		group1.setText(Messages.getString("schedule"));
		GridLayout gridLayout = new GridLayout(9, true);
		gridLayout.marginWidth = 10;
		gridLayout.marginHeight = 10;
		group1.setLayout(gridLayout);

		createType1(group1);
		createType2(group1);

		Group group2 = new Group(parent, SWT.NONE);
		group2.setText(Messages.getString("valid") + "/"
				+ Messages.getString("invalid"));
		rowLayout = new RowLayout();
		rowLayout.type = SWT.HORIZONTAL;
		rowLayout.spacing = 10;
		rowLayout.marginWidth = 10;
		rowLayout.marginHeight = 10;
		group2.setLayout(rowLayout);

		m_valid = new Button(group2, SWT.RADIO | SWT.CENTER);
		m_valid.setText(ValidConstant.STRING_VALID);
		m_valid.setLayoutData(new RowData(60, SizeConstant.SIZE_BUTTON_HEIGHT));
		m_valid.setSelection(true);

		m_invalid = new Button(group2, SWT.RADIO | SWT.CENTER);
		m_invalid.setText(ValidConstant.STRING_INVALID);
		m_invalid
		.setLayoutData(new RowData(60, SizeConstant.SIZE_BUTTON_HEIGHT));

		new Label(parent, SWT.SEPARATOR | SWT.HORIZONTAL);

		// 画面中央に
		Display display = m_shell.getDisplay();
		m_shell.setLocation(
				(display.getBounds().width - m_shell.getSize().x) / 2, (display
						.getBounds().height - m_shell.getSize().y) / 2);

		//スケジュール情報反映
		reflectJobSchedule();
	}

	/**
	 * 更新処理
	 * 
	 */
	public void update(){
		// 必須項目を明示
		if("".equals(this.m_scheduleId.getText())){
			this.m_scheduleId.setBackground(RequiredFieldColorConstant.COLOR_REQUIRED);
		}else{
			this.m_scheduleId.setBackground(RequiredFieldColorConstant.COLOR_UNREQUIRED);
		}
		if("".equals(this.m_scheduleName.getText())){
			this.m_scheduleName.setBackground(RequiredFieldColorConstant.COLOR_REQUIRED);
		}else{
			this.m_scheduleName.setBackground(RequiredFieldColorConstant.COLOR_UNREQUIRED);
		}
		if("".equals(this.m_jobId.getText())){
			this.m_jobId.setBackground(RequiredFieldColorConstant.COLOR_REQUIRED);
		}else{
			this.m_jobId.setBackground(RequiredFieldColorConstant.COLOR_UNREQUIRED);
		}

		if(this.m_comboMinutes1.getEnabled() && "".equals(this.m_comboMinutes1.getText())){
			this.m_comboMinutes1.setBackground(RequiredFieldColorConstant.COLOR_REQUIRED);
		}else{
			this.m_comboMinutes1.setBackground(RequiredFieldColorConstant.COLOR_UNREQUIRED);
		}

		if(this.m_comboDayOfWeek.getEnabled() && "".equals(this.m_comboDayOfWeek.getText())){
			this.m_comboDayOfWeek.setBackground(RequiredFieldColorConstant.COLOR_REQUIRED);
		}else{
			this.m_comboDayOfWeek.setBackground(RequiredFieldColorConstant.COLOR_UNREQUIRED);
		}

		if(this.m_comboMinutes2.getEnabled() && "".equals(this.m_comboMinutes2.getText())){
			this.m_comboMinutes2.setBackground(RequiredFieldColorConstant.COLOR_REQUIRED);
		}else{
			this.m_comboMinutes2.setBackground(RequiredFieldColorConstant.COLOR_UNREQUIRED);
		}
	}

	/**
	 * スケジュール日時指定を生成します。
	 * 
	 * @param parent 親グループ
	 */
	protected void createType1(Group parent) {
		m_type1 = new Button(parent, SWT.RADIO);
		GridData gridData = new GridData();
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		m_type1.setLayoutData(gridData);
		m_type1.addSelectionListener(new SelectionListener() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				Button check = (Button) e.getSource();
				if (check.getSelection()) {
					m_type2.setSelection(false);
					m_comboMonth.setEnabled(true);
					m_comboDay.setEnabled(true);
					m_comboHours1.setEnabled(true);
					m_comboMinutes1.setEnabled(true);
					m_comboDayOfWeek.setEnabled(false);
					m_comboHours2.setEnabled(false);
					m_comboMinutes2.setEnabled(false);
				}
				update();
			}

			@Override
			public void widgetDefaultSelected(SelectionEvent e) {

			}
		});

		m_comboMonth = new Combo(parent, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		m_comboMonth.setLayoutData(gridData);
		m_comboMonth.setTextLimit(2);
		m_comboMonth.setVisibleItemCount(10);
		m_comboMonth.addKeyListener(new NumberKeyListener());
		m_comboMonth.addModifyListener(new ComboModifyListener());
		DecimalFormat format = new DecimalFormat("00");
		m_comboMonth.add("");
		for (int month = 1; month <= 12; month++) {
			m_comboMonth.add(format.format(month));
		}

		Label labelMonth = new Label(parent, SWT.NONE);
		labelMonth.setText(Messages.getString("month"));
		gridData = new GridData();
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		labelMonth.setLayoutData(gridData);

		m_comboDay = new Combo(parent, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		m_comboDay.setLayoutData(gridData);
		m_comboDay.setTextLimit(2);
		m_comboDay.setVisibleItemCount(10);
		m_comboDay.addKeyListener(new NumberKeyListener());
		m_comboDay.addModifyListener(new ComboModifyListener());
		format = new DecimalFormat("00");
		m_comboDay.add("");
		for (int day = 1; day <= 31; day++) {
			m_comboDay.add(format.format(day));
		}

		Label labelDay = new Label(parent, SWT.NONE);
		labelDay.setText(Messages.getString("monthday"));
		gridData = new GridData();
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		labelDay.setLayoutData(gridData);

		m_comboHours1 = new Combo(parent, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		m_comboHours1.setLayoutData(gridData);
		m_comboHours1.setTextLimit(2);
		m_comboHours1.setVisibleItemCount(10);
		m_comboHours1.addKeyListener(new NumberKeyListener());
		m_comboHours1.addModifyListener(new ComboModifyListener());
		format = new DecimalFormat("00");
		m_comboHours1.add("");
		for (int hour = 0; hour < 24; hour++) {
			m_comboHours1.add(format.format(hour));
		}
		this.m_comboHours1.addModifyListener(new ModifyListener(){
			@Override
			public void modifyText(ModifyEvent arg0) {
				update();
			}
		});

		Label labelHours = new Label(parent, SWT.NONE);
		labelHours.setText(Messages.getString("hour"));
		gridData = new GridData();
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		labelHours.setLayoutData(gridData);

		m_comboMinutes1 = new Combo(parent, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		m_comboMinutes1.setLayoutData(gridData);
		m_comboMinutes1.setTextLimit(2);
		m_comboMinutes1.setVisibleItemCount(10);
		m_comboMinutes1.addKeyListener(new NumberKeyListener());
		m_comboMinutes1.addModifyListener(new ComboModifyListener());
		format = new DecimalFormat("00");
		m_comboMinutes1.add("");
		for (int minutes = 0; minutes < 60; minutes++) {
			m_comboMinutes1.add(format.format(minutes));
		}
		this.m_comboMinutes1.addModifyListener(new ModifyListener(){
			@Override
			public void modifyText(ModifyEvent arg0) {
				update();
			}
		});

		Label labelMinutes = new Label(parent, SWT.NONE);
		labelMinutes.setText(Messages.getString("minute"));
		gridData = new GridData();
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		labelMinutes.setLayoutData(gridData);
	}

	/**
	 * スケジュール曜日指定を生成します。
	 * 
	 * @param parent 親グループ
	 */
	protected void createType2(Group parent) {
		m_type2 = new Button(parent, SWT.RADIO);
		GridData gridData = new GridData();
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		m_type2.setLayoutData(gridData);
		m_type2.addSelectionListener(new SelectionListener() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				Button check = (Button) e.getSource();
				if (check.getSelection()) {
					m_type1.setSelection(false);
					m_comboMonth.setEnabled(false);
					m_comboDay.setEnabled(false);
					m_comboHours1.setEnabled(false);
					m_comboMinutes1.setEnabled(false);
					m_comboDayOfWeek.setEnabled(true);
					m_comboHours2.setEnabled(true);
					m_comboMinutes2.setEnabled(true);
				}
				update();
			}

			@Override
			public void widgetDefaultSelected(SelectionEvent e) {

			}
		});

		Label labelDayOfWeek = new Label(parent, SWT.NONE);
		labelDayOfWeek.setText(Messages.getString("weekday"));
		gridData = new GridData();
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		labelDayOfWeek.setLayoutData(gridData);

		m_comboDayOfWeek = new Combo(parent, SWT.READ_ONLY);
		gridData = new GridData();
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		gridData.horizontalSpan = 2;
		m_comboDayOfWeek.setLayoutData(gridData);
		m_comboDayOfWeek.setTextLimit(3);
		m_comboDayOfWeek.setVisibleItemCount(10);
		m_comboDayOfWeek.add(DayOfWeekConstant.STRING_SUNDAY);
		m_comboDayOfWeek.add(DayOfWeekConstant.STRING_MONDAY);
		m_comboDayOfWeek.add(DayOfWeekConstant.STRING_TUESDAY);
		m_comboDayOfWeek.add(DayOfWeekConstant.STRING_WEDNESDAY);
		m_comboDayOfWeek.add(DayOfWeekConstant.STRING_THURSDAY);
		m_comboDayOfWeek.add(DayOfWeekConstant.STRING_FRIDAY);
		m_comboDayOfWeek.add(DayOfWeekConstant.STRING_SATURDAY);
		this.m_comboDayOfWeek.addModifyListener(new ModifyListener(){
			@Override
			public void modifyText(ModifyEvent arg0) {
				update();
			}
		});

		Label labelDummy = new Label(parent, SWT.NONE);
		labelDummy.setText("");
		gridData = new GridData();
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		labelDummy.setLayoutData(gridData);

		m_comboHours2 = new Combo(parent, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		m_comboHours2.setLayoutData(gridData);
		m_comboHours2.setTextLimit(2);
		m_comboHours2.setVisibleItemCount(10);
		m_comboHours2.addKeyListener(new NumberKeyListener());
		m_comboHours2.addModifyListener(new ComboModifyListener());
		DecimalFormat format = new DecimalFormat("00");
		m_comboHours2.add("");
		for (int hour = 0; hour < 24; hour++) {
			m_comboHours2.add(format.format(hour));
		}
		this.m_comboHours2.addModifyListener(new ModifyListener(){
			@Override
			public void modifyText(ModifyEvent arg0) {
				update();
			}
		});

		Label labelHours = new Label(parent, SWT.NONE);
		labelHours.setText(Messages.getString("hour"));
		gridData = new GridData();
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		labelHours.setLayoutData(gridData);

		m_comboMinutes2 = new Combo(parent, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		m_comboMinutes2.setLayoutData(gridData);
		m_comboMinutes2.setTextLimit(2);
		m_comboMinutes2.setVisibleItemCount(10);
		m_comboMinutes2.addKeyListener(new NumberKeyListener());
		m_comboMinutes2.addModifyListener(new ComboModifyListener());
		format = new DecimalFormat("00");
		m_comboMinutes2.add("");
		for (int minutes = 0; minutes < 60; minutes++) {
			m_comboMinutes2.add(format.format(minutes));
		}
		this.m_comboMinutes2.addModifyListener(new ModifyListener(){
			@Override
			public void modifyText(ModifyEvent arg0) {
				update();
			}
		});

		Label labelMinutes = new Label(parent, SWT.NONE);
		labelMinutes.setText(Messages.getString("minute"));
		gridData = new GridData();
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		labelMinutes.setLayoutData(gridData);
	}

	/**
	 * スケジュール情報を設定します。
	 * 
	 * @param info スケジュール情報
	 */
	public void setSchedule(ArrayList info) {
		m_schedule = info;
	}

	/**
	 * スケジュール情報を返します。
	 * 
	 * @return スケジュール情報
	 */
	public ArrayList getSchedule() {
		return m_schedule;
	}

	/**
	 * ダイアログにスケジュール情報を反映します。
	 * 
	 * @see com.clustercontrol.jobmanagement.bean.ScheduleTableDefine
	 */
	protected void reflectJobSchedule() {

		if (m_schedule instanceof ArrayList) {
			//スケジュールID設定
			String scheduleId = (String) m_schedule
			.get(GetScheduleTableDefine.SCHE_ID);
			m_scheduleId.setText(scheduleId);

			//スケジュール名設定
			String scheduleName = (String) m_schedule
			.get(GetScheduleTableDefine.SCHE_NAME);
			m_scheduleName.setText(scheduleName);

			//ジョブID設定
			String jobId = (String) m_schedule.get(GetScheduleTableDefine.JOB_ID);
			m_jobId.setText(jobId);

			//ジョブ名設定
			String jobName = (String) m_schedule
			.get(GetScheduleTableDefine.JOB_NAME);
			m_jobName.setText(jobName);

			//ジョブユニットIDを設定
			String jobunitId = (String) m_schedule.get(GetScheduleTableDefine.JOBUNIT_ID);
			this.setJobunitId(jobunitId);


			//カレンダID
			String calendarId = (String) m_schedule.get(GetScheduleTableDefine.CALENDAR_ID);
			if (calendarId != null && calendarId.length() > 0) {
				m_calendarId.select(0);
				if (!m_calendarId.getEnabled()) {
					// カレンダー参照権限がない場合
					m_calendarId.add(calendarId);
				}
				for (int i = 0; i < m_calendarId.getItemCount(); i++) {
					if (calendarId.equals(m_calendarId.getItem(i))) {
						m_calendarId.select(i);
						break;
					}
				}
			}

			//スケジュール設定
			Schedule schedule = (Schedule) m_schedule
			.get(GetScheduleTableDefine.SCHEDULE);

			//日時を設定
			DecimalFormat format = new DecimalFormat("00");
			if (schedule.getType() == ScheduleConstant.TYPE_DAY) {
				//月を設定
				m_comboMonth.select(0);
				for (int i = 0; i < m_comboMonth.getItemCount(); i++) {
					if (schedule.getMonth() == null) {
						break;
					}
					String month = format.format(schedule.getMonth());
					if (month.equals(m_comboMonth.getItem(i))) {
						m_comboMonth.select(i);
						break;
					}
				}
				//日を設定
				m_comboDay.select(0);
				for (int i = 0; i < m_comboDay.getItemCount(); i++) {
					if (schedule.getDay() == null) {
						break;
					}
					String day = format.format(schedule.getDay());
					if (day.equals(m_comboDay.getItem(i))) {
						m_comboDay.select(i);
						break;
					}
				}
				//時を設定
				m_comboHours1.select(0);
				m_comboHours2.select(0);
				for (int i = 0; i < m_comboHours1.getItemCount(); i++) {
					if (schedule.getHour() == null) {
						break;
					}
					String hours = format.format(schedule.getHour());
					if (hours.equals(m_comboHours1.getItem(i))) {
						m_comboHours1.select(i);
						break;
					}
				}
				//分を設定
				m_comboMinutes1.select(0);
				m_comboMinutes2.select(0);
				for (int i = 0; i < m_comboMinutes1.getItemCount(); i++) {
					if (schedule.getMinute() == null) {
						break;
					}
					String minutes = format.format(schedule.getMinute());
					if (minutes.equals(m_comboMinutes1.getItem(i))) {
						m_comboMinutes1.select(i);
						break;
					}
				}
			} else {
				//曜日を設定
				m_comboDayOfWeek.select(0);
				for (int i = 0; i < m_comboDayOfWeek.getItemCount(); i++) {
					if (schedule.getWeek() == null) {
						break;
					}
					String dayOfWeek = DayOfWeekConstant.typeToString(schedule.getWeek());
					if (dayOfWeek.equals(m_comboDayOfWeek.getItem(i))) {
						m_comboDayOfWeek.select(i);
						break;
					}
				}
				//時を設定
				m_comboHours1.select(0);
				m_comboHours2.select(0);
				for (int i = 0; i < m_comboHours1.getItemCount(); i++) {
					if (schedule.getHour() == null) {
						break;
					}
					String hours = format.format(schedule.getHour());
					if (hours.equals(m_comboHours2.getItem(i))) {
						m_comboHours2.select(i);
						break;
					}
				}
				//分を設定
				m_comboMinutes1.select(0);
				m_comboMinutes2.select(0);
				for (int i = 0; i < m_comboMinutes1.getItemCount(); i++) {
					if (schedule.getMinute() == null) {
						break;
					}
					String minutes = format.format(schedule.getMinute());
					if (minutes.equals(m_comboMinutes1.getItem(i))) {
						m_comboMinutes2.select(i);
						break;
					}
				}
			}

			//指定方式を設定
			if (schedule.getType() == ScheduleConstant.TYPE_DAY) {
				m_type1.setSelection(true);
				m_comboMonth.setEnabled(true);
				m_comboDay.setEnabled(true);
				m_comboHours1.setEnabled(true);
				m_comboMinutes1.setEnabled(true);
				m_comboDayOfWeek.setEnabled(false);
				m_comboHours2.setEnabled(false);
				m_comboMinutes2.setEnabled(false);
			} else {
				m_type2.setSelection(true);
				m_comboMonth.setEnabled(false);
				m_comboDay.setEnabled(false);
				m_comboHours1.setEnabled(false);
				m_comboMinutes1.setEnabled(false);
				m_comboDayOfWeek.setEnabled(true);
				m_comboHours2.setEnabled(true);
				m_comboMinutes2.setEnabled(true);
			}

			//有効/無効設定
			Integer effective = (Integer) m_schedule
			.get(GetScheduleTableDefine.VALID);
			if (effective.intValue() == ValidConstant.TYPE_VALID) {
				m_valid.setSelection(true);
				m_invalid.setSelection(false);
			} else {
				m_valid.setSelection(false);
				m_invalid.setSelection(true);
			}
		} else {
			m_type1.setSelection(true);
			m_comboMonth.setEnabled(true);
			m_comboDay.setEnabled(true);
			m_comboHours1.setEnabled(true);
			m_comboMinutes1.setEnabled(true);
			m_comboDayOfWeek.setEnabled(false);
			m_comboHours2.setEnabled(false);
			m_comboMinutes2.setEnabled(false);
		}

		// 必須入力項目を可視化
		this.update();
	}

	/**
	 * ダイアログの情報からスケジュール情報を作成します。
	 * 
	 * @return 入力値の検証結果
	 * 
	 * @see com.clustercontrol.jobmanagement.bean.GetScheduleTableDefine
	 */
	protected ValidateResult createJobSchedule() {
		ValidateResult result = null;

		m_jobSchedule = new JobSchedule();

		//ジョブユニットID取得
		if (getJobunitId() != null) {
			m_jobSchedule.setJobunitId(getJobunitId());
		}

		//スケジュールID取得
		if (m_scheduleId.getText().length() > 0) {
			m_jobSchedule.setId(m_scheduleId.getText());
		} else {
			result = new ValidateResult();
			result.setValid(false);
			result.setID(Messages.getString("message.hinemos.1"));
			result.setMessage(Messages.getString("message.job.24"));
			return result;
		}

		//スケジュール名取得
		if (m_scheduleName.getText().length() > 0) {
			m_jobSchedule.setName(m_scheduleName.getText());
		} else {
			result = new ValidateResult();
			result.setValid(false);
			result.setID(Messages.getString("message.hinemos.1"));
			result.setMessage(Messages.getString("message.job.25"));
			return result;
		}

		//ジョブID取得
		if (m_jobId.getText().length() > 0) {
			m_jobSchedule.setJobId(m_jobId.getText());
		} else {
			result = new ValidateResult();
			result.setValid(false);
			result.setID(Messages.getString("message.hinemos.1"));
			result.setMessage(Messages.getString("message.job.22"));
			return result;
		}

		//ジョブ名取得
		if (m_jobName.getText().length() > 0) {
			m_jobSchedule.setJobName(m_jobName.getText());
		} else {
			result = new ValidateResult();
			result.setValid(false);
			result.setID(Messages.getString("message.hinemos.1"));
			result.setMessage(Messages.getString("message.job.23"));
			return result;
		}

		//カレンダID
		if (m_calendarId.getText().length() > 0) {
			m_jobSchedule.setCalendarId(m_calendarId.getText());
		}
		else{
			m_jobSchedule.setCalendarId("");
		}

		//条件設定
		Schedule schedule = new Schedule();
		Integer month = null;
		Integer day = null;
		Integer hours = null;
		Integer minutes = null;
		Integer week = null;
		if (m_type1.getSelection()) {
			//月を取得
			if (m_comboMonth.getText().length() > 0) {
				month = new Integer(m_comboMonth.getText());
				m_log.debug("908 " + month);
			}
			//日を取得
			if (m_comboDay.getText().length() > 0) {
				day = new Integer(m_comboDay.getText());
			}
			//時を取得
			if (m_comboHours1.getText().length() > 0) {
				hours = new Integer(m_comboHours1.getText());
			}
			//分を取得
			if (m_comboMinutes1.getText().length() > 0) {
				minutes = new Integer(m_comboMinutes1.getText());
			}

			//スケジュール種別ごとにチェック
			schedule.setType(ScheduleConstant.TYPE_DAY);
		} else if (m_type2.getSelection()) {
			schedule.setType(ScheduleConstant.TYPE_WEEK);

			//曜日を取得
			if (m_comboDayOfWeek.getText().length() > 0) {
				week = new Integer(DayOfWeekConstant
						.stringToType(m_comboDayOfWeek.getText()));
			}
			//時を取得
			if (m_comboHours2.getText().length() > 0) {
				hours = new Integer(m_comboHours2.getText());
			}
			//分を取得
			if (m_comboMinutes2.getText().length() > 0) {
				minutes = new Integer(m_comboMinutes2.getText());
			}
		}

		//日時を設定
		schedule.setMonth(month);
		schedule.setDay(day);
		schedule.setWeek(week);
		schedule.setHour(hours);
		schedule.setMinute(minutes);
		m_jobSchedule.setSchedule(schedule);

		//有効/無効取得
		if (m_valid.getSelection()) {
			m_jobSchedule.setValid(ValidConstant.TYPE_VALID);
		} else {
			m_jobSchedule.setValid(ValidConstant.TYPE_INVALID);
		}

		return result;
	}

	/**
	 * ＯＫボタンテキスト取得
	 * 
	 * @return ＯＫボタンのテキスト
	 */
	@Override
	protected String getOkButtonText() {
		return Messages.getString("register");
	}

	/**
	 * キャンセルボタンテキスト取得
	 * 
	 * @return キャンセルボタンのテキスト
	 */
	@Override
	protected String getCancelButtonText() {
		return Messages.getString("cancel");
	}

	/**
	 * 入力値を保持したプロパティを返します。<BR>
	 * プロパティオブジェクトのコピーを返します。
	 * 
	 * @return プロパティ
	 * 
	 * @see com.clustercontrol.util.PropertyUtil#copy(Property)
	 */
	@Override
	protected ValidateResult validate() {
		ValidateResult result = null;

		result = createJobSchedule();
		if (result != null) {
			return result;
		}

		return null;
	}

	/**
	 * 所属ジョブユニットのジョブIDを返します。<BR>
	 * @return 所属ジョブユニットのジョブID
	 */
	public String getJobunitId() {
		return m_jobunitId;
	}

	/**
	 * 所属ジョブユニットのジョブIDを設定します。<BR>
	 * @param jobunitId 所属ジョブユニットのジョブID
	 */
	public void setJobunitId(String jobunitId) {
		m_jobunitId = jobunitId;
	}

	@Override
	protected boolean action() {
		boolean result = false;
		try {
			if (m_schedule != null) {
				//登録者、登録日時、更新日時を設定
				m_jobSchedule.setCreateUser((String) m_schedule
						.get(GetScheduleTableDefine.CREATE_USER));
				m_jobSchedule.setCreateTime(((Date) m_schedule
						.get(GetScheduleTableDefine.CREATE_TIME)).getTime());
				m_jobSchedule.setUpdateTime(((Date) m_schedule
						.get(GetScheduleTableDefine.UPDATE_TIME)).getTime());

				JobEndpointWrapper.modifySchedule(m_jobSchedule);
				MessageDialog.openInformation(null, Messages.getString("successful"),
						Messages.getString("message.job.77"));
			} else {
				JobEndpointWrapper.addSchedule(m_jobSchedule);
				MessageDialog.openInformation(null, Messages.getString("successful"),
						Messages.getString("message.job.79"));
			}
			result = true;
		} catch (InvalidRole_Exception e) {
			// アクセス権なしの場合、エラーダイアログを表示する
			MessageDialog.openInformation(
					null,
					Messages.getString("message"),
					Messages.getString("message.accesscontrol.16"));
		} catch (InvalidUserPass_Exception e) {
			MessageDialog.openError(
					null,
					Messages.getString("failed"),
					Messages.getString("message.job.74") + ", " + e.getMessage());
		} catch (InvalidSetting_Exception e) {
			MessageDialog.openError(
					null,
					Messages.getString("failed"),
					Messages.getString("message.job.74") + ", " + e.getMessage());
		} catch (Exception e) {
			m_log.warn("action(), " + e.getMessage(), e);
			MessageDialog.openError(
					null,
					Messages.getString("failed"),
					Messages.getString("message.hinemos.failure.unexpected") + ", " + e.getMessage());
		}
		return result;
	}
}