/*

Copyright (C) 2007 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.maintenance.dialog;

import java.text.DecimalFormat;
import java.util.List;

import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Text;

import com.clustercontrol.bean.DayOfWeekConstant;
import com.clustercontrol.bean.Property;
import com.clustercontrol.bean.RequiredFieldColorConstant;
import com.clustercontrol.bean.ScheduleConstant;
import com.clustercontrol.bean.ValidConstant;
import com.clustercontrol.calendar.composite.CalendarIdListComposite;
import com.clustercontrol.composite.ComboModifyListener;
import com.clustercontrol.composite.NumberKeyListener;
import com.clustercontrol.dialog.CommonDialog;
import com.clustercontrol.dialog.ValidateResult;
import com.clustercontrol.maintenance.action.GetMaintenance;
import com.clustercontrol.maintenance.action.ModifyMaintenance;
import com.clustercontrol.maintenance.composite.MaintenanceTypeListComposite;
import com.clustercontrol.maintenance.util.MaintenanceEndpointWrapper;
import com.clustercontrol.maintenance.util.MaintenanceUtil;
import com.clustercontrol.notify.composite.NotifyInfoComposite;
import com.clustercontrol.util.Messages;
import com.clustercontrol.ws.common.Schedule;
import com.clustercontrol.ws.maintenance.InvalidRole_Exception;
import com.clustercontrol.ws.maintenance.MaintenanceDuplicate_Exception;
import com.clustercontrol.ws.maintenance.MaintenanceInfo;
import com.clustercontrol.ws.notify.NotifyRelationInfo;

/**
 * メンテナンス[メンテナンスの作成・変更]ダイアログクラスです。
 * 
 * @version 4.0.0
 * @since 2.2.0
 */
public class MaintenanceDialog extends CommonDialog {

	public static final int WIDTH_TITLE = 4;
	public static final int WIDTH_TEXT = 8;

	/** メンテナンスID用テキスト */
	private Text m_textMaintenanceId = null;
	/** 説明 */
	private Text m_textDescription = null;
	/** メンテナンス種別 */
	private MaintenanceTypeListComposite m_maintenance_type = null;
	/** 保存期間(日)用テキスト*/
	private Text m_textDataRetentionPeriod = null;
	/** カレンダID用テキスト */
	private CalendarIdListComposite m_calendarId = null;
	/** スケジュール日時指定用ラジオボタン */
	private Button m_type1 = null;
	/** スケジュール曜日指定用ラジオボタン */
	private Button m_type2 = null;
	/** 月指定用コンボボックス */
	private Combo m_comboMonth = null;
	/** 日指定用コンボボックス */
	private Combo m_comboDay = null;
	/** 時指定用コンボボックス */
	private Combo m_comboHours1 = null;
	/** 分指定用コンボボックス */
	private Combo m_comboMinutes1 = null;
	/** 曜日指定用コンボボックス */
	private Combo m_comboDayOfWeek = null;
	/** 時指定用コンボボックス */
	private Combo m_comboHours2 = null;
	/** 分指定用コンボボックス */
	private Combo m_comboMinutes2 = null;
	/** メンテナンスID */
	private String maintenanceId = null;
	/** 通知情報 */
	private NotifyInfoComposite notifyInfo = null;
	/** この設定を有効にする */
	private Button confirmValid = null;

	private MaintenanceInfo maintenanceInfo = null;

	/**
	 * コンストラクタ
	 * 
	 * @param parent 親シェル
	 */
	public MaintenanceDialog(Shell parent) {
		super(parent);
	}

	public MaintenanceDialog(Shell parent, String maintenanceId, boolean referenceFlg) {
		super(parent);

		this.maintenanceId = maintenanceId;
	}

	/**
	 * ダイアログエリアを生成します。
	 * 
	 * @param parent 親コンポジット
	 */
	@Override
	protected void customizeDialog(Composite parent) {
		Shell shell = this.getShell();

		// タイトル
		shell.setText(Messages.getString("dialog.history.delete.settings.modify"));

		// 変数として利用されるラベル
		Label label = null;
		// 変数として利用されるグリッドデータ
		GridData gridData = null;

		// レイアウト
		GridLayout layout = new GridLayout(1, true);
		layout.marginWidth = 10;
		layout.marginHeight = 10;
		layout.numColumns = 15;
		parent.setLayout(layout);

		/*
		 * メンテナンスID
		 */
		// ラベル
		label = new Label(parent, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = WIDTH_TITLE;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);
		label.setText(Messages.getString("maintenance.id") + " : ");
		// テキスト
		this.m_textMaintenanceId = new Text(parent, SWT.BORDER | SWT.LEFT);
		gridData = new GridData();
		gridData.horizontalSpan = WIDTH_TEXT;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		this.m_textMaintenanceId.setLayoutData(gridData);
		if(this.maintenanceId != null && !"".equals(this.maintenanceId.trim())){
			this.m_textMaintenanceId.setEnabled(false);
		}
		this.m_textMaintenanceId.addModifyListener(new ModifyListener(){
			@Override
			public void modifyText(ModifyEvent arg0) {
				update();
			}
		});

		// 空白
		label = new Label(parent, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = 2;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);

		/*
		 * 説明
		 */
		// ラベル
		label = new Label(parent, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = WIDTH_TITLE;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);
		label.setText(Messages.getString("description") + " : ");
		// テキスト
		this.m_textDescription = new Text(parent, SWT.BORDER | SWT.LEFT);
		gridData = new GridData();
		gridData.horizontalSpan = WIDTH_TEXT;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		this.m_textDescription.setLayoutData(gridData);
		this.m_textDescription.addModifyListener(new ModifyListener(){
			@Override
			public void modifyText(ModifyEvent arg0) {
				update();
			}
		});

		// 空白
		label = new Label(parent, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = 2;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);
		/*
		 * 種別
		 */
		this.m_maintenance_type = new MaintenanceTypeListComposite(parent, SWT.NONE, true);
		gridData = new GridData();
		gridData.horizontalSpan = 15;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		this.m_maintenance_type.setLayoutData(gridData);
		this.m_maintenance_type.addModifyListener(new ModifyListener(){
			@Override
			public void modifyText(ModifyEvent arg0) {
				update();
			}
		});

		/*
		 * 保存期間(日)
		 */
		// ラベル
		label = new Label(parent, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = WIDTH_TITLE;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);
		label.setText(Messages.getString("maintenance.retention.period") + " : ");
		// テキスト
		this.m_textDataRetentionPeriod = new Text(parent, SWT.BORDER | SWT.LEFT);
		this.m_textDataRetentionPeriod.setToolTipText(Messages.getString("message.maintenance.23"));
		gridData = new GridData();
		gridData.horizontalSpan = WIDTH_TEXT;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		this.m_textDataRetentionPeriod.setLayoutData(gridData);
		this.m_textDataRetentionPeriod.addModifyListener(new ModifyListener(){
			@Override
			public void modifyText(ModifyEvent arg0) {
				update();
			}
		});

		/*
		 * カレンダ
		 */
		this.m_calendarId = new CalendarIdListComposite(parent, SWT.NONE, true);
		gridData = new GridData();
		gridData.horizontalSpan = 15;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		m_calendarId.setLayoutData(gridData);

		label = new Label(parent, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = 15;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);

		/*
		 * スケジュール
		 */
		Group group1 = new Group(parent, SWT.NONE);
		layout = new GridLayout(1, true);
		layout.marginWidth = 5;
		layout.marginHeight = 5;
		layout.numColumns = 20;
		group1.setLayout(layout);
		gridData = new GridData();
		gridData.horizontalSpan = 15;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		group1.setLayoutData(gridData);
		group1.setText(Messages.getString("schedule"));

		// スケジュール日時指定、曜日指定
		createType1(group1);
		createType2(group1);

		/*
		 * 通知情報の属性グループ
		 */
		// グループ
		Group groupNotifyAttribute = new Group(parent, SWT.NONE);
		layout = new GridLayout(1, true);
		layout.marginWidth = 5;
		layout.marginHeight = 5;
		layout.numColumns = 1;
		groupNotifyAttribute.setLayout(layout);
		groupNotifyAttribute.setText(Messages.getString("notify.attribute"));
		gridData = new GridData();
		gridData.horizontalSpan = 15;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		groupNotifyAttribute.setLayoutData(gridData);

		// 通知情報（通知ID，アプリケーションID）
		this.notifyInfo = new NotifyInfoComposite(groupNotifyAttribute, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = 1;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		notifyInfo.setLayoutData(gridData);

		/*
		 * 有効／無効
		 */
		this.confirmValid = new Button(parent, SWT.CHECK);
		gridData = new GridData();
		gridData.horizontalSpan = 15;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		this.confirmValid.setLayoutData(gridData);
		this.confirmValid.setText(Messages.getString("setting.valid.confirmed"));
		this.confirmValid.setSelection(true);

		// ラインを引く
		Label line = new Label(parent, SWT.SEPARATOR | SWT.HORIZONTAL);
		gridData = new GridData();
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		gridData.horizontalSpan = 15;
		line.setLayoutData(gridData);

		// サイズを最適化
		// グリッドレイアウトを用いた場合、こうしないと横幅が画面いっぱいになります。
		shell.pack();
		shell.setSize(new Point(550, shell.getSize().y));

		// 画面中央に
		Display display = shell.getDisplay();
		shell.setLocation((display.getBounds().width - shell.getSize().x) / 2,
				(display.getBounds().height - shell.getSize().y) / 2);


		//最後に変更であれば情報を表示する
		MaintenanceInfo info = null;
		if(this.maintenanceId != null){
			info = new GetMaintenance().getMaintenanceInfo(this.maintenanceId);
		}
		this.setInputData(info);
		this.reflectMaintenanceSchedule();
	}

	/* (非 Javadoc)
	 * @see org.eclipse.swt.widgets.Control#setEnabled(boolean)
	 */
	private void update(){
		// 各項目が必須項目であることを明示
		if(this.m_textMaintenanceId.getEnabled() && "".equals(this.m_textMaintenanceId.getText())){
			this.m_textMaintenanceId.setBackground(RequiredFieldColorConstant.COLOR_REQUIRED);
		}else{
			this.m_textMaintenanceId.setBackground(RequiredFieldColorConstant.COLOR_UNREQUIRED);
		}
		if("".equals(this.m_maintenance_type.getText())){
			this.m_maintenance_type.setBackground(RequiredFieldColorConstant.COLOR_REQUIRED);
		}else{
			this.m_maintenance_type.setBackground(RequiredFieldColorConstant.COLOR_UNREQUIRED);
		}
		if("".equals(this.m_textDataRetentionPeriod.getText())){
			this.m_textDataRetentionPeriod.setBackground(RequiredFieldColorConstant.COLOR_REQUIRED);
		}else{
			this.m_textDataRetentionPeriod.setBackground(RequiredFieldColorConstant.COLOR_UNREQUIRED);
		}

		if(this.m_comboHours1.getEnabled() && "".equals(this.m_comboHours1.getText())){
			this.m_comboHours1.setBackground(RequiredFieldColorConstant.COLOR_REQUIRED);
		}else{
			this.m_comboHours1.setBackground(RequiredFieldColorConstant.COLOR_UNREQUIRED);
		}

		if(this.m_comboMinutes1.getEnabled() && "".equals(this.m_comboMinutes1.getText())){
			this.m_comboMinutes1.setBackground(RequiredFieldColorConstant.COLOR_REQUIRED);
		}else{
			this.m_comboMinutes1.setBackground(RequiredFieldColorConstant.COLOR_UNREQUIRED);
		}

		if(this.m_comboDayOfWeek.getEnabled() && "".equals(this.m_comboDayOfWeek.getText())){
			this.m_comboDayOfWeek.setBackground(RequiredFieldColorConstant.COLOR_REQUIRED);
		}else{
			this.m_comboDayOfWeek.setBackground(RequiredFieldColorConstant.COLOR_UNREQUIRED);
		}

		if(this.m_comboHours2.getEnabled() && "".equals(this.m_comboHours2.getText())){
			this.m_comboHours2.setBackground(RequiredFieldColorConstant.COLOR_REQUIRED);
		}else{
			this.m_comboHours2.setBackground(RequiredFieldColorConstant.COLOR_UNREQUIRED);
		}

		if(this.m_comboMinutes2.getEnabled() && "".equals(this.m_comboMinutes2.getText())){
			this.m_comboMinutes2.setBackground(RequiredFieldColorConstant.COLOR_REQUIRED);
		}else{
			this.m_comboMinutes2.setBackground(RequiredFieldColorConstant.COLOR_UNREQUIRED);
		}
	}


	/**
	 * スケジュール日時指定を生成します。
	 * 
	 * @param parent 親グループ
	 */
	private void createType1(Group parent) {
		m_type1 = new Button(parent, SWT.RADIO);
		GridData gridData = new GridData();
		gridData.horizontalSpan = 2;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		m_type1.setLayoutData(gridData);
		m_type1.addSelectionListener(new SelectionListener() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				Button check = (Button) e.getSource();
				if (check.getSelection()) {
					m_type2.setSelection(false);
					m_comboMonth.setEnabled(true);
					m_comboDay.setEnabled(true);
					m_comboHours1.setEnabled(true);
					m_comboMinutes1.setEnabled(true);
					m_comboDayOfWeek.setEnabled(false);
					m_comboHours2.setEnabled(false);
					m_comboMinutes2.setEnabled(false);
				}
				update();
			}

			@Override
			public void widgetDefaultSelected(SelectionEvent e) {

			}
		});

		m_comboMonth = new Combo(parent, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = 2;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		m_comboMonth.setLayoutData(gridData);
		m_comboMonth.setTextLimit(2);
		m_comboMonth.setVisibleItemCount(10);
		m_comboMonth.addKeyListener(new NumberKeyListener());
		m_comboMonth.addModifyListener(new ComboModifyListener());
		DecimalFormat format = new DecimalFormat("00");
		m_comboMonth.add("");
		for (int month = 1; month <= 12; month++) {
			m_comboMonth.add(format.format(month));
		}

		Label labelMonth = new Label(parent, SWT.NONE);
		labelMonth.setText(Messages.getString("month"));
		gridData = new GridData();
		gridData.horizontalSpan = 2;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		labelMonth.setLayoutData(gridData);

		m_comboDay = new Combo(parent, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = 2;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		m_comboDay.setLayoutData(gridData);
		m_comboDay.setTextLimit(2);
		m_comboDay.setVisibleItemCount(10);
		m_comboDay.addKeyListener(new NumberKeyListener());
		m_comboDay.addModifyListener(new ComboModifyListener());
		format = new DecimalFormat("00");
		m_comboDay.add("");
		for (int day = 1; day <= 31; day++) {
			m_comboDay.add(format.format(day));
		}

		Label labelDay = new Label(parent, SWT.NONE);
		labelDay.setText(Messages.getString("monthday"));
		gridData = new GridData();
		gridData.horizontalSpan = 2;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		labelDay.setLayoutData(gridData);

		m_comboHours1 = new Combo(parent, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = 2;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		m_comboHours1.setLayoutData(gridData);
		m_comboHours1.setTextLimit(2);
		m_comboHours1.setVisibleItemCount(10);
		m_comboHours1.addKeyListener(new NumberKeyListener());
		m_comboHours1.addModifyListener(new ComboModifyListener());
		format = new DecimalFormat("00");
		m_comboHours1.add("");
		for (int hour = 0; hour < 24; hour++) {
			m_comboHours1.add(format.format(hour));
		}
		this.m_comboHours1.addModifyListener(new ModifyListener(){
			@Override
			public void modifyText(ModifyEvent arg0) {
				update();
			}
		});

		Label labelHours = new Label(parent, SWT.NONE);
		labelHours.setText(Messages.getString("hour"));
		gridData = new GridData();
		gridData.horizontalSpan = 2;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		labelHours.setLayoutData(gridData);

		m_comboMinutes1 = new Combo(parent, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = 2;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		m_comboMinutes1.setLayoutData(gridData);
		m_comboMinutes1.setTextLimit(2);
		m_comboMinutes1.setVisibleItemCount(10);
		m_comboMinutes1.addKeyListener(new NumberKeyListener());
		m_comboMinutes1.addModifyListener(new ComboModifyListener());
		format = new DecimalFormat("00");
		m_comboMinutes1.add("");
		for (int minutes = 0; minutes < 60; minutes++) {
			m_comboMinutes1.add(format.format(minutes));
		}
		this.m_comboMinutes1.addModifyListener(new ModifyListener(){
			@Override
			public void modifyText(ModifyEvent arg0) {
				update();
			}
		});

		Label labelMinutes = new Label(parent, SWT.NONE);
		labelMinutes.setText(Messages.getString("minute"));
		gridData = new GridData();
		gridData.horizontalSpan = 2;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		labelMinutes.setLayoutData(gridData);

		// 空白
		Label label = new Label(parent, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = 2;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);
	}

	/**
	 * スケジュール曜日指定を生成します。
	 * 
	 * @param parent 親グループ
	 */
	private void createType2(Group parent) {
		m_type2 = new Button(parent, SWT.RADIO);
		GridData gridData = new GridData();
		gridData.horizontalSpan = 2;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		m_type2.setLayoutData(gridData);
		m_type2.addSelectionListener(new SelectionListener() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				Button check = (Button) e.getSource();
				if (check.getSelection()) {
					m_type1.setSelection(false);
					m_comboMonth.setEnabled(false);
					m_comboDay.setEnabled(false);
					m_comboHours1.setEnabled(false);
					m_comboMinutes1.setEnabled(false);
					m_comboDayOfWeek.setEnabled(true);
					m_comboHours2.setEnabled(true);
					m_comboMinutes2.setEnabled(true);
				}
				update();
			}

			@Override
			public void widgetDefaultSelected(SelectionEvent e) {

			}
		});

		Label labelDayOfWeek = new Label(parent, SWT.NONE);
		labelDayOfWeek.setText(Messages.getString("weekday"));
		gridData = new GridData();
		gridData.horizontalSpan = 2;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		labelDayOfWeek.setLayoutData(gridData);

		m_comboDayOfWeek = new Combo(parent, SWT.READ_ONLY);
		gridData = new GridData();
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		gridData.horizontalSpan = 4;
		m_comboDayOfWeek.setLayoutData(gridData);
		m_comboDayOfWeek.setTextLimit(3);
		m_comboDayOfWeek.setVisibleItemCount(10);
		m_comboDayOfWeek.add(DayOfWeekConstant.STRING_SUNDAY);
		m_comboDayOfWeek.add(DayOfWeekConstant.STRING_MONDAY);
		m_comboDayOfWeek.add(DayOfWeekConstant.STRING_TUESDAY);
		m_comboDayOfWeek.add(DayOfWeekConstant.STRING_WEDNESDAY);
		m_comboDayOfWeek.add(DayOfWeekConstant.STRING_THURSDAY);
		m_comboDayOfWeek.add(DayOfWeekConstant.STRING_FRIDAY);
		m_comboDayOfWeek.add(DayOfWeekConstant.STRING_SATURDAY);
		this.m_comboDayOfWeek.addModifyListener(new ModifyListener(){
			@Override
			public void modifyText(ModifyEvent arg0) {
				update();
			}
		});

		Label labelDummy = new Label(parent, SWT.NONE);
		labelDummy.setText("");
		gridData = new GridData();
		gridData.horizontalSpan = 2;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		labelDummy.setLayoutData(gridData);

		m_comboHours2 = new Combo(parent, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = 2;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		m_comboHours2.setLayoutData(gridData);
		m_comboHours2.setTextLimit(2);
		m_comboHours2.setVisibleItemCount(10);
		m_comboHours2.addKeyListener(new NumberKeyListener());
		m_comboHours2.addModifyListener(new ComboModifyListener());
		DecimalFormat format = new DecimalFormat("00");
		m_comboHours2.add("");
		for (int hour = 0; hour < 24; hour++) {
			m_comboHours2.add(format.format(hour));
		}
		this.m_comboHours2.addModifyListener(new ModifyListener(){
			@Override
			public void modifyText(ModifyEvent arg0) {
				update();
			}
		});

		Label labelHours = new Label(parent, SWT.NONE);
		labelHours.setText(Messages.getString("hour"));
		gridData = new GridData();
		gridData.horizontalSpan = 2;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		labelHours.setLayoutData(gridData);

		m_comboMinutes2 = new Combo(parent, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = 2;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		m_comboMinutes2.setLayoutData(gridData);
		m_comboMinutes2.setTextLimit(2);
		m_comboMinutes2.setVisibleItemCount(10);
		m_comboMinutes2.addKeyListener(new NumberKeyListener());
		m_comboMinutes2.addModifyListener(new ComboModifyListener());
		format = new DecimalFormat("00");
		m_comboMinutes2.add("");
		for (int minutes = 0; minutes < 60; minutes++) {
			m_comboMinutes2.add(format.format(minutes));
		}
		this.m_comboMinutes2.addModifyListener(new ModifyListener(){
			@Override
			public void modifyText(ModifyEvent arg0) {
				update();
			}
		});

		Label labelMinutes = new Label(parent, SWT.NONE);
		labelMinutes.setText(Messages.getString("minute"));
		gridData = new GridData();
		gridData.horizontalSpan = 2;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		labelMinutes.setLayoutData(gridData);

		// 空白
		Label label = new Label(parent, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = 2;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);
	}

	/**
	 * ダイアログにスケジュール情報を反映します。
	 * 
	 * @see com.clustercontrol.jobmanagement.bean.ScheduleTableDefine
	 */
	private void reflectMaintenanceSchedule() {

		// 新規作成の場合はこちらのルートを通る。
		if (maintenanceId == null) {

			m_type1.setSelection(true);
			m_comboMonth.setEnabled(true);
			m_comboDay.setEnabled(true);
			m_comboHours1.setEnabled(true);
			m_comboMinutes1.setEnabled(true);
			m_comboDayOfWeek.setEnabled(false);
			m_comboHours2.setEnabled(false);
			m_comboMinutes2.setEnabled(false);
			this.update();
			return;
		}else{
			maintenanceInfo = new GetMaintenance().getMaintenanceInfo(maintenanceId);
			//スケジュール設定
			Schedule schedule = maintenanceInfo.getSchedule();

			DecimalFormat format = new DecimalFormat("00");
			//日時を設定
			if (schedule.getType() == ScheduleConstant.TYPE_DAY) {
				m_comboMonth.select(0);
				//月を設定
				for (int i = 0; i < m_comboMonth.getItemCount(); i++) {
					if (schedule.getMonth() == null) {
						break;
					}
					String month = format.format(schedule.getMonth());
					if (month.equals(m_comboMonth.getItem(i))) {
						m_comboMonth.select(i);
						break;
					}
				}
				//日を設定
				m_comboDay.select(0);
				for (int i = 0; i < m_comboDay.getItemCount(); i++) {
					if (schedule.getDay() == null) {
						break;
					}
					String day = format.format(schedule.getDay());
					if (day.equals(m_comboDay.getItem(i))) {
						m_comboDay.select(i);
						break;
					}
				}
				//時を設定
				m_comboHours1.select(0);
				m_comboHours2.select(0);
				for (int i = 0; i < m_comboHours1.getItemCount(); i++) {
					if (schedule.getHour() == null) {
						break;
					}
					String hours = format.format(schedule.getHour());
					if (hours.equals(m_comboHours1.getItem(i))) {
						m_comboHours1.select(i);
						break;
					}
				}
				//分を設定
				m_comboMinutes1.select(0);
				m_comboMinutes2.select(0);
				for (int i = 0; i < m_comboMinutes1.getItemCount(); i++) {
					if (schedule.getMinute() == null) {
						break;
					}
					String minutes = format.format(schedule.getMinute());
					if (minutes.equals(m_comboMinutes1.getItem(i))) {
						m_comboMinutes1.select(i);
						break;
					}
				}
			} else {
				//曜日を設定
				m_comboDayOfWeek.select(0);
				String dayOfWeek = DayOfWeekConstant.typeToString(schedule.getWeek());
				for (int i = 0; i < m_comboDayOfWeek.getItemCount(); i++) {
					if (dayOfWeek.equals(m_comboDayOfWeek.getItem(i))) {
						m_comboDayOfWeek.select(i);
						break;
					}
				}
				//時を設定
				m_comboHours1.select(0);
				m_comboHours2.select(0);
				for (int i = 0; i < m_comboHours2.getItemCount(); i++) {
					if (schedule.getHour() == null) {
						break;
					}
					String hours = format.format(schedule.getHour());
					if (hours.equals(m_comboHours2.getItem(i))) {
						m_comboHours2.select(i);
						break;
					}
				}
				//分を設定
				m_comboMinutes1.select(0);
				m_comboMinutes2.select(0);
				for (int i = 0; i < m_comboMinutes2.getItemCount(); i++) {
					if (schedule.getMinute() == null) {
						break;
					}
					String minutes = format.format(schedule.getMinute());
					if (minutes.equals(m_comboMinutes2.getItem(i))) {
						m_comboMinutes2.select(i);
						break;
					}
				}

			}

			//指定方式を設定
			if (schedule.getType() == ScheduleConstant.TYPE_DAY) {
				m_type1.setSelection(true);
				m_comboMonth.setEnabled(true);
				m_comboDay.setEnabled(true);
				m_comboHours1.setEnabled(true);
				m_comboMinutes1.setEnabled(true);
				m_comboDayOfWeek.setEnabled(false);
				m_comboHours2.setEnabled(false);
				m_comboMinutes2.setEnabled(false);
			} else if (schedule.getType() == ScheduleConstant.TYPE_WEEK) {
				m_type2.setSelection(true);
				m_comboMonth.setEnabled(false);
				m_comboDay.setEnabled(false);
				m_comboHours1.setEnabled(false);
				m_comboMinutes1.setEnabled(false);
				m_comboDayOfWeek.setEnabled(true);
				m_comboHours2.setEnabled(true);
				m_comboMinutes2.setEnabled(true);
			}
			this.update();
		}
	}

	/**
	 * ダイアログの情報からスケジュール情報を作成します。
	 * 
	 * @return 入力値の検証結果
	 * 
	 * @see com.clustercontrol.jobmanagement.bean.ScheduleTableDefine
	 */
	private ValidateResult createMaintenanceInfo() {
		ValidateResult result = null;

		if (maintenanceInfo == null) {
			maintenanceInfo = new MaintenanceInfo();
		}

		//メンテナンスID取得
		if (m_textMaintenanceId.getText().length() > 0) {
			maintenanceInfo.setMaintenanceId(m_textMaintenanceId.getText());
		}

		//説明取得
		if (m_textDescription.getText().length() > 0) {
			maintenanceInfo.setDescription(m_textDescription.getText());
		}

		//種別
		if(m_maintenance_type.getSelectionIndex() >= 0){
			maintenanceInfo.setTypeId(m_maintenance_type.getSelectionTypeId());
		}
		//保存期間(日)となる期間を指定
		if(m_textDataRetentionPeriod.getText().length() > 0){
			try{
				int period = Integer.valueOf(m_textDataRetentionPeriod.getText());
				maintenanceInfo.setDataRetentionPeriod(new Integer(period));
			} catch (NumberFormatException e) {
				result = new ValidateResult();
				result.setValid(false);
				result.setID(Messages.getString("message.hinemos.1"));
				result.setMessage(Messages.getString("message.maintenance.17"));
				return result;
			}
		} else {
			maintenanceInfo.setDataRetentionPeriod(null);
		}

		//カレンダID
		if (m_calendarId.getText().length() > 0) {
			maintenanceInfo.setCalendarId(m_calendarId.getText());
		} else {
			maintenanceInfo.setCalendarId(null);
		}

		//条件設定
		Schedule schedule = new Schedule();
		Integer month = null;
		Integer day = null;
		Integer hours = null;
		Integer minutes = null;
		Integer week = null;
		if (m_type1.getSelection()) {
			//月を取得
			if (m_comboMonth.getText().length() > 0) {
				month = new Integer(m_comboMonth.getText());
			}
			//日を取得
			if (m_comboDay.getText().length() > 0) {
				day = new Integer(m_comboDay.getText());
			}
			//時を取得
			if (m_comboHours1.getText().length() > 0) {
				hours = new Integer(m_comboHours1.getText());
			}
			//分を取得
			if (m_comboMinutes1.getText().length() > 0) {
				minutes = new Integer(m_comboMinutes1.getText());
			}

			//スケジュール種別
			schedule.setType(ScheduleConstant.TYPE_DAY);

		} else if (m_type2.getSelection()) {
			schedule.setType(ScheduleConstant.TYPE_WEEK);

			//曜日を取得
			if (m_comboDayOfWeek.getText().length() > 0) {
				week = new Integer(DayOfWeekConstant
						.stringToType(m_comboDayOfWeek.getText()));
			}
			//時を取得
			if (m_comboHours2.getText().length() > 0) {
				hours = new Integer(m_comboHours2.getText());
			}
			//分を取得
			if (m_comboMinutes2.getText().length() > 0) {
				minutes = new Integer(m_comboMinutes2.getText());
			}
		}

		//日時を設定
		schedule.setMonth(month);
		schedule.setDay(day);
		schedule.setWeek(week);
		schedule.setHour(hours);
		schedule.setMinute(minutes);
		maintenanceInfo.setSchedule(schedule);

		//通知関連情報の設定
		//通知グループIDの設定
		maintenanceInfo.setNotifyGroupId(MaintenanceUtil.createNotifyGroupIdMaintenance(maintenanceInfo.getMaintenanceId()));
		//通知情報へのNotifyGroupIdのセット
		this.notifyInfo.setNotifyGroupId(maintenanceInfo.getNotifyGroupId());
		//コンポジットから通知情報を取得します。
		List<NotifyRelationInfo> notifyRelationInfoList = maintenanceInfo.getNotifyId();
		notifyRelationInfoList.clear();
		if (this.notifyInfo.getNotify() != null) {
			notifyRelationInfoList.addAll(this.notifyInfo.getNotify());
		}

		//アプリケーションを取得
		maintenanceInfo.setApplication(notifyInfo.getApplication());

		//有効/無効取得
		if (confirmValid.getSelection()) {
			maintenanceInfo.setValidFlg(ValidConstant.TYPE_VALID);
		} else {
			maintenanceInfo.setValidFlg(ValidConstant.TYPE_INVALID);
		}

		return result;
	}

	private void setInputData(MaintenanceInfo info) {
		if(info != null){
			if (info.getMaintenanceId() != null) {
				this.maintenanceId = info.getMaintenanceId();
				this.m_textMaintenanceId.setText(this.maintenanceId);
			}

			if (info.getDescription() != null) {
				this.m_textDescription.setText(info.getDescription());
			}

			if (info.getDataRetentionPeriod() != null) {
				this.m_textDataRetentionPeriod.setText(info.getDataRetentionPeriod().toString());
			}

			if (info.getCalendarId() != null) {
				if (!m_calendarId.getEnabled()) {
					// カレンダー参照権限がない場合
					m_calendarId.addText(info.getCalendarId());
				}
				this.m_calendarId.setText(info.getCalendarId());
			}

			if (info.getTypeId() != null) {
				this.m_maintenance_type.setText(m_maintenance_type.getMaintenanceTypeName(info.getTypeId()));
			}

			//通知情報の設定
			if(info.getNotifyId() != null)
				this.notifyInfo.setNotify(info.getNotifyId());


			if (info.getApplication() != null) {
				this.notifyInfo.setApplication(info.getApplication());
				this.notifyInfo.update();
			}

			if (info.getValidFlg() != null) {
				this.confirmValid.setSelection(ValidConstant.typeToBoolean(info.getValidFlg().intValue()));
			}
		}
		// 各項目が必須項目であることを明示
		this.update();

	}

	/**
	 * ＯＫボタンテキスト取得
	 * 
	 * @return ＯＫボタンのテキスト
	 */
	@Override
	protected String getOkButtonText() {
		return Messages.getString("ok");
	}

	/**
	 * キャンセルボタンテキスト取得
	 * 
	 * @return キャンセルボタンのテキスト
	 */
	@Override
	protected String getCancelButtonText() {
		return Messages.getString("cancel");
	}

	/**
	 * 入力値を保持したプロパティを返します。<BR>
	 * プロパティオブジェクトのコピーを返します。
	 * 
	 * @return プロパティ
	 * 
	 * @see com.clustercontrol.util.PropertyUtil#copy(Property)
	 */
	@Override
	protected ValidateResult validate() {
		ValidateResult result = null;

		result = createMaintenanceInfo();
		if (result != null) {
			return result;
		}

		return null;
	}

	/**
	 * 入力値をマネージャに登録します。
	 * 
	 * @return true：正常、false：異常
	 * 
	 * @see com.clustercontrol.dialog.CommonDialog#action()
	 */
	@Override
	protected boolean action() {
		boolean result = false;

		if(this.maintenanceInfo != null){
			if(this.maintenanceId == null){
				// 作成の場合
				String[] args = { maintenanceInfo.getMaintenanceId() };
				try {
					MaintenanceEndpointWrapper.addMaintenance(maintenanceInfo);

					MessageDialog.openInformation(
							null,
							Messages.getString("successful"),
							Messages.getString("message.maintenance.1", args));

					result = true;

				} catch (MaintenanceDuplicate_Exception e) {
					// メンテナンスIDが重複している場合、エラーダイアログを表示する
					MessageDialog.openInformation(
							null,
							Messages.getString("message"),
							Messages.getString("message.maintenance.11", args));
				} catch (Exception e) {
					String errMessage = "";
					if (e instanceof InvalidRole_Exception) {
						MessageDialog.openInformation(null, Messages.getString("message"),
								Messages.getString("message.accesscontrol.16"));
					} else {
						errMessage = ", " + e.getMessage();
					}
					MessageDialog.openError(
							null,
							Messages.getString("failed"),
							Messages.getString("message.maintenance.2", args) + errMessage);
				}

			} else{
				// 変更の場合
				result = new ModifyMaintenance().modify(maintenanceInfo);
			}
		}

		return result;
	}
}