/*

Copyright (C) since 2006 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.commons.jmx;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.IOException;
import java.net.InetSocketAddress;
import java.net.URL;
import java.security.KeyStore;
import java.security.SecureRandom;
import java.sql.Connection;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.Collections;
import java.util.concurrent.Executors;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

import javax.management.ObjectName;
import javax.naming.InitialContext;
import javax.naming.NamingException;
import javax.net.ssl.KeyManagerFactory;
import javax.net.ssl.SSLContext;
import javax.net.ssl.TrustManagerFactory;
import javax.xml.ws.Endpoint;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jboss.jmx.adaptor.rmi.RMIAdaptor;
import org.jboss.system.ServiceMBeanSupport;
import org.quartz.Scheduler;
import org.quartz.SchedulerException;
import org.quartz.Trigger;

import com.clustercontrol.commons.util.ConnectionManager;
import com.clustercontrol.commons.util.HinemosProperties;
import com.clustercontrol.commons.util.QuartzCommonJobSetupUtil;
import com.clustercontrol.hinemosagent.util.AgentConnectUtil;
import com.clustercontrol.util.XMLUtil;
import com.clustercontrol.util.apllog.AplLogger;
import com.clustercontrol.ws.endpoint.AccessEndpoint;
import com.clustercontrol.ws.endpoint.AgentEndpoint;
import com.clustercontrol.ws.endpoint.CalendarEndpoint;
import com.clustercontrol.ws.endpoint.CloudCommonEndpoint;
import com.clustercontrol.ws.endpoint.CollectiveRunEndpoint;
import com.clustercontrol.ws.endpoint.CollectorEndpoint;
import com.clustercontrol.ws.endpoint.JobEndpoint;
import com.clustercontrol.ws.endpoint.JobMapEndpoint;
import com.clustercontrol.ws.endpoint.MailTemplateEndpoint;
import com.clustercontrol.ws.endpoint.MaintenanceEndpoint;
import com.clustercontrol.ws.endpoint.MonitorEndpoint;
import com.clustercontrol.ws.endpoint.MonitorSettingEndpoint;
import com.clustercontrol.ws.endpoint.MonitorSnmpTrapEndpoint;
import com.clustercontrol.ws.endpoint.NodeMapEndpoint;
import com.clustercontrol.ws.endpoint.NotifyEndpoint;
import com.clustercontrol.ws.endpoint.PerformanceCollectMasterEndpoint;
import com.clustercontrol.ws.endpoint.RepositoryEndpoint;
import com.sun.net.httpserver.HttpsConfigurator;
import com.sun.net.httpserver.HttpsServer;

/**
 * Hinemosサービスの制御クラス<BR>
 *
 * @version 4.0.0
 * @since 3.2.0
 */
public class HinemosService extends ServiceMBeanSupport implements HinemosServiceMBean {

	private static Log m_log = LogFactory.getLog( HinemosService.class );

	/** XML のInvalidな文字を置換する場合の置換文字のキー */
	private static final String MESSAGE_REPLACE_METHOD_KEY = "common.xml.invalid.char.replace";
	
	/** XML のInvalidな文字を置換する場合の置換文字のキー */
	private static final String MESSAGE_REPLACE_CHAR_KEY = "common.xml.invalid.char.replace.to";

	/** DBMSスケジューラのJNDI名 */
	private String jndiNameDBMSScheduler = null;

	/** RAMスケジューラのJNDI名 */
	private String jndiNameRAMScheduler = null;

	/** スケジューラの開始遅延時間（起動処理に時間がかかる場合にスケジューラの開始を遅らせることが可能） */
	private int delay = 10;

	private static ScheduledExecutorService scheduler;
	
	public static final int _threadPoolSize;
	public static final int _queueSize;
	private static final ThreadPoolExecutor _threadPool;
	
	public static final int _threadPoolSizeForAgent;
	public static final int _queueSizeForAgent;
	private static final ThreadPoolExecutor _threadPoolForAgent;

	/**
	 * 停止時に利用するEndpointのリスト。
	 */
	private static ArrayList<Endpoint> endpointList = new ArrayList<Endpoint>();
	
	/**
	 * HTTPS通信時に利用
	 */
	private static HttpsServer httpsServer = null;

	static {
		_threadPoolSize = Integer.parseInt(HinemosProperties.getProperty("common.ws.threadpool.size" , "4"));
		_queueSize = Integer.parseInt(HinemosProperties.getProperty("common.ws.queue.size" , "300"));
		
		_threadPool = new ThreadPoolExecutor(_threadPoolSize, _threadPoolSize, 0L, TimeUnit.MICROSECONDS, 
				new LinkedBlockingQueue<Runnable>(_queueSize), 
				new ThreadFactory() {
				private volatile int _count = 0;
					@Override
					public Thread newThread(Runnable r) {
						return new Thread(r, "WebServiceWorker-" + _count++);
					}
				}, new ThreadPoolExecutor.AbortPolicy());
		
		_threadPoolSizeForAgent = Integer.parseInt(HinemosProperties.getProperty("agent.ws.threadpool.size" , "8"));
		_queueSizeForAgent = Integer.parseInt(HinemosProperties.getProperty("agent.ws.queue.size" , "1200"));
		
		_threadPoolForAgent = new ThreadPoolExecutor(_threadPoolSizeForAgent, _threadPoolSizeForAgent, 0L, TimeUnit.MICROSECONDS, 
				new LinkedBlockingQueue<Runnable>(_queueSizeForAgent), 
				new ThreadFactory() {
				private volatile int _count = 0;
					@Override
					public Thread newThread(Runnable r) {
						return new Thread(r, "WebServiceWorkerForAgent-" + _count++);
					}
				}, new ThreadPoolExecutor.AbortPolicy());
		
		String replaceMethodString = HinemosProperties.getProperty(MESSAGE_REPLACE_METHOD_KEY, "false");
		if("true".equals(replaceMethodString)){
			XMLUtil.setReplace(true);
		} else {
			XMLUtil.setReplace(false);
		}
		
		String replaceCharString = HinemosProperties.getProperty(MESSAGE_REPLACE_CHAR_KEY, "?");
		if(replaceCharString != null){
			XMLUtil.setReplaceChar(replaceCharString);
		}
	}
	
	/**
	 * HinemosServiceのコンストラクタ
	 */
	public HinemosService() {
		jndiNameDBMSScheduler = null;
		jndiNameRAMScheduler = null;

		scheduler = Executors.newSingleThreadScheduledExecutor(
				new ThreadFactory() {
					@Override
					public Thread newThread(Runnable r) {
						return new Thread(r, "HinemosServiceScheduler");
					}
				}
		);
	}

	/**
	 * DBMSスケジューラのJNDI名を登録する
	 */
	@Override
	public void setJndiNameDBMSScheduler(String jndiName) {
		jndiNameDBMSScheduler = jndiName;
	}

	/**
	 * 設定されているDBMSスケジューラのJNDI名を取得する
	 */
	@Override
	public String getJndiNameDBMSScheduler() {
		return jndiNameDBMSScheduler;
	}

	/**
	 * RAMスケジューラのJNDI名を登録する
	 */
	@Override
	public void setJndiNameRAMScheduler(String jndiName) {
		jndiNameRAMScheduler = jndiName;
	}

	/**
	 * 設定されているRAMスケジューラのJNDI名を取得する
	 */
	@Override
	public String getJndiNameRAMScheduler() {
		return jndiNameRAMScheduler;
	}

	/**
	 * 設定されているスケジューラの状態を取得する
	 */
	@Override
	public String printSchedulerStatus() throws Exception {
		/** ローカル変数 */
		InitialContext initCtx = null;
		Scheduler scheduler = null;
		String msg1 = null;
		String msg2 = null;

		/** メイン処理 */
		msg1 = "";
		msg2 = "";
		try {
			initCtx = new InitialContext();

			scheduler = (Scheduler)initCtx.lookup(jndiNameDBMSScheduler);
			msg1 = "status of DBMS scheduler : ";
			if (scheduler.isPaused()) {
				msg1 = msg1 + "[ PAUSED ]";
			} else if (scheduler.isShutdown()) {
				msg1 = msg1 + "[ DISABLED ]";
			} else {
				msg1 = msg1 + "[ RUNNING ]";
			}

			scheduler = (Scheduler)initCtx.lookup(jndiNameRAMScheduler);
			msg2 = "status of RAM scheduler  : ";
			if (scheduler.isPaused()) {
				msg2 = msg2 + "[ PAUSED ]";
			} else if (scheduler.isShutdown()) {
				msg2 = msg2 + "[ DISABLED ]";
			} else {
				msg2 = msg2 + "[ RUNNING ]";
			}
		} catch (NamingException e) {
			log.warn("failure to lookup scheduler.", e);
			throw e;
		}

		return msg1 + System.getProperty("line.separator") + msg2;
	}

	/**
	 * 設定されているスケジューラ情報を取得する
	 */
	@Override
	public String printSchedulerDetail() throws Exception {
		/** ローカル変数 */
		InitialContext initCtx = null;
		Scheduler scheduler = null;
		Trigger trigger = null;

		String startFireTime = null;
		String prevFireTime = null;
		String nextFireTime = null;
		int state = 0;
		String msg = null;

		String lineSeparator = null;

		/** メイン処理 */
		lineSeparator = System.getProperty("line.separator");
		msg = "";
		try {
			initCtx = new InitialContext();

			msg = msg + lineSeparator;
			msg = msg + "- DBMS Scheduler - " + lineSeparator;
			msg = msg + lineSeparator;
			scheduler = (Scheduler)initCtx.lookup(jndiNameDBMSScheduler);
			for (String triggerGroupName : scheduler.getTriggerGroupNames()) {
				for (String triggerName : scheduler.getTriggerNames(triggerGroupName)) {
					trigger = scheduler.getTrigger(triggerName, triggerGroupName);
					state = scheduler.getTriggerState(triggerName, triggerGroupName);
					startFireTime = String.format("%1$tY-%1$tm-%1$td %1$tH:%1$tM:%1$tS", trigger.getStartTime());
					prevFireTime = String.format("%1$tY-%1$tm-%1$td %1$tH:%1$tM:%1$tS", trigger.getPreviousFireTime());
					nextFireTime = String.format("%1$tY-%1$tm-%1$td %1$tH:%1$tM:%1$tS", trigger.getNextFireTime());

					msg = msg + String.format("%s (in %s) :", triggerName, triggerGroupName) + lineSeparator;
					msg = msg + String.format("   start fire time - %s", startFireTime) + lineSeparator;
					msg = msg + String.format("   last fire time  - %s", prevFireTime) + lineSeparator;
					msg = msg + String.format("   next fire time  - %s", nextFireTime) + lineSeparator;
					msg = msg + String.format("   current state   - %s", state == Trigger.STATE_PAUSED ? "PAUSED" : "ACTIVE (not PAUSED)") + lineSeparator;
					msg = msg + lineSeparator;
				}
			}

			msg = msg + lineSeparator;
			msg = msg + "- RAM Scheduler -" + lineSeparator;
			msg = msg + lineSeparator;
			scheduler = (Scheduler)initCtx.lookup(jndiNameRAMScheduler);
			for (String triggerGroupName : scheduler.getTriggerGroupNames()) {
				for (String triggerName : scheduler.getTriggerNames(triggerGroupName)) {
					trigger = scheduler.getTrigger(triggerName, triggerGroupName);
					state = scheduler.getTriggerState(triggerName, triggerGroupName);
					startFireTime = String.format("%1$tY-%1$tm-%1$td %1$tH:%1$tM:%1$tS", trigger.getStartTime());
					prevFireTime = String.format("%1$tY-%1$tm-%1$td %1$tH:%1$tM:%1$tS", trigger.getPreviousFireTime());
					nextFireTime = String.format("%1$tY-%1$tm-%1$td %1$tH:%1$tM:%1$tS", trigger.getNextFireTime());

					msg = msg + String.format("%s (in %s) :", triggerName, triggerGroupName) + lineSeparator;
					msg = msg + String.format("   start fire time - %s", startFireTime) + lineSeparator;
					msg = msg + String.format("   last fire time  - %s", prevFireTime) + lineSeparator;
					msg = msg + String.format("   next fire time  - %s", nextFireTime) + lineSeparator;
					msg = msg + String.format("   current state   - %s", state == Trigger.STATE_PAUSED ? "PAUSED" : "ACTIVE (not PAUSED)") + lineSeparator;
					msg = msg + lineSeparator;
				}
			}

		} catch (NamingException e) {
			log.warn("failure to lookup scheduler.", e);
			throw e;
		}

		return msg;
	}

	/**
	 * 通知抑制の履歴情報（最終重要度および通知日時）をリセットする
	 */
	@Override
	public void resetNotificationLogger() throws Exception {
		/** ローカル変数 */
		Connection con = null;
		Statement stmt = null;
		InitialContext ic = new InitialContext();

		/** メイン処理 */
		log.info("resetting notification counter...");

		// DB内の抑制情報を初期化
		con = ConnectionManager.getConnectionManager().getConnection();
		con.setAutoCommit(false);
		stmt = con.createStatement();

		log.info("clean up cc_monitor_statue table (latest priority)...");
		stmt.executeUpdate("TRUNCATE TABLE cc_monitor_status");
		log.info("clean up cc_notify_history table (latest date)...");
		stmt.executeUpdate("TRUNCATE TABLE cc_notify_history");

		con.commit();
		stmt.close();
		con.close();

		// EJB Container内の抑制情報を初期化
		RMIAdaptor server = (RMIAdaptor) ic.lookup("jmx/invoker/RMIAdaptor");

		log.info("flush MonitorStatusLocal cache (latest priority)...");
		server.invoke(new ObjectName("jboss.j2ee:jndiName=MonitorStatusLocal,service=EJB"), "flushCache", null, null);
		log.info("flush NotifyHistoryLocal cache (latest date)...");
		server.invoke(new ObjectName("jboss.j2ee:jndiName=NotifyHistoryLocal,service=EJB"), "flushCache", null, null);

		log.info("notify counter resetted successfully.");
	}

	/**
	 * MBeanの初期化処理
	 */
	@Override
	public void createService() throws Exception {
		/** メイン処理 */
		log.info("creating a MBean of HinemosService...");
		log.info("successful in creating a MBean of HinemosService...");
	}

	/**
	 * MBeanの開始処理
	 */
	@Override
	public void startService() throws Exception {
		/** ローカル変数 */
		String mode = null;

		/** メイン処理 */
		log.info("starting a MBean of HinemosService...");
		mode = System.getProperty("hinemos.manager.mode");
		if (mode != null && mode.equals("maintenance")) {
			log.info("do nothing, because of maintenance mode.");
		} else {
			startScheduler(delay);
		}

		AplLogger apllog = new AplLogger("MNG", "mng");
		String[] args = {"start"};
		apllog.put("SYS", "001", args);
		
		String preAddress = HinemosProperties.getProperty("common.ws.address" , "http://0.0.0.0:8080");
		String preAgentAddress = preAddress;
		
		URL url = new URL(preAddress);

		if("https".equals(url.getProtocol())){
			preAgentAddress = HinemosProperties.getProperty("common.ws.agent.address" , "http://0.0.0.0:8080");
			
			String protocol = HinemosProperties.getProperty("common.ws.https.protocol", "TLS");
			String keystorePath = HinemosProperties.getProperty("common.ws.https.keystore.path", "/root/keystore");
			String keystorePassword = HinemosProperties.getProperty("common.ws.https.keystore.password", "hinemos");
			String keystoreType = HinemosProperties.getProperty("common.ws.https.keystore.type", "PKCS12");
			
			log.info("Starting HTTPS Server...");
			log.info("SSLContext: " + protocol + ", KeyStore: " + keystoreType);
			
			createHttpsServer(url.getHost(), url.getPort(), protocol, keystorePath, keystorePassword, keystoreType);
		}

		/** Webサービスの起動処理 */
		publish(preAgentAddress, "/HinemosWS/AgentEndpoint", new AgentEndpoint(), _threadPoolForAgent);
		publish(preAddress, "/HinemosWS/CalendarEndpoint", new CalendarEndpoint(), _threadPool);
		publish(preAgentAddress, "/HinemosWS/CloudCommonEndpoint", new CloudCommonEndpoint(), _threadPoolForAgent);
		publish(preAddress, "/HinemosWS/CollectiveRunEndpoint", new CollectiveRunEndpoint(), _threadPool);
		publish(preAddress, "/HinemosWS/CollectorEndpoint", new CollectorEndpoint(), _threadPool);
		publish(preAddress, "/HinemosWS/JobEndpoint", new JobEndpoint(), _threadPool);
		publish(preAddress, "/HinemosWS/MailTemplateEndpoint", new MailTemplateEndpoint(), _threadPool);
		publish(preAddress, "/HinemosWS/MaintenanceEndpoint", new MaintenanceEndpoint(), _threadPool);
		publish(preAddress, "/HinemosWS/MonitorEndpoint", new MonitorEndpoint(), _threadPool);
		publish(preAddress, "/HinemosWS/MonitorSettingEndpoint", new MonitorSettingEndpoint(), _threadPool);
		publish(preAddress, "/HinemosWS/MonitorSnmpTrapEndpoint", new MonitorSnmpTrapEndpoint(), _threadPool);
		publish(preAddress, "/HinemosWS/NotifyEndpoint", new NotifyEndpoint(), _threadPool);
		publish(preAddress, "/HinemosWS/RepositoryEndpoint", new RepositoryEndpoint(), _threadPool);
		publish(preAddress, "/HinemosWS/PerformanceCollectMasterEndpoint", new PerformanceCollectMasterEndpoint(), _threadPool);

		if (isNodeMap()) {
			publish(preAddress, "/HinemosWS/NodeMapEndpoint", new NodeMapEndpoint(), _threadPool);
		}

		if (isJobMap()) {
			publish(preAddress, "/HinemosWS/JobMapEndpoint", new JobMapEndpoint(), _threadPool);
		}

		// ログインは最後にpublishする。
		publish(preAddress, "/HinemosWS/AccessEndpoint", new AccessEndpoint(), _threadPool);

		if(httpsServer != null){
			httpsServer.start();
		}
	
		log.info("successful in starting a MBean of HinemosService...");
	}

	private boolean isJobMap() {
		return isOption("jobmap.key", "787d5539825b8c1bbff05b28de1ea502");
	}

	private boolean isNodeMap() {
		return isOption("nodemap.key", "e78d72f931f7d4f544022a68957b6a3b");
	}

	private boolean isOption(String keyFile, String md5) {
		boolean ret = false;
		String etcdir = System.getProperty("hinemos.manager.etc.dir");
		String keyPath = etcdir + File.separator + keyFile;
		FileReader fileReader = null;
		try {
			fileReader = new FileReader(keyPath);
			char[] cbuf = new char[128];
			fileReader.read(cbuf, 0, 128);
			String str = new String(cbuf);
			if (str != null && str.startsWith(md5)) {
				ret = true;
			}
		} catch (FileNotFoundException e) {
		} catch (Exception e){
			m_log.info("isOption " + e.getMessage());
		} finally {
			try {
				if (fileReader != null) {
					fileReader.close();
				}
			} catch (IOException e) {
			}
		}

		if (ret) {
			m_log.info(keyFile + " is active");
		} else {
			m_log.info(keyFile + " is not active");
		}
		return ret;
	}

	private void publish (String preAddress, String postAddress, Object endpointInstance, ThreadPoolExecutor threadPool) {
		try {
			String address = preAddress + postAddress;
			log.info("publish " + address);

			Endpoint endpoint = Endpoint.create(endpointInstance);
			endpoint.setExecutor(threadPool);
			if(httpsServer == null || endpointInstance instanceof AgentEndpoint || endpointInstance instanceof CloudCommonEndpoint){
				endpoint.publish(address);
			}
			else {
				endpoint.publish(httpsServer.createContext(postAddress));
			}
			endpointList.add(endpoint);
		} catch (Exception e) {
			log.error("publish " + e.getClass().getSimpleName() + ", " + e.getMessage(), e);
		}
	}
	
	private void createHttpsServer(String managerIP, int port, String protocol, String keyStorePath, String keyStorePassword, String keyStoreType) throws Exception{
		SSLContext ssl = SSLContext.getInstance(protocol);
		KeyManagerFactory keyFactory = KeyManagerFactory.getInstance(KeyManagerFactory.getDefaultAlgorithm());
		KeyStore store = KeyStore.getInstance(keyStoreType);
		store.load(new FileInputStream(keyStorePath), keyStorePassword.toCharArray());

		keyFactory.init(store, keyStorePassword.toCharArray());

		TrustManagerFactory trustFactory = TrustManagerFactory.getInstance(TrustManagerFactory.getDefaultAlgorithm());

		trustFactory.init(store);

		ssl.init(keyFactory.getKeyManagers(),
		trustFactory.getTrustManagers(), new SecureRandom());

		HttpsConfigurator configurator = new HttpsConfigurator(ssl);

		httpsServer = HttpsServer.create(new InetSocketAddress(managerIP, port), 0);
		httpsServer.setHttpsConfigurator(configurator);
	}

	/**
	 * MBeanの停止処理
	 */
	@Override
	public void stopService() throws Exception {
		/** メイン処理 */
		log.info("stopping a MBean of HinemosService...");
		_threadPool.shutdownNow();
		shutdownScheduler();

		/**
		 * webサービスの停止
		 */
		for (Endpoint endpoint : endpointList) {
			log.info("endpoint stop");
			try {
				/**
				 * JAX-WSの不具合により、0.0.0.0でlistenしているwebサービスは
				 * stop時にNullPointerExceptionが出て、stopできないようだ。
				 * http://java.net/jira/browse/JAX_WS-941
				 * 
				 * このため、JBoss終了時にwebサービスのアクセスがあると、
				 * jboss.logに、エラーのスタックトレースが出力される。
				 */
				endpoint.stop();
			} catch (NullPointerException e) {
				log.info("stop endpoint :  " + e.getMessage());
			} catch (Exception e) {
				log.warn("stop endpoint : " + e.getMessage());
			}
		}
		
		if(httpsServer != null){
			httpsServer.stop(0);
		}

		AplLogger apllog = new AplLogger("MNG", "mng");
		String[] args = {"shutdown"};
		apllog.put("SYS", "002", args);

		log.info("successful in stopping a MBean of HinemosService...");
	}

	/**
	 * MBeanの削除処理
	 */
	@Override
	public void destroyService() throws Exception {
		/** メイン処理 */
		log.info("destroying a MBean of HinemosService...");
		jndiNameDBMSScheduler = null;
		jndiNameRAMScheduler = null;
		log.info("successful in destroying a MBean of HinemosService...");
	}

	/**
	 * 全スケジューラを開始する
	 */
	@Override
	public void startScheduler() throws Exception {
		/** メイン処理 */
		startScheduler(0);
	}

	/**
	 * 遅延起時間を指定してのスケジューラを開始。
	 */
	public void startScheduler(int delay) throws Exception {

		// ジョブ管理および共通ジョブ用スケジューラを起動する。
		// ジョブはDBで永続化されているため起動直後からスケジュールされたジョブが実行される。
		startDBMSScheduler();

		// 監視機能用のスケジューラを起動する。
		// 監視設定はDBで永続化されていないためスケジューラを起動しても監視は再開されない。
		// 以下の遅延起動処理の中で起動する。
		startRAMScheduler();

		/** メイン処理2遅延起動 */
		log.info("start scheduler after " + delay + " second...");
		scheduler.schedule(
				new HinemosDelayServiceStarter(this),
				delay * 1000,
				TimeUnit.MILLISECONDS
		);
	}

	/**
	 * DBMSスケジューラを開始する
	 */
	@Override
	public void startDBMSScheduler() throws Exception {
		/** メイン処理 */
		log.info("starting a scheduler using DBMS...");
		QuartzCommonJobSetupUtil.setup();
		startScheduler(jndiNameDBMSScheduler);
		log.info("successful in starting a scheduler using DBMS...");
	}

	/**
	 * RAMスケジューラを開始する
	 */
	@Override
	public void startRAMScheduler() throws Exception {
		/** メイン処理 */
		log.info("starting a scheduler using RAM...");
		startScheduler(jndiNameRAMScheduler);
		log.info("successful in starting a scheduler using RAM...");
	}

	/**
	 * 全スケジューラを一時停止する
	 */
	@Override
	public void pauseScheduler() throws Exception {
		/** メイン処理 */
		pauseDBMSScheduler();
		pauseRAMScheduler();
	}

	/**
	 * DBMSスケジューラを一時停止する
	 */
	@Override
	public void pauseDBMSScheduler() throws Exception {
		/** メイン処理 */
		log.info("pausing a scheduler using DBMS...");
		pauseScheduler(jndiNameDBMSScheduler);
		log.info("successful in pausing a scheduler using DBMS...");
	}

	/**
	 * RAMスケジューラを一時停止する
	 */
	@Override
	public void pauseRAMScheduler() throws Exception {
		/** メイン処理 */
		log.info("pausing a scheduler using RAM...");
		pauseScheduler(jndiNameRAMScheduler);
		log.info("successful in pausing a scheduler using RAM...");
	}

	/**
	 * 全スケジューラを停止する
	 */
	public void shutdownScheduler() throws Exception {
		/** メイン処理 */
		shutdownDBMSScheduler();
		shutdownRAMScheduler();
	}

	/**
	 * DBMSスケジューラを停止する
	 * @throws Exception
	 */
	public void shutdownDBMSScheduler() throws Exception {
		/** メイン処理 */
		log.info("shutting down a scheduler using DBMS...");
		shutdownScheduler(jndiNameDBMSScheduler);
		log.info("successful in shutting down a scheduler using DBMS...");
	}

	/**
	 * RAMスケジューラを停止する
	 * @throws Exception
	 */
	public void shutdownRAMScheduler() throws Exception {
		/** メイン処理 */
		log.info("shutting down a scheduler using RAM...");
		shutdownScheduler(jndiNameRAMScheduler);
		log.info("successful in shutting down a scheduler using RAM...");
	}

	/**
	 * スケジューラを開始する
	 * @param jndiNameSchduler スケジューラのJNDI名
	 * @throws NamingException
	 * @throws SchedulerException
	 */
	private void startScheduler(String jndiNameSchduler) throws NamingException, SchedulerException {
		/** ローカル変数 */
		InitialContext initCtx = null;
		Scheduler scheduler = null;

		/** メイン処理 */
		try {
			initCtx = new InitialContext();
			scheduler = (Scheduler)initCtx.lookup(jndiNameSchduler);

			scheduler.start();
		} catch (NamingException e) {
			log.warn("failure to lookup scheduler. (jndiName = " + jndiNameSchduler + "", e);
			throw e;
		} catch (SchedulerException e) {
			log.warn("failure to start scheduler. (jndiName = " + jndiNameSchduler + "", e);
			throw e;
		}
	}

	/**
	 * スケジューラを一時停止する
	 * @param jndiNameSchduler スケジューラのJNDI名
	 * @throws NamingException
	 * @throws SchedulerException
	 */
	private void pauseScheduler(String jndiNameSchduler) throws NamingException, SchedulerException {
		/** ローカル変数 */
		InitialContext initCtx = null;
		Scheduler scheduler = null;

		/** メイン処理 */
		try {
			initCtx = new InitialContext();
			scheduler = (Scheduler)initCtx.lookup(jndiNameSchduler);

			scheduler.pause();
		} catch (NamingException e) {
			log.warn("failure to lookup scheduler. (jndiName = " + jndiNameSchduler + "", e);
			throw e;
		} catch (SchedulerException e) {
			log.warn("failure to pause scheduler. (jndiName = " + jndiNameSchduler + "", e);
			throw e;
		}
	}

	/**
	 * スケジューラを停止する
	 * @param jndiNameSchduler スケジューラのJNDI名
	 * @throws NamingException
	 * @throws SchedulerException
	 */
	private void shutdownScheduler(String jndiNameSchduler) throws NamingException, SchedulerException {
		/** ローカル変数 */
		InitialContext initCtx = null;
		Scheduler scheduler = null;

		/** メイン処理 */
		try {
			initCtx = new InitialContext();
			scheduler = (Scheduler)initCtx.lookup(jndiNameSchduler);

			scheduler.shutdown();
		} catch (NamingException e) {
			log.warn("failure to lookup scheduler. (jndiName = " + jndiNameSchduler + "", e);
			throw e;
		} catch (SchedulerException e) {
			log.warn("failure to shutdown scheduler. (jndiName = " + jndiNameSchduler + "", e);
			throw e;
		}
	}

	@Override
	public int getDelay() {
		return delay;
	}

	@Override
	public void setDelay(int delay) {
		this.delay = delay;
	}

	/**
	 * エージェント一覧を返します
	 * 下記のコマンドでエージェント一覧が取得可能
	 * /opt/hinemos/jboss-as/bin/twiddle.sh -s jnp://<IP-Address>:1099
	 *  invoke "user:service=HinemosAgentService,name=HinemosAgent" printValidAgent
	 * @return エージェント一覧
	 *
	 * @jmx.managed-attribute
	 */
	@Override
	public String printValidAgent() {
		log.info("printValidAgent()");
		ArrayList<String> validAgent = AgentConnectUtil.getValidAgent();
		Collections.sort(validAgent);
		String ret = "";
		for (String facilityId : validAgent) {
			String agentString = AgentConnectUtil.getAgentString(facilityId);
			if (agentString == null) {
				continue;
			}
			ret += facilityId + "," + agentString + "\n";
		}
		return ret;
	}
	
	public static HttpsServer getHttpsServer(){
		return httpsServer;
	}
	
	public static ThreadPoolExecutor getThreadPool(){
		return _threadPool;
	}
}
