/*

Copyright (C) 2011 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.custom.factory;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import javax.ejb.CreateException;
import javax.ejb.FinderException;
import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.fault.CalendarInvalid;
import com.clustercontrol.fault.CalendarNotFound;
import com.clustercontrol.fault.FacilityNotFound;
import com.clustercontrol.fault.HinemosUnknown;
import com.clustercontrol.fault.CustomInvalid;
import com.clustercontrol.fault.MonitorNotFound;
import com.clustercontrol.fault.NotifyNotFound;
import com.clustercontrol.bean.HinemosModuleConstant;
import com.clustercontrol.bean.ValidConstant;
import com.clustercontrol.calendar.bean.CalendarDTO;
import com.clustercontrol.calendar.ejb.session.CalendarControllerLocal;
import com.clustercontrol.calendar.ejb.session.CalendarControllerUtil;
import com.clustercontrol.custom.CustomConstant;
import com.clustercontrol.custom.bean.CustomCheckInfo;
import com.clustercontrol.custom.bean.CommandExecuteDTO;
import com.clustercontrol.custom.bean.CommandVariableDTO;
import com.clustercontrol.custom.ejb.entity.MonitorCustomInfoLocal;
import com.clustercontrol.custom.ejb.entity.MonitorCustomInfoUtil;
import com.clustercontrol.monitor.run.bean.MonitorInfo;
import com.clustercontrol.monitor.run.factory.SelectMonitor;
import com.clustercontrol.repository.bean.NodeInfo;
import com.clustercontrol.repository.ejb.session.RepositoryControllerBean;
import com.clustercontrol.repository.ejb.session.RepositoryControllerLocal;
import com.clustercontrol.repository.ejb.session.RepositoryControllerUtil;
import com.clustercontrol.repository.util.RepositoryUtil;

/**
 * コマンド監視の特有設定に対する参照処理実装クラス
 * @version 4.0
 * @author Tomoya Takahata
 */
public class SelectCustom extends SelectMonitor {

	private static Log log = LogFactory.getLog( SelectCustom.class );

	/**
	 * コマンド監視特有の設定を返す。<br/>
	 */
	@Override
	protected CustomCheckInfo getCommandCheckInfo() throws FinderException, NamingException {
		// Local Variables
		MonitorCustomInfoLocal monitorInfo = null;
		CustomCheckInfo checkInfo = null;

		// MAIN
		monitorInfo = MonitorCustomInfoUtil.getLocalHome().findByPrimaryKey(m_monitorId);

		try {
			checkInfo = new CustomCheckInfo(m_monitorTypeId, m_monitorId,
					monitorInfo.getCommandExecType(),
					monitorInfo.getSelectedFacilityId(), monitorInfo.getEffectiveUser(), monitorInfo.getCommand(), monitorInfo.getTimeout());
		} catch (CustomInvalid e) {
			log.warn("instance creation failure.", e);
		}

		return checkInfo;
	}

	private static Object customLock = new Object();
	private static ArrayList<MonitorInfo> customCache = null;
	
	public static void clearCustomCache () {
		log.info("clearCustomCache()");
		synchronized (customLock) {
			customCache = null;
		}
	}
	
	/**
	 * 要求されたエージェントが実行すべきコマンド実行情報の一覧を返す。<br/>
	 * @param requestedFacilityId 要求したエージェントのファシリティID
	 * @return コマンド実行情報の一覧
	 * @throws CustomInvalid コマンド監視設定
	 * @throws HinemosUnknown
	 */
	@SuppressWarnings("unchecked")
	public ArrayList<CommandExecuteDTO> getCommandExecuteDTO(String requestedFacilityId) throws CustomInvalid, HinemosUnknown {
		// Local Variables
		CommandExecuteDTO dto = null;
		ArrayList<CommandExecuteDTO> dtos = new ArrayList<CommandExecuteDTO>();

		CalendarControllerLocal calendarCtrl = null;
		RepositoryControllerLocal repositoryCtrl = null;

		CustomCheckInfo cmdInfo = null;
		CalendarDTO calendar = null;
		ArrayList<String> facilityIds = null;
		NodeInfo nodeInfo = null;

		List<CommandVariableDTO> variables = null;

		// MAIN
		if (requestedFacilityId == null) {
			throw new HinemosUnknown("facilityId is null. (facilityId = " + requestedFacilityId + ")");
		}

		try {
			calendarCtrl = CalendarControllerUtil.getLocalHome().create();
			repositoryCtrl = RepositoryControllerUtil.getLocalHome().create();

			ArrayList<MonitorInfo> monitorList = null;
			synchronized(customLock) {
				if (customCache == null) {
					long start = System.currentTimeMillis();
					customCache = getMonitorList(HinemosModuleConstant.MONITOR_CUSTOM);
					long end = System.currentTimeMillis();
					log.info("refresh customCache " + (end - start) + "ms. size=" + customCache.size());
				}
				// Cacheがrefreshされるタイミングによっては、
				// 下のfor文でNullPointerExceptionが発生するので、
				// monitorListに値を移しておく。
				monitorList = customCache;
			}

			for (MonitorInfo info : monitorList) {
				if (info.getMonitorFlg() == ValidConstant.TYPE_INVALID && info.getCollectorFlg() == ValidConstant.TYPE_INVALID) {
					if (log.isDebugEnabled())
						log.debug("command monitor is not enabled. (monitorId = " + info.getMonitorId() + ", facilityId = " + requestedFacilityId + ")");
					continue;
				}

				cmdInfo = info.getCustomCheckInfo();
				calendar = calendarCtrl.getCalendarDTO(info.getCalendarId());

				if (cmdInfo.getCommandExecType() == CustomConstant.CommandExecType.INDIVIDUAL) {
					// コマンド監視の対象スコープに含まれる有効ノードとして、問い合わせたエージェントが含まれるかどうか
					facilityIds = repositoryCtrl.getNodeFacilityIdList(info.getFacilityId(), RepositoryControllerBean.ALL, false, true);
					if (! facilityIds.contains(requestedFacilityId)) {
						if (log.isDebugEnabled())
							log.debug("CommandExcecuteDTO is required from not-contained facility. (monitorId = " + info.getMonitorId() + ", facilityId = " + requestedFacilityId + ")");
						continue;
					}
					facilityIds = new ArrayList<String>();
					facilityIds.add(requestedFacilityId);
				} else {
					// コマンド監視の指定ノードと問い合わせたエージェントが一致するかどうか
					if (! requestedFacilityId.equals(cmdInfo.getSelectedFacilityId())) {
						if (log.isDebugEnabled())
							log.debug("CommandExcecuteDTO is required from not-selected facility. (monitorId = " + info.getMonitorId() + ", facilityId = " + requestedFacilityId + ")");
						continue;
					}
					if (! repositoryCtrl.getNode(requestedFacilityId).isValid()) {
						if (log.isDebugEnabled()) {
							log.debug("requestedFacilityId is not enabled. (monitorId = " + info.getMonitorId() + ", facilityId = " + requestedFacilityId + ")");
						}
						continue;
					}
					facilityIds = repositoryCtrl.getNodeFacilityIdList(info.getFacilityId(), RepositoryControllerBean.ALL, false, true);
				}

				variables = new ArrayList<CommandVariableDTO>();
				for (String facilityId : facilityIds) {
					if (log.isDebugEnabled())
						log.debug("facility variables are assigned to CommandExecuteDTO. (monitroId = " + info.getMonitorId() + ", facilityId = " + facilityId + ")");
					nodeInfo = repositoryCtrl.getNode(facilityId);
					Map<String, String> variable = RepositoryUtil.createNodeParameter(nodeInfo);
					variables.add(new CommandVariableDTO(facilityId, variable));
				}

				dto = new CommandExecuteDTO(info.getMonitorId(), cmdInfo.getEffectiveUser(), cmdInfo.getCommand(), cmdInfo.getTimeout(), info.getRunInterval() * 1000, calendar, variables);
				dtos.add(dto);

				log.info("CommandExecuteDTO is retured to agent. (requestedFacilityId = " + requestedFacilityId + ", dto = " + dto + ")");
			}

		} catch (NotifyNotFound e) {
			log.warn("configuration of custom is not valid.", e);
			throw new CustomInvalid("configuration of custom is not valid.", e);
		} catch (MonitorNotFound e) {
			log.warn("configuration of custom is not valid.", e);
			throw new CustomInvalid("configuration of custom is not valid.", e);
		} catch (CalendarInvalid e) {
			log.warn("configuration of custom is not valid.", e);
			throw new CustomInvalid("configuration of custom is not valid.", e);
		} catch (CalendarNotFound e) {
			log.warn("configuration of custom is not valid.", e);
			throw new CustomInvalid("configuration of custom is not valid.", e);
		} catch (FacilityNotFound e) {
			log.warn("configuration of custom is not valid.", e);
			throw new CustomInvalid("configuration of custom is not valid.", e);
		} catch (CreateException e) {
			log.warn("unexpected internal failure occurred.", e);
			throw new HinemosUnknown("unexpected internal failure occurred.", e);
		} catch (NamingException e) {
			log.warn("unexpected internal failure occurred.", e);
			throw new HinemosUnknown("unexpected internal failure occurred.", e);
		}

		return dtos;
	}

}
