/*

Copyright (C) 2011 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.logfile.ejb.session;

import java.rmi.RemoteException;
import java.util.ArrayList;

import javax.ejb.EJBException;
import javax.ejb.SessionBean;
import javax.ejb.SessionContext;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.fault.HinemosUnknown;
import com.clustercontrol.fault.MonitorNotFound;
import com.clustercontrol.bean.HinemosModuleConstant;
import com.clustercontrol.calendar.bean.CalendarDTO;
import com.clustercontrol.calendar.ejb.session.CalendarControllerLocal;
import com.clustercontrol.calendar.ejb.session.CalendarControllerUtil;
import com.clustercontrol.logfile.factory.SelectMonitorLogfile;
import com.clustercontrol.monitor.run.bean.MonitorInfo;
import com.clustercontrol.repository.ejb.session.RepositoryControllerLocal;
import com.clustercontrol.repository.ejb.session.RepositoryControllerUtil;

/**
 * ログファイル監視の管理を行う Session Bean <BR>
 * クライアントからの Entity Bean へのアクセスは、Session Bean を介して行います。
 * 
 * @ejb.bean name="MonitorLogfileController"
 *           jndi-name="MonitorLogfileController"
 *           type="Stateless"
 *           transaction-type="Container"
 *           view-type="local"
 * 
 * @ejb.transaction type="Required"
 * 
 * @ejb.permission
 *     unchecked="true"
 *     method-intf="LocalHome"
 * 
 * @ejb.permission
 *     unchecked="true"
 *     method-intf="Local"
 */
public abstract class MonitorLogfileControllerBean implements SessionBean {

	/** ログ出力のインスタンス。 */
	private static Log m_log = LogFactory.getLog( MonitorLogfileControllerBean.class );

	/** コンテキスト情報。 */
	@SuppressWarnings("unused")
	private SessionContext m_context;

	/**
	 * コンテキスト情報を設定します。<BR>
	 * Session Bean の生成時に行う処理を実装します。
	 * @see javax.ejb.SessionBean#setSessionContext(javax.ejb.SessionContext)
	 */
	@Override
	public void setSessionContext(SessionContext ctx) throws EJBException, RemoteException {
		m_context = ctx;
	}


	/**
	 * ログファイル監視一覧リストを返します。
	 * 
	 * @ejb.interface-method
	 * 
	 * @jboss.method-attributes
	 *     read-only="true"
	 * 
	 * @return Objectの2次元配列
	 * @throws MonitorNotFound
	 * @throws HinemosUnknown
	 */
	public ArrayList<MonitorInfo> getLogfileList() throws MonitorNotFound, HinemosUnknown{

		SelectMonitorLogfile logfile = new SelectMonitorLogfile();
		ArrayList<MonitorInfo> list = null;
		try {
			list = logfile.getMonitorList(HinemosModuleConstant.MONITOR_LOGFILE);
		} catch (MonitorNotFound e) {
			throw e;
		} catch (Exception e) {
			m_log.error("getLogfileList " + e.getMessage(), e);
			throw new HinemosUnknown(e.getMessage(), e);
		}
		return list;
	}

	private static Object logfileLock = new Object();
	private static ArrayList<MonitorInfo> logfileCache = null;
	
	public static void clearLogfileCache () {
		m_log.info("clearLogfileCache()");
		synchronized (logfileLock) {
			logfileCache = null;
		}
	}

	/**
	 * facilityIDごとのログファイル監視一覧リストを返します。
	 * withCalendarをtrueにするとMonitorInfoのcalendarDTOに値が入ります。
	 * 
	 * @ejb.interface-method
	 * 
	 * @jboss.method-attributes
	 *     read-only="true"
	 * 
	 * @return Objectの2次元配列
	 * @throws HinemosUnknown
	 * @throws MonitorNotFound
	 * 
	 */
	public ArrayList<MonitorInfo> getLogfileListForFacilityId (String facilityId, boolean withCalendar)
	throws MonitorNotFound, HinemosUnknown {
		ArrayList<MonitorInfo> ret = new ArrayList<MonitorInfo>();

		RepositoryControllerLocal local = null;
		try {
			local = RepositoryControllerUtil.getLocalHome().create();
		} catch (Exception e) {
			String message = "getLogfileListForFacilityId1 " +
			e.getClass().getCanonicalName() + " : " + e.getMessage();
			m_log.warn(message);
			throw new HinemosUnknown(message, e);
		}

		CalendarControllerLocal calendarLocal = null;
		try {
			calendarLocal = CalendarControllerUtil.getLocalHome().create();
		} catch (Exception e) {
			String message = "getLogfileListForFacilityId2 " +
			e.getClass().getCanonicalName() + " : " + e.getMessage();
			m_log.warn(message);
			throw new HinemosUnknown(message, e);
		}

		ArrayList<MonitorInfo> monitorList = null;
		synchronized(logfileLock) {
			if (logfileCache == null) {
				long start = System.currentTimeMillis();
				logfileCache = getLogfileList();
				long end = System.currentTimeMillis();
				m_log.info("refresh logfileCache " + (end - start) + "ms. size=" + logfileCache.size());
			}
			// Cacheがrefreshされるタイミングによっては、
			// 下のfor文でNullPointerExceptionが発生するので、
			// monitorListに値を移しておく。
			monitorList = logfileCache;
		}
		
		for (MonitorInfo monitorInfo : monitorList) {
			String scope = monitorInfo.getFacilityId();
			ArrayList<String> facilityIdList = local.getExecTargetFacilityIdList(scope);
			if (facilityIdList != null && facilityIdList.contains(facilityId)) {
				if (withCalendar) {
					String calendarId = monitorInfo.getCalendarId();
					try {
						CalendarDTO calendar = calendarLocal.getCalendarDTO(calendarId);
						monitorInfo.setCalendarDTO(calendar);
					} catch (Exception e) {
						String message = "getLogfileListForFacilityId3 " +
						e.getClass().getCanonicalName() + " : " + e.getMessage();
						m_log.warn(message, e);
						throw new HinemosUnknown(message, e);
					}
				}
				ret.add(monitorInfo);
			}
		}

		return ret;
	}
}
