/*

Copyright (C) 2006 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.notify.util;

import java.io.UnsupportedEncodingException;
import java.rmi.RemoteException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.Locale;
import java.util.Map;
import java.util.StringTokenizer;
import java.util.Vector;

import javax.ejb.CreateException;
import javax.ejb.FinderException;
import javax.mail.AuthenticationFailedException;
import javax.mail.Message;
import javax.mail.MessagingException;
import javax.mail.Session;
import javax.mail.Transport;
import javax.mail.internet.AddressException;
import javax.mail.internet.InternetAddress;
import javax.mail.internet.MimeMessage;
import javax.mail.internet.MimeUtility;
import javax.naming.Context;
import javax.naming.InitialContext;
import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.bean.PriorityConstant;
import com.clustercontrol.commons.util.HinemosProperties;
import com.clustercontrol.notify.bean.NotifyRequestMessage;
import com.clustercontrol.notify.bean.OutputBasicInfo;
import com.clustercontrol.notify.ejb.entity.NotifyMailInfoData;
import com.clustercontrol.notify.ejb.entity.NotifyMailInfoLocal;
import com.clustercontrol.notify.ejb.entity.NotifyMailInfoPK;
import com.clustercontrol.notify.ejb.entity.NotifyMailInfoUtil;
import com.clustercontrol.notify.mail.bean.MailTemplateInfo;
import com.clustercontrol.notify.mail.ejb.session.MailTemplateControllerUtil;
import com.clustercontrol.util.Messages;
import com.clustercontrol.util.StringBinder;
import com.clustercontrol.util.apllog.AplLogger;
import com.sun.mail.smtp.SMTPAddressFailedException;

/**
 * メールを送信するクラス<BR>
 * 
 * @version 3.0.0
 * @since 3.0.0
 */
public class SendMail implements Notifier {
	/** ログ出力のインスタンス。 */
	private static Log m_log = LogFactory.getLog(SendMail.class);

	/** 日時フォーマット。 */
	private static final String SUBJECT_DATE_FORMAT = "yyyy/MM/dd HH:mm:ss";

	/** 差出人アドレス。 */
	static private String FROM_ADDRESS = null;
	/** 差出人個人名。 */
	static private String FROM_PERSONAL_NAME = null;
	/** 返信の送信先アドレス。 */
	static private String REPLY_TO_ADDRESS = null;
	/** 返信の送信先個人名。 */
	static private String REPLY_TO_PERSONAL_NAME = null;
	/** エラー送信先アドレス。 */
	static private String ERRORS_TO_ADDRESS = null;
	/** 再送回数。 */
	static private int RETRY_COUNT = 0;
	/** 再送スリープ */
	static private int RETRY_SLEEP = 10000; // デフォルトは10秒。
	/** ロケール情報。 */
	private Locale m_local = Locale.getDefault();

	/** アドレスロケール */
	public static final String _CHARSET_ADDRESS;
	public static final String _CHARSET_ADDRESS_DEFAULT = "UTF-8";
	
	/** メール件名ロケール */
	public static final String _CHARSET_SUBJECT;
	public static final String _CHARSET_SUBJECT_DEFAULT = "UTF-8";
	
	/**`メール本文ロケール */
	public static final String _CHARSET_CONTENT;
	public static final String _CHARSET_CONTENT_DEFAULT = "UTF-8";
	
	static {
		String charsetAddress = _CHARSET_ADDRESS_DEFAULT;
		String charsetSubject = _CHARSET_SUBJECT_DEFAULT;
		String charsetContent = _CHARSET_CONTENT_DEFAULT;
		
		/**
		 * メールの設定をプロパティファイルから取得
		 */
		// FIXME RuntimeのExceptionが発生する恐れがあるためtry-catchとする
		try {
			FROM_ADDRESS = HinemosProperties
			.getProperty("common.mail.from.address");
			FROM_PERSONAL_NAME = HinemosProperties
			.getProperty("common.mail.from.personal.name");
			REPLY_TO_ADDRESS = HinemosProperties
			.getProperty("common.mail.reply.to.address");
			REPLY_TO_PERSONAL_NAME = HinemosProperties
			.getProperty("common.mail.reply.personal.name");
			ERRORS_TO_ADDRESS = HinemosProperties
			.getProperty("common.mail.errors.to.address");
			RETRY_COUNT = Integer.parseInt(HinemosProperties
					.getProperty("common.mail.retry.count"));
			RETRY_SLEEP = Integer.parseInt(HinemosProperties
					.getProperty("common.mail.retry.sleep"));
			
			charsetAddress = HinemosProperties.getProperty("common.mail.charset.address", _CHARSET_ADDRESS_DEFAULT);
			charsetSubject = HinemosProperties.getProperty("common.mail.charset.subject", _CHARSET_SUBJECT_DEFAULT);
			charsetContent = HinemosProperties.getProperty("common.mail.charset.content", _CHARSET_CONTENT_DEFAULT);
			
		} catch (Exception e) {
			m_log.warn(e.getMessage(), e);
		}
		
		_CHARSET_ADDRESS = charsetAddress;
		_CHARSET_SUBJECT = charsetSubject;
		_CHARSET_CONTENT = charsetContent;
		
		m_log.info("initialized mail sender : from_address = " + FROM_ADDRESS 
				+ ", From = " + FROM_PERSONAL_NAME + " <" + REPLY_TO_ADDRESS + ">"
				+ ", Reply-To = " + REPLY_TO_PERSONAL_NAME + " <" + REPLY_TO_ADDRESS + ">"
				+ ", Errors-To = " + ERRORS_TO_ADDRESS
				+ ", retries = " + RETRY_COUNT
				+ ", retry-sleep = " + RETRY_SLEEP
				+ ", Charset [address:subject:content] = [" + _CHARSET_ADDRESS + ":" + _CHARSET_SUBJECT + ":" + _CHARSET_CONTENT + "]");
	}

	/**
	 * メールの送信を行います。
	 * 
	 * @param outputInfo 出力・通知情報
	 * @throws UnsupportedEncodingException
	 * @throws RemoteException
	 * @throws NamingException
	 * @throws MessagingException
	 * @throws CreateException
	 * @throws FinderException
	 */
	@Override
	public synchronized void notify(NotifyRequestMessage message)
	throws UnsupportedEncodingException, RemoteException,
	NamingException, CreateException, FinderException {

		sendMail(message.getOutputInfo(), message.getNotifyId());
	}

	/**
	 * メールの送信を行います。
	 * 
	 * @throws MessagingException
	 * @throws
	 */
	private void sendMail(OutputBasicInfo outputInfo, String notifyId) {
		if (m_log.isDebugEnabled()) {
			m_log.debug("notify() " + outputInfo);
		}

		try {
			NotifyMailInfoLocal mailInfo = NotifyMailInfoUtil.getLocalHome()
			.findByPrimaryKey(
					new NotifyMailInfoPK(notifyId, outputInfo
							.getPriority()));

			NotifyMailInfoData mailInfoData = mailInfo.getData();

			// メールの件名を指定
			String subject = getSubject(outputInfo, mailInfoData);

			// メールの内容を指定
			String content = getContent(outputInfo, mailInfoData);

			/**
			 * メール送信
			 */
			String address = mailInfo.getMailAddress();
			StringTokenizer t = new StringTokenizer(address, ";");
			ArrayList<String> addressList = new ArrayList<String>();
			while (t.hasMoreTokens()) {
				addressList.add(t.nextToken());
			}
			String[] addresses = addressList.toArray(new String[0]);

			if (addresses == null || addresses.length <= 0) {
				m_log.debug("sendMail() : mail address is empty");
				return;
			}

			try {
				this.sendMail(addresses, subject, content);
			} catch (AuthenticationFailedException e) {
				String detailMsg = "cannot connect to the mail server due to an Authentication Failure";
				m_log.error("sendMail():" + e.getMessage() + " : " + detailMsg);
				internalErrorNotify(notifyId, "007", detailMsg);
			} catch (SMTPAddressFailedException e) {
				String detailMsg = e.getMessage() + "(SMTPAddressFailedException)";
				m_log.error("sendMail():" + detailMsg);
				internalErrorNotify(notifyId, "007", detailMsg);
			} catch (MessagingException e) {
				String detailMsg = e.getCause() != null ? e.getMessage() + "\nCause : " + e.getCause().getMessage() : e.getMessage();
				m_log.error("sendMail():" + e.getMessage(), e);
				internalErrorNotify(notifyId, "007", detailMsg);
			}
		} catch (Exception e1) {
			String detailMsg = e1.getCause() != null ? e1.getMessage() + "\nCause : " + e1.getCause().getMessage() : e1.getMessage();
			m_log.error("sendMail():" + e1.getMessage(), e1);
			internalErrorNotify(notifyId, "007", detailMsg);
		}
	}

	/**
	 * メールを送信します。
	 * 
	 * <p>
	 * 下記の情報は、ファイルより取得します。
	 * <p>
	 * <ul>
	 * <li>差出人アドレス</li>
	 * <li>差出人個人名</li>
	 * <li>返信の送信先アドレス</li>
	 * <li>返信の送信先個人名</li>
	 * <li>エラー送信先アドレス</li>
	 * </ul>
	 * 
	 * @param address
	 *            送信先アドレス
	 * @param source
	 *            出力内容
	 * @return 送信に成功した場合、<code> true </code>
	 * @throws NamingException
	 * @throws MessagingException
	 * @throws NamingException
	 * @throws UnsupportedEncodingException
	 * @throws NamingException
	 * @throws UnsupportedEncodingException
	 */
	public void sendMail(String[] address, String subject, String content)
	throws MessagingException, NamingException, UnsupportedEncodingException {

		if (address == null || address.length <= 0) {
			// 何もせず終了
			return;
		}

		Context initialContext = new InitialContext();

		// 再送信フラグ設定
		boolean retryFlg = true;

		// JavaMail Sessionリソース検索
		Session session = (Session) initialContext.lookup("java:/Mail");

		Message mes = new MimeMessage(session);

		// 送信元メールアドレスと送信者名を指定
		if (FROM_ADDRESS != null && FROM_PERSONAL_NAME != null) {
			mes.setFrom(new InternetAddress(FROM_ADDRESS, FROM_PERSONAL_NAME, _CHARSET_ADDRESS));
		} else if (FROM_ADDRESS != null && FROM_PERSONAL_NAME == null) {
			mes.setFrom(new InternetAddress(FROM_ADDRESS));
		}
		// REPLY-TOを指定
		if (REPLY_TO_ADDRESS != null && REPLY_TO_PERSONAL_NAME != null) {
			InternetAddress reply[] = { new InternetAddress(REPLY_TO_ADDRESS, REPLY_TO_PERSONAL_NAME, _CHARSET_ADDRESS) };
			mes.setReplyTo(reply);
			mes.reply(true);
		} else if (REPLY_TO_ADDRESS != null && REPLY_TO_PERSONAL_NAME == null) {
			InternetAddress reply[] = { new InternetAddress(REPLY_TO_ADDRESS) };
			mes.setReplyTo(reply);
			mes.reply(true);
		}

		// ERRORS-TOを指定
		if (ERRORS_TO_ADDRESS != null) {
			mes.setHeader("Errors-To", ERRORS_TO_ADDRESS);
		}

		// 送信先メールアドレスを指定
		InternetAddress[] toAddress = this.getAddress(address);
		if (toAddress == null || toAddress.length <= 0) {
			// 何もせず終了
			return;
		}
		mes.setRecipients(javax.mail.Message.RecipientType.TO, toAddress);

		// メールの件名を指定
		mes.setSubject(MimeUtility.encodeText(subject, _CHARSET_SUBJECT, "B"));

		// メールの内容を指定
		mes.setContent(content, "text/plain; charset=" + _CHARSET_CONTENT);

		// 送信日付を指定
		mes.setSentDate(new Date());

		// 再送信フラグがtrueかつ再送回数以内の場合
		for (int i = 0; i < RETRY_COUNT + 1 && retryFlg; i++) {
			try {
				// メール送信
				Transport.send(mes);
				retryFlg = false;
			} catch (AuthenticationFailedException e) {
				throw e;
			} catch (SMTPAddressFailedException e) {
				throw e;
			} catch (MessagingException me) {
				// メール再送信フラグを挙げる
				retryFlg = true;
				if (i < RETRY_COUNT) {
					m_log.warn("sendMail() : resend mail");
				} else {
					throw me;
				}
			}
			if (retryFlg) {
				try {
					m_log.debug("sendMail() : resend sleep (" + RETRY_SLEEP + ")");
					Thread.sleep(RETRY_SLEEP);
				} catch (InterruptedException e) {
					m_log.warn("InterruptedException " + e.getMessage());
				}
			}
		}
	}

	/**
	 * 引数で指定された送信先アドレスの<code> InternetAddress </code>オブジェクトを返します。
	 * 
	 * @param address
	 *            送信先アドレスの文字列配列
	 * @return <code> InternetAddress </code>オブジェクトの配列
	 */
	private InternetAddress[] getAddress(String[] address) {

		InternetAddress toAddress[] = null;
		Vector<InternetAddress> list = new Vector<InternetAddress>();

		if (address != null) {

			for (int index = 0; index < address.length; index++) {
				try {
					list.add(new InternetAddress(address[index]));
				} catch (AddressException ae) {
					m_log.warn(ae.getMessage(), ae);
				}
			}

			if (list.size() > 0) {
				toAddress = new InternetAddress[list.size()];
				list.copyInto(toAddress);
			}
		}
		return toAddress;
	}

	/**
	 * メール件名を返します。
	 * 
	 * @param source
	 *            出力内容
	 * @param mailInfo
	 *            通知内容
	 * @return メール件名
	 * @throws CreateException
	 * @throws NamingException
	 * @throws FinderException
	 * @throws RemoteException
	 */
	public String getSubject(OutputBasicInfo source,
			NotifyMailInfoData mailInfo) {

		String subject = null;

		try {
			if (mailInfo != null && mailInfo.getMailTemplateId() != null) {
				MailTemplateInfo templateData = MailTemplateControllerUtil
				.getLocalHome().create().getMailTemplateInfo(
						mailInfo.getMailTemplateId());
				Map<String, String> param = NotifyUtil.createParameter(source);
				StringBinder binder = new StringBinder(param);
				subject = binder.bindParam(templateData.getSubject());
			} else {
				subject = Messages.getString("mail.subject", m_local) + "("
				+ PriorityConstant.typeToString(source.getPriority())
				+ ")";
			}
		} catch (Exception e) {
			m_log.warn(e.getMessage());
			// 例外発生時のメールサブジェクト
			return "Hinemos Notification";
		}

		return subject;
	}

	/**
	 * メール本文を返します。
	 * 
	 * @param source
	 *            出力内容
	 * @param mailInfo
	 *            通知内容
	 * @return メール本文
	 * @throws CreateException
	 * @throws NamingException
	 * @throws FinderException
	 * @throws RemoteException
	 */
	public String getContent(OutputBasicInfo source, NotifyMailInfoData mailInfo) {

		StringBuffer buf = new StringBuffer();
		SimpleDateFormat sdf = new SimpleDateFormat(SUBJECT_DATE_FORMAT);

		try {
			if (mailInfo != null && mailInfo.getMailTemplateId() != null) {
				MailTemplateInfo mailData = MailTemplateControllerUtil
				.getLocalHome().create().getMailTemplateInfo(
						mailInfo.getMailTemplateId());
				Map<String, String> param = NotifyUtil.createParameter(source);
				StringBinder binder = new StringBinder(param);
				buf.append(binder.bindParam(mailData.getBody() + "\n"));
			} else {
				buf.append(Messages.getString("generation.time", m_local)
						+ " : "
						+ sdf.format(source.getGenerationDate()) + "\n");
				buf.append(Messages.getString("application", m_local) + " : "
						+ source.getApplication() + "\n");
				buf.append(Messages.getString("priority", m_local) + " : "
						+ PriorityConstant.typeToString(source.getPriority())
						+ "\n");
				buf.append(Messages.getString("message", m_local) + " : "
						+ source.getMessage() + "\n");
				buf.append(Messages.getString("scope", m_local) + " : "
						+ source.getScopeText() + "\n");
			}
		} catch (Exception e) {
			m_log.warn(e.getMessage());
			// 例外発生時のメール本文
			return "An error occurred creating message.";
		}
		
		// 改行コードをLFからCRLFに変更する。
		// 本来はJavaMailが変換するはずだが、変換されないこともあるので、
		// 予め変更しておく。
		String ret = buf.toString().replaceAll("\r\n", "\n").replaceAll("\n", "\r\n");
		
		return ret;
	}

	/**
	 * 通知失敗時の内部エラー通知を定義します
	 */
	@Override
	public void internalErrorNotify(String notifyId, String msgID, String detailMsg) {
		AplLogger apllog = new AplLogger("NOTIFY", "notify");
		String[] args = { notifyId };
		// 通知失敗メッセージを出力
		apllog.put("SYS", msgID, args, detailMsg);
	}
}
