/*

Copyright (C) 2006 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.repository.factory;

import java.util.ArrayList;
import java.util.Collection;
import java.util.concurrent.ConcurrentHashMap;

import javax.ejb.EJBException;
import javax.ejb.FinderException;
import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.fault.FacilityNotFound;
import com.clustercontrol.repository.bean.NodeCpuInfo;
import com.clustercontrol.repository.bean.NodeDeviceInfo;
import com.clustercontrol.repository.bean.NodeDiskInfo;
import com.clustercontrol.repository.bean.NodeFilesystemInfo;
import com.clustercontrol.repository.bean.NodeHostnameInfo;
import com.clustercontrol.repository.bean.NodeInfo;
import com.clustercontrol.repository.bean.NodeMemoryInfo;
import com.clustercontrol.repository.bean.NodeNetworkInterfaceInfo;
import com.clustercontrol.repository.bean.NodeNoteInfo;
import com.clustercontrol.repository.bean.NodeVariableInfo;
import com.clustercontrol.repository.ejb.entity.FacilityLocal;
import com.clustercontrol.repository.ejb.entity.NodeCpuLocal;
import com.clustercontrol.repository.ejb.entity.NodeDeviceLocal;
import com.clustercontrol.repository.ejb.entity.NodeDiskLocal;
import com.clustercontrol.repository.ejb.entity.NodeFilesystemLocal;
import com.clustercontrol.repository.ejb.entity.NodeHostnameLocal;
import com.clustercontrol.repository.ejb.entity.NodeLocal;
import com.clustercontrol.repository.ejb.entity.NodeMemoryLocal;
import com.clustercontrol.repository.ejb.entity.NodeNetworkInterfaceLocal;
import com.clustercontrol.repository.ejb.entity.NodeNoteLocal;
import com.clustercontrol.repository.ejb.entity.NodeUtil;
import com.clustercontrol.repository.ejb.entity.NodeVariableLocal;

/**
 * ノード用プロパティを作成するクラス<BR>
 * 
 * @version 1.0.0
 * @since 1.0.0
 */
public class NodeProperty {

	private static Log m_log = LogFactory.getLog(NodeProperty.class);

	/** ----- 初期値キャッシュ ----- */

	private static Object nodeLock = new Object();
	private static ConcurrentHashMap<String, NodeInfo> nodeCache =
		new ConcurrentHashMap<String, NodeInfo>();

	public static void removeNode (String facilityId) {
		m_log.info("remove NodeCache : " + facilityId);
		synchronized(nodeLock) {
			nodeCache.remove(facilityId);
		}
	}

	public static void clearCache () {
		m_log.info("clear NodeCache");
		synchronized(nodeLock) {
			nodeCache.clear();
		}
	}

	/**
	 * 与えられたファシリティIDに基づき、該当するノード情報を返す。<BR>
	 * 
	 * @param facilityId ファシリティID
	 * @param mode ノード情報扱い種別（参照、追加、変更）
	 * @return ノード情報
	 * @throws FacilityNotFound
	 * @throws FinderException
	 * @throws NamingException
	 */
	public static NodeInfo getProperty(String facilityId) throws FacilityNotFound {
		m_log.debug("getProperty() : facilityId = " + facilityId);

		/** ローカル変数 */
		NodeInfo property = null;
		NodeLocal node = null;
		FacilityLocal facility = null;

		/** メイン処理 */
		if (facilityId == null || facilityId.compareTo("") == 0) {
			return new NodeInfo();
		}

		synchronized(nodeLock) {
			property = nodeCache.get(facilityId);
			if (property != null) {
				return property;
			} else {
				property = new NodeInfo();
			}
			
			try {
				node = NodeUtil.getLocalHome().findByPrimaryKey(facilityId);
				facility = node.getFacility();
			} catch (FinderException e) {
				throw new FacilityNotFound(e);

			} catch (Exception e) {
				m_log.warn("failure to get node's properties. (facilityId = " + facilityId + ")", e);
				throw new EJBException("failure to get node's properties. (facilityId = " + facilityId + ")", e);
			}

			// ----- ファシリティ関連 -----
			// ファシリティID
			property.setFacilityId(facility.getFacilityId());
			// ファシリティ名
			property.setFacilityName(facility.getFacilityName());
			// 説明
			property.setDescription(facility.getDescription());
			// 有効/無効
			property.setValid(facility.isValid());
			// 登録ユーザID
			property.setCreateUserId(facility.getCreateUserId());
			// 登録日時
			if (facility.getCreateDatetime() != null
					&& facility.getCreateDatetime().getTime() != 0) {
				property.setCreateDatetime(facility.getCreateDatetime().getTime());
			}
			// 最終更新ユーザID
			property.setModifyUserId(facility.getModifyUserId());
			// 最終更新日時
			if (facility.getModifyDatetime() != null
					&& facility.getModifyDatetime().getTime() != 0) {
				property.setModifyDatetime(facility.getModifyDatetime().getTime());
			}

			// ----- ノード基本情報 -----

			// ----- HW -----
			// プラットフォーム
			property.setPlatformFamily(node.getPlatformFamily());
			// H/Wタイプ
			property.setHardwareType(node.getHardwareType());
			// 画面アイコンイメージ
			property.setIconImage(node.getIconImage());

			// ----- IPアドレス関連 -----
			// IPバージョン
			if (node.getIpAddressVersion() != -1) {
				property.setIpAddressVersion(node.getIpAddressVersion());
			}
			// IPアドレスV4
			property.setIpAddressV4(node.getIpAddressV4());
			// IPアドレスV6
			property.setIpAddressV6(node.getIpAddressV6());
			// ホスト名
			ArrayList<NodeHostnameInfo> hostnameList = new ArrayList<NodeHostnameInfo>();
			if (node.getNodeHostname() != null) {
				for (NodeHostnameLocal nodeHostname : (Collection<NodeHostnameLocal>)node.getNodeHostname()) {
					// ホスト名
					hostnameList.add(new NodeHostnameInfo(nodeHostname.getHostname()));
				}
			}
			property.setNodeHostnameInfo(hostnameList);

			// ----- OS関連 -----
			// ノード名
			property.setNodeName(node.getNodeName());
			// OS名
			property.setOsName(node.getOsName());
			// OSリリース
			property.setOsRelease(node.getOsRelease());
			// OSバージョン
			property.setOsVersion(node.getOsVersion());
			// 文字セット
			property.setCharacterSet(node.getCharacterSet());

			// ----- サービス -----

			// ----- SNMP関連 -----
			// SNMPポート番号
			if (node.getSnmpPort() != -1) {
				property.setSnmpPort(node.getSnmpPort());
			}
			// SNMPコミュニティ名
			property.setSnmpCommunity(node.getSnmpCommunity());
			// SNMPバージョン
			property.setSnmpVersion(node.getSnmpVersion());
			// SNMPタイムアウト
			if (node.getSnmpTimeout() != -1) {
				property.setSnmpTimeout(node.getSnmpTimeout());
			}
			// SNMPリトライ回数
			if (node.getSnmpRetryCount() != -1) {
				property.setSnmpRetryCount(node.getSnmpRetryCount());
			}

			// ----- WBEM関連 -----
			// WBEM接続ポート番号
			if (node.getWbemPort() != -1) {
				property.setWbemPort(node.getWbemPort());
			}
			// WBEM接続ユーザ
			property.setWbemUser(node.getWbemUser());
			// WBEM接続ユーザパスワード
			property.setWbemUserPassword(node.getWbemUserPassword());
			// WBEM接続プロトコル
			property.setWbemProtocol(node.getWbemProtocol());
			// WBEM接続タイムアウト
			if (node.getWbemTimeout() != -1) {
				property.setWbemTimeout(node.getWbemTimeout());
			}
			// WBEM接続リトライ回数
			if (node.getWbemRetryCount() != -1) {
				property.setWbemRetryCount(node.getWbemRetryCount());
			}

			// ----- IPMI関連 -----
			// IPMI接続アドレス
			property.setIpmiIpAddress(node.getIpmiIpAddress());
			// IPMIポート番号
			if (node.getIpmiPort() != -1) {
				property.setIpmiPort(node.getIpmiPort());
			}
			// IPMI接続ユーザ
			property.setIpmiUser(node.getIpmiUser());
			// IPMI接続ユーザパスワード
			property.setIpmiUserPassword(node.getIpmiUserPassword());
			// IPMI接続タイムアウト
			if (node.getIpmiTimeout() != -1) {
				property.setIpmiTimeout(node.getIpmiTimeout());
			}
			// IPMI接続リトライ回数
			if (node.getIpmiRetryCount() != -1) {
				property.setIpmiRetries(node.getIpmiRetryCount());
			}
			// IPMI接続プロトコル
			property.setIpmiProtocol(node.getIpmiProtocol());
			// IPMI特権レベル
			property.setIpmiLevel(node.getIpmiLevel());

			// ----- WinRM関連 -----
			// WinRM接続ユーザ
			property.setWinrmUser(node.getWinrmUser());
			// WinRM接続ユーザパスワード
			property.setWinrmUserPassword(node.getWinrmUserPassword());
			// WinRMバージョン
			property.setWinrmVersion(node.getWinrmVersion());
			// WinRM接続ポート番号
			if (node.getWinrmPort() != -1) {
				property.setWinrmPort(node.getWinrmPort());
			}
			// WinRM接続プロトコル
			property.setWinrmProtocol(node.getWinrmProtocol());
			// WinRM接続タイムアウト
			if (node.getWinrmTimeout() != -1) {
				property.setWinrmTimeout(node.getWinrmTimeout());
			}
			// WinRM接続リトライ回数
			if (node.getWinrmRetryCount() != -1) {
				property.setWinrmRetries(node.getWinrmRetryCount());
			}

			// ----- デバイス関連 -----

			// ----- 汎用デバイス情報 -----
			ArrayList<NodeDeviceInfo> deviceList = new ArrayList<NodeDeviceInfo>();
			if (node.getNodeDevice() != null) {
				int cnt = 0;
				for (NodeDeviceLocal nodeDevice : (Collection<NodeDeviceLocal>)node.getNodeDevice()) {
					NodeDeviceInfo device = new NodeDeviceInfo();

					// デバイス種別
					device.setDeviceType(nodeDevice.getDeviceType());
					// デバイス表示名
					device.setDeviceDisplayName(nodeDevice.getDeviceDisplayName());
					// デバイスINDEX
					device.setDeviceIndex(nodeDevice.getDeviceIndex());
					// デバイス名
					device.setDeviceName(nodeDevice.getDeviceName());
					// デバイスサイズ
					device.setDeviceSize(nodeDevice.getDeviceSize());
					// デバイスサイズ単位
					device.setDeviceSizeUnit(nodeDevice.getDeviceSizeUnit());
					// 説明
					device.setDeviceDescription(nodeDevice.getDeviceDescription());
					cnt++;
					deviceList.add(device);
				}
			}
			property.setNodeDeviceInfo(deviceList);

			// ----- CPUデバイス情報 -----
			ArrayList<NodeCpuInfo> cpuList = new ArrayList<NodeCpuInfo>();
			if (node.getNodeCpu() != null) {
				int cnt = 0;
				for(NodeCpuLocal nodeCpu : (Collection<NodeCpuLocal>)node.getNodeCpu()){
					NodeCpuInfo cpu = new NodeCpuInfo();

					// デバイス種別
					cpu.setDeviceType(nodeCpu.getDeviceType());
					// デバイス表示名
					cpu.setDeviceDisplayName(nodeCpu.getDeviceDisplayName());
					// デバイスINDEX
					cpu.setDeviceIndex(nodeCpu.getDeviceIndex());
					// デバイス名
					cpu.setDeviceName(nodeCpu.getDeviceName());
					// デバイスサイズ
					cpu.setDeviceSize(nodeCpu.getDeviceSize());
					// デバイスサイズ単位
					cpu.setDeviceSizeUnit(nodeCpu.getDeviceSizeUnit());
					// 説明
					cpu.setDeviceDescription(nodeCpu.getDeviceDescription());
					cnt++;
					cpuList.add(cpu);
				}
			}
			property.setNodeCpuInfo(cpuList);

			// ----- MEMデバイス情報 -----
			ArrayList<NodeMemoryInfo> memList = new ArrayList<NodeMemoryInfo>();
			if (node.getNodeMemory() != null) {
				int cnt = 0;
				for(NodeMemoryLocal nodeMem : (Collection<NodeMemoryLocal>)node.getNodeMemory()){
					NodeMemoryInfo mem = new NodeMemoryInfo();

					// デバイス種別
					mem.setDeviceType(nodeMem.getDeviceType());
					// デバイス表示名
					mem.setDeviceDisplayName(nodeMem.getDeviceDisplayName());
					// デバイスINDEX
					mem.setDeviceIndex(nodeMem.getDeviceIndex());
					// デバイス名
					mem.setDeviceName(nodeMem.getDeviceName());
					// デバイスサイズ
					mem.setDeviceSize(nodeMem.getDeviceSize());
					// デバイスサイズ単位
					mem.setDeviceSizeUnit(nodeMem.getDeviceSizeUnit());
					// 説明
					mem.setDeviceDescription(nodeMem.getDeviceDescription());
					cnt++;
					memList.add(mem);
				}
			}
			property.setNodeMemoryInfo(memList);

			// ----- NICデバイス情報 -----
			ArrayList<NodeNetworkInterfaceInfo> nicList = new ArrayList<NodeNetworkInterfaceInfo>();
			if (node.getNodeNetworkInterface() != null) {
				int cnt = 0;
				for(NodeNetworkInterfaceLocal nodeNic : (Collection<NodeNetworkInterfaceLocal>)node.getNodeNetworkInterface()){
					NodeNetworkInterfaceInfo nic = new NodeNetworkInterfaceInfo();

					// デバイス種別
					nic.setDeviceType(nodeNic.getDeviceType());
					// デバイス表示名
					nic.setDeviceDisplayName(nodeNic.getDeviceDisplayName());
					// デバイスINDEX
					nic.setDeviceIndex(nodeNic.getDeviceIndex());
					// デバイス名
					nic.setDeviceName(nodeNic.getDeviceName());
					// デバイスサイズ
					nic.setDeviceSize(nodeNic.getDeviceSize());
					// デバイスサイズ単位
					nic.setDeviceSizeUnit(nodeNic.getDeviceSizeUnit());
					// 説明
					nic.setDeviceDescription(nodeNic.getDeviceDescription());
					// NIC ip
					nic.setNicIpAddress(nodeNic.getDeviceNicIpAddress());
					// NIC MAC
					nic.setNicMacAddress(nodeNic.getDeviceNicMacAddress());
					cnt++;
					nicList.add(nic);
				}
			}
			property.setNodeNetworkInterfaceInfo(nicList);

			// ----- DISKデバイス情報 -----
			ArrayList<NodeDiskInfo> diskList = new ArrayList<NodeDiskInfo>();
			if (node.getNodeDisk() != null) {
				int cnt = 0;
				for(NodeDiskLocal nodeDisk : (Collection<NodeDiskLocal>)node.getNodeDisk()){
					NodeDiskInfo disk = new NodeDiskInfo();

					// デバイス種別
					disk.setDeviceType(nodeDisk.getDeviceType());
					// デバイス表示名
					disk.setDeviceDisplayName(nodeDisk.getDeviceDisplayName());
					// デバイスINDEX
					disk.setDeviceIndex(nodeDisk.getDeviceIndex());
					// デバイス名
					disk.setDeviceName(nodeDisk.getDeviceName());
					// デバイスサイズ
					disk.setDeviceSize(nodeDisk.getDeviceSize());
					// デバイスサイズ単位
					disk.setDeviceSizeUnit(nodeDisk.getDeviceSizeUnit());
					// 説明
					disk.setDeviceDescription(nodeDisk.getDeviceDescription());
					// 回転数
					disk.setDiskRpm(nodeDisk.getDeviceDiskRpm());
					cnt++;
					diskList.add(disk);
				}
			}
			property.setNodeDiskInfo(diskList);

			// ---- ファイルシステム情報 -----
			ArrayList<NodeFilesystemInfo> filesystemList = new ArrayList<NodeFilesystemInfo>();
			if (node.getNodeFilesystem() != null) {
				int cnt = 0;
				for (NodeFilesystemLocal nodeFilesystem : (Collection<NodeFilesystemLocal>)node.getNodeFilesystem()) {
					NodeFilesystemInfo filesystem = new NodeFilesystemInfo();

					// デバイス種別
					filesystem.setDeviceType(nodeFilesystem.getDeviceType());
					// デバイス表示名
					filesystem.setDeviceDisplayName(nodeFilesystem.getDeviceDisplayName());
					// デバイスINDEX
					filesystem.setDeviceIndex(nodeFilesystem.getDeviceIndex());
					// デバイス名
					filesystem.setDeviceName(nodeFilesystem.getDeviceName());
					// デバイスサイズ
					filesystem.setDeviceSize(nodeFilesystem.getDeviceSize());
					// デバイスサイズ単位
					filesystem.setDeviceSizeUnit(nodeFilesystem.getDeviceSizeUnit());
					// 説明
					filesystem.setDeviceDescription(nodeFilesystem.getDeviceDescription());
					// ファイルシステム種別
					filesystem.setFilesystemType(nodeFilesystem.getDeviceFilesystemType());
					cnt++;
					filesystemList.add(filesystem);
				}
			}
			property.setNodeFilesystemInfo(filesystemList);

			// ----- サーバ仮想化関連 -----
			// 仮想化ノード種別
			property.setVirtualizationNodeType(node.getVirtualizationNodeType());
			// 仮想マシン管理ノード
			property.setVmManagementNode(node.getVmManagementNode());
			// 仮想マシンINDEX
			if (node.getVmIndex() != -1) {
				property.setVmIndex(node.getVmIndex());
			}
			// 仮想マシン名
			property.setVmName(node.getVmName());
			// 仮想化ソリューション
			property.setVirtualizationSolution(node.getVirtualizationSolution());
			// 仮想マシンID
			property.setVmId(node.getVmId());
			// 仮想化ソフト接続ユーザ
			property.setVmUser(node.getVmUser());
			// 仮想化ソフト接続ユーザパスワード
			property.setVmUserPassword(node.getVmUserPassword());
			// 仮想化ソフト接続プロトコル
			property.setVmProtocol(node.getVmProtocol());

			// ----- ネットワーク仮想化関連 -----
			// 仮想スイッチ種別
			property.setvNetSwitchType(node.getVNetSwitchType());
			// 仮想スイッチホストノード
			property.setvNetHostNode(node.getVNetHostNode());
			// OpenFlow データパスID
			property.setOpenFlowDataPathId(node.getOpenFlowDatapathId());
			// OpenFlow コントローラIPアドレス
			property.setOpenFlowCtrlIpAddress(node.getOpenFlowCtrlIpAddress());

			// ----- 	ノード変数 -----
			ArrayList<NodeVariableInfo> nodeVariableList = new ArrayList<NodeVariableInfo>();
			if (node.getNodeVariable() != null) {
				int cnt = 0;
				for (NodeVariableLocal nodeVariable : (Collection<NodeVariableLocal>)node.getNodeVariable()) {
					NodeVariableInfo variable = new NodeVariableInfo();

					// ノード変数名
					variable.setNodeVariableName(nodeVariable.getNodeVariableName());
					// ノード変数値
					variable.setNodeVariableValue(nodeVariable.getNodeVariableValue());
					cnt++;
					nodeVariableList.add(variable);
				}
			}
			property.setNodeVariableInfo(nodeVariableList);

			// ----- 管理情報関連 -----
			// 連絡先
			property.setContact(node.getContact());
			// 管理者
			property.setAdministrator(node.getAdministrator());


			// ----- 備考 -----
			ArrayList<NodeNoteInfo> noteList = new ArrayList<NodeNoteInfo>();
			if (node.getNodeNote() != null) {
				int cnt = 0;
				for (NodeNoteLocal nodeNote : (Collection<NodeNoteLocal>)node.getNodeNote()) {
					// 備考
					noteList.add(new NodeNoteInfo(cnt, nodeNote.getNote()));
					cnt++;
				}
			}
			property.setNodeNoteInfo(noteList);

			if (nodeCache.get(facilityId) == null) {
				m_log.debug("add NodeCache : " + facilityId);
				nodeCache.put(facilityId, property);
			}
		}
		return property;
	}
}
