/*

Copyright (C) 2011 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */
package com.clustercontrol.repository.util;

import java.net.Inet4Address;
import java.net.Inet6Address;
import java.net.InetAddress;
import java.net.UnknownHostException;

import javax.ejb.FinderException;
import javax.naming.NamingException;

import com.clustercontrol.fault.FacilityNotFound;
import com.clustercontrol.fault.InvalidSetting;
import com.clustercontrol.bean.SnmpVersionConstant;
import com.clustercontrol.commons.util.CommonValidator;
import com.clustercontrol.repository.bean.FacilityConstant;
import com.clustercontrol.repository.bean.FacilityInfo;
import com.clustercontrol.repository.bean.NodeCpuInfo;
import com.clustercontrol.repository.bean.NodeDeviceInfo;
import com.clustercontrol.repository.bean.NodeDiskInfo;
import com.clustercontrol.repository.bean.NodeFilesystemInfo;
import com.clustercontrol.repository.bean.NodeHostnameInfo;
import com.clustercontrol.repository.bean.NodeInfo;
import com.clustercontrol.repository.bean.NodeMemoryInfo;
import com.clustercontrol.repository.bean.NodeNetworkInterfaceInfo;
import com.clustercontrol.repository.bean.ScopeInfo;
import com.clustercontrol.repository.ejb.entity.CollectorPlatformMstUtil;
import com.clustercontrol.repository.factory.NodeProperty;
import com.clustercontrol.repository.factory.ScopeProperty;
import com.clustercontrol.util.Messages;

/**
 * リポジトリ管理の入力チェッククラス
 * 
 * @author hinemos
 *
 */
public class RepositoryValidator {

	public static void validateNodeInfo(NodeInfo nodeInfo) throws InvalidSetting{
		validateFacilityInfo(nodeInfo);

		// hardware
		CommonValidator.validateString(Messages.getString("hardware.type"), nodeInfo.getHardwareType(), false, 0, 128);
		// platformFamily
		CommonValidator.validateString(Messages.getString("platform.family.name"), nodeInfo.getPlatformFamily(), true, 1, 128);
		try{
			CollectorPlatformMstUtil.getLocalHome().findByPrimaryKey(nodeInfo.getPlatformFamily());
		} catch (FinderException e) {
			throw new InvalidSetting("platform " + nodeInfo.getPlatformFamily() + " does not exist!");
		} catch (NamingException e) {
			throw new InvalidSetting("platform " + nodeInfo.getPlatformFamily() + " check unknown error.", e);
		}

		// facilityType
		if(nodeInfo.getFacilityType() != FacilityConstant.TYPE_NODE){
			throw new InvalidSetting("Node FacilityType is  " + FacilityConstant.TYPE_NODE);
		}

		//IPアドレスの入力チェック
		if(nodeInfo.getIpAddressVersion() == null){
			throw new InvalidSetting("IpAddressVersion is null.");
		}
		int ipaddressVersion = nodeInfo.getIpAddressVersion().intValue();
		if(ipaddressVersion == 4){
			//versionが空か4の場合には、
			if(nodeInfo.getIpAddressV4() == null || "".equals(nodeInfo.getIpAddressV4())){
				throw new InvalidSetting(Messages.getString("message.repository.24"));
			}

			// ipv4形式チェック
			try{
				InetAddress address = InetAddress.getByName(nodeInfo.getIpAddressV4());
				if (address instanceof Inet4Address){
					//IPv4の場合はさらにStringをチェック
					if (!nodeInfo.getIpAddressV4().matches(".{1,3}?\\..{1,3}?\\..{1,3}?\\..{1,3}?")){
						throw new InvalidSetting(Messages.getString("message.repository.24") + "(" + nodeInfo.getIpAddressV4() + ")");
					}
				} else{
					throw new InvalidSetting(Messages.getString("message.repository.24") + "(" + nodeInfo.getIpAddressV4() + ")");
				}
			}catch (UnknownHostException e) {
				throw new InvalidSetting(Messages.getString("message.repository.24") + "(" + nodeInfo.getIpAddressV4() + ")");
			}
		}
		else if(ipaddressVersion == 6){
			//	versionが6の場合には、
			if(nodeInfo.getIpAddressV6() == null || "".equals(nodeInfo.getIpAddressV6())){
				throw new InvalidSetting(Messages.getString("message.repository.25"));
			}

			// ipv6形式チェック
			try{
				InetAddress address = InetAddress.getByName(nodeInfo.getIpAddressV6());
				if (address instanceof Inet6Address){
				} else{
					throw new InvalidSetting(Messages.getString("message.repository.25") + "(" + nodeInfo.getIpAddressV6() + ")");
				}
			} catch (UnknownHostException e) {
				throw new InvalidSetting(Messages.getString("message.repository.25") + "(" + nodeInfo.getIpAddressV6() + ")");
			}
		}
		else{
			throw new InvalidSetting("IpAddressVersion is not 4 / 6.");
		}

		//ノード名の入力チェック
		CommonValidator.validateString(Messages.getString("node.name"), nodeInfo.getNodeName(), true, 1, 128);

		//ノード種別指定の必須条件
		String getVirtualizationNodeType = nodeInfo.getVirtualizationNodeType();
		//host指定の場合
		if(getVirtualizationNodeType != null && (getVirtualizationNodeType.equals("host") || getVirtualizationNodeType.equals("controller"))){

			String virtSolution = nodeInfo.getVirtualizationSolution();
			String vmUser = nodeInfo.getVmUser();
			String vmUserpassword = nodeInfo.getVmUserPassword();
			String vmProtocol = nodeInfo.getVmProtocol();

			//必須項目を全て入力すること
			if(virtSolution == null || virtSolution.compareTo("") == 0
					|| vmUser == null || vmUser.compareTo("") == 0
					|| vmUserpassword == null || vmUserpassword.compareTo("") == 0
					|| vmProtocol == null || vmProtocol.compareTo("") == 0){
				throw new InvalidSetting(Messages.getString("message.repository.42"));
			}

			//guest指定の場合
		}else if(getVirtualizationNodeType != null && getVirtualizationNodeType.equals("guest")){

			String virtSolution = nodeInfo.getVirtualizationSolution();
			String vmManagementNode = nodeInfo.getVmManagementNode();
			String vmName = nodeInfo.getVmName();
			String vmId = nodeInfo.getVmId();

			//必須項目を全て入力すること
			if(virtSolution == null || virtSolution.compareTo("") == 0
					|| vmManagementNode == null || vmManagementNode.compareTo("") == 0
					|| vmName == null || vmName.compareTo("") == 0
					|| vmId == null || vmId.compareTo("") == 0){

				throw new InvalidSetting(Messages.getString("message.repository.43"));
			}
		}

		// OS名
		CommonValidator.validateString(Messages.getString("os.name"), nodeInfo.getOsName(), false, 0, 256);
		// OSリリース
		CommonValidator.validateString(Messages.getString("os.release"), nodeInfo.getOsRelease(), false, 0, 256);
		// OSバージョン
		CommonValidator.validateString(Messages.getString("os.version"), nodeInfo.getOsVersion(), false, 0, 256);
		// 文字セット
		CommonValidator.validateString(Messages.getString("character.set"), nodeInfo.getCharacterSet(), false, 0, 16);

		//デバイスの入力チェック
		if(nodeInfo.getNodeCpuInfo() != null){
			String DeviceTypeName = Messages.getString("cpu");
			for(NodeCpuInfo info : nodeInfo.getNodeCpuInfo()){
				CommonValidator.validateString(DeviceTypeName + "[" + Messages.getString("device.name") + "]", info.getDeviceName(), true, 1, 128);
				CommonValidator.validateString(DeviceTypeName + "[" + Messages.getString("device.type") + "]", info.getDeviceType(), true, 1, 32);
				CommonValidator.validateString(DeviceTypeName + "[" + Messages.getString("device.display.name") + "]", info.getDeviceDisplayName(), true, 1, 256);
				CommonValidator.validateDouble(DeviceTypeName + "[" + Messages.getString("device.index") + "]", info.getDeviceIndex(), 0, Integer.MAX_VALUE);
			}
		}
		if(nodeInfo.getNodeMemoryInfo() != null){
			String DeviceTypeName = Messages.getString("memory");
			for(NodeMemoryInfo info : nodeInfo.getNodeMemoryInfo()){
				CommonValidator.validateString(DeviceTypeName + "[" + Messages.getString("device.name") + "]", info.getDeviceName(), true, 1, 128);
				CommonValidator.validateString(DeviceTypeName + "[" + Messages.getString("device.type") + "]", info.getDeviceType(), true, 1, 32);
				CommonValidator.validateString(DeviceTypeName + "[" + Messages.getString("device.display.name") + "]", info.getDeviceDisplayName(), true, 1, 256);
				CommonValidator.validateDouble(DeviceTypeName + "[" + Messages.getString("device.index") + "]", info.getDeviceIndex(), 0, Integer.MAX_VALUE);
			}
		}
		if(nodeInfo.getNodeDiskInfo() != null){
			String DeviceTypeName = Messages.getString("disk");
			for(NodeDiskInfo info : nodeInfo.getNodeDiskInfo()){
				CommonValidator.validateString(DeviceTypeName + "[" + Messages.getString("device.name") + "]", info.getDeviceName(), true, 1, 128);
				CommonValidator.validateString(DeviceTypeName + "[" + Messages.getString("device.type") + "]", info.getDeviceType(), true, 1, 32);
				CommonValidator.validateString(DeviceTypeName + "[" + Messages.getString("device.display.name") + "]", info.getDeviceDisplayName(), true, 1, 256);
				CommonValidator.validateDouble(DeviceTypeName + "[" + Messages.getString("device.index") + "]", info.getDeviceIndex(), 0, Integer.MAX_VALUE);
			}
		}
		if(nodeInfo.getNodeNetworkInterfaceInfo() != null){
			String DeviceTypeName = Messages.getString("network.interface");
			for(NodeNetworkInterfaceInfo info : nodeInfo.getNodeNetworkInterfaceInfo()){
				CommonValidator.validateString(DeviceTypeName + "[" + Messages.getString("device.name") + "]", info.getDeviceName(), true, 1, 128);
				CommonValidator.validateString(DeviceTypeName + "[" + Messages.getString("device.type") + "]", info.getDeviceType(), true, 1, 32);
				CommonValidator.validateString(DeviceTypeName + "[" + Messages.getString("device.display.name") + "]", info.getDeviceDisplayName(), true, 1, 256);
				CommonValidator.validateDouble(DeviceTypeName + "[" + Messages.getString("device.index") + "]", info.getDeviceIndex(), 0, Integer.MAX_VALUE);
			}
		}
		if(nodeInfo.getNodeFilesystemInfo() != null){
			String DeviceTypeName = Messages.getString("file.system");
			for(NodeFilesystemInfo info : nodeInfo.getNodeFilesystemInfo()){
				CommonValidator.validateString(DeviceTypeName + "[" + Messages.getString("device.name") + "]", info.getDeviceName(), true, 1, 128);
				CommonValidator.validateString(DeviceTypeName + "[" + Messages.getString("device.type") + "]", info.getDeviceType(), true, 1, 32);
				CommonValidator.validateString(DeviceTypeName + "[" + Messages.getString("device.display.name") + "]", info.getDeviceDisplayName(), true, 1, 256);
				CommonValidator.validateDouble(DeviceTypeName + "[" + Messages.getString("device.index") + "]", info.getDeviceIndex(), 0, Integer.MAX_VALUE);
			}
		}
		if(nodeInfo.getNodeDeviceInfo() != null){
			String DeviceTypeName = Messages.getString("general.device");
			for(NodeDeviceInfo info : nodeInfo.getNodeDeviceInfo()){
				CommonValidator.validateString(DeviceTypeName + "[" + Messages.getString("device.name") + "]", info.getDeviceName(), true, 1, 128);
				CommonValidator.validateString(DeviceTypeName + "[" + Messages.getString("device.type") + "]", info.getDeviceType(), true, 1, 32);
				CommonValidator.validateString(DeviceTypeName + "[" + Messages.getString("device.display.name") + "]", info.getDeviceDisplayName(), true, 1, 256);
				CommonValidator.validateDouble(DeviceTypeName + "[" + Messages.getString("device.index") + "]", info.getDeviceIndex(), 0, Integer.MAX_VALUE);
			}
		}
		if(nodeInfo.getNodeHostnameInfo() != null){
			for(NodeHostnameInfo info : nodeInfo.getNodeHostnameInfo()){
				CommonValidator.validateString(Messages.getString("host.name"), info.getHostname(), false, 0, 128);
			}
		}

		//サービスのチェック(SNMP)
		CommonValidator.validateString(Messages.getString("community.name"), nodeInfo.getSnmpCommunity(), false, 0, 64);
		if(nodeInfo.getSnmpVersion() == null
				|| (!"".equals(nodeInfo.getSnmpVersion())
						&& !SnmpVersionConstant.STRING_V1.equals(nodeInfo.getSnmpVersion())
						&& !SnmpVersionConstant.STRING_V2.equals(nodeInfo.getSnmpVersion()))){
			throw new InvalidSetting("SNMP Version is " + SnmpVersionConstant.STRING_V1 + " or " + SnmpVersionConstant.STRING_V2);
		}
		CommonValidator.validateInt(Messages.getString("snmp.retries"),
				nodeInfo.getSnmpRetryCount(), 1, 10);

		//サービスのチェック(WBEM)
		if(nodeInfo.getWbemProtocol() == null
				|| (!"".equals(nodeInfo.getWbemProtocol())
						&& !"http".equals(nodeInfo.getWbemProtocol())
						&& !"https".equals(nodeInfo.getWbemProtocol()))){
			throw new InvalidSetting("WBEM Protocol is http or https");
		}
		CommonValidator.validateInt(Messages.getString("wbem.retries"),
				nodeInfo.getWbemRetryCount(), 1, 10);

		//サービスのチェック(IPMI)
		CommonValidator.validateString(Messages.getString("ipmi.protocol"), nodeInfo.getIpmiProtocol(), false, 0, 32);
		CommonValidator.validateString(Messages.getString("ipmi.level"), nodeInfo.getIpmiLevel(), false, 0, 32);
		CommonValidator.validateInt(Messages.getString("ipmi.retries"),
				nodeInfo.getIpmiRetries(), 1, 10);

		//サービスのチェック(WinRM)
		if(nodeInfo.getWinrmProtocol() == null
				|| (!"".equals(nodeInfo.getWinrmProtocol())
						&& !"http".equals(nodeInfo.getWinrmProtocol())
						&& !"https".equals(nodeInfo.getWinrmProtocol()))){
			throw new InvalidSetting("WinRM Protocol is http or https");
		}
		CommonValidator.validateInt(Messages.getString("winrm.retries"),
				nodeInfo.getWinrmRetries(), 1, 10);

		// administrator
		CommonValidator.validateString(Messages.getString("administrator"), nodeInfo.getAdministrator(), false, 0, 256);

	}

	/**
	 * スコープ情報の妥当性チェック
	 * 
	 * @param scopeInfo
	 * @throws InvalidSetting
	 */
	public static void validateScopeInfo(String parentFacilityId, ScopeInfo scopeInfo, boolean parentCheck) throws InvalidSetting{
		validateFacilityInfo(scopeInfo);

		// parentFacilityId
		if(parentCheck){
			try{
				ScopeProperty.getProperty(parentFacilityId);
			} catch (FacilityNotFound e) {
				throw new InvalidSetting("Scope does not exist! facilityId = " + parentFacilityId);
			} catch (NamingException e) {
				throw new InvalidSetting("Unknown Error facilityId = " + parentFacilityId, e);
			}
		}

		// facilityType
		if(scopeInfo.getFacilityType() != FacilityConstant.TYPE_SCOPE){
			throw new InvalidSetting("Scope FacilityType is  " + scopeInfo.getFacilityType());
		}
	}

	public static void validateFacilityInfo (FacilityInfo facilityInfo) throws InvalidSetting {
		// facilityId
		CommonValidator.validateId(Messages.getString("facility.id"), facilityInfo.getFacilityId(), 64);

		// facilityName
		CommonValidator.validateString(Messages.getString("facility.name"), facilityInfo.getFacilityName(), true, 1, 128);

		// description
		CommonValidator.validateString(Messages.getString("description"), facilityInfo.getDescription(), false, 0, 256);
	}

	/**
	 * ノード割り当て時のチェック
	 * 
	 * @param parentFacilityId
	 * @param facilityIds
	 * @throws InvalidSetting
	 */
	public static void validateaAssignNodeScope(String parentFacilityId, String[] facilityIds) throws InvalidSetting{
		// parentFacilityId
		try{
			ScopeProperty.getProperty(parentFacilityId);
		} catch (FacilityNotFound e) {
			throw new InvalidSetting("Scope does not exist! facilityId = " + parentFacilityId);
		} catch (NamingException e) {
			throw new InvalidSetting("Unknown Error facilityId = " + parentFacilityId, e);
		}

		// facilityIds
		if(facilityIds == null || facilityIds.length == 0){
			throw new InvalidSetting(Messages.getString("message.repository.2"));
		}
		for (int i = 0; i < facilityIds.length; i++) {
			try{
				NodeProperty.getProperty(facilityIds[i]);
			} catch (FacilityNotFound e) {
				throw new InvalidSetting("Node does not exist! facilityId = " + facilityIds[i]);
			}
		}
	}

}
