/*

Copyright (C) 2010 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.selfcheck.monitor;

import java.sql.Connection;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.Date;

import javax.naming.InitialContext;
import javax.sql.DataSource;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.bean.PriorityConstant;
import com.clustercontrol.notify.bean.OutputBasicInfo;
import com.clustercontrol.util.Messages;

/**
 * データソースへの疎通を確認する処理の実装クラス
 */
public class DataSourceMonitor extends SelfCheckMonitorBase {

	private static Log m_log = LogFactory.getLog( DataSourceMonitor.class );

	private String dataSourceName = null;
	private String jndiName = null;
	private final String jndiPrefix = "java:/";
	private static final String query = "SELECT 1";

	private static final String monitorIdPrefix = "SYS_DS";
	private String monitorId = "";
	private String application = "SELFCHECK (DataSource)";

	/**
	 * コンストラクタ
	 * @param dataSourceName データソース名（"HinemosDS"など）
	 */
	public DataSourceMonitor(String dataSourceName) {
		this.dataSourceName = dataSourceName;
		this.jndiName = jndiPrefix + dataSourceName;

		this.monitorId = monitorIdPrefix + "_" + dataSourceName;
	}

	/**
	 * セルフチェック処理名
	 */
	@Override
	public String toString() {
		return "monitoring datasource - " + dataSourceName;
	}

	/**
	 * 監視項目ID
	 */
	@Override
	public String getMonitorId() {
		return monitorId;
	}

	/**
	 * データソースへの疎通確認
	 * @return 通知情報（アプリケーション名は未格納）
	 */
	@Override
	public OutputBasicInfo execute() {
		/** ローカル変数 */
		InitialContext initCtx = null;
		DataSource dataSource = null;
		Connection conn = null;
		Statement stmt = null;

		boolean avaiable = false;

		OutputBasicInfo notifyInfo = null;

		/** メイン処理 */
		if (m_log.isDebugEnabled()) m_log.debug("monitoring datasource. (dataSource = " + dataSourceName + ", jndi="+ jndiName + ")");
		try {
			// データソースのオブジェクトをJNDI経由で取得し、取得したコネクションが正しく動作するかを確認する
			initCtx = new InitialContext();
			dataSource = (DataSource)initCtx.lookup(jndiName);

			conn = dataSource.getConnection();
			stmt = conn.createStatement();
			if (stmt.execute(query)) {
				avaiable = true;
			} else {
				avaiable = false;
			}
		} catch (Exception e) {
			m_log.warn("executing query failure. (dataSource = " + dataSourceName + ")", e);
			avaiable = false;
		} finally {
			try {
				if (stmt != null) {
					stmt.close();
				}
				if (conn != null) {
					conn.close();
				}
			} catch (SQLException e) {
				m_log.warn("closing connection failure. (dataSource = " + dataSourceName + ")", e);
			}
		}

		if (avaiable) {
			if (m_log.isDebugEnabled()) m_log.debug("datasource is available. (dataSource = " + dataSourceName + ")");
			notifyInfo = new OutputBasicInfo();

			// set result, but do not notify (priority == info)
			notifyInfo = new OutputBasicInfo();
			notifyInfo.setPluginId(PLUGIN_ID);
			notifyInfo.setMonitorId(monitorId);
			notifyInfo.setPriority(PriorityConstant.TYPE_INFO);
		} else {
			m_log.warn("datasource is not avaiable. (dataSource = " + dataSourceName + ")");
			notifyInfo = new OutputBasicInfo();

			String[] msgAttr1 = { dataSourceName };
			String[] msgAttr2 = { dataSourceName, query };

			notifyInfo.setPluginId(PLUGIN_ID);
			notifyInfo.setMonitorId(monitorId);
			notifyInfo.setApplication(application);
			notifyInfo.setMessageId("001");
			notifyInfo.setMessage(Messages.getString("message.selfcheck.notify.db.failure.msg", msgAttr1));
			notifyInfo.setMessageOrg(Messages.getString("message.selfcheck.notify.db.failure.origmsg", msgAttr2));
			notifyInfo.setFacilityId(FACILITY_ID);
			notifyInfo.setScopeText(FACILITY_TEXT);
			notifyInfo.setPriority(PriorityConstant.TYPE_WARNING);
			notifyInfo.setGenerationDate(new Date().getTime());
		}

		return notifyInfo;
	}

}
