/*

Copyright (C) 2011 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.systemlog.service;

import java.io.IOException;
import java.net.DatagramPacket;
import java.net.DatagramSocket;
import java.net.InetAddress;
import java.net.SocketException;
import java.util.Arrays;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.commons.util.HinemosProperties;
import com.clustercontrol.systemlog.SyslogReceiverConfig;
import com.clustercontrol.systemlog.service.MatcherTask;
import com.clustercontrol.systemlog.service.queue.Channel;
import com.clustercontrol.systemlog.service.queue.Consumer;

/**
 * syslogを受信し、マッチング処理を行う機構に受け取ったデータを渡します。
 */
public class SyslogReceiver {
	/** ログ出力のインスタンス  */
	private Log _log = LogFactory.getLog(this.getClass());

	private final SyslogReceiverConfig _config;

	private DatagramSocket _socket;
	private volatile boolean _isStop = true;

	private ExecutorService _executorService;
	private Channel _channel;

	// 受信したメッセージの数
	private volatile long _messageCounter = 0;

	// 指定の間隔でログに統計情報を出力する
	private final int _statsInterval;

	public enum Mode {
		/** マッチング処理用にプールされたスレッドを利用するモード*/
		POOLED_MULTI_THREAD,
		/** ノード毎に独立したキューを利用するモード*/
		NODE_QUEUE
	}

	/**
	 * 新しいSyslogReceiverオブジェクトを割り当てます
	 * @param port ポート番号
	 * @param bindAddress バインド対象IPアドレス
	 * @param threadPoolSize プールするスレッド数
	 * @param charsetName 文字セット
	 * @param multiId HA構成時にインスタンスを特定するためのID
	 * @param mode 動作モード
	 */
	public SyslogReceiver(SyslogReceiverConfig config){

		_log.info("initializing SyslogReceiver. [" + config + "]");
		this._config = config;

		_statsInterval = 1000;

		init();
	}

	// 初期化
	private void init(){
		try {
			_socket = new DatagramSocket(_config.getListenPort(), InetAddress.getByName(_config.getListenAddress()));

			_log.info("SyslogReceiver  default buffer size : " + _socket.getReceiveBufferSize());
			int size = Integer.parseInt(HinemosProperties.getProperty("monitor.systemlog.udp.socket.buffer.size", "8388608"));
			_socket.setReceiveBufferSize(size);
			_log.info("SyslogReceiver  setting buffer size : " + _socket.getReceiveBufferSize());
		} catch (IOException e) {
			_log.error(e.getMessage(), e);
		}

		// 動作モード毎の初期化処理
		switch (_config.getQueueMode()) {
		case NODE_QUEUE :
			_channel = new Channel();
			new Thread(new Consumer(_channel, _config), "NodeQueueConsumer").start();
			break;
		default :
			_executorService = new ThreadPoolExecutor(_config.getThreadPoolSize(), _config.getThreadPoolSize(),
					0L, TimeUnit.MILLISECONDS, new LinkedBlockingQueue<Runnable>(_config.getFilterTaskQueueSize()),
					new ThreadFactory() {
				private volatile int _count = 0;

				@Override
				public Thread newThread(Runnable r) {
					String threadName = "SyslogMatcherTask-" + _count++;
					_log.info("create thread : " + threadName);
					return new Thread(r, threadName);
				}
			}, new SyslogRejectionHandler());

		}
	}

	private class SyslogRejectionHandler extends ThreadPoolExecutor.DiscardPolicy {

		@Override
		public void rejectedExecution(Runnable r, ThreadPoolExecutor e) {
			if (r instanceof MatcherTask) {
				_log.warn("too many syslog, filtering queue is full. (msg = " + ((MatcherTask)r).getMsgString() + ")");
			}
		}

	}

	/**
	 * syslog受信処理を開始します。
	 */
	public void start(){
		_isStop = false;

		Thread receiverThread = new Thread(new ReceiverTask());
		receiverThread.setName("SyslogReceiver");
		receiverThread.start();
	}

	/**
	 * 起動時から受信したsyslog数を返します。
	 * @return
	 */
	public long getMessageCount(){
		return _messageCounter;
	}

	/**
	 * syslog受信処理を終了します。
	 */
	public void stop(){
		_isStop = true;

		_executorService.shutdown();

		// 排他制御していないため、closeを呼ばれれた後、receiveがコールされる可能性がある。
		// その場合、既にcloseしたはずのDatagramSocketのファイルディスクリプタは開放され、
		// 他のSocketで利用されている可能性がある。
		if(_socket != null){
			_socket.close();
		}

		try {
			_log.info("call awaitTermination()");
			// 1分間終了を待つ。
			if (!_executorService.awaitTermination(60 * 1000l, TimeUnit.MILLISECONDS)){
				// タイムアウトした場合は、全てのスレッドを中断(interrupt)してスレッドプールを破棄する。
				_log.info("call shutdownNow()");
				_executorService.shutdownNow();
			}
			_log.info("called awaitTermination()");
		} catch (InterruptedException e) {
			// awaitTerminationの処理中にinterruptされた場合も、全てのスレッドを中断する
			_executorService.shutdownNow();
		}
	}

	private class ReceiverTask implements Runnable {
		private byte[] buffer = new byte[8192];

		@Override
		public void run() {
			DatagramPacket packet = new DatagramPacket(buffer, buffer.length);
			while(!_isStop){
				try {
					_socket.receive(packet);

					synchronized (this) {
						if (_messageCounter < Long.MAX_VALUE) {
							_messageCounter++;

							if (_statsInterval != 0 && _messageCounter % _statsInterval == 0) {
								_log.info("The number of received syslog : " + _messageCounter);
							}
						} else {
							_messageCounter = 0;
							_log.info("syslog received counter is reseted.");
						}
					}

					// 受信したパケットのデータをbyte型の配列にコピーする
					byte[] msgData = Arrays.copyOf(packet.getData(), packet.getLength());

					// 動作モード毎にマッチング処理
					switch (_config.getQueueMode()) {
					case NODE_QUEUE :
						_channel.addMessage(packet.getAddress(), msgData);
						break;
					default :
						_executorService.execute(new MatcherTask(_config, msgData));
					}
				} catch (Exception e) {
					if (e instanceof SocketException) {
						// Hinemosマネージャ停止時に、ここを通ることがある。
						_log.warn("ReceiverTask : SocketException, " + e.getMessage());
					} else {
						_log.error(e.getMessage(), e);
					}

					// 例外と初期化が繰り返されることで高負荷にならないようにスリープを入れる
					try {
						Thread.sleep(100);
					} catch (InterruptedException e1) {
						_log.error(e1.getMessage(), e1);
					}

					// ソケットやスレッドプーリングを再作成。
					init();
					packet = new DatagramPacket(buffer, buffer.length);
				}
			}
		}
	}
}
