/*
Copyright (C) 2010 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.
 */
package com.clustercontrol.ws.endpoint;

import java.util.ArrayList;

import javax.annotation.Resource;
import javax.ejb.CreateException;
import javax.naming.NamingException;
import javax.xml.ws.WebServiceContext;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jnp.interfaces.NamingContext;

import com.clustercontrol.fault.HinemosUnknown;
import com.clustercontrol.fault.InvalidRole;
import com.clustercontrol.fault.InvalidSetting;
import com.clustercontrol.fault.InvalidUserPass;
import com.clustercontrol.fault.MailTemplateDuplicate;
import com.clustercontrol.fault.MailTemplateNotFound;
import com.clustercontrol.accesscontrol.bean.RoleConstant;
import com.clustercontrol.bean.HinemosModuleConstant;
import com.clustercontrol.notify.mail.bean.MailTemplateInfo;
import com.clustercontrol.notify.mail.ejb.session.MailTemplateControllerLocal;
import com.clustercontrol.notify.mail.ejb.session.MailTemplateControllerLocalHome;

/**
 * メールテンプレート用のWebAPIエンドポイント
 */
@javax.jws.WebService(targetNamespace = "http://mailtemplate.ws.clustercontrol.com")
public class MailTemplateEndpoint {
	@Resource
	WebServiceContext wsctx;

	private static Log m_log = LogFactory.getLog( MailTemplateEndpoint.class );
	private static Log m_opelog = LogFactory.getLog("HinemosOperation");

	/**
	 * echo(WebサービスAPI疎通用)
	 * 
	 * 権限必要なし（ユーザ名チェックのみ実施）
	 * 
	 * @param str
	 * @return
	 * @throws InvalidUserPass
	 * @throws InvalidRole
	 * @throws HinemosUnknown
	 */
	public String echo(String str) throws InvalidUserPass, InvalidRole, HinemosUnknown {
		ArrayList<String> roleList = new ArrayList<String>();
		HttpAuthenticator.authCheck(wsctx, roleList);

		return str + ", " + str;
	}

	/**
	 * メールテンプレート用のSessionBeanの取得
	 * 
	 * @return
	 * @throws HinemosUnknown
	 * @throws InvalidUserPass
	 */
	private MailTemplateControllerLocal getMailTemplateControllerLocal() throws HinemosUnknown, InvalidUserPass {
		MailTemplateControllerLocal local = null;
		try{
			String account = HttpAuthenticator.getAccount(wsctx);
			NamingContext namingContext = HttpAuthenticator.getContext(account);
			MailTemplateControllerLocalHome localHome =
				(MailTemplateControllerLocalHome)
				namingContext.lookup(MailTemplateControllerLocalHome.JNDI_NAME);
			local = localHome.create();
		} catch (CreateException e) {
			m_log.error("getMailTemplateControllerLocal CreateException : " + e,e);
			throw new HinemosUnknown(e.getMessage(), e);
		} catch (NamingException e) {
			m_log.error("getMailTemplateControllerLocal NamingException : " + e,e);
			throw new HinemosUnknown(e.getMessage(), e);
		}
		return local;
	}

	/**
	 * メールテンプレート情報をマネージャに登録します。<BR>
	 * 
	 * NotifyWrite権限が必要
	 * 
	 * @param info 作成対象のメールテンプレート情報
	 * @return 作成に成功した場合、<code> true </code>
	 * @throws MailTemplateDuplicate
	 * @throws HinemosUnknown
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 * 
	 * @see com.clustercontrol.notify.factory.AddMailTemplate#add(MailTemplateInfo)
	 */
	public boolean addMailTemplate(MailTemplateInfo data) throws HinemosUnknown, MailTemplateDuplicate, InvalidUserPass, InvalidRole,InvalidSetting {
		m_log.debug("addMailTemplate");
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.NOTIFY_WRITE);
		HttpAuthenticator.authCheck(wsctx, roleList);

		// 認証済み操作ログ
		if(data != null){
			StringBuffer msg = new StringBuffer();
			msg.append(", MailTemplateID=");
			msg.append(data.getMailTemplateId());
			m_opelog.info(HinemosModuleConstant.LOG_PREFIX_MAIL_TEMPLATE + " Add, Method=addMailTemplate, User="
					+ HttpAuthenticator.getUserAccountString(wsctx)
					+ msg.toString());
		}

		return getMailTemplateControllerLocal().addMailTemplate(data);
	}

	/**
	 * マネージャ上のメールテンプレート情報を変更します。<BR>
	 * 
	 * NotifyWrite権限が必要
	 * 
	 * @param info 変更対象のメールテンプレート情報
	 * @return 変更に成功した場合、<code> true </code>
	 * @throws MailTemplateNotFound
	 * @throws HinemosUnknown
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 * 
	 * @see com.clustercontrol.notify.factory.ModifyMailTemplate#modify(MailTemplateInfo)
	 */
	public boolean modifyMailTemplate(MailTemplateInfo data) throws HinemosUnknown, MailTemplateNotFound, InvalidUserPass, InvalidRole,InvalidSetting {
		m_log.debug("modifyMailTemplate");
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.NOTIFY_WRITE);
		HttpAuthenticator.authCheck(wsctx, roleList);

		// 認証済み操作ログ
		if(data != null){
			StringBuffer msg = new StringBuffer();
			msg.append(", MailTemplateID=");
			msg.append(data.getMailTemplateId());
			m_opelog.info(HinemosModuleConstant.LOG_PREFIX_MAIL_TEMPLATE + " Change, Method=modifyMailTemplate, User="
					+ HttpAuthenticator.getUserAccountString(wsctx)
					+ msg.toString());
		}

		return getMailTemplateControllerLocal().modifyMailTemplate(data);
	}

	/**
	 * メールテンプレート情報をマネージャから削除します。<BR>
	 * 
	 * NotifyWrite権限が必要
	 * 
	 * @param mailTemplateId 削除対象のメールテンプレートID
	 * @return 削除に成功した場合、<code> true </code>
	 * @throws HinemosUnknown
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 * 
	 * @see com.clustercontrol.notify.factory.DeleteMailTemplate#delete(String)
	 */
	public boolean deleteMailTemplate(String mailTemplateId) throws HinemosUnknown, InvalidUserPass, InvalidRole {
		m_log.debug("deleteMailTemplate");
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.NOTIFY_WRITE);
		HttpAuthenticator.authCheck(wsctx, roleList);

		// 認証済み操作ログ
		StringBuffer msg = new StringBuffer();
		msg.append(", MailTemplateID=");
		msg.append(mailTemplateId);
		m_opelog.info(HinemosModuleConstant.LOG_PREFIX_MAIL_TEMPLATE + " Delete, Method=deleteMailTemplate, User="
				+ HttpAuthenticator.getUserAccountString(wsctx)
				+ msg.toString());

		return getMailTemplateControllerLocal().deleteMailTemplate(mailTemplateId);
	}

	/**
	 * 引数で指定されたメールテンプレート情報を返します。
	 * 
	 * NotifyRead権限が必要
	 * 
	 * @param mailTemplateId 取得対象のメールテンプレートID
	 * @return メールテンプレート情報
	 * @throws HinemosUnknown
	 * @throws MailTemplateNotFound
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 * 
	 * @see com.clustercontrol.notify.factory.SelectMailTemplate#getMailTemplateInfo(String)
	 */
	public MailTemplateInfo getMailTemplateInfo(String mailTemplateId) throws MailTemplateNotFound, HinemosUnknown, InvalidUserPass, InvalidRole {
		m_log.debug("getMailTemplateInfo");
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.NOTIFY_READ);
		HttpAuthenticator.authCheck(wsctx, roleList);

		// 認証済み操作ログ
		StringBuffer msg = new StringBuffer();
		msg.append(", MailTemplateID=");
		msg.append(mailTemplateId);
		m_opelog.debug(HinemosModuleConstant.LOG_PREFIX_MAIL_TEMPLATE + " Get, Method=getMailTemplateInfo, User="
				+ HttpAuthenticator.getUserAccountString(wsctx)
				+ msg.toString());

		return getMailTemplateControllerLocal().getMailTemplateInfo(mailTemplateId);
	}

	/**
	 * メールテンプレート情報一覧を取得します。<BR>
	 * 
	 * NotifyRead権限が必要
	 * 
	 * @return メールテンプレート情報一覧（Objectの2次元配列）
	 * @throws MailTemplateNotFound
	 * @throws HinemosUnknown
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 * 
	 * @see com.clustercontrol.notify.factory.SelectMailTemplate#getMailTemplateList()
	 */
	@SuppressWarnings("unchecked")
	public ArrayList<MailTemplateInfo> getMailTemplateList() throws HinemosUnknown, MailTemplateNotFound, InvalidUserPass, InvalidRole {
		m_log.debug("getMailTemplateList");
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.NOTIFY_READ);
		HttpAuthenticator.authCheck(wsctx, roleList);

		// 認証済み操作ログ
		m_opelog.debug(HinemosModuleConstant.LOG_PREFIX_MAIL_TEMPLATE + " Get, Method=getMailTemplateList, User="
				+ HttpAuthenticator.getUserAccountString(wsctx));

		return getMailTemplateControllerLocal().getMailTemplateList();
	}
}