/*

 Copyright (C) 2011 NTT DATA Corporation

 This program is free software; you can redistribute it and/or
 Modify it under the terms of the GNU General Public License
 as published by the Free Software Foundation, version 2.

 This program is distributed in the hope that it will be
 useful, but WITHOUT ANY WARRANTY; without even the implied
 warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
 PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.agent.log;

import java.io.File;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.concurrent.ConcurrentHashMap;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.agent.SendQueue;
import com.clustercontrol.agent.util.AgentProperties;
import com.clustercontrol.bean.HinemosModuleConstant;
import com.clustercontrol.bean.ValidConstant;
import com.clustercontrol.ws.agent.OutputBasicInfo;
import com.clustercontrol.ws.monitor.MonitorInfo;

/**
 * ログ転送スレッドを管理するクラス<BR>
 * 
 * 転送対象ログファイル情報を受け取り、ログ転送スレッドを制御します。
 * 
 */
public class LogfileMonitorManager {

	//ロガー
	private Log m_log = LogFactory.getLog(LogfileMonitorManager.class);

	/** ファイルパスとファイルの読み込み状態を保持しているマップ */
	private static ConcurrentHashMap<String, LogfileMonitor> m_logfileMonitorCache =
		new ConcurrentHashMap<String, LogfileMonitor>();

	/** Queue送信  */
	private SendQueue m_sendQueue;

	/** ログファイル監視間隔 */
	private static int m_runInterval = 10000; // 10sec
	
	/** 監視可能ファイル数 */
	private static int m_fileMax = 500;
	
	/** このエージェントの監視設定 */
	private static ArrayList<MonitorInfo> m_monitorList = new ArrayList<MonitorInfo>();
	
	/** 正規表現モード */
	private static boolean m_regexp = false;

	/** セパレータ */
	private static String m_separator = "\"";
	
	/**
	 * ログ転送管理情報クラス<BR>
	 *
	 */
	class LogThreadManageInfo{

		String m_fileName;

		LogfileMonitor m_thread;

		public String getFileName() {
			return m_fileName;
		}

		public LogfileMonitor getThread() {
			return m_thread;
		}

		public LogThreadManageInfo(String fileName) {
			super();
			m_fileName = fileName;
		}

		public void setThread(LogfileMonitor thread) {
			m_thread = thread;
		}
	}

	/**
	 * コンストラクタ
	 * 
	 * @param ejbConnectionManager EJBコネクション管理
	 * @param sendQueue 監視管理Queue送信
	 * @param props ログ転送エージェントプロパティ
	 */
	public LogfileMonitorManager(SendQueue sendQueue) {
		m_sendQueue = sendQueue;
		String key1 = "logfile.run.interval";
		try {
			String runIntervalStr = AgentProperties.getProperty(key1, Integer.toString(m_runInterval));
			m_runInterval = Integer.parseInt(runIntervalStr);
		} catch (Exception e) {
			m_log.warn("LogfileThread : " + e.getMessage());
		}
		String key2 = "logfile.file.max";
		try {
			String fileMaxStr = AgentProperties.getProperty(key2, Integer.toString(m_fileMax));
			m_fileMax = Integer.parseInt(fileMaxStr);
		} catch (Exception e) {
			m_log.warn("LogfileThread : " + e.getMessage());
		} 
		String key3 = "logfile.regexp";
		try {
			String regexpStr = AgentProperties.getProperty(key3, Boolean.toString(m_regexp));
			m_regexp = Boolean.parseBoolean(regexpStr);
		} catch (Exception e) {
			m_log.warn("LogfileThread : " + e.getMessage());
		}
		String key4 = "logfile.separator";
		try {
			m_separator = AgentProperties.getProperty(key4, m_separator);
		} catch (Exception e) {
			m_log.warn("LogfileThread : " + e.getMessage());
		}
		m_log.info(key1 + "=" + m_runInterval + ", " + key2 + "=" + m_fileMax + ", " +
				key3 + "=" + m_regexp + ", " + key4 + "=" + m_separator);
	}

	private static Object globalLock = new Object();

	/**
	 * 監視設定をスレッドに反映します。<BR>
	 * 
	 * @param list 転送対象ログファイル情報一覧
	 */
	public void setLogfileMonitor(ArrayList<MonitorInfo> monitorList) {
		if (!m_regexp) {
			setLogfileMonitor1(monitorList);
		} else {
			setLogfileMonitor2(monitorList);
		}
	}

	private void setLogfileMonitor1(ArrayList<MonitorInfo> monitorList) {
		HashMap <String, ArrayList<MonitorInfo>> newMonitorMap =
			new HashMap<String, ArrayList<MonitorInfo>>();

		try {
			String runIntervalStr = AgentProperties.getProperty("run.interval",
					Integer.toString(m_runInterval));
			m_runInterval = Integer.parseInt(runIntervalStr);
		} catch (Exception e) {
			m_log.warn("LogfileThread : " + e.getMessage());
		}
		/*
		 * logfileMonitorはログファイルごとにオブジェクトが生成される。
		 * logfileMonitor.monitorInfoListに監視設定が登録される。
		 * (logfileMonitorとmonitorInfoは1対多の関係)
		 */
		/*
		 * 1. logfileMonitorを生成する。
		 */
		for (MonitorInfo monitorInfo : monitorList) {
			if (monitorInfo.getMonitorFlg() == ValidConstant.TYPE_INVALID) {
				continue;
			}
			String filePath = monitorInfo.getLogfileCheckInfo().getLogfile();

			LogfileMonitor logfileMonitor = m_logfileMonitorCache.get(filePath);
			if(logfileMonitor == null){
				// ファイル監視オブジェクトを生成。
				logfileMonitor = new LogfileMonitor(this, filePath, m_runInterval, false);
				m_logfileMonitorCache.put(filePath, logfileMonitor);
			}

			ArrayList<MonitorInfo> list = newMonitorMap.get(filePath);
			if (list == null){
				list = new ArrayList<MonitorInfo> ();
				newMonitorMap.put(filePath, list);
			}
			list.add(monitorInfo);
		}

		/*
		 * 2. logfileMonitor.monitorInfoListを登録する。
		 */
		ArrayList<String> noMonitorFileList = new ArrayList<String>();
		for (String filePath : m_logfileMonitorCache.keySet()) {
			LogfileMonitor thread = m_logfileMonitorCache.get(filePath);
			ArrayList<MonitorInfo> list = newMonitorMap.get(filePath);
			thread.setMonitor(list);
			/*
			 * 監視設定が登録されていないファイルは閉じる。
			 */
			if (thread.clean()) {
				noMonitorFileList.add(filePath);
			}
		}
		for (String file : noMonitorFileList) {
			m_logfileMonitorCache.remove(file);
		}
	}
	
	private void setLogfileMonitor2(ArrayList<MonitorInfo> monitorList) {
		 m_log.info("setLogfileMonitor() : m_monitorList.size=" + monitorList.size());
		 m_monitorList = monitorList;
	}
	
	/*
	 * 新規の監視項目なのか、既存の監視項目なのか見分けるためのリスト
	 */
	private static ArrayList<String> monitorIdList = new ArrayList<String>();
	
	private void refresh() {
		HashMap <String, ArrayList<MonitorInfo>> newMonitorMap =
			new HashMap<String, ArrayList<MonitorInfo>>();

		/*
		 * logfileMonitorはログファイルごとにオブジェクトが生成される。
		 * logfileMonitor.monitorInfoListに監視設定が登録される。
		 * (logfileMonitorとmonitorInfoは1対多の関係)
		 */
		/*
		 * 1. logfileMonitorを生成する。
		 */
		int fileN = 0;
		m_log.debug("refresh() : m_monitorList.size=" + m_monitorList.size());
		ArrayList<String> newMonitorIdList = new ArrayList<String>();
		for (MonitorInfo monitorInfo : m_monitorList) {
			if (monitorInfo.getMonitorFlg() == ValidConstant.TYPE_INVALID) {
				continue;
			}
			String filePathTmp = monitorInfo.getLogfileCheckInfo().getLogfile();
			
			if(filePathTmp.split(m_separator).length < 2) {
				m_log.info("refresh : filename is not regexp");
				continue;
			}
			
			String directoryStr = filePathTmp.split(m_separator)[0];
			String filenamePattern = filePathTmp.split(m_separator)[1];
			m_log.debug("refresh() directory=" + directoryStr + ", filenamePattern=" + filenamePattern);
			
			Pattern pattern = null;
			// 大文字・小文字を区別しない場合 
			pattern = Pattern.compile(filenamePattern, Pattern.DOTALL | Pattern.CASE_INSENSITIVE);
			// 大文字・小文字を区別する場合 
			// pattern = Pattern.compile(filenamePattern, Pattern.DOTALL);
			
			File directory = new File(directoryStr);
			if (!directory.isDirectory()) {
				m_log.info("refresh() " + directoryStr + " is not directory");
				continue;
			}
			String monitorId = monitorInfo.getMonitorId();
			for (File file : directory.listFiles()) {
				m_log.debug("refresh() file=" + file.getName());
				if (!file.isFile()) {
					m_log.debug(file.getName() + " is not file");
					continue;
				}
				Matcher matcher = pattern.matcher(file.getName());
				if(!matcher.matches()) {
					m_log.debug("refresh() don't match. filename=" + file.getName() + ", pattern=" + filenamePattern);
					continue;
				}
				fileN ++;
				if (fileN > m_fileMax) {
					m_log.warn("refresh() too many files for logfile-monitor. not monitor=" + file.getName());
					continue;
				}
				String filePath = file.getAbsolutePath();
				LogfileMonitor logfileMonitor = m_logfileMonitorCache.get(filePath);
				if(logfileMonitor == null){
					// 初めから読むか、途中から読むか決める。
					boolean flag = monitorIdList.contains(monitorId);
					
					// ファイル監視オブジェクトを生成。 
					logfileMonitor = new LogfileMonitor(this, filePath, m_runInterval, flag);
					m_logfileMonitorCache.put(filePath, logfileMonitor);
				}
				ArrayList<MonitorInfo> list = newMonitorMap.get(filePath);
				if (list == null){
					list = new ArrayList<MonitorInfo> ();
					newMonitorMap.put(filePath, list);
				}
				list.add(monitorInfo);
			}
			if (!newMonitorIdList.contains(monitorId)) {
				newMonitorIdList.add(monitorId);
			}
		}
		monitorIdList = newMonitorIdList;
		
		/*
		 * 2. logfileMonitor.monitorInfoListを登録する。
		 */
		ArrayList<String> noMonitorFileList = new ArrayList<String>();
		for (String filePath : m_logfileMonitorCache.keySet()) {
			LogfileMonitor thread = m_logfileMonitorCache.get(filePath);
			ArrayList<MonitorInfo> list = newMonitorMap.get(filePath);
			thread.setMonitor(list);
			/*
			 * 監視設定が登録されていないファイルは閉じる。
			 */
			if (thread.clean()) {
				noMonitorFileList.add(filePath);
			}
		}
		for (String file : noMonitorFileList) {
			m_logfileMonitorCache.remove(file);
		}
	}

	public void start() {
		LogfileThread thread = new LogfileThread();
		thread.setName("LogFileMonitor");
		thread.start();
	}

	private class LogfileThread extends Thread {
		@Override
		public void run() {
			m_log.info("run LogfileThread");
			while (true) {
				try {
					if (m_regexp) {
						refresh();
					}
					for (String filePath : m_logfileMonitorCache.keySet()) {
						LogfileMonitor logfileMonitor = m_logfileMonitorCache.get(filePath);
						logfileMonitor.run();
					}
				} catch (Exception e) {
					m_log.warn("LogfileThread : " + e.getClass().getCanonicalName() + ", " +
							e.getMessage(), e);
				} catch (Throwable e) {
					m_log.error("LogfileThread : " + e.getClass().getCanonicalName() + ", " +
							e.getMessage(), e);
				}
				try {
					Thread.sleep(m_runInterval);
				} catch (InterruptedException e) {
					m_log.info("LogfileThread is Interrupted");
					break;
				}
			}
		}
	}

	/**
	 * 監視管理のJMSに情報を通知します。<BR>
	 * 
	 * @param priority 重要度
	 * @param app アプリケーション
	 * @param msgId メッセージID
	 * @param msg メッセージ
	 * @param msgOrg オリジナルメッセージ
	 */
	public void sendMessage(String filePath, int priority, String app, String msgId, String msg, String msgOrg, String monitorId) {
		// ログ出力情報
		OutputBasicInfo output = new OutputBasicInfo();
		output.setPluginId(HinemosModuleConstant.MONITOR_LOGFILE);
		output.setPriority(priority);
		output.setApplication(app);
		output.setMessageId(msgId);
		output.setMessage(msg);
		output.setMessageOrg(msgOrg);

		output.setGenerationDate(new Date().getTime());
		output.setMonitorId(monitorId);
		output.setFacilityId(""); // マネージャがセットする。
		output.setScopeText(""); // マネージャがセットする。

		this.m_sendQueue.put(output);
	}

}
