#include "htlib2.h"
#include "Receiver.h"
#include "Header.h"
#include "Socket.h"
#include "HtLib_.h"

static void
prepareBuffer(HTLIB* o)
{
	/* manage buffer for 100 Continue response */

	int len;
	if ((len=o->_read_index - o->_body_index)>0) {
		LOGDBG("memove %d:{%.20s...} to head\n",
			   o->_body_index, o->_rec_buffer);
		memmove(o->_rec_buffer, o->_rec_buffer + o->_body_index, len);
	}
	o->_header = NULL;
	o->_read_index = len;
	o->_body_index = 0;
	return;
}

static int
findBody(char* buffer, int size, BOOL req)
{
	LOGDBG("findBody(%s, %d)\n", buffer, size);
	BOOL lf = FALSE;
	int spcs = 0;
	BOOL first = TRUE;

	int i;
	for (i=0; i<size; i++) {
		switch (buffer[i]) {
		case ' ':
			spcs++;
			continue;
			
		case '\r':
			continue;

		case '\n':
			if (lf ||
				(req && first && spcs==1 /* HTTP0.9 */)) {
				buffer[i] = 0;
				if (i>0 && buffer[i-1]=='\r') {
					buffer[i-1] = 0;
				}
				return i+1;
			}
			lf = TRUE;
			first = FALSE;
			break;

		default:
			lf = FALSE;
			break;
		}
	}
	return 0;
}

static BOOL
manageHeaders(HTLIB* o, const HTLIB_Header* header_buffer, USHORT blen,
			  BOOL no_clen_means_0, BOOL isReq, HTLIB_ERROR* err)
{
	/*
	 * Connection:
	 * manage keeping-alive for the next time
	 */
	if ((o->remote_version==HTTP_VER_11 &&
		 M(Header,HasConnectionClose)(header_buffer, blen))||
		(o->remote_version==HTTP_VER_10 &&
		 M(Header,HasConnectionKeepAlive)(header_buffer, blen)==FALSE)) {
		/* "close" in HTTP1.1, or not "Keep-Alive" in HTTP1.0 */
		LOGINFO("CLR KEEP_ALIVE\n");
		FLAG_CLR(FLAG_KEEP_ALIVE, o->_flags);
	} else {
		/* can't be set TRUE,
		   because it may be already set FALSE when the request. */
	}

	/*
	 * Content-Length:
	 */
	o->_rec_content_transfered = 0;

	int index;
	if ((index=M(Header,Find)(header_buffer, blen,
							  "Transfer-Encoding"))>=0) {
		if (strcasecmp(header_buffer[index].value, "chunked")!=0) {
			/* no other tokens allowed */
			LOGERR("Transfer-Encoding specified, but not chunked\n");
			*err = HTLIB_E_NOT_IMPLEMENTED;
			return FALSE;
		}
		FLAG_SET(FLAG_RECV_CHUNKED, o->_flags);
		LOGINFO("SET RECV_CHUNKED\n");

		/* (RFC2616 4.4.2) */
		
		o->_rec_content_length = 0; /* initialized before chunk */

	} else if ((index=M(Header,Find)(header_buffer, blen,
									 "Content-Length"))!=-1 &&
			   header_buffer[index].value != NULL) {
		if (sscanf(header_buffer[index].value, "%lld",
				   &o->_rec_content_length)==0) {
			*err = HTLIB_E_INVALID_SYNTAX;
			LOGERR("Content-Length cannot be recognized\n");
			return FALSE;
		}
		/* Content-Length was set properly */
		
	} else if (no_clen_means_0) {
		/* GET, HEAD, etc. */
		o->_rec_content_length = 0;
		
	} else if (FLAG_ISSET(FLAG_KEEP_ALIVE, o->_flags)==FALSE && isReq==FALSE) {
		/* content end will be specified by close (RFC2616 4.4.2) */
		o->_rec_content_length = -1;

	} else {
		*err = HTLIB_E_INVALID_SYNTAX;
		LOGERR("Content-Length not found when keep-alive\n");
		return FALSE;

	}

	return TRUE;
}

static int
parseHeader(HTLIB* o,
			HTLIB_Header* header_buffer, USHORT blen,
			HTLIB_NEEDS_FUNC needs,	 
			HTLIB_ERROR* err)
{
	LOGNOTICE("ParseHeader(%d)\n", blen);

	if (o->_header == NULL || o->_body_index==0) {
		*err = HTLIB_E_INVALID_STATE;
		LOGERR("didn't receive all headers yet\n");
		return -1;
	}
	return M(Header,Parse)(o,
						   header_buffer, blen,
						   needs,
						   err);
}

static void
proceedUntilSpace(HTLIB* o)
{
	int index = o->_body_index;
	while (index < o->_read_index && isspace(o->_rec_buffer[index])) {
		index++;
	}
	if (o->_body_index < index) {
		int mlen = index - o->_body_index;
		memmove(o->_rec_buffer + o->_body_index,
				o->_rec_buffer + index,
				o->_read_index - index);
		o->_read_index -= mlen;
	}
}

static BOOL
waitAndRead(HTLIB* o, int timeout_millis, HTLIB_ERROR* err)
{
	if (M(Socket,Wait)(o, timeout_millis, HTLIB_W_READ,
					   M(HTLIB,CheckCanceled), err)==FALSE) {
		if (*err == HTLIB_E_TIMEOUT) {
			LOGINFO("Socket_Wait timeout\n");
		} else {
			LOGERR("Socket_Wait failed\n");
		}
		return FALSE;
	}
	
	int len = o->_rec_buffer_len - o->_read_index;
	if (len == 0) {
		*err = HTLIB_E_OUT_OF_MEMORY;
		LOGERR("buffer full\n");
		return FALSE;
	}
	switch ((len=M(Socket,Read)(o,
								o->_rec_buffer+o->_read_index,
								len,
								err))) {
	case -1:
		LOGERR("Socket_Read failed err=%d errno=%d\n", *err, o->system_errno);
		return FALSE;

	case 0:
		*err = HTLIB_E_DISCONNECTED;
		LOGERR("Socket disconnected\n", *err);
		return FALSE;

	default:
		o->_read_index += len;
		proceedUntilSpace(o);
		break; /* into the while-loop again */
	}
	return TRUE;
}

static BOOL
receiveMessage(HTLIB* o,
			   int timeout_millis,
			   HTLIB_Header* header_buffer, USHORT* blen,
			   HTLIB_NEEDS_FUNC needs,
			   BOOL req,
			   HTLIB_ERROR* err)
{
	if (o->_header) {
		/* already receiveMessage returned non-NULL before */
		prepareBuffer(o);
	}
	while (TRUE) {
		/*
		 * First, check if the head-end can be found.
		 * (In case of 1XX, it may be already received.)
		 */
		
		if ((o->_body_index=findBody(o->_rec_buffer, o->_read_index, req))>0) {

			/* "CR LF CR LF" found */

			/* skip the first line */
			LOGDBG("check 1st line: (len=%d) %s\n",
					o->_read_index,  o->_rec_buffer);
			o->_header = strchr(o->_rec_buffer, '\n');
			if (o->_header == NULL) {
				/* HTTP0.9 */
				LOGWARN("it seemds like HTTP0.9...(%s)\n", o->_rec_buffer);
				*blen = 0;
				return TRUE;
			}
			*o->_header++ = 0;
			char* p;
			if ((p=strchr(o->_rec_buffer, '\r'))==NULL) {
				*p = 0;
			}

			int blen2;
			if ((blen2=parseHeader(o,
								   header_buffer, *blen,
								   needs,
								   err))==-1) {
				LOGERR("parseHeader failed err=%d\n", *err);
				return FALSE;
			}
			*blen = (USHORT)blen2;

			return TRUE;
		}

		if (o->_rec_buffer_len <= o->_read_index) {
			/* memory cannot be assigned any more */
			break;
		}

		/*
		 * Now try, to receive from network
		 */
		if (waitAndRead(o, timeout_millis, err)==FALSE) {
			return FALSE;
		}
	} /* while */

	*err = HTLIB_E_OUT_OF_MEMORY;
	LOGERR("ReceiveMessage: outof memory(%d >= %d)\n",
		   o->_read_index, o->_rec_buffer_len);
	return FALSE;
}

static USHORT
getVersion(const char* p, HTLIB_ERROR* err)
{
	ULONG maj;
	ULONG min;

	if (sscanf(p, "%lu.%lu", &maj, &min)!=2) {
		*err = HTLIB_E_INVALID_SYNTAX;
		LOGERR("unrecoginzed HTTP version ?/?.\n");
		return 0;
	}
	
	USHORT ver = (USHORT)((maj << 8)&0xff00) | (min & 0x00ff);
	LOGNOTICE("remote HTTP-Ver: %hx\n", ver);
	return ver;
}

static BOOL
expectedBody(HTLIB_HANDLE o)
{
	if (FLAG_ISSET(FLAG_RECV_CHUNKED, o->_flags)) {
		return TRUE;
	}
	
	if (o->_rec_content_length<=0) {
		return FALSE;
	}
	if (o->_rec_content_length <= o->_rec_content_transfered) {
		return FALSE;
	}
	return TRUE;
}

static BOOL
readBodyAway(HTLIB* o, int timeout_millis, HTLIB_ERROR* err)
{
	while (expectedBody(o)) {
		char dummy[1000];
		int len;
		if ((len=M(Receiver,ReceiveBody)(o, timeout_millis,
										 dummy, sizeof(dummy), err))==-1) {
			LOGNOTICE("readBodyAway failed err=%d\n", *err);
			return FALSE;
		}
		LOGINFO("dispose %d bytes of body\n", len);
	}
	return TRUE;
}
const char*
M(Receiver,ReceiveRequest)(HTLIB* o,
						   int timeout_millis,
						   const char** method,
						   HTLIB_Header* header_buffer, USHORT* blen,
						   HTLIB_NEEDS_FUNC needs,
						   HTLIB_ERROR* err)
{
	if (o->remote_version == HTTP_VER_09) {
		*err = HTLIB_E_INVALID_STATE;
		LOGERR("HTTP0.9 is not capable of keep-alive\n");
		return NULL;
	}
	if (readBodyAway(o, timeout_millis, err)==FALSE) {
		return NULL;
	}

	if (FLAG_ISSET(FLAG_KEEP_ALIVE, o->_flags)==FALSE) {
		LOGNOTICE("no more KEEP_ALIVE, quit.\n");
		*err = HTLIB_E_DISCONNECTED;
		return NULL;
	}

	if (receiveMessage(o,
					   timeout_millis,
					   header_buffer, blen,
					   needs, TRUE,
					   err)==FALSE) {
		LOGERR("ReceiveMessage failed err=%d\n", *err);
		return NULL;
	}

	/* process request-line */

	LOGINFO("request-line: %s\n", o->_rec_buffer);

	/* method */
	char* saved;
	char* p = strtok_r(o->_rec_buffer, "\r\n\t ", &saved);
	if (p == NULL) {
		*err = HTLIB_E_INVALID_SYNTAX;
		LOGERR("no space after method\n");
		return NULL;
	}
	*method = o->_rec_buffer;

	/* url */
	char* url;
	if ((url=strtok_r(NULL, "\r\n\t ", &saved))==NULL) {
		*err = HTLIB_E_INVALID_SYNTAX;
		LOGERR("no url\n");
		return NULL;
	}

	char* host = NULL;
	if (strncmp(url, "http://", 7)==0) {
		/* absoluteURI */
		host = url+7;

		char* path = strpbrk(host, " /");
		if (path == NULL) {
			int l = strlen(host);
			path = host+l;
		}
		memmove(url, host, path - host);
		url[path-host] = 0;
		host = url;
		if (*path == ' '||*path == 0) {
			path = "/";
		}
		url = path;
	}
	if (*url != '/') {
		*err = HTLIB_E_INVALID_SYNTAX;
		LOGERR("unrecognized URL %s\n", p);
		return NULL;
	}

	/* p == " HTTP/1.?" */
	p = strtok_r(NULL, "\r\n\t ", &saved);
	if (p == NULL) {
		/* HTTP0.9 ? */
		if (strcmp(*method, "GET")!=0) {
			*err = HTLIB_E_INVALID_SYNTAX;
			LOGERR("HTTP0.9 but not GET (%s)\n", method);
			return NULL;
		}
		o->remote_version = HTTP_VER_09;
		return url;
	} else {
		if (strncmp(p, "HTTP/", 5)!=0) {
			*err = HTLIB_E_INVALID_SYNTAX;
			LOGERR("no HTTP/ (%.5s)\n", p);
			return NULL;
		}
		if ((o->remote_version=getVersion(p+5, err))==0) {
			return NULL;
		}
	}

	BOOL no_clen_means_0 = TRUE;
	if (strcmp(*method, "POST")==0 || strcmp(*method, "PUT")==0) {
		no_clen_means_0 = FALSE;
	}
	if (manageHeaders(o, header_buffer, *blen, no_clen_means_0, TRUE,
					  err)==FALSE) {
		LOGERR("manageHeaders failed err=%d\n", *err);
		return FALSE;
	}

	if (host != NULL) {
		int i = M(Header,Find)(header_buffer, *blen, "Host");
		if (i != -1) {
			header_buffer[i].value = host;
		}
	}

	return url;
}

int
M(Receiver,ReceiveResponse)(HTLIB* o,
						 int timeout_millis,
						 HTLIB_Header* header_buffer, USHORT* blen,
						 HTLIB_NEEDS_FUNC needs,
						 HTLIB_ERROR* err)
{
	if (readBodyAway(0, timeout_millis, err)==FALSE) {
		return -1;
	}

	if (receiveMessage(o,
					   timeout_millis,
					   header_buffer, blen,
					   needs, FALSE,
					   err)==FALSE) {
		LOGERR("ReceiveMessage failed err=%d\n", *err);
		return -1;
	}

	int status;

	LOGINFO("response-line: %.20s...\n", o->_rec_buffer);
	/*
	 * check HTTP version
	 */
	char* saved;
	char* p = strtok_r(o->_rec_buffer, "\r\n\t ", &saved);
	if (p == NULL) {
		*err = HTLIB_E_INVALID_SYNTAX;
		LOGERR("no 1st line\n");
		return -1;
	}
	if (strncmp(p, "HTTP/", 5)!=0) {
		*err = HTLIB_E_INVALID_SYNTAX;
		LOGERR("no HTTP/ (%.5s)\n", p);
		return -1;
	}
	if ((o->remote_version=getVersion(p+5, err))==0) {
		return -1;
	}
	if ((p=strtok_r(NULL, "\r\n\t ", &saved))==NULL) {
		*err = HTLIB_E_INVALID_SYNTAX;
		LOGERR("no status\n");
		return -1;
	}
	if (sscanf(p, "%d", &status)!=1) {
		*err = HTLIB_E_INVALID_SYNTAX;
		LOGERR("unrecognized status: %d\n", p);
		return -1;
	}

	if ((int)(status/100)==1) {
		prepareBuffer(o);
		return status;
	}

	BOOL no_clen_means_0 = FALSE;
 	if (status == 204 || status == 304) {
		/* (RFC2616 4.4) */
		no_clen_means_0 = TRUE;
	}
	if (manageHeaders(o, header_buffer, *blen, no_clen_means_0, FALSE,
					  err)==FALSE) {
		LOGERR("manageHeaders failed err=%d\n", *err);
		return FALSE;
	}
	if (FLAG_ISSET(FLAG_HEAD_REQUEST, o->_flags)||
		status == 204 || status == 304) {
		/* (RFC2616 4.4.1) */
		o->_rec_content_length = 0;
	}
	return status;
}

static void
memmoveRest(HTLIB* o)
{
	int restlen = o->_read_index - o->_body_index;
	memmove(o->_rec_buffer,
			o->_rec_buffer + o->_body_index,
			restlen);
	o->_read_index = restlen;
	o->_body_index = 0;
}
static void
manageBuffer(HTLIB* o)
{
	if (o->_rec_content_length>=0 &&
		o->_rec_content_length <= o->_rec_content_transfered) {

		LOGINFO("transfering %lld bytes completed.\n",
				o->_rec_content_transfered);

		/* for the next request */
		memmoveRest(o);
		o->_header = NULL;
	}
}

int
M(Receiver,ReceiveBody)(HTLIB* o,
				  int timeout_millis,
				  char* buffer,
				  USHORT buffer_len,
				  HTLIB_ERROR* err)
{
	/* ignore when HTTP0.9 */
	if (o->remote_version == HTTP_VER_09 && expectedBody(o)==FALSE) {
		/* no more body */
		manageBuffer(o);
		return 0;
	}

	if (M(HTLIB,CheckCanceled)(o, err)==FALSE) {
		return -1;
	}

	/*
	 * chunked consideration
	 */
	while (FLAG_ISSET(FLAG_RECV_CHUNKED, o->_flags)&&
		   o->_rec_content_transfered == o->_rec_content_length) {

		int rest = o->_read_index - o->_body_index;
		if (0 < rest) {
			int i;
			for (i=2; i<rest; i++) {
				if (o->_rec_buffer[o->_body_index + i]=='\n') {
					int clen = 0;
					o->_rec_buffer[o->_body_index + i] = 0;
					sscanf(o->_rec_buffer + o->_body_index, "%x", &clen);
					o->_rec_content_length += clen;
					o->_body_index += i + 1;
					break;
				}
			}
			if (i<rest) {
				/* chunk size was found */
				/* last CRLF will be ignored in the next receive */
				break;
			}
		}
		/* chunk size not found */
		if (waitAndRead(o, timeout_millis, err)==FALSE) {
			return FALSE;
		}
	}
	
	/*
	 * Connection-Length consideration
	 */
	long long body_rest;
	if (o->_rec_content_length !=-1 &&
		(body_rest = o->_rec_content_length
		 - o->_rec_content_transfered) < buffer_len) {
		LOGINFO("buffer_len shortened (%d => %d)\n",
				buffer_len, body_rest);
		buffer_len = body_rest;
		if (buffer_len == 0) {
			/* reached the end */
			FLAG_CLR(FLAG_RECV_CHUNKED, o->_flags);
			LOGINFO("CLR RECV_CHUNKED\n");
			manageBuffer(o);
			return 0;
		}
	}
	/*
	 * First, check if some bytes already read.
	 */
	if (o->_body_index < o->_read_index) {
		
		int len = o->_read_index - o->_body_index;
		if (buffer_len < len) {
			len = buffer_len;
		}
		memcpy(buffer, o->_rec_buffer + o->_body_index, len);
		o->_body_index += len;
		o->_rec_content_transfered += len;
		manageBuffer(o);
		return len;
	}

	/*
	 * Now try to read from the socket.
	 */
	{
		int len;

		if (M(Socket,Wait)(o, timeout_millis, HTLIB_W_READ,
						   M(HTLIB,CheckCanceled),
						   err)==FALSE) {
			return -1;
		}
		
		len = M(Socket,Read)(o, buffer, buffer_len, err);
		if (len>0) {
			o->_rec_content_transfered += len;
		}
		manageBuffer(o);
		return len;
	}
}


